/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/*
 *  $Revision: 1.4 $
 *
 *  author: Jeff Moyer <moyer@mclinux.com>
 */

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <sys/types.h>
#include <sys/errno.h>

#include "fdlist.h"

#ifdef DEBUG
#define Dprintf(fmt,args...)  printf(fmt,##args)
#else
#define Dprintf(fmt,args...)
#endif

/*
 * Local prototypes
 */
struct fd_list *fdlist_lookup(int fd);


/*
 * Global list variable
 */
struct fd_list_head fdlist;


void
fdlist_init(void)
{
    fdlist.head = fdlist.tail = NULL;
}


int
fdlist_setstate(int fd, int state)
{
    struct fd_list *fdnode;

    fdnode = fdlist_lookup(fd);
    if (fdnode == NULL)
	return -1;

    fdnode->state |= state;
    return 0;
}


int
fdlist_getstate(int fd)
{
    struct fd_list *fdnode;

    fdnode = fdlist_lookup(fd);
    if (fdnode == NULL)
	return -1;

    return fdnode->state;
}


int
fdlist_add(int fd, int state, struct sockaddr_in *local)
{
    struct fd_list *node;

    node = (struct fd_list *)malloc(sizeof(struct fd_list));
    if (node == NULL) {
	Dprintf("Unable to allocate memory.  Error: %s\n", strerror(errno));
	return -1;
    }

    node->fd = fd;
    node->state = state;
    node->local_addr = local;
    node->next = NULL;

    if (fdlist.tail) {
	fdlist.tail->next = node;
    }
    fdlist.tail = node;

    if (fdlist.head == NULL)
	fdlist.head = node;

    return 0;
}

/* if local is 1, we return local addr, remote for local==0 */
struct sockaddr_in *
fdlist_getaddr(int fd)
{
    struct fd_list *fdnode;

    fdnode = fdlist_lookup(fd);
    if (fdnode == NULL)
	return NULL;

    return fdnode->local_addr;
}


int
fdlist_delete(int fd)
{
    struct fd_list *curr, *prev;

    if (!fdlist.head) {
	return -1;
    }

    if (fdlist.head->fd == fd) {
	curr = fdlist.head;
	fdlist.head = fdlist.head->next;
	if (fdlist.tail == curr) {
	    fdlist.tail = fdlist.head; /* NULL */
	}
	free(curr);
	return 0;
    }

    prev = fdlist.head;
    for (curr = fdlist.head->next; curr != NULL; curr = curr->next) {
	if (curr->fd == fd) {
	    prev->next = curr->next;
	    if (curr == fdlist.tail)
		fdlist.tail = prev;
	    free(curr);
	    return 0;
	}
	prev = curr;
    }
    /* Entry not found */
    return -1;
}

struct fd_list *
fdlist_lookup(int fd)
{
    struct fd_list *curr;

    for (curr = fdlist.head; curr != NULL; curr = curr->next) {
	if (curr->fd == fd) {
	    return curr;
	}
    }
    return NULL;
}
/*
 * Local variables:
 *  c-basic-offset: 4
 *  c-indent-level: 4
 *  tab-width: 8
 * End:
 */

