/***************************************************************************
 *   Copyright (C) 2004 by Emil Stoyanov                                   *
 *   emosto@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

class QAction;
class KiaxMainWindow;
class QApplication;

#include "kiaxmainwindow.h"
#include <qevent.h>

KiaxMainWindow::KiaxMainWindow(QApplication* app, QWidget* parent, const char* name, WFlags fl, IaxWrapper* wrp)
: KiaxMainWindowUi(parent,name,fl) {

    application = app;

    wrapper = wrp;
    appTranslator = NULL;
    i18n();

    logWindow = new KiaxLogWindow(this, "Log Window", false, 0);

    connect(logWindow , SIGNAL( showStatistics(bool) ), this, SLOT( showStatistics(bool)) );

    /* Handling IaxClientWrapper Events */

    connect(wrapper , SIGNAL( signalCalling(int, QString) ), this, SLOT( handleIaxWrapperEventCalling(int, QString) ) );

    connect(wrapper , SIGNAL( signalAnsweredCall(int) ), this, SLOT( handleIaxWrapperEventAnsweredCall(int) ) );

    connect(wrapper , SIGNAL( signalHoldCall(int) ), this, SLOT( handleIaxWrapperEventHoldCall(int) ) );

    connect(wrapper , SIGNAL( signalResumedCall(int) ), this, SLOT( handleIaxWrapperEventResumedCall(int) ) );

    connect(wrapper , SIGNAL( signalOutgoingRinging(int) ), this, SLOT( handleIaxWrapperEventOutgoingRinging(int) ) );

    connect(wrapper , SIGNAL( signalIncomingRinging(int) ), this, SLOT( handleIaxWrapperEventIncomingRinging(int) ) );

    connect(wrapper , SIGNAL( signalComplete(int) ), this, SLOT( handleIaxWrapperEventComplete(int) ) );

    connect(wrapper , SIGNAL( signalInactive(int) ), this, SLOT( handleIaxWrapperEventInactive(int) ) );

    connect(wrapper , SIGNAL( signalRegistered() ), this, SLOT( handleIaxWrapperEventRegistered() ) );

    connect(wrapper , SIGNAL( signalUnregistered() ), this, SLOT( handleIaxWrapperEventUnregistered() ) );

    connect(wrapper , SIGNAL( signalLevels(int, int, int)), this, SLOT( handleIaxWrapperEventLevels(int,int, int) ) );

    connect(wrapper , SIGNAL( signalRegistrationRequested(KiaxAccount*) ), this, SLOT( handleIaxWrapperEventRegistrationRequested(KiaxAccount*) ) );

    connect(wrapper , SIGNAL( fatalError(QString) ), this, SLOT( fatalError(QString) ));

    connect(wrapper , SIGNAL( signalDebug(const char*) ), this, SLOT( slotLogWindow(const char*) ));

    connect(wrapper , SIGNAL(callTime(int, int, int, int ) ), this, SLOT(updateCallDuration(int, int, int, int )) );

    initOK = wrapper->init_iax_client();

    statusIndicator = new QLabel((QWidget*)statusBar(),0,0);
    // add indication to the form statusbar

    indicatorsFrame = new QWidget(this);
    registrationIndicatorsContainer = new QHBoxLayout(indicatorsFrame);
    registrationIndicatorsContainer->setAlignment( AlignVCenter | AlignRight );
    statusBar()->addWidget(statusIndicator,0,false);
    statusBar()->addWidget(indicatorsFrame,1,false);
    registrationIndicatorsContainer->activate();
    // the context menu, right click on contact in the contact list
    contactListMenu = new QPopupMenu(this);
    callDialAction->addTo(contactListMenu);
    contactListMenu->insertSeparator();
    contactListChangeAction->addTo(contactListMenu);
    contactListDeleteAction->addTo(contactListMenu);

    // the context menu, right click on contact in the contact list
    callRegisterMenu = new QPopupMenu(this);
    callDialAction->addTo(callRegisterMenu);
    callRegisterMenu->insertSeparator();
    callRegisterAddUserAction->addTo(callRegisterMenu);

    connect (contactListChangeAction, SIGNAL(activated()), this, SLOT(contactListChangeAction_activated()));
    connect (contactListDeleteAction, SIGNAL(activated()), this, SLOT(contactListDeleteAction_activated()));

    connect (callRegisterAddUserAction, SIGNAL(activated()), this, SLOT(callRegisterAddUserAction_activated()));

    // dial contact when press enter or doubleclick on contact item
    connect (contactsListBox, SIGNAL(selectionChanged(QListViewItem*)), this, SLOT(contactsListBox_highlighted(QListViewItem* )));
    connect (contactsListBox, SIGNAL(clicked(QListViewItem*)), this, SLOT(contactsListBox_highlighted(QListViewItem* )));
    connect (contactsListBox, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(callDialAction_activated()));
    connect (contactsListBox, SIGNAL(contextMenuRequested (QListViewItem *, const QPoint &, int)), this, SLOT(contactsListBox_contextMenuRequested(QListViewItem*, const QPoint&, int)));
    // dial number from record when press enter or doubleclick on call record item
    connect (callRegisterListBox, SIGNAL(selected(int)), this, SLOT(callDialAction_activated()));

    connect(toolsLogWindowAction , SIGNAL(activated() ), logWindow, SLOT(show()) );

    // load contact list
    addressBook = new KiaxAddressBook();
    addressBook->load();
    refreshContactsListBox();
    // load call register list
    callRegister = new KiaxCallRegister();
    callRegister->load();
    refreshCallRegisterListBox();

    // keep pointers to pages to hide and show them

    contactsPage = tabs->page(0);
    dialpadPage = tabs->page(1);
    callRegisterPage = tabs->page(2);
    tabs->removePage(callRegisterPage);
    tabs->removePage(dialpadPage);
    currentPage = contactsPage;

    setCaption( tr("Kiax") );

    systemTray = new KiaxSystemTray (this, tr("Kiax"));

    connect(systemTray , SIGNAL( showSettingsWindow() ), this, SLOT( fileSettingsAction_activated()) );
    connect(systemTray , SIGNAL( quitApplication() ), this, SLOT( fileExitAction_activated()));

    KiaxPrefs * settings = wrapper->getSettings();

    if (settings->isDocked()) {
        systemTray->dock();
        docked = true;
    } else
        docked = false;

    transferButton->hide();

    viewToolbarAction->setOn(settings->getShowToolbar());

    viewDialHangup_ButtonsAction->setOn(settings->getShowDialHangupButtons());

    viewAccountDial_CombosAction->setOn(settings->getShowAccountNumberCombos());

    viewCall_RegisterAction->setOn(settings->getShowCallRegister());

    viewDialPadAction->setOn(settings->getShowDialPad());

    registeredAccounts = 0;

    tabs->showPage(contactsPage);


    //statusIndicator->setText(tr("Disconnected"));

    setGeometry(wrapper->getSettings()->getWidgetGeometry());
    move(wrapper->getSettings()->getWindowPosition());

    // show avaliable accounts in the combobox
    updateAccountsComboBox(accountsComboBox);

    contactsListBox->setFocusPolicy(NoFocus);
    callRegisterListBox->setFocusPolicy(NoFocus);

    //numberComboBox->installEventFilter(this);


    dialPixmap = new QPixmap( getIconsPath()  + "dial.png" );
    hangupPixmap = new QPixmap( getIconsPath()  + "hangup.png" );
    holdPixmap = new QPixmap( getIconsPath() + "hold.png" );
    resumePixmap = new QPixmap( getIconsPath() + "resume.png");
    DialButton->setPixmap(*dialPixmap);
    HangupButton->setPixmap(*hangupPixmap);


    int argc = app->argc();
    char** args = app->argv();

    if (argc>1) {
        char* accountId=NULL;
        char* exten=NULL;

        char* command = args[1]; /* command*/
        if (strcmp(command,"list")==0) {
            for (uint i=0;i<wrapper->getSettings()->getAccounts().count();i++) {
                KiaxAccount* account=wrapper->getSettings()->getAccount(i);
                fprintf(stderr, "account id=%s, alias=%s, server=%s, username=%s, codec=%s\n",(const char*)account->accId, (const char*)account->accAlias, (const char*)account->iaxServer, (const char*)account->username, (const char*)account->codec);
            }
            initOK=false; /* terminate program */
        }
        if (strcmp(command,"dial")==0) {
            for (int i=2;i<argc;i++) {
                if (strcmp(args[i],"--account-id")==0) {
                    accountId=args[i+1];
                    i++;
                    KiaxAccount* account=wrapper->getSettings()->findAccountById(accountId);
                    fprintf(stderr,"account id=%s, alias=%s, server=%s, username=%s, codec=%s",accountId, (const char*)account->accAlias, (const char*)account->iaxServer, (const char*)account->username, (const char*)account->codec);
                }
                if (strcmp(args[i],"--extension")==0) {
                    exten=args[i+1];
                    i++;
                    fprintf(stderr,"extension %s", exten);
                }
            }
            if (exten) {
                show();
                fileConnectAction_activated(); //
                if (!accountId)
                    accountId = (char*)(const char*)   wrapper->getSettings()->getDefaultAccountId();
                wrapper->dial(accountId,exten,exten);
            } else {
                fprintf(stderr, "usage: kiax [list] | [dial [--account-id id] --extension exten ");
                initOK=false; /* terminate program */
            }
        }
    } else {
        if (!wrapper->getSettings()->isHiddenOnStartup()) {
            show();
	}
        if (wrapper->getSettings()->isConnectedOnStartup()) {
            fileConnectAction_activated();
	}
        else {
            fileDisconnectAction_activated();
	}
    }

    accountWizard = new KiaxAccountWizard(this, "Account Wizard", true, 0);
    connect(accountWizard, SIGNAL(accountCreated()), this, SLOT(updateAccounts()));

    KiaxKeyFilter* keyFilter = new KiaxKeyFilter(this);
    numberComboBox->installEventFilter(keyFilter);


}


KiaxMainWindow::~KiaxMainWindow() {}

void KiaxMainWindow::i18n() {
    if (QString(QTextCodec::locale())!=wrapper->getSettings()->getPreferredLanguage())
        i18n(wrapper->getSettings()->getPreferredLanguage());
}

void KiaxMainWindow::i18n(QString localeStr) {
    if (appTranslator) {
        application->removeTranslator(appTranslator);
        delete appTranslator;
        appTranslator = NULL;
    }

    appTranslator = new QTranslator(0);
    appTranslator->load( QString( "kiax_" ) + localeStr, QString( I18N_DIRECTORY ) );
    application->installTranslator(appTranslator);
}

void KiaxMainWindow::fileNewAccountAction_activated() {
    accountWizard->init();
    accountWizard->show();
}

void KiaxMainWindow::toolsLanguageBulgarianAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("bg");
    i18n("bg");
}

void KiaxMainWindow::toolsLanguageEnglishAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("en");
    i18n("en");
}

void KiaxMainWindow::toolsLanguageDeutschAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("de");
    i18n("de");
}

void KiaxMainWindow::toolsLanguageFrenchAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("fr");
    i18n("fr");
}

void KiaxMainWindow::toolsLanguageCzechAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("cs");
    i18n("cs");
}

void KiaxMainWindow::toolsLanguageHungarianAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("hu");
    i18n("hu");
}

void KiaxMainWindow::toolsLanguageItalianoAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("it");
    i18n("it");
}

void KiaxMainWindow::toolsLanguagePortugueseAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("pt");
    i18n("pt");
}

void KiaxMainWindow::toolsLanguageMacedonianAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("mk");
    i18n("mk");
}

void KiaxMainWindow::toolsLanguageMalayAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("bm");
    i18n("bm");
}

void KiaxMainWindow::toolsLanguagePolishAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("pl");
    i18n("pl");
}

void KiaxMainWindow::toolsLanguageSpanishAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("es");
    i18n("es");
}

void KiaxMainWindow::toolsLanguageHebrewAction_activated() {
    wrapper->getSettings()->setPreferredLanguage("he");
    i18n("he");
}
 
void KiaxMainWindow::callRegisterCleanButton_pressed() {
    callRegister->removeAll();
    refreshCallRegisterListBox();
}


void KiaxMainWindow::answerButtonPressed() {
    wrapper->answer_call(getCallNumberFromActiveTab());
    wrapper->select_call(getCallNumberFromActiveTab());
}

void KiaxMainWindow::hangupButtonPressed() {
    if (wrapper->callSession[getCallNumberFromActiveTab()].isOutgoing()) { //if we are in a call then action is hangup
        wrapper->interrupt_call();
    } else { // if we are receiving call then we are rejecting
        wrapper->reject_call(getCallNumberFromActiveTab());
    }
}

void KiaxMainWindow::callDialAction_activated() {
    if (callDialAction->isEnabled()) {
	KiaxContactListItem * currentItem = (KiaxContactListItem*)contactsListBox->currentItem();
	int ci = -1;
	if (currentItem!=NULL)
        	ci = currentItem->getIndex();
        if ((ci>-1)&&(tabs->currentPage()==contactsPage)&&(contactsListBox->isSelected(contactsListBox->currentItem()))&&(currentItem->isContact())) {
            KiaxContact * contact = addressBook->getContact(ci);
            QString dest = contact->getNumber();
            QString accId = contact->getAccountId();
            wrapper->dial(contact->getAccountId(),contact->getName(), contact->getNumber());
        } else {
            int ci = callRegisterListBox->currentItem();
            if ((ci>-1)&&(tabs->currentPage()==callRegisterPage)&&(callRegisterListBox->isSelected(ci))) {
                KiaxCallRecord * record = callRegister->getRecord(callRegisterListBox->currentItem());
                QString dest = record->getCallerIdNumber();
                QString accountId = record->getAccountId();
                wrapper->dial(record->getAccountId(),record->getCallerIdName(),record->getCallerIdNumber());
            } else if (tabs->currentPage()==dialpadPage) {
                QString dest = numberComboBox->currentText();
       	        wrapper->dial(wrapper->getSettings()->getAccount(accountsComboBox->currentItem())->accId,dest, dest);
	    }
        }
        numberComboBox->clearEdit();
    }
}

/*
   Used to dial/answer/hold/unhold a call
   when wrapper->incomming is true it acts as button Answer
   when a calltab is selected it acts as hold/unhold
*/

void KiaxMainWindow::dialButton_pressed() {
    if ((int)wrapper->callSession.count()<wrapper->getSettings()->getMaxCalls()&&((tabs->currentPage()==contactsPage)||(tabs->currentPage()==dialpadPage)||(tabs->currentPage()==callRegisterPage)))
        callDialAction_activated();
    else {// we are on one of the calltabs
        KiaxCallTab* callTab = (KiaxCallTab*) tabs->currentPage();
        int callNumber = callTab->getCallNumber();
        if
        ((!wrapper->callSession[callNumber].isOutgoing())&&(!wrapper->callSession[callNumber].isAnswered())) { // else act as answerButton. misleading method names, has to be fixed
            answerButtonPressed();
        } else {
            QString callHolded = wrapper->callSession[callNumber].getCustomProperty("isHold");
            if (callHolded=="false") {
                callHoldAction_activated();
            } else {
                callResumeAction_activated();
            }
        }
    }
}

void KiaxMainWindow::transferButton_pressed() {
    QString exten = numberComboBox->currentText();
    wrapper->transfer_call(getCallNumberFromActiveTab(),exten);
}

void KiaxMainWindow::numberComboBox_activated(const QString& dest) {
    /*number combobox participates in dialing and in transferring*/
    if ((tabs->currentPage()==contactsPage)||(tabs->currentPage()==dialpadPage||(tabs->currentPage()==callRegisterPage))) {
       //callDialAction_activated();

    	if (callDialAction->isEnabled()) {
               QString dest = numberComboBox->currentText();
       	        wrapper->dial(wrapper->getSettings()->getAccount(accountsComboBox->currentItem())->accId,dest, dest);
    	}

     } else {
	transferButton_pressed();
    }
     numberComboBox->clearEdit();

}

void KiaxMainWindow::digitPressed(char digit) {
    if (!wrapper->callSession.count()>0) /* if we dont have any call then print the chars in the combobox */
    {
        QString lineStr = numberComboBox->currentText();
        lineStr.append(digit);
        numberComboBox->setCurrentText(lineStr);

    }  /* else we send dtmf with the current call */
    else
        wrapper->send_dtmf(QChar(digit));
}

void KiaxMainWindow::digit9Pressed() {
    digitPressed('9');
}

void KiaxMainWindow::digit8Pressed() {
    digitPressed('8');
}

void KiaxMainWindow::digit7Pressed() {
    digitPressed('7');
}

void KiaxMainWindow::digit6Pressed() {
    digitPressed('6');
}

void KiaxMainWindow::digit5Pressed() {
    digitPressed('5');
}

void KiaxMainWindow::digit4Pressed() {
    digitPressed('4');
}

void KiaxMainWindow::digit3Pressed() {
    digitPressed('3');
}

void KiaxMainWindow::digit2Pressed() {
    digitPressed('2');
}

void KiaxMainWindow::digit1Pressed() {
    digitPressed('1');
}

void KiaxMainWindow::digit0Pressed() {
    digitPressed('0');
}

void KiaxMainWindow::digitStarPressed() {
    digitPressed('*');
}

void KiaxMainWindow::digitPoundPressed() {
    digitPressed('#');
}

void KiaxMainWindow::registerAccount(KiaxAccount* account) {
    wrapper->registerAccount(account);
}

void KiaxMainWindow::unregisterAccount(KiaxAccount* account) {
    wrapper->unregisterAccount(account);
}

void KiaxMainWindow::fileSettingsAction_activated() {

    KiaxPrefsWindow* settingsWindow = new KiaxPrefsWindow(wrapper -> getSettings(), this, tr("Settings"), true, 0);
    int result =settingsWindow->exec();
    if (result==QDialog::Accepted) {
        updateAccountsComboBox(accountsComboBox);
        if (wrapper->getSettings()->isDocked()) {
            if (!docked) {
                systemTray->dock();
                docked = true;
            }
        } else if (docked) {
            systemTray->undock();
            docked = false;
        }
    }
    delete settingsWindow;
}

void  KiaxMainWindow::fileExitAction_activated() {
    // make it clean
    if (wrapper->registered)
	fileDisconnectAction_activated(); 
	
    wrapper->shutdown_iax_client();
    
    wrapper->getSettings()->setShowDialPad(viewDialPadAction->isOn());
    wrapper->getSettings()->setShowToolbar(viewToolbarAction->isOn());
    wrapper->getSettings()->setShowCallRegister(viewCall_RegisterAction->isOn());
    wrapper->getSettings()->setWindowPosition(frameGeometry().topLeft());
    wrapper->getSettings()->setWidgetGeometry(geometry());

    wrapper->getSettings()->saveSessionSettings();

    QApplication::exit();
}

void KiaxMainWindow::helpAboutAction_activated() {
    KiaxAboutBox * aboutBox = new KiaxAboutBox(this,tr("About"), true, 0, KIAX_VERSION, wrapper->getIaxClientVersion());
    aboutBox->exec();
    delete aboutBox;
}

void KiaxMainWindow::viewDialPadAction_toggled(bool flag) {
    flag ? tabs->insertTab(dialpadPage,tr("Dialpad"), 1):(tabs->removePage(dialpadPage));
    tabs->setCurrentPage(1);
}

void KiaxMainWindow::viewToolbarAction_toggled(bool flag) {
    flag ? toolBars(DockTop).at(0)->show():toolBars(DockTop).at(0)->hide();
    tabs->setCurrentPage(1);
}

void KiaxMainWindow::viewCall_RegisterAction_toggled(bool flag) {
    flag ? tabs->insertTab(callRegisterPage,tr("Call Register"),2):tabs->removePage(callRegisterPage);
    tabs->setCurrentPage(2);
}

void KiaxMainWindow::fileConnectAction_activated() {

    statusIndicator->setText(tr("Connecting.."));
    /* we have to check how many registrations we have to do */
    int toBeRegistered = 0;
    int accountsNumber = wrapper->getSettings()->getAccounts().count();
    for (int i=0;i<accountsNumber;i++)
        if ((wrapper->getSettings()->getAccount(i)->registerAccount)&&(wrapper->getSettings()->getAccount(i)->iaxServer!=""))
            toBeRegistered++;

    /* and if we have no accounts to register,
        tell it to the people and do nothing */
    if (toBeRegistered>0) {
        wrapper->registerWithServer();
   }
    else if (accountsNumber>0) { /* do we have at all some accounts?*/
        statusIndicator->setText(tr("OK"));
        /* anyway allow calling */
        setWidgetsRegistered();
        /* start the iaxclient thread */
        wrapper->start_iax_client();
        /* this has to be removed, but needed for now.. */
        wrapper->registered = true;
    } else {
        fileSettingsAction_activated(); /*pop up settings dialog to add accounts*/
        if (wrapper->getSettings()->getAccounts().count()==0) {
            handleIaxWrapperEventUnregistered();
            statusIndicator->setText(tr("No existing accounts."));
        } else {
            handleIaxWrapperEventUnregistered();
            statusIndicator->setText(tr("Press connect button to start "));
        }
        /* just think unregistered*/
    }
}

void KiaxMainWindow::fileDisconnectAction_activated() {
    RegistrationIndicatorsMap::Iterator it;

    wrapper->unregisterFromServer();

    for ( it = registrationIndicators.begin(); it != registrationIndicators.end(); ++it ) {
        KiaxRegistrationIndicator* indicator = it.data();
//        statusBar()->removeWidget(indicator);
        indicator->disconnect(SIGNAL(signalRegisterAccountAction(KiaxAccount*)));
        indicator->disconnect(SIGNAL(signalUnregisterAccountAction(KiaxAccount*)));
        delete indicator;
    }
    registrationIndicators.clear();

}

/* Wrapper signals handling */

void KiaxMainWindow::handleIaxWrapperEventCalling(int callNumber, QString callerStatus) {
    addTab(callNumber, callerStatus);
    if (!dialHangupButtonsFrame->isVisible())
        dialHangupButtonsFrame->show();
}

void KiaxMainWindow::handleIaxWrapperEventAnsweredCall(int callNo) {
    if (callTabs[callNo]) {
        callTabs[callNo]->setCallerStatus(tr("Answering.."));
        setControls();
    }
}

void KiaxMainWindow::handleIaxWrapperEventHoldCall(int callNo) {
    callHoldAction->setEnabled(false);
    callResumeAction->setEnabled(true);
    DialButton->setPixmap(*resumePixmap);
    callTabs[callNo]->setCallerStatus(tr("Call Hold"));
}

void KiaxMainWindow::handleIaxWrapperEventResumedCall(int callNo) {
    callHoldAction->setEnabled(true);
    callResumeAction->setEnabled(false);
    DialButton->setPixmap(*holdPixmap);
    callTabs[callNo]->setCallerStatus(tr("Call Resumed"));
}

void KiaxMainWindow::handleIaxWrapperEventRegistered() {
    registeredAccounts=0;
    setWidgetsRegistered();
}

void KiaxMainWindow::handleIaxWrapperEventUnregistered() {
    setWidgetsUnregistered();
}

void KiaxMainWindow::handleIaxWrapperEventLevels(int inValue, int outValue, int threshold) {
    /*
       notify all tabs (calls respectively) for 
       the level of input and output signals
     */
    QMutexLocker locker(&callTabsMutex);
    int callTabsNumber = callTabs.count();
    for (int i=0; i<callTabsNumber;i++) {
        KiaxCallTab * tab = callTabs[i];
        if (tab) {
            tab->setMicLevel(inValue,threshold);
            tab->setAudioLevel(outValue, 0);
        }
    }
}

void KiaxMainWindow::handleIaxWrapperEventOutgoingRinging(int callNo) {
    //KiaxCallRecord * callRecord = wrapper->callSession.at(callNo);
    if (callTabs[callNo])
        callTabs[callNo]->setCallerStatus(tr("Ringing out.."));
     if (!wrapper->getSettings()->getOutgoingRingMute())
       wrapper->startRing();
}

void KiaxMainWindow::handleIaxWrapperEventRegistrationInactive(KiaxAccount* account) {
    KiaxRegistrationIndicator* registrationIndicator = NULL;
    fprintf(stderr, "Deactivated account  %s\n", (const char*)account->accAlias);
    if (account->iaxRegistrationId!=-1) {
        fprintf(stderr, "Account has idt  %d\n", (const char*) account->accId);
        registrationIndicator = registrationIndicators[account->accId];
    }
    if (registrationIndicator!=NULL) {  //should never be NULL
        registrationIndicator->registrationInactive();
    }
    statusIndicator->setText(tr("Inactive") + " " + account->accAlias);
}

void KiaxMainWindow::handleIaxWrapperEventRegistrationRequested(KiaxAccount* account) {
    KiaxRegistrationIndicator* registrationIndicator = NULL;
    if (account->iaxRegistrationId!=-1)
        registrationIndicator = registrationIndicators[account->accId];
    if (registrationIndicator==NULL) {
        KiaxRegistrationIndicator* registrationIndicator = new KiaxRegistrationIndicator(indicatorsFrame,account->accAlias,account);
        registrationIndicators.insert(account->accId, registrationIndicator);
        connect(account, SIGNAL(signalRegistrationConnecting(KiaxAccount*)), this,  SLOT(handleIaxWrapperEventRegistrationConnecting(KiaxAccount*)));
        connect(account, SIGNAL(signalRegistrationAccepted(KiaxAccount*)), this,  SLOT(handleIaxWrapperEventRegistrationAccepted(KiaxAccount*)));
        connect(account, SIGNAL(signalRegistrationRejected(KiaxAccount*)), this,  SLOT(handleIaxWrapperEventRegistrationRejected(KiaxAccount*)));
        connect(account, SIGNAL(signalRegistrationInactive(KiaxAccount*)), this,  SLOT(handleIaxWrapperEventRegistrationInactive(KiaxAccount*)));
        connect(registrationIndicator, SIGNAL(signalRegisterAccountAction(KiaxAccount*)), this,  SLOT(registerAccount(KiaxAccount*)));
        connect(registrationIndicator, SIGNAL(signalUnregisterAccountAction(KiaxAccount*)), this,  SLOT(unregisterAccount(KiaxAccount*)));
        // add indication to the form statusbar
	QPixmap groupPm ( getIconsPath() + "account.png" );
	registrationIndicator->setPixmap(groupPm);
        registrationIndicatorsContainer->addWidget(registrationIndicator, 0, AlignRight);
	registrationIndicator->show();
        registrationIndicator->registrationConnecting();
    } else {
        registrationIndicator->registrationConnecting();
    }
    statusIndicator->setText(tr("Connecting") + " " + account->accAlias);
    fprintf(stderr, "Registration requested. Got ID = %d\n", account->iaxRegistrationId);

}

void KiaxMainWindow::handleIaxWrapperEventRegistrationConnecting(KiaxAccount* account) {
    statusIndicator->setText(tr("Connecting") + " " + account->accAlias);
}

void KiaxMainWindow::handleIaxWrapperEventRegistrationAccepted(KiaxAccount* account) {
    registeredAccounts++;
    setWidgetsRegistered();
    statusIndicator->setText(tr("Registered") + " " + account->accAlias);
}

void KiaxMainWindow::handleIaxWrapperEventRegistrationRejected(KiaxAccount* account) {
    statusIndicator->setText(tr("Rejected") + " " + account->accAlias);
}

void KiaxMainWindow::handleIaxWrapperEventRegistrationTimeout(KiaxAccount* account) {
    statusIndicator->setText(tr("Timeout") + " " + account->accAlias);
}

void KiaxMainWindow::handleIaxWrapperEventIncomingRinging(int callNo) {
    //  setWidgetsIncoming();
    KiaxCallRecord& record = wrapper->callSession[callNo];

    /*
       we try to guess who is calling from the CallID Number because
       there is no reference in the iaxc_event to the actual used registration 
       to connect to the server which sends us the event
     */
    KiaxContact * possibleContact = guessContact(record.getCallerIdNumber());

    if (possibleContact) {
        record.setAccountId(possibleContact->getAccountId());
        record.setCallerIdName(possibleContact->getName());
        record.setCallerIdNumber(possibleContact->getNumber());
    }

    if (!dialHangupButtonsFrame->isVisible())
        dialHangupButtonsFrame->show();
    if (!callTabs[callNo])// avoid duplication of tabs
        addTab(callNo, tr("Ringing in.."));

   if (!wrapper->getSettings()->getIncomingRingMute())
       wrapper->startRing();

    DialButton->setEnabled(true);
    callDialAction->setEnabled(true);

    // show the main window (if configured so)
    if (wrapper->getSettings()->getRaiseWindow()) {
        if (!this->isVisible())
            this->show();
        this->raise();
    }
    setControls();
    if (wrapper->getSettings()->getExecCommandOnRing()) {
	const char* command = wrapper->getSettings()->getCommandOnRing();
    	wrapper->debug("Starting %s\n", command);
	int result = system(command);
	wrapper->debug("Got result %d", result);
    }
}

void KiaxMainWindow::handleIaxWrapperEventComplete(int callNo) {
    if (callTabs[callNo])
        callTabs[callNo]->setCallerStatus(tr("Call in progress.."));
    wrapper->stopRing();
    setControls();
}

void KiaxMainWindow::handleIaxWrapperEventInactive(int callNo) {
    QMutexLocker locker(&(wrapper->callSessionMutex));
    wrapper->stopRing();
    int position = callRegister->addRecord(wrapper->callSession[callNo].copy());
    addRecordToCallRegisterListBox(&(wrapper->callSession[callNo]), position);
    callRegisterListBox->setCurrentItem(position);
    callRegisterListBox->setSelected(position, true);
    setControls();
    if (!wrapper->getSettings()->getShowDialHangupButtons())
        dialHangupButtonsFrame->hide();
    wrapper->callSession.erase(callNo);
    if (wrapper->callSession.count()>0) {
        KiaxCallTab* tab = callTabs[wrapper->callSession.keys().first()];
        tabs->showPage(tab);
    } else if (currentPage) {
        tabs->setCurrentPage(tabs->indexOf(currentPage));
        //tabs->showPage(currentPage);
    }
    removeTab(callNo);
}

/* Contact List related */

void KiaxMainWindow::contactsListBox_contextMenuRequested(QListViewItem* item, const QPoint& location, int col) {	
	if (item!=NULL) {
		KiaxContactListItem * clItem = (KiaxContactListItem*) item;
		if (clItem->isContact())
        		contactListMenu->popup(location,0);
	}
}

void KiaxMainWindow::toolsAdd_ContactAction_activated() {
    /* the button + will take the selected call record
     * if we are on the call register */
    
   if (tabs->currentPage()==callRegisterPage) {
        if (callRegisterListBox->currentItem()>-1) {
            callRegisterAddUserAction_activated();
            return;
        }
    }

    KiaxContactWindow * cw = new KiaxContactWindow(this,0,true,0);
    updateAccountsComboBox(cw->accountsComboBox);
    int result = cw->exec();
    if (result == QDialog::Accepted) {
        QString cname = cw->userAliasEdit->text();
        QString cnumber = cw->userNumberEdit->text();
        QString caccountId = "0";
        // check if we have any accounts
        if (cw->accountsComboBox->count()>0)
            caccountId = wrapper->getSettings()->getAccount(cw->accountsComboBox->currentItem())->accId;
        KiaxContact * contact = new KiaxContact(cname, cnumber, caccountId);
        int position = addressBook->addContact(contact);
        contact->save();
        addContactToContactsListBox(contact, position);
        //contactsListBox->setSelected(item, true);
    }
    delete cw;

}

QString KiaxMainWindow::getIconsPath() {
	return QString( ICONS_PATH );
}

void KiaxMainWindow::contactListChangeAction_activated() {
    KiaxContactWindow * cw = new KiaxContactWindow(this,0,true,0);
    cw->setCaption(tr("Update User Data"));
    cw->buttonOk->setText(tr("Update"));
    int selectedContact = ((KiaxContactListItem*)contactsListBox->selectedItem())->getIndex();
    KiaxContact * contact = addressBook->getContact(selectedContact);
    fprintf(stderr, "selected contact number = %d\n", selectedContact);
    if (contact==NULL) {
    	fprintf(stderr, "contact is NULL\n");
    fflush(stderr);
    } else {
  
    cw->userAliasEdit->setText(contact->getName());
    cw->userNumberEdit->setText(contact->getNumber());

    updateAccountsComboBox(cw->accountsComboBox,contact->getAccountId());
    bool result = cw->exec();
    if ((result == QDialog::Accepted)) {
        contact->setName(cw->userAliasEdit->text());
        contact->setNumber(cw->userNumberEdit->text());
        // check if we have any accounts
        if (cw->accountsComboBox->count()>0)
            contact->setAccountId(wrapper->getSettings()->getAccount(cw->accountsComboBox->currentItem())->accId);
        else //if we don't have, save the id as 0, better as nothing
            contact->setAccountId("0");
        contact->save();
    	refreshContactsListBox();
    	setControls();
    }
   }
    delete cw;

}

void KiaxMainWindow::contactListDeleteAction_activated() {

    int currentContact = ((KiaxContactListItem*)contactsListBox->selectedItem())->getIndex();
    QListViewItem* currentContactItem = contactsListBox->currentItem();
    KiaxContact * contact = addressBook->getContact(currentContact);
    contact->remove();
    addressBook->removeContact(currentContact);
    contactsListBox->removeItem(currentContactItem);
    refreshContactsListBox();
    setControls();

}

void KiaxMainWindow::callRegisterAddUserAction_activated() {
    KiaxContactWindow * cw = new KiaxContactWindow(this,0,true,0);
    int selectedRecord = callRegisterListBox->currentItem();
    KiaxCallRecord * record = callRegister->getRecord(selectedRecord);
    KiaxContact * contact = new KiaxContact(record->getCallerIdName(),record->getCallerIdNumber(),record->getAccountId() );
    cw->userAliasEdit->setText(contact->getName());
    cw->userNumberEdit->setText(contact->getNumber());
    updateAccountsComboBox(cw->accountsComboBox,contact->getAccountId());
    bool result = cw->exec();
    if (result == QDialog::Accepted) {
        contact->setName(cw->userAliasEdit->text());
        contact->setNumber(cw->userNumberEdit->text());
        // check if we have any accounts
        if (cw->accountsComboBox->count()>0)
            contact->setAccountId(wrapper->getSettings()->getAccount(cw->accountsComboBox->currentItem())->accId);
        else //if we don't have, save the id as 0, better as nothing
            contact->setAccountId("0");
        contact->save();
        int position = addressBook->addContact(contact);
        addContactToContactsListBox(contact,position);
        //contactsListBox->setSelected(item, true);
    }
    delete cw;
}

void KiaxMainWindow::callRegisterListBox_contextMenuRequested(QListBoxItem* item,const QPoint& point) {
    if (callRegisterListBox->count()>0)
        callRegisterMenu->popup(point,0);
}

void KiaxMainWindow::numberComboBox_textChanged(const QString& text) {
    contactsListBox->setSelected(contactsListBox->currentItem(), (text==""));
    callRegisterListBox->setSelected(callRegisterListBox->currentItem(),(text==""));
    setControls();
}

void KiaxMainWindow::contactsListBox_highlighted(QListViewItem* item) {
    setControls();
}

void KiaxMainWindow::callRegisterListBox_highlighted(int item) {
    setControls();
}

void KiaxMainWindow::tabs_selected(const QString& newTab) {
    if ((tabs->currentPage()!=contactsPage)&&(tabs->currentPage()!=dialpadPage)&&(tabs->currentPage()!=callRegisterPage)) {
        int callNumber = ((KiaxCallTab*)tabs->currentPage())->getCallNumber();
        int selectedCall = wrapper->getSelectedCall();
        if ((callNumber!=selectedCall)&&(wrapper->callSession[callNumber].isAnswered())) {
            wrapper->select_call(callNumber);
        }
    } else {
        currentPage = tabs->currentPage();
    }
    setControls();
}

void KiaxMainWindow::callRegisterCheckBox_toggled(bool status) {
    refreshCallRegisterListBox();
}

void KiaxMainWindow::viewDialHangup_ButtonsAction_toggled(bool toggled) {
    if (toggled) {
        dialHangupButtonsFrame->show();
        wrapper->getSettings()->setShowDialHangupButtons(true);
    } else {
        dialHangupButtonsFrame->hide();
        wrapper->getSettings()->setShowDialHangupButtons(false);
    }
}

void KiaxMainWindow::viewAccountDial_CombosAction_toggled(bool toggled) {
    if (toggled) {
        numberEditFrame->show();
        wrapper->getSettings()->setShowAccountNumberCombos(true);
    } else {
        numberEditFrame->hide();
        wrapper->getSettings()->setShowAccountNumberCombos(false);
    }
}

void KiaxMainWindow::callHoldAction_activated() {
    wrapper->hold_call(getCallNumberFromActiveTab());
}

void KiaxMainWindow::callResumeAction_activated() {
    wrapper->resume_call(getCallNumberFromActiveTab());
}

void KiaxMainWindow::updateAccounts() {
    wrapper->getSettings()->loadAccounts();
    updateAccountsComboBox(this->accountsComboBox);
}

void KiaxMainWindow::updateCallDuration(int callNo, int hours, int minutes, int seconds) {
    QMutexLocker locker(&callTabsMutex);
    QString hoursStr = QString::number(hours);
    if (hours<10)
        hoursStr = "0" + hoursStr;
    QString minutesStr = QString::number(minutes);
    if (minutes<10)
        minutesStr = "0" + minutesStr;
    QString secondsStr = QString::number(seconds);
    if (seconds<10)
        secondsStr = "0" + secondsStr;
    int callTabsNumber = callTabs.count();
    if (callTabsNumber>0) {
        if (callTabs.keys().contains(callNo)>0) {
            KiaxCallTab * tab = callTabs[callNo];
            if (tab) {
                tab->setCallerStatus(tr("Call duration") + " " + hoursStr+":"+minutesStr+":"+secondsStr);
            }
        }
    }

}

KiaxContact * KiaxMainWindow::guessContact(QString remoteExtension) {
    for (uint i=0; i<addressBook->count();i++) {
        if (addressBook->getContact(i)->getNumber()==remoteExtension)
            return addressBook->getContact(i);
    }
    return NULL;
}

/*
    setControls() is called whenever the state of the dialing buttons/actions has
    to be changed (disabled/enabled) depending on the currently focused widget
*/
void KiaxMainWindow::setControlsInitialState() {
    transferButton->hide();
    accountsComboBox->show();
    DialButton->setPixmap(*dialPixmap);
    DialButton->setEnabled(false);
    HangupButton->setEnabled(false);
    callHangupAction->setEnabled(false);
    callHoldAction->setEnabled(false);
    callResumeAction->setEnabled(false);
    callAnswerAction->setEnabled(false);
}

void KiaxMainWindow::setControlsEnableDialing() {
    DialButton->setEnabled(true);
    callDialAction->setEnabled(true);
    numberComboBox->setEnabled(true);
    accountsComboBox->setEnabled(true);
    DialButton->setEnabled(true);
}

void KiaxMainWindow::setControlsMaxCallsReached() {
    numberComboBox->setEnabled(false);
    accountsComboBox->setEnabled(false);
    DialButton->setEnabled(false);
    callDialAction->setEnabled(false);
}

void KiaxMainWindow::setControlsOutgoing() {
    DialButton->setPixmap(*holdPixmap);
    callHoldAction->setEnabled(true);
    callResumeAction->setEnabled(false);
    HangupButton->setEnabled(true);
    callHangupAction->setEnabled(true);
    callDialAction->setEnabled(false);
    callAnswerAction->setEnabled(false);
}

void KiaxMainWindow::setControlsIncoming() {
    DialButton->setPixmap(*dialPixmap);
    HangupButton->setEnabled(true);
    callHoldAction->setEnabled(false);
    callResumeAction->setEnabled(false);
    callAnswerAction->setEnabled(true);
    callHangupAction->setEnabled(true);
}

void KiaxMainWindow::setControls() {
    if (wrapper->registered)
        if ((tabs->currentPage()==contactsPage)) {
            setControlsInitialState();
	    KiaxContactListItem * currentItem = (KiaxContactListItem*)contactsListBox->selectedItem();
	    int currentItemIndex = -1;
	    if (currentItem!=NULL)
            	currentItemIndex = currentItem->getIndex();
            if (((int)wrapper->callSession.count()<wrapper->getSettings()->getMaxCalls()&&(currentItemIndex>-1))||(numberComboBox->currentText().length()>0)) {
		if (currentItem!=NULL)
		if (currentItem->isContact()) {
                	setControlsEnableDialing(); 
		} else ;
		else if (numberComboBox->currentText().length()>0){
                	setControlsEnableDialing(); 
		}
		else {
		        DialButton->setEnabled(false);
        		callDialAction->setEnabled(false);
		}
            } else if ((int)wrapper->callSession.count()>=wrapper->getSettings()->getMaxCalls()) {
                setControlsMaxCallsReached();
            } else if (wrapper->callSession.count()==0) {
                transferButton->hide();
                accountsComboBox->show();
            }
        } else if ((tabs->currentPage()==callRegisterPage)) {
            setControlsInitialState();
            if ((int)wrapper->callSession.count()<wrapper->getSettings()->getMaxCalls()&&
                    ((callRegisterListBox->currentItem()>-1)||(numberComboBox->currentText().length()>0))) {
                setControlsEnableDialing();
            } else if ((int)wrapper->callSession.count()>=wrapper->getSettings()->getMaxCalls()) {
                setControlsMaxCallsReached();
            } else if (wrapper->callSession.count()==0) {
                transferButton->hide();
                accountsComboBox->show();
            }
        } else if (tabs->currentPage()==dialpadPage) {
            setControlsInitialState();
            if ((int)wrapper->callSession.count()<wrapper->getSettings()->getMaxCalls()&&
                    (numberComboBox->currentText().length()>0)) {
                setControlsEnableDialing();
            } else if ((int)wrapper->callSession.count()>=wrapper->getSettings()->getMaxCalls()) {
                setControlsMaxCallsReached();
            } else if (wrapper->callSession.count()==0) {
                transferButton->hide();
                accountsComboBox->show();
            }
        } else {
            DialButton->setEnabled(true);
            if (!(wrapper->callSession[wrapper->getSelectedCall()].getCustomProperty("isHold")=="true")) {
                KiaxCallTab* selectedTab = (KiaxCallTab*) tabs->currentPage();
                int callNumber = selectedTab->getCallNumber();
                if   (!wrapper->callSession[callNumber].isOutgoing()&&(!wrapper->callSession[callNumber].isAnswered())) {
                    setControlsIncoming();
                } else {
                    setControlsOutgoing();
                }
                /* these are needed anyway */
                numberComboBox->setEnabled(true);
                accountsComboBox->setEnabled(true);
                transferButton->show();
                accountsComboBox->hide();
            } else {
                DialButton->setPixmap(*resumePixmap);
                callHoldAction->setEnabled(false);
                callResumeAction->setEnabled(true);
            }
        }
    else {
        DialButton->setEnabled(false);
        callDialAction->setEnabled(false);
    }
}

void KiaxMainWindow::refreshContactsListBox() { 

    contactsListBox->clear();
    groups.clear();
    contactsListBox->header()->hide();
    contactsListBox->header()->setStretchEnabled(true);
    if (contactsListBox->columns()<1)
    	int n = contactsListBox->addColumn("Users");

    for(uint i=0;i<addressBook->count();i++) {
	// root group of the contact
	KiaxContactListItem* contactRoot = NULL;
	// Find if group is already added
	QString accountId = addressBook->getContact(i)->getAccountId();
	contactRoot = groups[accountId];
	if (contactRoot==NULL) {
	        QPixmap groupPm ( QString( ICONS_PATH ) + "account.png" );
		contactRoot = new KiaxContactListItem(contactsListBox,0, false);
		KiaxAccount* account =  wrapper->getSettings()->findAccountById(accountId);
		if (account!=NULL) {
			contactRoot->setText(0,account->accAlias);
			contactRoot->setPixmap(0,groupPm);
			groups[addressBook->getContact(i)->getAccountId()] = contactRoot;
		}
	}
        QPixmap pm ( QString( ICONS_PATH ) + "contact.png" );
        KiaxContactListItem * lbi = new KiaxContactListItem(contactRoot,i, true);
	//fprintf(stderr,"conact index = %d, name = %s\n", i, (const char*) addressBook->getContact(i)->getName());
	//fflush(stderr);
	KiaxContact * contact = addressBook->getContact(i);
	if (contact!=NULL) {
		lbi->setText(0,contact->getName());
		lbi->setPixmap(0,pm);
		contactsListBox->ensureItemVisible(lbi);
	}
    }
    setControls();
}

void KiaxMainWindow::refreshCallRegisterListBox() {
    uint selectedRecord = callRegisterListBox->currentItem();
    callRegisterListBox->clear();
    for(uint i=0;i<callRegister->count();i++)
        addRecordToCallRegisterListBox(callRegister->getRecord(i),i);
    if ((int)selectedRecord>-1)
        if (selectedRecord>callRegisterListBox->count()-1)
            callRegisterListBox->setSelected(callRegisterListBox->count()-1,true);
        else
            callRegisterListBox->setSelected(selectedRecord,true);

}

void KiaxMainWindow::addRecordToCallRegisterListBox(KiaxCallRecord* record, int position) {
    QString cidName = record->getCallerIdName();
    QString cidNumber = record->getCallerIdNumber();
    QString accId = record->getAccountId();
    bool outgoing = record->isOutgoing();
    bool answered = record->isAnswered();
    if ((outgoingCheckBox->isChecked()&&outgoing)||(incomingCheckBox->isChecked()&&(!outgoing)&&(answered))||((missedCheckBox->isChecked())&&(!answered)&&(!outgoing))) {
        QString accountName;
        KiaxAccount * account = wrapper->getSettings()->findAccountById(accId);
        if (account)
            accountName = account->accAlias;
        else
            accountName = "unknown";
        QString recordIcon = "outgoing.png";
        if (!outgoing) {
            recordIcon = "incoming.png";
            if (!answered)
                recordIcon ="missed.png";
        } else
            if (!answered)
                recordIcon="failed.png";
        QString tooltipText = cidName + " ["+cidNumber+"@"+accountName+"]";
        QDateTime* time = new QDateTime();
        time->setTime_t(record->getCallStartTime().toUInt());
        QString recordText = cidName + ", " + time->toString();
        QPixmap pm ( QString( ICONS_PATH) + QString( recordIcon)  );
        //    QListBoxPixmap * lbi = new QListBoxPixmap(pm, recordText);
        QString iconsPath = QString(ICONS_PATH);
        KiaxCallRegisterItem * lbi = new KiaxCallRegisterItem(record, iconsPath);
        callRegisterListBox->insertItem(lbi,position);
    }
}

void KiaxMainWindow::addContactToContactsListBox(KiaxContact* contact, int position) {
        refreshContactsListBox();
}

void KiaxMainWindow::setWidgetsRegistered() {
    if (!wrapper->callSession.count()>0) // check to see if we have active calls
    {
        setControls();

        fileConnectAction->setEnabled(false);
        fileDisconnectAction->setEnabled(true);
        HangupButton->setEnabled(false);
        DialPadFrame->setEnabled(true);
        numberComboBox->setEnabled(true);
        accountsComboBox->setEnabled(true);
    }
}

void KiaxMainWindow::setWidgetsUnregistered() {
    statusIndicator->setText(tr("Press connect button to start "));
    DialButton->setEnabled(false);
    HangupButton->setEnabled(false);
    fileConnectAction->setEnabled(true);
    fileDisconnectAction->setEnabled(false);
    callDialAction->setEnabled(false); // deactivate any dialing options
    callHangupAction->setEnabled(false);
    callAnswerAction->setEnabled(false);
    DialPadFrame->setEnabled(false);
    numberComboBox->setEnabled(false);
    accountsComboBox->setEnabled(false);
}


/*
   When call is made or received this adds a KiaxCallTab with the
   status of the selected call
*/
void KiaxMainWindow::addTab(int callNumber, QString callerStatus) {
    // add a user tab
    QMutexLocker locker(&callTabsMutex);
    KiaxCallRecord& record = wrapper->callSession[callNumber];
    KiaxCallTab * newCallTab = new KiaxCallTab(this,0,0,record.getCallerIdName()+" ("+record.getCallerIdNumber()+")", callerStatus);
    newCallTab->setCallNumber(callNumber);
    callTabs.insert(callNumber, newCallTab);
    tabs->addTab(newCallTab, record.getCallerIdName());
    // and focus it
    tabs->setCurrentPage(tabs->count()-1);
    if (wrapper->callSession[callNumber].isOutgoing())
        DialButton->setPixmap(*holdPixmap);
    else
        DialButton->setPixmap(*dialPixmap);
}

/*
   Called when call reaches INACTIVE state. Removes the call tab.
*/
void KiaxMainWindow::removeTab(int &callNumber) {
    QMutexLocker locker(&callTabsMutex);
    KiaxCallTab * theTab = callTabs[callNumber];
    if (theTab) {
        theTab->close();
        tabs->removePage(theTab);
        delete theTab;
        theTab = NULL;
    }
    callTabs.erase(callNumber);
}

/*
   Refreshes the content of the accounts combobox in main window and selects
   the default account
*/
void KiaxMainWindow::updateAccountsComboBox(QComboBox* comboBox) {
    updateAccountsComboBox(comboBox,wrapper->getSettings()->getDefaultAccountId());
}

/*
   Refreshes the content of the accounts combobox in the Change/modify Contact
   dialog and selects the associated to the contact account
*/
void KiaxMainWindow::updateAccountsComboBox(QComboBox* comboBox, QString selectId) // after change in accounts settings
{
    int defaultAccount = 0;
    comboBox->clear();
    for (uint i=0;i<wrapper->getSettings()->getAccounts().count();i++) {
        comboBox->insertItem(wrapper->getSettings()->getAccount(i)->accAlias,i);
        if (wrapper->getSettings()->getAccount(i)->accId==selectId)
            defaultAccount = i;
    }
    // select the default account
    comboBox->setCurrentItem(defaultAccount);
}

int KiaxMainWindow::getCallNumberFromActiveTab() {
    if (tabs->currentPage()->isA("KiaxCallTab")) {
        KiaxCallTab* callTab = (KiaxCallTab*)tabs->currentPage();
        return callTab->getCallNumber();
    } else
        return wrapper->getSelectedCall();
}

void KiaxMainWindow::fatalError(QString errorMsg) {
    qFatal(tr("Error message:") + errorMsg + tr("\nExitting application. Possible reason: Device initialization failed.\n"));
    close();
    QApplication::exit();
}

void KiaxMainWindow::slotLogWindow(const char* logStr) {
    if (logWindow->logEdit->lines()>1000)
        logWindow->logEdit->clear();
    logWindow->logEdit->append(logStr);
}

void KiaxMainWindow::showStatistics(bool flag) {
    wrapper->setShowStats(flag);
}
