# vm.asm
#
# The kForth Virtual Machine
#
# Copyright (c) 1998--2000 Krishna Myneni, Creative Consulting for
#   Research and Education
#
# This software is provided under the terms of the General Public License.
#
# Usage from C++
#
#       extern "C" int vm (byte* ip);
#       ecode = vm(ip);
#
# Originally written for the A386 assembler
# Revisions:
#	3-21-98
#	8-25-1998 ported to GNU assembler under Linux
#	9-8-1998  additional functions added: c@, c!, f=, f<>
#	9-10-1998 added: and, or, not, xor
#	9-11-1998 added: = , <>, <, >, <=, >=
#	9-15-1998 added: +!, fsqrt, fsin, fcos, -rot, sp@, rp@
#	9-17-1998 added: a@
#	9-27-1998 added: mod, 2*, 2/
#	10-1-1998 added: pick
#	10-4-1998 fixed signed integer division and mod; added /MOD
#	10-6-1998 added ?dup
#	10-14-1998 added count
#	10-16-1998 fixed L_div error
#	10-19-1998 added 0<, 0=, 0>, true, false
#	10-20-1998 added 2+, 2-
#	02-09-1999 added execute
#	03-01-1999 added open, lseek, close, read, write
#	03-02-1999 added ioctl
#	03-03-1999 added usleep	
#	03-07-1999 added fill, cmove
#	03-09-1999 interchanged meaning of execute and call
#		   to be consistent with ANS Forth
#	03-27-1999 added +loop, unloop
#	03-29-1999 added roll
#	03-31-1999 added cmove>, key
#	05-05-1999 fixed +loop
#	05-06-1999 added fround
#	05-15-1999 added floor
#	05-26-1999 added fatan2, lshift, rshift
#	05-27-1999 added u<, quit, base
#	06-02-1999 added */, */mod
#	06-09-1999 call CPP functions from vm
#	07-18-1999 added find
#	09-06-1999 added pTIB, word, tick
#	09-12-1999 added system
#	10-04-1999 added create, variable, fvariable as intrinsic words
#	10-06-1999 added constant, fconstant as intrinsic words
#	10-07-1999 added chdir
#       10-08-1999 added erase, brackettick
#	10-09-1999 added time&date, ms, question, bl
#       10-10-1999 added char, forget, cold
#       10-20-1999 added >file, console
#	10-28-1999 added key?
#	12-24-1999 added hooks for f0=, f0<, f0>, u>, s>d, d>f, f>d, 
#	              um*, um/mod, m*, m+, m/, m*/
#	12-25-1999 added cells, cell+, dfloats, dfloat+
#	12-27-1999 added bye
#	01-08-2000 fixed f0<, increased vm loop efficiency
#	01-13-1999 added ?allot, fixed f0=
#	01-22-2000 modified + to remove ordering sensitivity for address arithmetic
#	01-23-2000 added 0<>, [char], .r, u.r, changed opcodes for 
#	             relational operators
#	01-24-2000 added CPP_literal, CPP_cquote, CPP_squote, CPP_dotquote
#	02-04-2000 implemented m*, um*
#	02-26-2000 fixed fm/mod, added sm/rem, um/mod
#	03-02-2000 modified QUIT to clear the return stacks, added CPP_do
#	03-05-2000 added CPP_begin, CPP_while, CPP_repeat, CPP_until, CPP_again,
#	             CPP_leave, CPP_if, CPP_else, CPP_then, CPP_lparen
#	05-17-2000 added CPP_does
#	05-18-2000 fix L_plusloop for negative increments
#	06-04-2000 fix L_roll to roll the typestack as well
#	06-11-2000 added CPP_case, CPP_endcase, CPP_of, and CPP_endof
#	06-15-2000 added CPP_querydo, CPP_abortquote
#
.equ WSIZE,	4

.equ OP_ADDR,	65
.equ OP_FVAL,	70
.equ OP_IVAL,	73
.equ OP_RET,	238

# Error Codes

.equ E_NOT_ADDR,	1
.equ E_DIV_ZERO,	4
.equ E_RET_STK_CORRUPT,	5
.equ E_UNKNOWN_OP,	6

.data
FCONST_180: .double 180.
JumpTable: .int L_false, L_true, L_cells, L_cellplus # 0 -- 3
           .int L_dfloats, L_dfloatplus, CPP_case__Fv, CPP_endcase__Fv  # 4 -- 7
           .int CPP_of__Fv, CPP_endof__Fv, C_open, C_lseek     # 8 -- 11
           .int C_close, C_read, C_write, C_ioctl # 12 -- 15
           .int L_usleep, L_ms, L_nop, L_nop      # 16 -- 19
           .int L_fill, L_cmove, L_cmovefrom, L_nop # 20 -- 23
           .int L_nop, CPP_tofile__Fv, CPP_console__Fv, L_nop  # 24 -- 27
           .int L_nop, CPP_squote__Fv, CPP_cr__Fv, L_bl    # 28 -- 31
           .int CPP_spaces__Fv, L_store, CPP_cquote__Fv, L_nop # 32 -- 35
           .int L_nop, L_mod, L_and, L_tick       # 36 -- 39
           .int CPP_lparen__Fv, L_nop, L_mul, L_add  # 40 -- 43
           .int L_nop, L_sub, CPP_dot__Fv, L_div  # 44 -- 47
           .int L_nop, L_nop, L_umstar, L_umslashmod   # 48 -- 51
           .int L_mstar, L_mplus, L_mslash, L_mstarslash # 52 -- 55
           .int L_fmslashmod, L_smslashrem, L_nop, L_nop # 56 -- 59
           .int L_lt, L_eq, L_gt, L_question      # 60 -- 63
           .int L_fetch, L_addr, L_base, L_call   # 64 -- 67
           .int L_nop, L_erase, L_fval, CPP_forget__Fv # 68 -- 71
           .int L_nop, L_ival, L_nop, C_key       # 72 -- 75
           .int L_lshift, L_slashmod, C_numberquery, CPP_dotr__Fv # 76 -- 79
           .int L_nop, C_keyquery, L_rshift, CPP_dots__Fv  # 80 -- 83
           .int C_accept, CPP_char__Fv, CPP_bracketchar__Fv, CPP_word__Fv  # 84 -- 87
           .int L_starslash, L_starslashmod, CPP_udotr__Fv, L_nop  # 88 -- 91
           .int L_nop, L_nop, L_xor, CPP_literal__Fv  # 92 -- 95
           .int CPP_queryallot__Fv, CPP_allot__Fv, L_binary, L_count # 96 -- 99
           .int L_decimal, CPP_emit__Fv, CPP_fdot__Fv, CPP_cold__Fv # 100 -- 103
           .int L_hex, L_i, L_j, CPP_brackettick__Fv         # 104 -- 107
           .int CPP_fvariable__Fv, C_timeanddate, CPP_find__Fv, CPP_constant__Fv # 108 -- 111
           .int L_nop, CPP_fconstant__Fv, CPP_create__Fv, CPP_dotquote__Fv  # 112 -- 115
           .int CPP_type__Fv, CPP_udot__Fv, CPP_variable__Fv, CPP_words__Fv # 116 -- 119
           .int CPP_does__Fv, C_system, C_chdir, L_nop   # 120 -- 123
           .int L_or, L_nop, L_not, L_nop         # 124 -- 127
           .int L_fsin, L_fcos, C_ftan, C_fasin   # 128 -- 131
           .int C_facos, C_fatan, C_fexp, C_fln   # 132 -- 135
           .int C_flog, L_fatan2, L_nop, L_nop    # 136 -- 139
           .int C_fmin, C_fmax, L_floor, L_fround # 140 -- 143
           .int L_nop, L_nop, L_nop, L_nop        # 144 -- 147
           .int L_nop, L_nop, L_stod, L_stof      # 148 -- 151
           .int L_dtof, L_ftos, L_ftod, L_degtorad  # 152 -- 155
           .int L_radtodeg, L_nop, L_nop, L_nop   # 156 -- 159
           .int L_inc, L_dec, L_abs, L_neg        # 160 -- 163
           .int L_min, L_max, L_twostar, L_twodiv # 164 -- 167
           .int L_twoplus, L_twominus, L_cfetch, L_cstore  # 168 -- 171
           .int L_wfetch, L_wstore, L_dffetch, L_dfstore  # 172 -- 175
           .int L_sffetch, L_sfstore, L_spfetch, L_plusstore # 176 -- 179
           .int L_fadd, L_fsub, L_fmul, L_fdiv    # 180 -- 183
           .int L_fabs, L_fneg, C_fpow, L_fsqrt   # 184 -- 187
           .int L_nop, L_nop, L_feq, L_fne        # 188 -- 191
           .int L_flt, L_fgt, L_fle, L_fge        # 192 -- 195
           .int L_fzeroeq, L_fzerolt, L_fzerogt, L_nop # 196 -- 199
           .int L_drop, L_dup, L_swap, L_over     # 200 -- 203
           .int L_rot, L_minusrot, L_nip, L_tuck  # 204 -- 207
           .int L_pick, L_roll, L_2drop, L_2dup   # 208 -- 211
           .int L_2swap, L_2over, L_2rot, L_depth # 212 -- 215
           .int L_querydup, CPP_if__Fv, CPP_else__Fv, CPP_then__Fv # 216 -- 219
           .int L_push, L_pop, L_puship, L_rfetch # 220 -- 223
           .int L_rpfetch, L_afetch, CPP_do__Fv, CPP_leave__Fv # 224 -- 227
           .int CPP_querydo__Fv, CPP_abortquote__Fv, L_jz, L_jnz  # 228 -- 231
           .int L_jmp, L_loop, L_plusloop, L_unloop  # 232 -- 235
           .int L_execute, CPP_recurse__Fv, L_ret, L_abort  # 236 -- 239
           .int L_quit, L_ge, L_le, L_ne          # 240 -- 243
           .int L_zeroeq, L_zerone, L_zerolt, L_zerogt # 244 -- 247
           .int L_ult, L_ugt, CPP_begin__Fv, CPP_while__Fv    # 248 -- 251
           .int CPP_repeat__Fv, CPP_until__Fv, CPP_again__Fv, CPP_bye__Fv  # 252 -- 255



.text
	.align 4
.global JumpTable
.global L_depth, L_tick, L_quit, L_abort, L_ret
.global vm
	.type	vm,@function
vm:	
        pushl %ebp
        pushl %ebx
        pushl %ecx
        pushl %edx
        movl %esp, %ebp
        movl 20(%ebp), %ebx     # load the Forth instruction pointer
        movl %ebx, GlobalIp     
next:
        movl GlobalIp, %ebx
        xor %eax, %eax
        movb (%ebx), %al         # get the opcode
        sall $2, %eax            # determine offset of opcode
	movl JumpTable(%eax), %ebx	# machine code address of word
	xor %eax, %eax          # clear error code
	call *%ebx		# call the word
	cmpl $0, %eax           # check for error
        jnz exitloop            
        incl GlobalIp           # increment the Forth instruction ptr
        jmp next
exitloop:
        cmpl $OP_RET, %eax         # return from vm?
        jnz vmexit
        xor %eax, %eax            # clear the error
vmexit:
	pop %edx
        pop %ecx
        pop %ebx
        pop %ebp
        ret
L_nop:
        movl $E_UNKNOWN_OP, %eax   # unknown operation
        ret
L_quit:
	movl BottomOfReturnStack, %eax	# clear the return stacks
	movl %eax, GlobalRp
	movl BottomOfReturnTypeStack, %eax
	movl %eax, GlobalRtp	
	movl $8, %eax		# exit the virtual machine
	ret
L_abort:
	movl BottomOfStack, %eax
	movl %eax, GlobalSp
	movl BottomOfTypeStack, %eax
	movl %eax, GlobalTp
	jmp L_quit
L_base:	
	movl GlobalSp, %ebx
	movl $Base, %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl GlobalTp
	xor %eax, %eax
	ret
L_binary:
	movl $Base, %ebx
	movl $2, (%ebx)
	ret
L_decimal:	
	movl $Base, %ebx
	movl $10, (%ebx)
	ret
L_hex:	
	movl $Base, %ebx
	movl $16, (%ebx)
	ret		
L_false:
	movl GlobalSp, %ebx
	movl $0, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	ret
L_true:
	movl GlobalSp, %ebx
	movl $-1, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	ret
L_cells:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	sall $2, %eax
	movl %eax, (%ebx)
	xor %eax, %eax
	ret
L_cellplus:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %eax
	movl %eax, (%ebx)
	xor %eax, %eax
	ret
L_dfloats:	
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	sall $3, %eax
	movl %eax, (%ebx)
	xor %eax, %eax
	ret
L_dfloatplus:	
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %eax
	addl $WSIZE, %eax
	movl %eax, (%ebx)
	xor %eax, %eax
	ret				
L_bl:
	movl GlobalSp, %ebx
	movl $32, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp	
	ret
L_ret:
	movl $WSIZE, %eax
        addl %eax, GlobalRp
        incl GlobalRtp
        movl GlobalRtp, %ebx
        cmpl BottomOfReturnTypeStack, %ebx
        jle ret1
        movl $OP_RET, %eax             # exhausted the return stack so exit vm
        jmp retexit
ret1:   movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jz ret2
        movl $E_RET_STK_CORRUPT, %eax   # indicate return stack corrupted
        jmp retexit
ret2:   movl GlobalRp, %ebx
        movl (%ebx), %eax
	movl %eax, GlobalIp		# reset the instruction ptr
        xor %eax, %eax
retexit:
        ret
L_tick:
	movl GlobalSp, %ebx
	movl $32, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	decl GlobalTp
	call CPP_word__Fv
	call CPP_find__Fv
	call L_drop
	ret
L_usleep:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	pushl %eax
	call usleep
	popl %eax
	xor %eax, %eax
	ret
L_ms:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	imull $1000, %eax
	movl %eax, WSIZE(%ebx)
	call L_usleep
	ret
L_fill:
	call L_swap
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz fill2
	popl %ebx
	popl %ebx
	movl $E_NOT_ADDR, %eax
	jmp fillexit
fill2:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call memset
	addl $12, %esp
	xor %eax, %eax
fillexit:	
	ret
L_erase:
	movl GlobalSp, %ebx
	movl $0, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_fill
	ret	
L_cmove:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call L_swap
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmove2
	popl %ebx
	movl $E_NOT_ADDR, %eax
	ret
cmove2:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmove3
	popl %ebx
	popl %ebx
	movl $E_NOT_ADDR, %eax
	ret
cmove3:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call memcpy
	addl $12, %esp
	xor %eax, %eax				
	ret		
L_cmovefrom:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx	# load count register
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmovefrom2
	movl $E_NOT_ADDR, %eax						
	ret
cmovefrom2:
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	movl %ecx, %eax
	decl %eax
	addl %eax, %ebx
	movl %ebx, %edx		# dest addr in %edx
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmovefrom3
	movl $E_NOT_ADDR, %eax
	ret
cmovefrom3:
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	movl %ecx, %eax
	decl %eax
	addl %eax, %ebx		# src addr in %ebx
cmovefromloop:	
	movb (%ebx), %al
	decl %ebx
	xchgl %ebx, %edx
	movb %al, (%ebx)
	decl %ebx
	xchgl %ebx, %edx
	loop cmovefromloop	
	xor %eax, %eax
	ret
L_call:	
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	call *(%ebx)
	xor %eax, %eax
	ret 		
L_push:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx	
        movl %ebx, GlobalSp
	movl (%ebx), %ecx
	movl GlobalRp, %ebx
	movl %ecx, (%ebx)
	subl %eax, %ebx
	movl %ebx, GlobalRp
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, GlobalTp
	movb (%ebx), %al
	movl GlobalRtp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalRtp
        xor %eax, %eax
        ret
L_pop:
	movl $WSIZE, %eax
	movl GlobalRp, %ebx
	addl %eax, %ebx
	movl %ebx, GlobalRp
	movl (%ebx), %ecx
	movl GlobalSp, %ebx
	movl %ecx, (%ebx)
	subl %eax, %ebx
	movl %ebx, GlobalSp
	movl GlobalRtp, %ebx
	incl %ebx
	movl %ebx, GlobalRtp
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	ret
L_puship:
        movl GlobalIp, %eax
        movl GlobalRp, %ebx
        mov %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalRp
        movl GlobalRtp, %ebx
	movb $OP_ADDR, %al
        movb %al, (%ebx)
        decl GlobalRtp
        xor %eax, %eax
        ret
L_execute:	
        movl GlobalIp, %ecx
        movl GlobalRp, %ebx
        movl %ecx, (%ebx)
	movl $WSIZE, %eax 
        subl %eax, %ebx 
	movl %ebx, GlobalRp
        movl GlobalRtp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
        movl %ebx, GlobalRtp
	movl GlobalSp, %ebx
        addl %eax, %ebx
	movl %ebx, GlobalSp
        movl (%ebx), %eax
	decl %eax
	movl %eax, GlobalIp
	incl GlobalTp
        xor %eax, %eax
        ret
L_rfetch:
        movl GlobalRp, %ebx
        addl $WSIZE, %ebx
        movl (%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        movl GlobalTp, %ebx
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
	ret
L_rpfetch:
	movl GlobalRp, %eax
	addl $WSIZE, %eax
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	ret
L_spfetch:
	movl GlobalSp, %eax
	movl %eax, %ebx
	addl $WSIZE, %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax 
	ret
L_i:
        movl GlobalRtp, %ebx
        movb 3(%ebx), %al
        movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
        movl GlobalRp, %ebx
        movl 3*WSIZE(%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, %ebx 
	movl %ebx, GlobalSp
        xor %eax, %eax
        ret
L_j:
        movl GlobalRtp, %ebx
        movb 6(%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
        movl GlobalRp, %ebx
        movl 6*WSIZE(%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, %ebx
	movl %ebx, GlobalSp
        xor %eax, %eax
        ret
L_loop:
        movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz loopbad
        movl GlobalRp, %ebx	
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
        movl (%ebx), %eax
        incl %eax
	cmpl %ecx, %eax	
        jz L_unloop
loop1:	
        movl %eax, (%ebx)	# set loop counter to next value
	movl %edx, GlobalIp	# set instruction ptr to start of loop
        xor %eax, %eax
        ret
L_unloop:  
	movl $WSIZE, %eax
	imul $3, %eax
	addl %eax, GlobalRp  # terminal count reached, discard top 3 items
	movl $3, %eax
        addl %eax, GlobalRtp
        xor %eax, %eax
        ret
loopbad:
        movl $E_RET_STK_CORRUPT, %eax
        ret	
L_plusloop:
	movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz loopbad
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ebp	# get loop increment 
	movl %ebx, GlobalSp
	incl GlobalTp		
        movl GlobalRp, %ebx
	addl %eax, %ebx		# get ip and save in edx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %ecx	# get terminal count in ecx
	addl %eax, %ebx
	movl (%ebx), %eax	# get current loop count 
	addl %ebp, %eax		# increment loop count
	cmpl $0, %ebp
	jl plusloop1
	cmpl %ecx, %eax
	jge L_unloop				
	movl %eax, (%ebx)	# set loop counter to incremented value
        movl %edx, GlobalIp     # set instruction ptr to start of loop
        xor %eax, %eax
        ret
plusloop1:
	cmpl %ecx, %eax
	jl L_unloop
	movl %eax, (%ebx)
	movl %edx, GlobalIp
	xor %eax, %eax
	ret
L_jz:
	movl $WSIZE, %eax
        movl GlobalSp, %ebx
	addl %eax, %ebx
	movl %ebx, GlobalSp
        incl GlobalTp
        movl (%ebx), %eax
        cmpl $0, %eax
        jz jz1
	movl $4, %eax
        addl %eax, GlobalIp       # do not jump
        jmp jzexit
jz1:    movl GlobalIp, %ebx
        incl %ebx
        movl (%ebx), %eax          # get the relative jump count
        decl %eax
        addl %eax, GlobalIp
jzexit: xor %eax, %eax
        ret
L_jnz:				# not implemented
	ret
L_jmp:
        movl GlobalIp, %ebx
        incl %ebx
        movl (%ebx), %eax          # get the relative jump count
        addl %eax, %ebx
        subl $2, %ebx
        movl %ebx, GlobalIp      # set instruction ptr
	xor %eax, %eax
        ret
L_count:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz counterror
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %ebx
	xor %eax, %eax
	movb (%ebx), %al
	movl GlobalSp, %ebx
	incl WSIZE(%ebx)
	movl %eax, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	xor %eax, %eax
	ret
counterror:
	movl $E_NOT_ADDR, %eax
	ret
L_ival:
        movl GlobalIp, %ebx
        incl %ebx
        movl (%ebx), %eax
	movl %ebx, %ecx
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	subl $WSIZE, %ebx
        movl %ebx, GlobalSp    # decrement the stack ptr
        movl GlobalTp, %ebx
	movb $OP_IVAL, %al
        movb %al, (%ebx)     # set type and dec
	decl %ebx
        movl %ebx, GlobalTp
	movl %ecx, %eax
	decl %eax
	addl $WSIZE, %eax
        movl %eax, GlobalIp    # advance the instruction ptr
        xor %eax, %eax            # no errors
        ret
L_addr:
	movl GlobalIp, %ebx
	incl %ebx
	movl (%ebx), %eax
	movl %ebx, %ecx
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, %al
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	movl %ecx, %eax
	decl %eax
	addl $WSIZE, %eax
	movl %eax, GlobalIp
	xor %eax, %eax
        ret
L_fval:
        movl GlobalIp, %ebx
        incl %ebx
        fldl (%ebx)
        movl GlobalSp, %ebx
        subl $WSIZE, %ebx
        fstpl (%ebx)
	movl $WSIZE, %eax
	sall $1, %eax
        subl %eax, GlobalSp
        movl GlobalTp, %ebx
        movb $OP_FVAL, (%ebx)
        decl %ebx
        movb $OP_FVAL, (%ebx)
	decl GlobalTp
	decl GlobalTp
        addl %eax, GlobalIp
	xor %eax, %eax
        ret
L_and:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	pushl %ebx
	movl (%ebx), %ebx
	andl %ebx, %eax
	popl %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_or:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	orl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_not:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	notl %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	ret
L_xor:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	xorl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_lshift:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx
	movl WSIZE(%ebx), %eax
	shll %cl, %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	ret
L_rshift:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx
	movl WSIZE(%ebx), %eax
	shrl %cl, %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	ret
L_eq:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %ebx, %eax
	jne eq2
eq1:	movl GlobalSp, %ebx
	movl $-1, WSIZE(%ebx)
	jmp eq3
eq2:    movl GlobalSp, %ebx
	movl $0, WSIZE(%ebx)
eq3:    movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_ne:
	call L_eq
	call L_not
	ret
L_ult:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jae eq2
	jmp eq1
	ret
L_ugt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jbe eq2
	jmp eq1	
	ret	
L_lt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jge eq2
	jmp eq1
L_gt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jle eq2
	jmp eq1
L_le:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jg eq2
	jmp eq1
L_ge:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jl eq2
	jmp eq1
L_zerolt:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	jl zerolt2
zerolt1:
	movl $0, WSIZE(%ebx)
	jmp zeroltexit
zerolt2:
	movl $-1, WSIZE(%ebx)
zeroltexit:
	movl GlobalTp, %ebx
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	ret
L_zeroeq:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	je zerolt2
	jmp zerolt1
L_zerone:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	je zerolt1
	jmp zerolt2
L_zerogt:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	jg zerolt2
	jmp zerolt1 	
L_querydup:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	je L_querydupexit
	movl %eax, (%ebx)
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	movb %al, (%ebx)
	decl GlobalTp
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	xor %eax, %eax
L_querydupexit:
	ret	
L_drop:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	xor %eax, %eax
        ret
L_dup:
        movl GlobalSp, %ebx
        movl WSIZE(%ebx), %eax
        movl %eax, (%ebx)
	movl %ebx, %eax
	subl $WSIZE, %eax
	movl %eax, GlobalSp
        movl GlobalTp, %ebx
        movb 1(%ebx), %al
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
        ret
L_swap:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        movl WSIZE(%ebx), %eax
        xchgl (%ebx), %eax
        movl %eax, WSIZE(%ebx)
        movl GlobalTp, %ebx
        incl %ebx
        movb 1(%ebx), %al
        xchgb (%ebx), %al
        movb %al, 1(%ebx)
        xor %eax, %eax
        ret
L_over:
        movl GlobalSp, %ebx
        movl 2*WSIZE(%ebx), %eax
        movl %eax, (%ebx)
	movl %ebx, %eax
	subl $WSIZE, %eax
	movl %eax, GlobalSp
        movl GlobalTp, %ebx
        movb 2(%ebx), %al
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
        ret
L_rot:
        call L_swap
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        movl 2*WSIZE(%ebx), %eax
        xchgl (%ebx), %eax
        xchgl 2*WSIZE(%ebx), %eax
        movl GlobalTp, %ebx
        incl %ebx
        movb  2(%ebx), %al
        xchgb (%ebx), %al
        xchgb 2(%ebx), %al
        xor %eax, %eax
        ret
L_minusrot:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	movl -2*WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	movb %al, (%ebx)
	incl %ebx
	movw 1(%ebx), %ax
	movw %ax, (%ebx)
	movb -1(%ebx), %al
	movb %al, 2(%ebx)
	xor %eax, %eax
	ret
L_nip:
        call L_swap
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	xor %eax, %eax
        ret
L_tuck:
        call L_swap
        call L_over
        ret
L_pick:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	addl $2, %eax
	imul $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %eax
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	movl WSIZE(%ebx), %eax
	addl $2, %eax
	movl GlobalTp, %ebx
	addl %eax, %ebx
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, 1(%ebx)
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	ret
L_roll:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx 
	movl (%ebx), %eax
	incl %eax
	pushl %eax
	pushl %eax
	pushl %eax
	pushl %ebx
	imul $WSIZE, %eax
	addl %eax, %ebx		# addr of item to roll
	movl (%ebx), %eax
	popl %ebx
	movl %eax, (%ebx)
	popl %eax		# number of cells to copy
	movl %eax, %ecx
	imul $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, %edx		# dest addr
	subl $WSIZE, %ebx	# src addr
rollloop:
	movl (%ebx), %eax
	sub $WSIZE, %ebx
	xchgl %ebx, %edx
	movl %eax, (%ebx)
	sub $WSIZE, %ebx
	xchgl %ebx, %edx
	loop rollloop

	popl %eax		# now we have to roll the typestack
	mov GlobalTp, %ebx	
	addl %eax, %ebx
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	popl %eax
	movl %eax, %ecx
	addl %eax, %ebx
	movl %ebx, %edx
	decl %ebx
rolltloop:
	movb (%ebx), %al
	decl %ebx
	xchgl %ebx, %edx
	movb %al, (%ebx)
	decl %ebx
	xchgl %ebx, %edx
	loop rolltloop
	xor %eax, %eax
	ret
L_depth:
	movl GlobalSp, %ebx
	movl BottomOfStack, %eax
	subl %ebx, %eax
	movl $WSIZE, (%ebx)
	movl $0, %edx
	idivl (%ebx)
	movl %eax, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	xor %eax, %eax
        ret
L_2drop:
	movl $WSIZE, %eax
	addl %eax, %eax
	addl %eax, GlobalSp
        addl $2, GlobalTp
	xor %eax, %eax
        ret
L_2dup:
	movl GlobalSp, %ebx
	movl 2*WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl 2*WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb 2(%ebx), %al
	movb %al, (%ebx)
	decl %ebx
	movb 2(%ebx), %al
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
        ret
L_2swap:
        call L_rot              # not very efficient --- recode later
        call L_push
        call L_rot
        call L_pop
        ret
L_2over:
        call L_push             # not very efficient --- recode later
        call L_push
        call L_2dup
        call L_pop
        call L_pop
        call L_2swap
        ret
L_2rot:
        call L_push             # not very efficient --- recode later
        call L_push
        call L_2swap
        call L_pop
        call L_pop
        call L_2swap
        ret
L_question:
	call L_fetch
	cmpl $0, %eax
	jnz questionexit
	call CPP_dot__Fv
questionexit:	
	ret	
L_fetch:
	movl GlobalSp, %edx
	movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_IVAL, (%ebx)
	addl $WSIZE, %edx
        movl %edx, %ebx	
        movl (%ebx), %ebx
        movl (%ebx), %eax
        movl %edx, %ebx
        movl %eax, (%ebx)
	xor %eax, %eax
	ret
fetcherror:
        movl $E_NOT_ADDR, %eax
        ret
L_store:
        movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
        addl %eax, %ebx
        movl (%ebx), %ecx	# address to store to in ecx
	addl %eax, %ebx
	movl (%ebx), %edx	# value to store in edx
	movl %ebx, GlobalSp
	movl %ecx, %ebx
	movl %edx, (%ebx)
	movl GlobalTp, %ebx
	incl %ebx
	incl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	ret
L_afetch:
	call L_fetch
	movl GlobalTp, %ebx
	movb $OP_ADDR, 1(%ebx)
	ret
L_cfetch:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movb $OP_IVAL, 1(%ebx)
	xor %eax, %eax
	movl GlobalSp, %ebx
	movl %ebx, %ecx
	movl WSIZE(%ebx), %ebx
	movb (%ebx), %al
	movl %ecx, %ebx
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
        ret
L_cstore:
	movl GlobalSp, %eax
	movl %eax, %ecx
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl %ecx, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	xchgl %ebx, %ecx
	movb %al, (%ebx)
	movl %ecx, %eax
	movl %eax, GlobalSp
	incl GlobalTp
	xor %eax, %eax
	ret	
L_wfetch:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movb $OP_IVAL, 1(%ebx)
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %ebx
	movw (%ebx), %ax
	cwde
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
        ret
L_wstore:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	pushl %eax
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	pop %ebx
	movw %ax, (%ebx)
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	xor %eax, %eax
        ret
L_sffetch:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_FVAL, (%ebx)
        decl %ebx
        movb $OP_FVAL, (%ebx)
        decl GlobalTp
	decl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %ebx
        flds (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalSp, %ebx
        fstpl (%ebx)
        subl %eax, GlobalSp
	xor %eax, %eax
        ret
L_sfstore:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)              # load the f number into NDP
        subl $WSIZE, %ebx
        movl (%ebx), %ebx          # load the dest address
        fstps (%ebx)             # store as single precision float
	movl $WSIZE, %eax
	sall $1, %eax
        addl %eax, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        ret
L_dffetch:
        movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx		
        movl (%ebx), %ecx	# address to fetch from in ecx
	subl %eax, %ebx
	xchgl %ecx, %ebx
	movl (%ebx), %edx	# first dword
	addl %eax, %ebx
	xchgl %ecx, %ebx
	movl %edx, (%ebx)
	addl %eax, %ebx
	xchgl %ebx, %ecx
	movl (%ebx), %edx	# second dword
	movl %ecx, %ebx
	movl %edx, (%ebx)
	subl %eax, %ebx
	subl %eax, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_FVAL, (%ebx)
	movb $OP_FVAL, 1(%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	ret
L_dfstore:
        movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
	movl $WSIZE, %eax	
        movl GlobalSp, %ebx
	addl %eax, %ebx
        movl (%ebx), %ecx	# address to store in ecx
	addl %eax, %ebx
	movl (%ebx), %edx	# first dword to store
	xchgl %ecx, %ebx
	movl %edx, (%ebx)
	addl %eax, %ebx
	xchgl %ecx, %ebx
	addl %eax, %ebx
	movl (%ebx), %edx	# second dword to store
	xchgl %ecx, %ebx
	movl %edx, (%ebx)
	movl %ecx, GlobalSp
	movl GlobalTp, %ebx
	addl $3, %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	ret
L_inc:
        movl GlobalSp, %ebx
        incl WSIZE(%ebx)
        ret
L_dec:
        movl GlobalSp, %ebx
        decl WSIZE(%ebx)
        ret
L_twoplus:
	movl GlobalSp, %ebx
	incl WSIZE(%ebx)
	incl WSIZE(%ebx)
	ret
L_twominus:
	movl GlobalSp, %ebx
	decl WSIZE(%ebx)
	decl WSIZE(%ebx)
	ret
L_abs:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	cmpl $0, %eax
	jl abs1
	xor %eax, %eax
	ret
abs1:	negl %eax
	movl %eax, (%ebx)
	xor %eax, %eax
        ret
L_neg:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	negl %eax
	movl %eax, (%ebx)
	xor %eax, %eax
        ret
L_max:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jl max1
	movl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	jmp maxexit
max1:
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
maxexit:
	incl GlobalTp
	xor %eax, %eax
        ret
L_min:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jg min1
	movl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	jmp minexit
min1:
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
minexit:
	incl GlobalTp
	xor %eax, %eax
        ret
L_twostar:
	movl GlobalSp, %ebx
	sall $1, WSIZE(%ebx)
	ret
L_twodiv:
	movl GlobalSp, %ebx
	sarl $1, WSIZE(%ebx)
	ret
L_add:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %eax
	addl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, GlobalTp
	movw (%ebx), %ax
	andb %ah, %al		# and the two types to preserve addr type
	incl %ebx
	movb %al, (%ebx)
        xor %eax, %eax
        ret
L_sub:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %eax
	subl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
        incl GlobalTp		# result will have type of first operand
        xor %eax, %eax
        ret
L_mul:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl %ecx, %eax
	imull (%ebx)
	movl %eax, (%ebx)
	incl GlobalTp
	xor %eax, %eax
        ret
L_div:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        cmpl $0, %eax
        jnz div1
        movl $E_DIV_ZERO, %eax
        jmp divexit
div1:	
	addl $WSIZE, %ebx
        movl (%ebx), %eax
	cdq
        idivl -WSIZE(%ebx)
        movl %eax, (%ebx)
	xor %eax, %eax
divexit:
        ret
L_mod:
	call L_div
	movl %edx, (%ebx)
	ret
L_slashmod:
	call L_div
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_swap
	ret
L_starslash:
	movl $WSIZE, %eax
	sall $1, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        movl WSIZE(%ebx), %eax
        imull (%ebx)
	idivl -WSIZE(%ebx)
	movl %eax, WSIZE(%ebx)
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax		
	ret
L_starslashmod:
	call L_starslash
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_swap
	ret
L_plusstore:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	push %ebx
	push %ebx
	push %ebx
	movl WSIZE(%ebx), %ebx
	movl (%ebx), %eax
	pop %ebx
	movl 2*WSIZE(%ebx), %ebx
	addl %ebx, %eax
	pop %ebx
	movl WSIZE(%ebx), %ebx
	movl %eax, (%ebx)
	pop %ebx
	movl $WSIZE, %eax
	sall $1, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
	ret
L_umstar:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl %ecx, %eax
	mull (%ebx)
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	xor %eax, %eax				
	ret
L_umslashmod:
	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
	divl %ecx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %eax, (%ebx)
	incl GlobalTp
	xor %eax, %eax		
	ret
L_mstar:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl %ecx, %eax
	imull (%ebx)
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	xor %eax, %eax		
	ret
L_mplus:	
	ret
L_mslash:	
	ret
L_mstarslash:	
	ret
L_fmslashmod:
	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
	idivl %ecx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %eax, (%ebx)
	cmpl $0, %edx
	jge L_fmslashmodexit
	decl %eax		# floor the result
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	addl %ecx, (%ebx)
L_fmslashmodexit:		
	incl GlobalTp
	xor %eax, %eax
	ret
L_smslashrem:
	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
	idivl %ecx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %eax, (%ebx)
	incl GlobalTp
	xor %eax, %eax		
	ret
L_stod:
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	pushl %edx
	cdq
	movl %edx, (%ebx)
	popl %edx
	movl $WSIZE, %eax
	subl %eax, %ebx
	movl %ebx, GlobalSp
	mov GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	xor %eax, %eax	
	ret
L_stof:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        fildl (%ebx)
        movl GlobalTp, %ebx
        movb $OP_FVAL, (%ebx)
        decl %ebx
        movb $OP_FVAL, (%ebx)
	decl GlobalTp
	decl GlobalTp
        movl GlobalSp, %ebx
	movl $WSIZE, %eax
        subl %eax, %ebx
        fstpl (%ebx)
	sall $1, %eax
        subl %eax, GlobalSp
	xor %eax, %eax
        ret
L_dtof:
	movl $WSIZE, %eax
        movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %eax
	xchgl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
        fildq (%ebx)
        fstpl (%ebx)
	xor %eax, %eax	
	ret	
L_ftos:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
        addl %eax, %ebx
        fistpl (%ebx)
        incl GlobalTp
        movl GlobalTp, %ebx
        incl %ebx
        movb $OP_IVAL, (%ebx)
	xor %eax, %eax
        ret
L_ftod:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	fldl (%ebx)
	subl %eax, %ebx
	fnstcw (%ebx)
	movl (%ebx), %ecx	# save NDP control word	
	movl %ecx, %edx
	movb $12, %dh		
	movl %edx, (%ebx)
	fldcw (%ebx)
	addl %eax, %ebx	
	fistpq (%ebx)
	subl %eax, %ebx
	movl %ecx, (%ebx)
	fldcw (%ebx)		# restore NDP control word
	addl %eax, %ebx 
	movl (%ebx), %eax
	xchgl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	xor %eax, %eax	
	ret	
L_degtorad:
	fldl FCONST_180
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fdivp %st, %st(1)
        fldpi
        fmulp %st, %st(1)
        fstpl (%ebx)
        ret
L_radtodeg:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fldpi
	fxch
        fdivp %st, %st(1)
        fldl FCONST_180
        fmulp %st, %st(1)
        fstpl (%ebx)
        ret
L_fne:
	call L_feq
	movl GlobalSp, %ebx
	notl WSIZE(%ebx)
	ret
L_feq:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	addl %eax, GlobalSp
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	fucompp
	fnstsw %ax
	andb $69, %ah
	xorb $64, %ah
	jne flt2
	jmp flt1
L_flt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	addl %eax, GlobalSp
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fcompl (%ebx)
	fnstsw %ax
	andb $69, %ah
	jne flt2
flt1:	movl GlobalSp, %ebx
	movl $-1, WSIZE(%ebx)
	jmp fltexit
flt2:   movl GlobalSp, %ebx
	movl $0, WSIZE(%ebx)
fltexit:
	movl $4, %eax
	addl %eax, GlobalTp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	xor %eax, %eax
	ret
L_fgt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	addl %eax, GlobalSp
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fcompl (%ebx)
	fnstsw %ax
	andb $69, %ah
	cmpb $1, %ah
	jne flt2
	jmp flt1	
L_fle:
	call L_2over
	call L_2over
	call L_feq
	call L_push
	call L_flt
	call L_pop
	call L_or
	ret
L_fge:
	call L_2over
	call L_2over
	call L_feq
	call L_push
	call L_fgt
	call L_pop
	call L_or
	ret
L_fzeroeq:
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ecx
	movl %ebx, GlobalSp
	addl %eax, %ebx
	movl (%ebx), %eax
	shll $1, %eax
	orl %ecx, %eax
	jnz fzeroeq2
fzeroeq1:	
	movl $-1, (%ebx)
	jmp fzeroeqexit
fzeroeq2:
	movl $0, (%ebx)
fzeroeqexit:	
	movl GlobalTp, %ebx
	incl %ebx
	incl %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	ret
L_fzerolt:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	fldl (%ebx)
	add %eax, %ebx
	fldz
	fcompp	
	fnstsw %ax
	andb $69, %ah
	jne fzeroeq2		
	jmpl fzeroeq1
L_fzerogt:
	ret				
L_fadd:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
	sall $1, %eax
        addl %eax, %ebx
        faddl (%ebx)
        fstpl (%ebx)
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        ret
L_fsub:
	movl $WSIZE, %eax
	sall $1, %eax
	addl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
	subl $WSIZE, %eax
        subl %eax, %ebx
        fsubl (%ebx)
        addl %eax, %ebx
        fstpl (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        ret
L_fmul:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
	sall $1, %eax
        addl %eax, %ebx
        fmull (%ebx)
        fstpl (%ebx)
	sarl $1, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        ret
L_fdiv:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
        ftst
        jnz fdiv1
        subl %eax, GlobalSp
        movl $E_DIV_ZERO, %eax
        jmp fdivexit
fdiv1:  addl $WSIZE, %ebx
	addl $WSIZE, %ebx
        fdivrl (%ebx)
        fstpl (%ebx)
        addl %eax, GlobalSp
        incl GlobalTp
	incl GlobalTp
fdivexit:
	xor %eax, %eax
	ret
L_fabs:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fabs
        fstpl (%ebx)
        ret
L_fneg:
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)
        fchs
        fstpl (%ebx)
        ret
L_floor:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	pushl %eax
	movl (%ebx), %eax
	pushl %eax
	call floor
	addl $8, %esp
	fstpl (%ebx)
	xor %eax, %eax		
	ret
L_fround:
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	fldl (%ebx)
	frndint
	fstpl (%ebx)
	ret
L_fsqrt:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fsqrt
	fstpl WSIZE(%ebx)
	ret
L_fcos:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fcos
	fstpl WSIZE(%ebx)
	ret
L_fsin:
	movl GlobalSp, %ebx
	fldl WSIZE(%ebx)
	fsin
	fstpl WSIZE(%ebx)
	ret
L_fatan2:
	movl GlobalSp, %ebx
	addl $2*WSIZE, %ebx
	fldl WSIZE(%ebx)
	fldl -WSIZE(%ebx)
	fpatan
	fstpl WSIZE(%ebx)
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	ret
L_fpow:
#        add GlobalSp, WSIZE
#        mov ebx, GlobalSp
#        FLD Q[ebx]
#        add ebx, 2*WSIZE
#        FLD Q[ebx]
#        FYL2X
#        # FLD1
#        # FSCALE
#        FSTP Q[ebx]
#        add GlobalSp, WSIZE
#        add GlobalTp, 2
        ret
#

	.comm GlobalSp,4,4
	.comm GlobalTp,4,4
	.comm GlobalIp,4,4
	.comm GlobalRp,4,4
	.comm GlobalRtp,4,4
	.comm BottomOfStack,4,4
	.comm BottomOfReturnStack,4,4
	.comm BottomOfTypeStack,4,4
	.comm BottomOfReturnTypeStack,4,4
	.comm Base,4,4
	.comm State,4,4
	.comm pTIB,4,4
	.comm TIB,256,1
	.comm WordBuf,256,1


	