\ lk.4th
\
\ Solve the Lang-Kobayashi equations describing a
\ semiconductor laser with delayed optical feedback:
\
\	dY/ds = (1 + i*alpha)ZY(s) + eta*exp(-i*phi)*Y(s-rho)
\	dZ/ds = (P - Z - (1 + 2Z)|Y(s)|^2)/T
\
\ See sl.4th for description of Y, Z, s, alpha, P, and T. 
\ The new quantities in the laser equations are rho, eta, and phi:
\
\	rho	normalized delay time for optical feedback
\	phi	feedback phase (0 to 2*pi)
\	eta	normalized feedback rate (typically 0 to 0.3)
\
\ The L-K equations represent the physical situation of a partially
\ reflecting mirror at some distance from the semiconductor laser, 
\ reflecting the beam back into the laser. The interference between
\ the laser field and the delayed feedback results in fast chaotic
\ dynamics for the state of the laser.
\
\ Copyright (c) 2002 Krishna Myneni, 2002-11-15
\
\ Requires:
\
\	matrix.4th
\	complex.4th
\	zmatrix.4th
\	sl.4th
\
\ Revisions:
\
\	2003-05-13  Replaced F>S with FROUND>S  km
\
\ Notes:
\
\  1.  Assumes feedback phase (phi) is 0.
\
\  2.  Assumes constant drive current: P is not time varying.
\
\  3.  To solve the L-K equations using the default parameters
\      for a fixed number of steps, execute "lk". The output
\      is time in ns, optical intensity, phase, and carrier density.
\      The output may be redirected to a file, e.g. lk.dat, by typing
\
\	  >file lk.dat lk console
\     
 
include matrix
include complex
include zmatrix
include sl

8192 1 zmatrix Epast	\ past complex field

fvariable eta		\ normalized feedback rate
fvariable rho		\ normalized roundtrip time (real time = rho*t_p)
fvariable dt		\ actual time step in secs

0.1e ds f!		\ use normalized time step of 0.1
588e rho f!		\ rho*t_p is actual roundtrip time in sec
0.18e eta f!

2.6e P f!		\ set a constant pump value
1.8e-9 t_s f!		\ set t_s to give T_ratio = 400

variable feedback_index
variable max_index

: !Epast ( re im -- | store the complex field )
	feedback_index @ 1 Epast zmat!
	feedback_index @ 1+ dup max_index @ > if drop 1 then
	feedback_index ! ; 

: @Epast ( -- re im | retrieve the delayed field )
	feedback_index @ 1+ dup max_index @ > if drop 1 then
	1 Epast zmat@ ; 

: lk_init ( -- | initialize necessary params for L-K calculation)
	init_params		\ from sl.4th
	t_p f@ ds f@ f* dt f!
	Epast zmat_zero
	1 feedback_index !
	rho f@ ds f@ f/ fround>s max_index !  \ ok if rho is whole number
;

: lkparams. ( -- | print revised laser and new LK parameters )
	." L-K parameters:" cr
	separator cr
	." P       = " P f@ f. cr
	." rho     = " rho f@ f. cr
	." eta     = " eta f@ f. cr
	." phi     = " 0 . cr
;
 
: dV_lk/ds ( a -- Re{dY/ds} Im{dY/ds} dZ/ds | derivative of the state vector )
	dV/ds 2>r @Epast eta f@ z*f z+ 2r> ;


\ rk4 is an *approximate* fourth-order Runge-Kutta ODE stepper, 
\   specific to this problem. This stepper is only valid when
\   the derivatives dV_lk/ds do not change appreciably over a
\   single step. It ignores the explicit time dependence
\   of the derivative at internal points in the time step.
\   This assumption is valid for the case here because the
\   current does not vary much over one time step of ds*t_p.
 
: rk4 ( a ds ad -- f1 ... fn | final state vector is returned on the stack )

	\ a is the address of the state vector
	\ ds is the step size
	\ ad is the address of a vector containing the initial derivatives
 
	adv !
	h f!
	av !
	h f@ 0.5e f* hh f!	\ hh = h/2;
	h f@ 6e f/ h6 f!	\ h6 = h/6;

	av a@ hh f@ adv a@ stepV \ first step to midpoint	
	yt sv!

	yt dV_lk/ds dyt sv!	\ second step
	av a@ hh f@ dyt stepV
	yt sv!

	yt dV_lk/ds dym sv!	\ third step
	av a@ h f@ dym stepV	
	yt sv!
	dyt dym sv+ dym sv! 	

	yt dV_lk/ds dyt sv!	\ fourth step
	2e dym svc* dym sv!
	dyt dym sv+ dyt sv!
	adv a@ dyt sv+ dyt sv!
	
	av a@ h6 f@ dyt stepV
;

variable output_flag

: lksteps ( u flag -- | step the L-K equations for u steps)
        \ Compute the solution for u steps; flag=true gives output
	output_flag !
        0 ?do
	  output_flag @ if
	    i s>f dt f@ f* 1e-9 f/ f.  \ output the time in ns
	  then

  	  V dV_lk/ds Vdot sv!	\ evaluate derivatives at current time	
	  V ds f@ Vdot rk4	\ step the state vector
	  V sv!
	  V z@ !Epast

  	  output_flag @ if
	    2 spaces
	    V intensity f.	\ output intensity
	    2 spaces
	    V phase fpi f/ f.	\ output normalized phase: 1.0 = pi radians
	    2 spaces
	    V 2 floats + f@ f. \ output normalized carrier density 
	    cr
	  then
	loop
;

: lk ( -- | solve the L-K equations and show output )
        lk_init
	1.9e 1.1e 0.1e V sv!      \ initialize the state vector
	V z@ !Epast
        20000 true lksteps
;

lk_init
params.
lkparams.
