/*********************************************************************
 * $Id: led_thread.c,v 1.3 2000/11/28 12:07:34 orjana Exp $
 * Filename:      led_thread.c
 * Description:   Monitor XKB indicator state
 * Status:        Experimental, do not distribute.
 * Author:        rjan Nygaard Austvold <austvold@acm.org>
 * Created at:    Mon Nov 26 10:27:18 2000
 * Modified at:   Tue Nov 28 01:35:03 2000
 * Modified by:   rjan Nygaard Austvold <austvold@acm.org>
 *                
 * Copyright (c) 2000 rjan Nygaard Austvold, All Rights Reserved.
 *                
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 ********************************************************************/

#include "led_thread.h"

extern gint errno;
extern gint led_pipe[2];



/*
 * Function led_thread (arg)
 *
 *    Led thread function. Winds up in a infite loop waiting for new
 *    Xkb Indicator state change events.
 *
 */
void led_thread (void *arg)
{
  led_thread_arg_t *led_thread_arg = arg;
  gint      major, minor, event_code, error_code, status_code, buffer, old;
  gint      numlock_led, capslock_led, scrollock_led;
  guint     led_state;
  XkbEvent  xkbevent;
  Display  *display;

  pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &old);

  /* open a connection to the X server, check for a compatible version of
     the Xkb extension in both the avialable library and the server, and
     initialize the extension */
  major = XkbMajorVersion;
  minor = XkbMinorVersion;
  display = XkbOpenDisplay(led_thread_arg->display_name, &event_code,
			   &error_code, &major, &minor, &status_code);
  if (status_code != XkbOD_Success) {
    switch (status_code) {
    case XkbOD_BadServerVersion:
      g_error (_("Server \"%s\" used incompatible version %d.%02d"),
	       led_thread_arg->display_name, major, minor);
      /* there should have been a fall fall-through here... */
      break;
    case XkbOD_BadLibraryVersion:
      g_error (_("%s was compiled with XKB version %d.%02d"),
	       PACKAGE, XkbMajorVersion, XkbMinorVersion);
      break;
    case XkbOD_ConnectionRefused:
      g_error (_("Can't open display \"%s\""),
	       led_thread_arg->display_name);
      break;
    case XkbOD_NonXkbServer:
      g_error (_("XKB extension not present on \"%s\""),
	       led_thread_arg->display_name);
      break;
    default:
      g_error (_("Unkown status from XkbOpenDisplay()!"));
      break;
    }
  }

  /* fetch the current led state and tell the GUI thread about it */
  if (XkbGetIndicatorState(display, XkbUseCoreKbd, &led_state) != Success)
    g_error (_("Couldn't get indicator state from Xkb!"));

  /* keep the current state so we can tell which keys change states later on */
  numlock_led   = led_state & NUM_LOCK_MASK;
  capslock_led  = led_state & CAPS_LOCK_MASK;
  scrollock_led = led_state & SCROLL_LOCK_MASK;

  buffer = led_state & (NUM_LOCK_MASK | CAPS_LOCK_MASK | SCROLL_LOCK_MASK);
  if (writen(led_pipe[1], (char *)&buffer, 4) == -1)
    g_error (_("write(): %s"), sys_errlist[MIN(errno, sys_nerr - 1)]);

  /* tell the XKB extension that we only want events
     about indicator state changes */
  XkbSelectEvents(display, XkbUseCoreKbd, XkbIndicatorStateNotifyMask,
		  XkbIndicatorStateNotifyMask);

  /* loop forever */
  while (1) {

    /* block until there's an event about a change in the indicator state */
    XNextEvent(display, &xkbevent.core);
    if (xkbevent.core.type == event_code + XkbEventCode) {
      if (xkbevent.any.xkb_type == XkbIndicatorStateNotify) {

	/* create a message telling which keys have change led state */
	buffer = 0;
	if (numlock_led ^ (xkbevent.indicators.state & NUM_LOCK_MASK))
	  buffer |= NUM_LOCK_MASK;
	if (capslock_led ^ (xkbevent.indicators.state & CAPS_LOCK_MASK))
	  buffer |= CAPS_LOCK_MASK;
	if (scrollock_led ^ (xkbevent.indicators.state & SCROLL_LOCK_MASK))
	  buffer |= SCROLL_LOCK_MASK;

	numlock_led   = xkbevent.indicators.state & NUM_LOCK_MASK;
	capslock_led  = xkbevent.indicators.state & CAPS_LOCK_MASK;
	scrollock_led = xkbevent.indicators.state & SCROLL_LOCK_MASK;

	/* tell the gui_thread about the new state change */
	if (writen(led_pipe[1], (char *)&buffer, 4) == -1)
	  g_error (_("write(): %s"), sys_errlist[MIN(errno, sys_nerr - 1)]);
      }
    }
  }
}
