#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) 2009 Sebastian Wiesner <basti.wiesner@gmx.net>

# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.


import os
import sys
import logging
from subprocess import Popen, PIPE, CalledProcessError, check_call
from glob import glob


SOURCEDIR = '../src'
PROJECT = 'tablet'
BUGADDR = ''

RC_FILE_EXTENSIONS = frozenset(['.rc', '.ui', '.kcfg'])
SOURCE_FILE_EXTENSIONS = frozenset(['.cpp', '.h', '.c'])

XGETTEXT_KEYWORDS = [
    'i18n:1', 'i18nc:1c,2', 'i18np:1,2', 'i18ncp:1c,2,3', 'tr2i18n:1',
    'I18N_NOOP:1', 'I18N_NOOP2:1c,2', 'aliasLocale', 'ki18n:1',
    'ki18nc:1c,2', 'ki18np:1,2', 'ki18ncp:1c,2,3']


def msgmerge(template, *catalogs):
    log = logging.getLogger('msgmerge')
    for catalog in catalogs:
        log.info(u'merging %s with %s', catalog, template)
        cmd = ['msgmerge', '--quiet', '--update', '--sort-output',
               '--backup=none', catalog, template]
        check_call(cmd)


def extractrc(filename):
    log = logging.getLogger('extractrc')
    log.info(u'extracting messages from rc file %s', filename)
    cmd = ['extractrc', filename]
    extractrc = Popen(cmd, stdout=PIPE)
    extracted = extractrc.communicate()[0]
    if extractrc.returncode != 0:
        raise CalledProcessError(extractrc.returncode, cmd)
    return extracted


def xgettext(source_files, package_name=None, bugs_address=None,
             outfile=None):
    log = logging.getLogger('xgettext')
    log.info(u'extracting messages')
    cmd = ['xgettext', '--sort-output', '--from-code=utf-8',
           '--c++', '--kde', '-ci18n']
    cmd.extend('-k{0}'.format(k) for k in XGETTEXT_KEYWORDS)
    if package_name:
        cmd.append('--package-name')
        cmd.append(package_name)
    if bugs_address:
        cmd.append('--msgid-bugs-address')
        cmd.append(bugs_address)
    if isinstance(outfile, unicode):
        outfile = outfile.encode(sys.getfilesystemencoding())
    cmd.extend(source_files)
    if isinstance(outfile, str):
        cmd.append('-o')
        cmd.append(outfile)
        check_call(cmd)
    elif not outfile:
        check_call(cmd)
    else:
        xgettext = Popen(cmd, stdout=PIPE)
        stdout = xgettext.communicate()[0]
        if xgettext.returncode != 0:
            raise CalledProcessError(xgettext.returncode, cmd)
        outfile.write(stdout)


def find_files_by_extensions(directory, extensions):
    for root, dirnames, filenames in os.walk(directory):
        for filename in (os.path.join(root, f) for f in filenames):
            if os.path.splitext(filename)[1] in extensions:
                yield filename


def find_catalogs(directory=None):
    if not directory:
        directory = os.getcwd()
        pattern = os.path.join(directory, '*.po')
        return glob(pattern)


class ColorfulFormatter(object):
    BROWN = u'\x1b[33m'
    DARKGREEN = u'\x1b[32m'
    RESET = u'\x1b[39;49;00m'

    TEMPLATE = (u'{0.BROWN}{1.name}{0.RESET}: '
                u'{0.DARKGREEN}{1.message}{0.RESET}')

    def format(self, record):
        params = vars(record)
        params['message'] = record.getMessage()
        return self.TEMPLATE.format(self, record)


def main():
    logging.getLogger().setLevel(logging.NOTSET)
    stdout = logging.StreamHandler(sys.stdout)
    stdout.setFormatter(ColorfulFormatter())
    stdout.setLevel(logging.INFO)
    logging.getLogger().addHandler(stdout)

    template = '{0}.pot'.format(PROJECT)
    rc_files = find_files_by_extensions(SOURCEDIR, RC_FILE_EXTENSIONS)
    rc_file_messsages = [extractrc(fn) for fn in rc_files]
    source_files = list(
        find_files_by_extensions(SOURCEDIR, SOURCE_FILE_EXTENSIONS))
    with open('rc.cpp', 'wb') as rc_cpp:
        rc_cpp.write(''.join(rc_file_messsages))
        source_files.append(rc_cpp.name)
    xgettext(source_files, package_name=PROJECT,
             bugs_address=BUGADDR, outfile=template)
    msgmerge(template, *find_catalogs())
    os.unlink('rc.cpp')


if __name__ == '__main__':
    main()


