/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "kz-bookmark-folder.h"

#include <string.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include "glib-utils.h"
#include "kz-marshalers.h"
#include "utils.h"
#include "kz-bookmark-file.h"
#include "kz-bookmarks-enum-types.h"

enum {
	INSERT_CHILD_SIGNAL,
	REMOVE_CHILD_SIGNAL,
	MOVE_CHILD_SIGNAL,
	CHILDREN_REORDERED_SIGNAL,
	GET_CHILDREN_SIGNAL,
	HAS_CHILDREN_SIGNAL,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_INTERVAL,
	PROP_CURRENT_POSITION,
	PROP_LOCK,
	PROP_AUTO_REFRESH,
	PROP_ENABLE_JAVASCRIPT
};

typedef struct _KzBookmarkFolderPrivate       KzBookmarkFolderPrivate;
struct _KzBookmarkFolderPrivate
{
	GList *children;
        guint current_position;
	gboolean folded;
	gboolean locked;
	gboolean auto_refresh;
	gboolean enable_javascript;
};
#define KZ_BOOKMARK_FOLDER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_BOOKMARK_FOLDER, KzBookmarkFolderPrivate))

static void dispose      (GObject         *object);
static void set_property (GObject         *object,
                          guint            prop_id,
                          const GValue    *value,
                          GParamSpec      *pspec);
static void get_property (GObject         *object,
                          guint            prop_id,
                          GValue          *value,
                          GParamSpec      *pspec);

static gint signals[LAST_SIGNAL] = {0};

G_DEFINE_TYPE(KzBookmarkFolder, kz_bookmark_folder, KZ_TYPE_BOOKMARK);

static void
kz_bookmark_folder_class_init (KzBookmarkFolderClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS(klass);

	object_class->dispose      = dispose;
	object_class->set_property = set_property;
	object_class->get_property = get_property;

	g_object_class_install_property(
		object_class,
		 PROP_CURRENT_POSITION,
		 g_param_spec_uint(
			 "current-position",
			 _("Current Position"),
			 _("Current position in history"),
			 0,
			 G_MAXUINT,
			 0,
			 G_PARAM_READWRITE));

	g_object_class_install_property(
		object_class,
		PROP_LOCK,
		g_param_spec_boolean(
			"lock",
			_("Lock state"),
			_("Lock state for tab"),
			FALSE,
			 G_PARAM_READWRITE));

	g_object_class_install_property(
		object_class,
		PROP_AUTO_REFRESH,
		g_param_spec_boolean(
			 "auto-refresh",
			 _("Auto Refresh state"),
			 _("Auto Refresh state for tab"),
			 FALSE,
			 G_PARAM_READWRITE));

	g_object_class_install_property(
		object_class,
		PROP_ENABLE_JAVASCRIPT,
		g_param_spec_boolean(
			 "enable-javascript",
			 _("Javascript state"),
			 _("Javascript state for tab"),
			 FALSE,
			 G_PARAM_READWRITE));

	signals[INSERT_CHILD_SIGNAL]
		= g_signal_new ("insert-child",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzBookmarkFolderClass, insert_child),
				NULL, NULL,
				_kz_marshal_VOID__OBJECT_OBJECT,
				G_TYPE_NONE, 2,
				KZ_TYPE_BOOKMARK, KZ_TYPE_BOOKMARK);

	signals[REMOVE_CHILD_SIGNAL]
		= g_signal_new ("remove-child",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzBookmarkFolderClass, remove_child),
				NULL, NULL,
				g_cclosure_marshal_VOID__OBJECT,
				G_TYPE_NONE, 1,
				KZ_TYPE_BOOKMARK);

	signals[MOVE_CHILD_SIGNAL]
		= g_signal_new ("move-child",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzBookmarkFolderClass, move_child),
				NULL, NULL,
				_kz_marshal_VOID__OBJECT_OBJECT,
				G_TYPE_NONE, 2,
				KZ_TYPE_BOOKMARK, KZ_TYPE_BOOKMARK);

	signals[CHILDREN_REORDERED_SIGNAL]
		= g_signal_new ("children-reordered",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (KzBookmarkFolderClass,
						 children_reordered),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE, 0);

	signals[GET_CHILDREN_SIGNAL]
		= g_signal_new ("get-children",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzBookmarkFolderClass, get_children),
				NULL, NULL,
				_kz_marshal_POINTER__VOID,
				G_TYPE_POINTER, 0);

	signals[HAS_CHILDREN_SIGNAL]
		= g_signal_new ("has-children",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzBookmarkFolderClass, has_children),
				NULL, NULL,
				_kz_marshal_BOOLEAN__VOID,
				G_TYPE_BOOLEAN, 0);

	g_type_class_add_private(object_class, sizeof(KzBookmarkFolderPrivate));
}

static void
kz_bookmark_folder_init (KzBookmarkFolder *folder)
{
	KzBookmarkFolderPrivate *priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);

	priv->children = NULL;
	priv->folded = TRUE;
	priv->current_position = 0;
	priv->locked = FALSE;
	priv->auto_refresh = FALSE;
	priv->enable_javascript = TRUE;
}

static void
dispose (GObject *object)
{
	kz_bookmark_folder_remove_all(KZ_BOOKMARK_FOLDER(object));

	if (G_OBJECT_CLASS (kz_bookmark_folder_parent_class)->dispose)
		G_OBJECT_CLASS (kz_bookmark_folder_parent_class)->dispose(object);
}

static void
set_property (GObject *object,
              guint prop_id,
              const GValue *value,
              GParamSpec *pspec)
{
	KzBookmarkFolderPrivate *priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(object);

	switch (prop_id) {
	case PROP_CURRENT_POSITION:
		priv->current_position = g_value_get_uint(value);
		break;
	case PROP_LOCK:
		priv->locked = g_value_get_boolean(value);
		break;
	case PROP_AUTO_REFRESH:
		priv->auto_refresh = g_value_get_boolean(value);
		break;
	case PROP_ENABLE_JAVASCRIPT:
		priv->enable_javascript = g_value_get_boolean(value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


static void
get_property (GObject *object,
              guint prop_id,
              GValue *value,
              GParamSpec *pspec)
{
	KzBookmarkFolderPrivate *priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(object);

	switch (prop_id) {
	case PROP_CURRENT_POSITION:
		g_value_set_uint(value, priv->current_position);
		break;
	case PROP_LOCK:
		g_value_set_boolean(value, priv->locked);
		break;
	case PROP_AUTO_REFRESH:
		g_value_set_boolean(value, priv->auto_refresh);
		break;
	case PROP_ENABLE_JAVASCRIPT:
		g_value_set_boolean(value, priv->enable_javascript);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


KzBookmark *
kz_bookmark_folder_new (const gchar *title)
{
	return KZ_BOOKMARK(g_object_new(KZ_TYPE_BOOKMARK_FOLDER,
                                        "title", title,
                                        NULL));
}

gboolean
kz_bookmark_folder_get_folded (KzBookmarkFolder *folder)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK_FOLDER(folder), FALSE);

	return KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->folded;
}

void
kz_bookmark_folder_set_folded (KzBookmarkFolder *folder, gboolean folded)
{
	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));

	KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->folded = folded;
}

void
kz_bookmark_folder_insert_before (KzBookmarkFolder *folder,
			   	  KzBookmark *child,
				  KzBookmark *sibling)
{
	KzBookmarkFolderPrivate *priv;
	GList *next = NULL;

	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));

	priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);
	if (sibling)
		next = g_list_find(priv->children, sibling);

	g_object_ref(child);
	priv->children = g_list_insert_before(priv->children, next, child);

        kz_bookmark_set_parent(child, folder);
	g_signal_emit(folder, signals[INSERT_CHILD_SIGNAL], 0,
		      child, sibling);
}


void
kz_bookmark_folder_append (KzBookmarkFolder *folder, KzBookmark *child)
{
	kz_bookmark_folder_insert_before(folder, child, NULL);
}

void
kz_bookmark_folder_prepend (KzBookmarkFolder *folder, KzBookmark *child)
{
	KzBookmarkFolderPrivate *priv;

	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));

	priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);

	g_object_ref(child);
	priv->children = g_list_prepend(priv->children, child);

        kz_bookmark_set_parent(child, folder);
	g_signal_emit(folder, signals[INSERT_CHILD_SIGNAL], 0,
		      child, NULL);
}

void
kz_bookmark_folder_remove (KzBookmarkFolder *folder, KzBookmark *child)
{
	KzBookmarkFolderPrivate *priv;

	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));

	priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);
	if (!g_list_find(priv->children, child))
		return;

	g_signal_emit(folder, signals[REMOVE_CHILD_SIGNAL], 0, child);
	priv->children = g_list_remove(priv->children, child);

	g_object_unref(child);
}


void
kz_bookmark_folder_remove_all (KzBookmarkFolder *folder)
{
	GList *children, *node, *prev;
	KzBookmarkFolderPrivate *priv;

	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));

	priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);
	children = g_list_copy(priv->children);

	node = g_list_last(children);
	while (node)
	{
		KzBookmark *child = node->data;
		prev = g_list_previous(node);
		kz_bookmark_folder_remove(folder, child);
		node = prev;
	}

	g_list_free(children);
}

GList *
kz_bookmark_folder_get_children (KzBookmarkFolder *folder)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK_FOLDER(folder), NULL);

	return g_list_copy(KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->children);
}

gboolean
kz_bookmark_folder_has_children (KzBookmarkFolder *folder)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK_FOLDER(folder), FALSE);

	return (KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->children != NULL);
}

/* return current activate tab position */
guint
kz_bookmark_folder_get_current_position (KzBookmarkFolder *folder)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK_FOLDER(folder), 0);

	return KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->current_position;
}
void
kz_bookmark_folder_set_current_position (KzBookmarkFolder *folder, guint pos)
{
	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));

	KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->current_position = pos;
	g_object_notify(G_OBJECT(folder), "current-position");
}

KzBookmark *
kz_bookmark_folder_get_current_bookmark (KzBookmarkFolder *folder)
{
	gpointer *current_bookmark;

	g_return_val_if_fail(KZ_IS_BOOKMARK_FOLDER(folder), NULL);

	current_bookmark = g_list_nth_data(KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->children,
					   kz_bookmark_folder_get_current_position(folder));

	return current_bookmark ? KZ_BOOKMARK(current_bookmark) : NULL;
}

gboolean
kz_bookmark_folder_get_lock (KzBookmarkFolder *folder)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK_FOLDER(folder), FALSE);

	return KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->locked;
}

void
kz_bookmark_folder_set_lock (KzBookmarkFolder *folder,
			     gboolean lock)
{
	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));
	
	KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->locked = lock;
	g_object_notify(G_OBJECT(folder), "lock");
}

gboolean
kz_bookmark_folder_get_auto_refresh (KzBookmarkFolder *folder)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK_FOLDER(folder), FALSE);

	return KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->auto_refresh;
}

void
kz_bookmark_folder_set_auto_refresh (KzBookmarkFolder *folder,
				     gboolean auto_refresh)
{
	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));
	
	KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->auto_refresh = auto_refresh;
	g_object_notify(G_OBJECT(folder), "auto-refresh");
}

gboolean
kz_bookmark_folder_get_javascript (KzBookmarkFolder *folder)
{
	g_return_val_if_fail(KZ_IS_BOOKMARK_FOLDER(folder), FALSE);

	return KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->enable_javascript;
}

void
kz_bookmark_folder_set_javascript (KzBookmarkFolder *folder,
				   gboolean javascript)
{
	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));
	
	KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder)->enable_javascript = javascript;
	g_object_notify(G_OBJECT(folder), "enable-javascript");
}

static gint 
compare_func (gconstpointer a, gconstpointer b)
{
	guint one = kz_bookmark_get_last_modified((KzBookmark*)a);
	guint two = kz_bookmark_get_last_modified((KzBookmark*)b);
	return two - one;
}


void
kz_bookmark_folder_sort (KzBookmarkFolder *folder, const gchar *type)
{
	KzBookmarkFolderPrivate *priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);

	g_return_if_fail(KZ_IS_BOOKMARK_FOLDER(folder));

	if (!type) return;

	/* FIXME! use hash table */
	priv->children = g_list_sort(priv->children, compare_func);

	g_signal_emit(folder, signals[CHILDREN_REORDERED_SIGNAL], 0);
}


void
kz_bookmark_folder_register_sort_func (const gchar *type, GCompareFunc *func)
{
	g_warning("kz_bookmark_folder_register_sort_func() is not implemented yet.");
}


KzBookmark *
kz_bookmark_folder_find_bookmark_from_uri (KzBookmarkFolder *folder, const gchar *key_uri)
{
	GList *node;
	KzBookmarkFolderPrivate *priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);

	g_return_val_if_fail(KZ_IS_BOOKMARK_FOLDER(folder), NULL);

	/* FIXME! Use hash? */
	for (node = priv->children; node; node = g_list_next(node))
	{
		KzBookmark *child = node->data;
		const gchar *uri = kz_bookmark_get_link(child);
		if (uri && key_uri && !g_strcmp(uri, key_uri))
			return child;

		if (!KZ_IS_BOOKMARK_FOLDER(child))
			continue;

		child = kz_bookmark_folder_find_bookmark_from_uri(KZ_BOOKMARK_FOLDER(child),
								  key_uri);
		if (child)
			return child;
	}

	return NULL;
}

void
kz_bookmark_folder_foreach_child (KzBookmarkFolder*folder,
				  GFunc       func,
				  gpointer    user_data)
{
	KzBookmarkFolderPrivate *priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);

	if (!priv->children)
		return;

        g_list_foreach(priv->children, func, user_data);
}

KzBookmark *
kz_bookmark_folder_get_nth_child (KzBookmarkFolder *folder, gint index)
{
	KzBookmarkFolderPrivate *priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);

	if (!priv->children)
		return NULL;

	if (g_list_length(priv->children) < index)
		return NULL;

	return KZ_BOOKMARK(g_list_nth_data(priv->children, index));
}

gint
kz_bookmark_folder_get_child_index (KzBookmarkFolder *folder, KzBookmark *child)
{
	KzBookmarkFolderPrivate *priv = KZ_BOOKMARK_FOLDER_GET_PRIVATE(folder);

	return priv->children ? g_list_index(priv->children, child) : -1;
}

