/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2007 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <glib/gi18n.h>

#define __USE_XOPEN
#include <time.h>
#include "kazehakase.h"
#include "kz-ie.h"
#include "kz-embed-prefs.h"
#include "kz-embed.h"
#include "kz-prompt-dialog.h"
#include "kz-proxy-item.h"
#include "gtk-utils.h"
#include "kz-missing.h"

typedef struct _KzIEPrivate	KzIEPrivate;
struct _KzIEPrivate
{
	glong current_progress;
	glong max_progress;
	gchar *location;
	gchar *title;
	gboolean lock;
};
#define KZ_IE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_IE, KzIEPrivate))

static GType           kz_type_ie = 0;
static GtkIEEmbedClass *kz_ie_parent_class;

GtkWidget  *kz_ie_new          (void);
static void kz_ie_class_init   (KzIEClass *klass);
static void kz_ie_iface_init   (KzEmbedIFace *iface);
static void kz_ie_init         (KzIE *ie);
static void kz_ie_dispose      (GObject      *object);

/* GtkIEEmbed signals */
static void	    kz_ie_embed_location	 (GtkIEEmbed   *ieembed,
						  const gchar  *location);
static void	    kz_ie_embed_title		 (GtkIEEmbed   *ieembed,
						  const gchar  *title);
static void	    kz_ie_embed_progress	 (GtkIEEmbed   *ieembed,
						  glong		current_progress,
						  glong		max_progress);
static void	    kz_ie_embed_net_start	 (GtkIEEmbed   *ieembed);
static void	    kz_ie_embed_net_stop	 (GtkIEEmbed   *ieembed);
static gboolean	    kz_ie_embed_dom_mouse_down   (GtkIEEmbed   *embed,
						  GtkIEEmbedDOMMouseEvent *event);
static gboolean	    kz_ie_embed_dom_mouse_up     (GtkIEEmbed   *embed,
						  GtkIEEmbedDOMMouseEvent *event);
static gboolean	    kz_ie_embed_dom_mouse_move   (GtkIEEmbed   *embed,
						  GtkIEEmbedDOMMouseEvent *event);
static gboolean	    kz_ie_embed_dom_mouse_click  (GtkIEEmbed   *embed,
						  GtkIEEmbedDOMMouseEvent *event);
static gboolean	    kz_ie_embed_new_window	 (GtkIEEmbed   *embed, GtkIEEmbed **newembed);
static gboolean	    kz_ie_embed_close_window	 (GtkIEEmbed   *embed);
static void	    kz_ie_embed_selection_changed(GtkIEEmbed   *embed);

/* embed iface */
static void         kz_ie_load_url         (KzEmbed      *kzembed,
					    const gchar  *url);
static const gchar *kz_ie_get_title        (KzEmbed      *kzembed);
static const gchar *kz_ie_get_location     (KzEmbed      *kzembed);
static gchar	   *kz_ie_ensure_title 	   (KzEmbed	 *kzembed);
static gdouble	    kz_ie_get_progress	   (KzEmbed	 *kzembed);
static void         kz_ie_reload           (KzEmbed      *kzembed,
					    KzEmbedReloadFlag flags);
static void         kz_ie_stop_load        (KzEmbed      *kzembed);
static gboolean	    kz_ie_is_loading	   (KzEmbed	 *kzembed);
static void         kz_ie_go_back          (KzEmbed      *kzembed);
static void         kz_ie_go_forward       (KzEmbed      *kzembed);
static gboolean     kz_ie_can_go_back      (KzEmbed      *kzembed);
static gboolean     kz_ie_can_go_forward   (KzEmbed      *kzembed);
static gboolean     kz_ie_can_cut_selection
					   (KzEmbed      *kzembed);
static gboolean     kz_ie_can_copy_selection
					   (KzEmbed      *kzembed);
static gboolean     kz_ie_can_paste        (KzEmbed      *kzembed);
static void         kz_ie_cut_selection    (KzEmbed      *kzembed);
static void         kz_ie_copy_selection   (KzEmbed      *kzembed);
static void         kz_ie_paste            (KzEmbed      *kzembed);
static void         kz_ie_select_all       (KzEmbed      *kzembed);
static void         kz_ie_print            (KzEmbed      *kzembed);
static void         kz_ie_print_preview    (KzEmbed      *kzembed);
static void	    kz_ie_set_text_size	   (KzEmbed      *kzembed, 
					    int           zoom, 
					    gboolean      reflow);
static gint	    kz_ie_get_text_size	   (KzEmbed      *kzembed);
static void	    kz_ie_set_encoding	   (KzEmbed      *kzembed,
					    const gchar  *encoding);
static void	    kz_ie_get_encoding	   (KzEmbed      *kzembed,
					    gchar       **encoding,
					    gboolean     *forced);
static guint	    kz_ie_get_last_modified
					   (KzEmbed	 *kzembed);
static gchar       *kz_ie_get_selection_string 
					   (KzEmbed      *kzembed);
static gboolean	    kz_ie_find		   (KzEmbed	 *kzembed,
					    const char   *keyword,
					    gboolean      backward);
static gboolean     kz_ie_get_lock         (KzEmbed      *kzembed);
static void         kz_ie_set_lock         (KzEmbed      *kzembed,
					    gboolean      lock);

/* KzEmbedPrefs interfaces */
/* no functions is implemented yet */
static void kz_ie_prefs_iface_init (KzEmbedPrefsIFace *iface);

void
kz_ie_register_type (GTypeModule *module)
{
	static const GTypeInfo kz_ie_info =
	{
		sizeof (KzIEClass),
		NULL,		/* base_init */
		NULL,		/* base_finalize */
		(GClassInitFunc) kz_ie_class_init,
		NULL,		/* class_finalize */
		NULL,		/* class_data */
		sizeof (KzIE),
		0,		/* n_preallocs */
		(GInstanceInitFunc) kz_ie_init,
	};

	const GInterfaceInfo kz_embed_info =
	{
		(GInterfaceInitFunc) kz_ie_iface_init,
		NULL,
		NULL
	};

	const GInterfaceInfo kz_embed_prefs_info =
	{
		(GInterfaceInitFunc) kz_ie_prefs_iface_init,
		NULL,
		NULL
	};

	kz_type_ie = g_type_module_register_type(module,
						 GTK_TYPE_IE_EMBED,
						 "KzIE",
						 &kz_ie_info,
						 (GTypeFlags)0);

	g_type_module_add_interface(module,
				    KZ_TYPE_IE,
				    KZ_TYPE_EMBED,
				    &kz_embed_info);

	g_type_module_add_interface(module,
				    KZ_TYPE_IE,
				    KZ_TYPE_EMBED_PREFS,
				    &kz_embed_prefs_info);
}

GType
kz_ie_get_type (void)
{
       return kz_type_ie;
}

static void
kz_ie_class_init (KzIEClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;
	GtkIEEmbedClass *ie_class;

	kz_ie_parent_class = GTK_IE_EMBED_CLASS(g_type_class_peek_parent(klass));
	object_class = G_OBJECT_CLASS(klass);
	widget_class = GTK_WIDGET_CLASS(klass);
	ie_class = GTK_IE_EMBED_CLASS(klass);

	object_class->dispose = kz_ie_dispose;

	ie_class->location          = kz_ie_embed_location;
	ie_class->title             = kz_ie_embed_title;
	ie_class->net_start         = kz_ie_embed_net_start;
	ie_class->net_stop          = kz_ie_embed_net_stop;
	ie_class->progress          = kz_ie_embed_progress;
	ie_class->dom_mouse_down    = kz_ie_embed_dom_mouse_down;
	ie_class->dom_mouse_move    = kz_ie_embed_dom_mouse_move;
	ie_class->dom_mouse_up      = kz_ie_embed_dom_mouse_up;
	ie_class->dom_mouse_click   = kz_ie_embed_dom_mouse_click;
	ie_class->new_window	    = kz_ie_embed_new_window;
	ie_class->close_window	    = kz_ie_embed_close_window;
	ie_class->selection_changed = kz_ie_embed_selection_changed;

	g_type_class_add_private(object_class, sizeof(KzIEPrivate));
}


static void
kz_ie_iface_init (KzEmbedIFace *iface)
{
	iface->load_url               = kz_ie_load_url;
	iface->view_source            = NULL;
	iface->is_loading             = kz_ie_is_loading;
	iface->get_title              = kz_ie_get_title;
	iface->get_location           = kz_ie_get_location;
	iface->ensure_title           = kz_ie_ensure_title;
	iface->get_link_message       = NULL;
	iface->get_progress           = kz_ie_get_progress;
	iface->can_cut_selection      = kz_ie_can_cut_selection;
	iface->can_copy_selection     = kz_ie_can_copy_selection;
	iface->can_paste              = kz_ie_can_paste;
	iface->cut_selection          = kz_ie_cut_selection;
	iface->copy_selection         = kz_ie_copy_selection;
	iface->paste                  = kz_ie_paste;
	iface->select_all             = kz_ie_select_all;
	iface->get_selection_string   = kz_ie_get_selection_string;
	iface->find                   = kz_ie_find;
	iface->incremental_search     = NULL;
	iface->selection_is_collapsed = NULL;
	iface->get_links              = NULL;
	iface->copy_page              = NULL;
	iface->shistory_copy          = NULL;
	iface->shistory_get_pos       = NULL;
	iface->shistory_get_nth       = NULL;
	iface->reload                 = kz_ie_reload;
	iface->stop_load              = kz_ie_stop_load;
	iface->go_back                = kz_ie_go_back;
	iface->go_forward             = kz_ie_go_forward;
	iface->can_go_back            = kz_ie_can_go_back;
	iface->can_go_forward         = kz_ie_can_go_forward;
	iface->can_go_nav_link        = NULL;
	iface->go_nav_link            = NULL;
	iface->append_nav_link        = NULL;
	iface->set_nav_link           = NULL;
	iface->set_nth_nav_link       = NULL;
	iface->get_nav_link           = NULL;
	iface->get_nth_nav_link       = NULL;
	iface->get_nav_links          = NULL;
	iface->go_history_index       = NULL;
	iface->do_command             = NULL;
	iface->can_do_command         = NULL;
	iface->get_lock               = kz_ie_get_lock;
	iface->set_lock               = kz_ie_set_lock;
	iface->get_body_text          = NULL;
#if 0
	iface->get_selection_source   = NULL;
#endif
	iface->set_encoding           = kz_ie_set_encoding;
	iface->get_encoding           = kz_ie_get_encoding;
	iface->print                  = kz_ie_print;
	iface->print_preview          = kz_ie_print_preview;
	iface->get_printer_list       = NULL;
	iface->create_thumbnail       = NULL;
	iface->save_with_content      = NULL;
	iface->set_text_into_textarea = NULL;
	iface->get_text_from_textarea = NULL;
	iface->zoom_set               = kz_ie_set_text_size;
	iface->zoom_get               = kz_ie_get_text_size;
	iface->set_text_size          = kz_ie_set_text_size;
	iface->get_text_size          = kz_ie_get_text_size;
	iface->get_html_with_contents = NULL;
	iface->set_history            = NULL;
	iface->get_history            = NULL;
	iface->get_last_modified      = kz_ie_get_last_modified;
	iface->fine_scroll            = NULL;
	iface->page_up                = NULL;
	iface->page_down              = NULL; 
	iface->get_allow_javascript   = NULL;
	iface->set_allow_javascript   = NULL; 
	iface->get_allow_images       = NULL;
	iface->set_allow_images       = NULL;
#if 0
	iface->set_edit_mode          = NULL;
	iface->set_view_mode          = NULL;
#endif

	iface->link_message           = NULL;
	iface->js_status              = NULL;
	iface->location               = NULL;
	iface->title                  = NULL;
	iface->progress               = NULL;
	iface->net_start              = NULL;
	iface->net_stop               = NULL;
	iface->new_window             = NULL;
	iface->open_uri               = NULL;
	iface->size_to                = NULL;
	iface->dom_key_down           = NULL;
	iface->dom_key_press          = NULL;
	iface->dom_key_up             = NULL;
	iface->dom_mouse_down         = NULL;
	iface->dom_mouse_up           = NULL;
	iface->dom_mouse_click        = NULL;
	iface->dom_mouse_dbl_click    = NULL;
	iface->dom_mouse_over         = NULL;
	iface->dom_mouse_out          = NULL;
	iface->security_change        = NULL;
	iface->status_change          = NULL;
}


static void
kz_ie_prefs_iface_init (KzEmbedPrefsIFace *iface)
{
	iface->init             = NULL; /* kz_ie_prefs_init;*/
	iface->get_font_list    = NULL; /* kz_ie_prefs_get_font_list; */
	iface->get_passwords    = NULL; /* kz_ie_prefs_get_passwords; */
	iface->remove_passwords = NULL; /* kz_ie_prefs_remove_passwords; */
}

static void
kz_ie_init (KzIE *ie)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE (ie);

	priv->location = NULL;
	priv->title = NULL;

	priv->current_progress = 0;
	priv->max_progress =0;

	priv->lock = FALSE;

	gtk_ie_embed_set_use_context_menu (GTK_IE_EMBED (ie), FALSE);
}

static void
kz_ie_dispose (GObject *object)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE(object);

	if (priv->location)
		g_free(priv->location);
	if (priv->title)
		g_free(priv->title);
	priv->location = NULL;
	priv->title = NULL;

	if (G_OBJECT_CLASS(kz_ie_parent_class)->dispose)
		G_OBJECT_CLASS(kz_ie_parent_class)->dispose(object);
}

GtkWidget *
kz_ie_new (void)
{
	return GTK_WIDGET(g_object_new(KZ_TYPE_IE,
				       "use-context-menu", FALSE,
				       NULL));
}

static void
kz_ie_embed_location (GtkIEEmbed *ieembed, const gchar  *location)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE(ieembed);

	if (priv->location)
		g_free(priv->location);
	priv->location = g_strdup(location);
	g_signal_emit_by_name(ieembed, "kz-location");
}

static void
kz_ie_embed_title (GtkIEEmbed *ieembed, const gchar *title)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE(ieembed);

	if (priv->title)
		g_free(priv->title);
	priv->title = g_strdup(title);
	g_signal_emit_by_name(ieembed, "kz-title");
}

static void
kz_ie_embed_progress (GtkIEEmbed *ieembed,
		      glong current_progress,
		      glong max_progress)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE(ieembed);
	priv->current_progress = current_progress;
	priv->max_progress = max_progress;

	g_signal_emit_by_name(ieembed, "kz-progress");
}

static void
kz_ie_embed_net_start (GtkIEEmbed *ieembed)
{
	g_signal_emit_by_name(ieembed, "kz-net-start");
}

static void
kz_ie_embed_net_stop (GtkIEEmbed *ieembed)
{
	if (GTK_IE_EMBED_CLASS(kz_ie_parent_class)->net_stop)
		GTK_IE_EMBED_CLASS(kz_ie_parent_class)->net_stop(ieembed);

	g_signal_emit_by_name(ieembed, "kz-net-stop");
}

static KzEmbedEventMouse *
create_kz_embed_event_mouse (GtkIEEmbedDOMMouseEvent *event)
{
	KzEmbedEventMouse *kzevent;

	kzevent = (KzEmbedEventMouse *) kz_embed_event_new(KZ_EMBED_EVENT_MOUSE);
	if (event->shift_key)
		kzevent->modifier |= KZ_SHIFT_KEY;
	if (event->control_key)
		kzevent->modifier |= KZ_CTRL_KEY;
	if (event->alt_key)
		kzevent->modifier |= KZ_ALT_KEY;
	if (event->meta_key)
		kzevent->modifier |= KZ_META_KEY;
	kzevent->x = event->x;
	kzevent->y = event->y;
	kzevent->button = event->button;
	kzevent->cinfo.context = KZ_CONTEXT_NONE;

	if (event->target) {
		const gchar *name;
		name = gtk_ie_embed_dom_event_target_get_name(event->target);
		if (!strcmp("a", name)) {
			const gchar *value;
			kzevent->cinfo.context = KZ_CONTEXT_LINK;
			value = gtk_ie_embed_dom_event_target_get_attribute_value(event->target,
									     	  "href");
			kzevent->cinfo.link = g_strdup(value);
		}
	}
	return kzevent;
}

static gboolean
emit_mouse_signal (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event,
		   const gchar *signal_name)
{
	KzEmbedEventMouse *kzevent;
	gint ret = FALSE;

	kzevent = create_kz_embed_event_mouse(event);
	g_signal_emit_by_name(embed, signal_name, kzevent, &ret);
	kz_embed_event_free((KzEmbedEvent *) kzevent);

	return ret;
}

static gboolean
kz_ie_embed_dom_mouse_down (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event)
{
	return emit_mouse_signal(embed, event, "kz-dom-mouse-down");
}

static gboolean
kz_ie_embed_dom_mouse_up (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event)
{
	return emit_mouse_signal(embed, event, "kz-dom-mouse-up");
}

static gboolean
kz_ie_embed_dom_mouse_move (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event)
{
	return emit_mouse_signal(embed, event, "kz-dom-mouse-move");
}

static gboolean
kz_ie_embed_dom_mouse_click (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event)
{
	return emit_mouse_signal(embed, event, "kz-dom-mouse-click");
}

static gboolean
kz_ie_embed_new_window (GtkIEEmbed *embed, GtkIEEmbed **newembed)
{
	g_signal_emit_by_name(embed, "kz-new-window", newembed);

	return newembed ? FALSE : TRUE;
}

static gboolean
kz_ie_embed_close_window (GtkIEEmbed *embed)
{
	gtk_widget_destroy(GTK_WIDGET(embed));
	return FALSE;
}

static void
kz_ie_embed_selection_changed (GtkIEEmbed *embed)
{
	g_signal_emit_by_name(embed, "kz-selection");
}

static void
kz_ie_load_url (KzEmbed *kzembed, const gchar  *url)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_load_url(GTK_IE_EMBED(kzembed), url);
}

static const gchar *
kz_ie_get_title (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), NULL);

	return KZ_IE_GET_PRIVATE(kzembed)->title;
}

static const gchar *
kz_ie_get_location (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), NULL);

	return KZ_IE_GET_PRIVATE(kzembed)->location;
}

static gchar *
kz_ie_ensure_title (KzEmbed *kzembed)
{
	const gchar *title;

	title = kz_ie_get_title(kzembed);
	if (title)
		return g_strdup(title);

	title = kz_ie_get_location(kzembed);
	if (title)
		return g_strdup(title);

	return g_strdup(_("No title"));
}

static gdouble
kz_ie_get_progress (KzEmbed *kzembed)
{
	gdouble progress;
	KzIEPrivate *priv;

	g_return_val_if_fail(KZ_IS_IE(kzembed), 0.0);

	priv = KZ_IE_GET_PRIVATE(kzembed);

	if (priv->max_progress <= 0 ||
	    priv->current_progress <= 0)
	{
		return 0.0;
	}

	progress = (gdouble) priv->current_progress
		/ (gdouble) priv->max_progress;

	return (progress < 1.0) ? progress : 1.0;
}

static void
kz_ie_reload (KzEmbed *kzembed, KzEmbedReloadFlag flags)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_reload(GTK_IE_EMBED(kzembed), GTK_IE_EMBED_RELOAD_NORMAL);
}


static void
kz_ie_stop_load (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_stop(GTK_IE_EMBED(kzembed));
}


static gboolean
kz_ie_is_loading (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_is_loading(GTK_IE_EMBED(kzembed));
}


static void
kz_ie_go_back (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_go_back(GTK_IE_EMBED(kzembed));
}


static void
kz_ie_go_forward (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_go_forward(GTK_IE_EMBED(kzembed));
}


static gboolean
kz_ie_can_go_back (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_go_back(GTK_IE_EMBED(kzembed));
}


static gboolean
kz_ie_can_go_forward (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_go_forward(GTK_IE_EMBED(kzembed));
}

static gboolean
kz_ie_can_cut_selection (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_cut_clipboard(GTK_IE_EMBED(kzembed));
}

static gboolean
kz_ie_can_copy_selection (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_copy_clipboard(GTK_IE_EMBED(kzembed));
}

static gboolean
kz_ie_can_paste (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_paste_clipboard(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_cut_selection (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_cut_clipboard(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_copy_selection (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_copy_clipboard(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_paste (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_paste_clipboard(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_select_all (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_select_all(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_print (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_print(GTK_IE_EMBED(kzembed));
}


static void
kz_ie_print_preview (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_print_preview(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_set_text_size (KzEmbed *kzembed, int zoom, gboolean reflow)
{
	GtkIEEmbedFontSize size;

	g_return_if_fail(KZ_IS_IE(kzembed));

	if (zoom > 400)
		size = GTK_IE_EMBED_FONT_LARGEST;
	else if (zoom > 200)
		size = GTK_IE_EMBED_FONT_LARGE;
	else if (zoom > 80)
		size = GTK_IE_EMBED_FONT_MEDIUM;
	else if (zoom > 50)
		size = GTK_IE_EMBED_FONT_SMALL;
	else 
		size = GTK_IE_EMBED_FONT_SMALLEST;
		
	gtk_ie_embed_set_font_size(GTK_IE_EMBED(kzembed), size);
}

static int
kz_ie_get_text_size (KzEmbed *kzembed)
{
	GtkIEEmbedFontSize size;
	gint ret_size;

	g_return_val_if_fail(KZ_IS_IE(kzembed), 100);

	size = gtk_ie_embed_get_font_size(GTK_IE_EMBED(kzembed));
	switch (size)
	{
	case GTK_IE_EMBED_FONT_LARGEST:
		ret_size = 400;
		break;
	case GTK_IE_EMBED_FONT_LARGE:
		ret_size = 200;
		break;
	case GTK_IE_EMBED_FONT_MEDIUM:
		ret_size = 80;
		break;
	case GTK_IE_EMBED_FONT_SMALL:
		ret_size = 50;
		break;
	case GTK_IE_EMBED_FONT_SMALLEST:
		ret_size = 20;
		break;
	default:
		ret_size = 100;
		break;
	}

	return ret_size;
}

static void
kz_ie_set_encoding (KzEmbed *kzembed, const gchar *encoding)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_set_charset(GTK_IE_EMBED(kzembed), encoding);
}

static void
kz_ie_get_encoding (KzEmbed *kzembed, gchar **encoding, gboolean *forced)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	*encoding = gtk_ie_embed_get_charset(GTK_IE_EMBED(kzembed));
}

static guint
kz_ie_get_last_modified (KzEmbed *kzembed)
{
	gchar *string;
	struct tm t = {0};

	g_return_val_if_fail(KZ_IS_IE(kzembed), 0);

	string = gtk_ie_embed_get_last_modified(GTK_IE_EMBED(kzembed));
	strptime(string, "%m/%d/%C %H:%M:%S", &t);
	g_free (string);

	return (guint) mktime(&t);
}

static gchar *
kz_ie_get_selection_string (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), NULL);

	return gtk_ie_embed_get_selected_text(GTK_IE_EMBED(kzembed));
}

static gboolean
kz_ie_find (KzEmbed *kzembed, const char *keyword, gboolean backward)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_find_string(GTK_IE_EMBED(kzembed), keyword, !backward, FALSE);
}

static gboolean
kz_ie_get_lock (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE (kzembed);
	return priv->lock;
}


static void
kz_ie_set_lock (KzEmbed *kzembed, gboolean lock)
{
	g_return_if_fail(KZ_IS_IE(kzembed));
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE (kzembed);
	priv->lock = lock;
}

