{ -- -*-Haskell-*-

{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module Parser where

import Char
import Language
import Debug.Trace
import System.IO.Unsafe
import Lib
import Module(importVersion)
import IO
import Lexer

}

%name mkparse TopFile
%name mkparseprog Program

%tokentype { Token }

%monad { P } { thenP } { returnP }

%lexer { lexer } { TokenEOF }


%token 
      int_lit         { TokenInt $$ }
      bool_lit        { TokenBool $$ }
      real_lit        { TokenReal $$ }
      name            { TokenName $$ }
      string          { TokenString $$ }
      char_lit        { TokenChar $$ }
      metavar         { TokenMetaVar $$ }
      '='             { TokenEq }
      '+'             { TokenPlus }
      '-'             { TokenMinus }
      '*'             { TokenTimes }
      '/'             { TokenDiv }
      power           { TokenPower }
      '%'             { TokenMod }
      '&'             { TokenAnd }
      '|'             { TokenOr }
      andbool         { TokenAndBool }
      orbool          { TokenOrBool }
      '^'             { TokenXOR }
      shleft          { TokenShLeft }
      shright         { TokenShRight }
      inc             { TokenInc }
      incby           { TokenIncBy }
      dec             { TokenDec }
      decby           { TokenDecBy }
      multby          { TokenMultBy }
      divby           { TokenDivBy }
      inteq           { TokenIntEq }
      intne           { TokenIntNE }
      le              { TokenLE }
      ge              { TokenGE }
--      app             { TokenApp }
      '('             { TokenOB }
      ')'             { TokenCB }
      '{'             { TokenOCB }
      '}'             { TokenCCB }
      '['             { TokenOSB }
      ']'             { TokenCSB }
      '<'             { TokenLT }
      '>'             { TokenGT }
      '!'             { TokenNot }
      '@'             { TokenAt }
      atbrack         { TokenAtBracket }
      '`'             { TokenBacktick }
--      ':'             { TokenColon }
      colons          { TokenColons }
      ';'             { TokenSemiColon }
      ','             { TokenComma }
--      '#'             { TokenHash }
      '.'             { TokenDot }
      dots             { TokenDots }
--      assign          { TokenAssign }
      arrow           { TokenArrow }

      data            { TokenData }
      abstract        { TokenAbstract }
      type            { TokenType }
      var             { TokenVar }
      int             { TokenIntType }
      char            { TokenCharType }
      bool            { TokenBoolType }
      real            { TokenRealType }
      stringtype      { TokenStringType }
      file            { TokenFileType }
      pointer         { TokenPointerType }
      void            { TokenVoidType }
      return          { TokenReturn }
      while           { TokenWhile }
      do              { TokenDo }
      repeat          { TokenRepeat }
      for             { TokenFor }
      try             { TokenTry }
      catch           { TokenCatch }
      finally         { TokenFinally }
      throw           { TokenThrow }
      exception       { TokenException }
      break           { TokenBreak }
      pass            { TokenPass }
--      to              { TokenTo }
      in              { TokenIn }
      case            { TokenCase }
      of              { TokenOf }
      if              { TokenIf }
      else            { TokenElse }
--      end             { TokenEnd }
--      size            { TokenSize }
      print           { TokenPrint }
--      inputnum        { TokenInputNum }
--      inputstr        { TokenInputStr }
      include         { TokenInclude }
      import          { TokenImport }
      extern          { TokenExtern }
      lifted          { TokenLifted }
      extdata         { TokenExtData }
      exttype         { TokenExtType }
      docstring       { TokenDocstring }
      imported        { TokenImported }
      link            { TokenLink }
      test            { TokenTest }
      vm              { TokenVM }
      program         { TokenProgram }
      shebang         { TokenShebang }
      module          { TokenModule }
      webapp          { TokenWebapp }
      webprog         { TokenWebprog }
      cinclude        { TokenCInclude }
      foreign         { TokenForeign }
      globals         { TokenGlobal }
      lambda          { TokenLambda }
      public          { TokenPublic }
      private         { TokenPrivate }
      pure            { TokenPure }
      default         { TokenDefault }

%nonassoc NONE
%nonassoc assign
%nonassoc print
%nonassoc else
%nonassoc return
%left ';'
%nonassoc LENGTH
%left '|' orbool
%left '^'
%left '&' andbool
%left inteq intne
%left '<' '>' le ge
%left shleft shright
%left '+' '-'
%left '*' '/' '%' power
%left NEG
%left '!' 
%left '(' atbrack
%left '.'
%left '[' 
%left '`'
%nonassoc GUARD
%nonassoc RAPP

%%

TopFile :: { ParseResult }
        : intro name ';' Program { PR $1 $2 $4 }
        | shebang Module Program { PR Shebang $2 $3 }
--(\s fn mod l a -> returnP a s fn $2 l) $ PR $1 $2 $4 } 
        -- | error {% reportError "badly formed input file" }

intro :: { InputType }
      : program { Program }
      | module { Module }
      | webapp { Webapp }
      | webprog { Webprog }
   -- TODO: Make this more general; allow user specified preambles?
   -- On getting preamble, read in appropriate startup code. ($1.tdph)
      -- | error {% reportError "expected 'program', 'module' or 'webapp'" }

--Imports :: { [RawDecl] }
--       : ImportMod Imports { $1 ++ $2 }
--       | { [] }

-- ImportMod :: { [RawDecl] }

Program :: { [RawDecl] }
        : Binding Program { $1:$2 }
	| Binding { [$1] }
	| globals '{' Globals '}' Program { $3 ++ $5 }
	| globals '{' Globals '}' { $3 }
	| foreign string '{' Foreigns '}' Program { (mkForeign $2 $4) ++ $6 }
	| foreign string '{' Foreigns '}' { (mkForeign $2 $4) }
	| include Module '(' string ')' ';' Program Libdirs {%
	     let rest = $7 in
	     let pt = unsafePerformIO (readFile $4) in
		case (mkparseprog pt $4 $8 $2 1) of
		   Success x -> returnP (x ++ rest)
		   Failure err file ln -> failP err
	  }
        | Import Module ModulePath ';' Program Libdirs {%
             let rest = $5 in
             let mf = $3 ++ ".ki" in
             let findmod = unsafePerformIO (findFile $6 mf) in
                case findmod of
		   Nothing -> returnP $ (SearchImport $3):rest
--		   Nothing -> failP $ "Can't find module " ++ $3
		   Just pt -> case (mkparseprog pt mf $6 (UN $3) 1) of
				 Success x -> returnP ((exportall $1 x) ++ (SearchImport $3):(Imp $3):rest)
				 Failure err file ln -> failP err
          }
	| error {% reportError "Expected binding" }

Import: import public { True }
     | import private { False }
     | import { False } -- default private


ModulePath : name { let (UN n) = $1 in n }
	   | name '.' ModulePath { let (UN n) = $1 in
				     n ++ "/" ++ $3 }
	   | error {% reportError "Expected module path" }

Name :: { Name }
     : name { $1 }
     | name colons Name { NS $1 $3 }
     | module colons Name Module { NS $4 $3 }

Binding :: { RawDecl }
	: DocString FOpts Type name '(' ArgList ')' File Line Module FBody
--          { FB ($6,$7,$2, mkType $4 $1, [], Defined (mkBody $6 $7 [] $4 $9)) }
          { FB ($8,$9,NS $10 $4, mkType $6 $3, Export:$2, Defined (mkBody $8 $9 $6 $11)) $1 }
	| DocString FOpts Type name File Line Module FBody
          { FB ($5,$6,NS $7 $4, mkType [] $3, NoArgs:Export:$2, Defined (mkBody $5 $6 [] $8)) $1 }
--          { FB ($3,$4,$2, mkType [] $1, [NoArgs], Defined (mkBody $3 $4 [] [] $6)) }
        | extern KIVersion FOpts Type Name '(' ArgTypeList ')' ';' File Line
          { FB ($10,$11,$5, mkType' $7 $4, $3, Unbound) "" }
        | extern KIVersion FOpts Type Name ';' File Line
          { FB ($7,$8,$5, Fn [] [] $4, $3, Unbound) "" }
        | lifted string { FMN $2 }
        | cinclude string ';' { CInc $2 }
	| imported string ';' { Imp $2 }
	| link string ';' { Link $2 }
	| type name DParams '=' Type File Line Module ';' { TSyn ($6,$7,$2,$3,$5,True) }
	| exttype KIVersion name DParams '=' Type File Line ';' 
            { TSyn ($7,$8,$3,$4,$6,False) }
--	    { DDecl $6 $7 (NS $8 $2) $3 (mkRet (NS $8 $2) $3 $5 True) }
	| extdata KIVersion DOpts Name DParams '=' ExtDataDecl File Line ';'
	    { DDecl $8 $9 $3 $4 $5 (mkRet $4 $5 $7 False) "" }
	| Data { $1 }
--	| abstract Name DParams ';' { ADecl $2 $3 }
	--| error {% reportError "Invalid binding" }

Data :: { RawDecl }
     : DocString DOpts data name DParams '=' DataDecl File Line Module ';' 
--         { DDecl $8 $9 (DExport:$2) $4 $5 (mkRet $4 $5 $7 True) $1 }
         { DDecl $8 $9 (DExport:$2) (NS $10 $4) $5 (mkRet $4 $5 $7 True) $1 }
    | DocString DOpts data name DParams '(' CTypeList ')' File Line Module ';'
         { let condecl = Con (NS $11 $4) (Fn [] (fst $7) (Prim Void))
	                     (snd $7) True
--                in DDecl $9 $10 (DExport:$2) $4 $5 (mkRet $4 $5 [condecl] True) $1 
                in DDecl $9 $10 (DExport:$2) (NS $11 $4) $5 (mkRet $4 $5 [condecl] True) $1 
	    }

KIVersion :: { Int }
          : int_lit {% if $1 /= importVersion 
                        then reportError "Incorrect .ki format"
                        else returnP $1
                    }

Foreigns :: { [Foreign] }
        : ForeignDecl ';' Foreigns { $1:$3 }
	| ForeignDecl ';' { [$1] }

ForeignDecl :: { Foreign }
          : DocString FOpts Type name 
	       '(' ArgList ')' File Line '=' name Module 
          { ForeignDecl $8 $9 $2 $3 
	                  (map (\ (x,y,z,w) -> (x,y)) $6) 
	                  (NS $12 $4) $11 $1 }


FOpts:: { [FOpt] }
     : public FOpts{ Public:$2 }
     | private FOpts { $2 }
     | pure FOpts { Pure:$2 }
     | default FOpts { DefaultDef:$2 }
     | { [] }

DOpts :: { [DOpt] }
     : abstract DOpts { DAbstract:$2 }
     | public DOpts { DPublic:$2 }
     | private DOpts { $2 }
     | { [] }

DocString :: { String }
         : string { $1 }
	 | { "" }

FBody :: { Raw }
     : '{' Expr '}' { $2 }
     | '{' '}' File Line { RNoop $3 $4 }
     | File Line Rvalue ';' { RReturn $1 $2 $3 }
--     | File Line '=' TypedExpr ';' { RReturn $1 $2 $4 }
--     | File Line Guarded File Line { RReturn $1 $2 (mkIf $1 $2 $3) }

Rvalue :: { Raw }
      : '=' TypedExpr { $2 }
      | File Line GuardRval %prec GUARD{ mkIf $1 $2 $3 }

GuardRval :: { [(Raw, Raw)] }
         : '|' Guard %prec GUARD { [$2] }
         | '|' Guard GuardRval %prec GUARD { $2:$3 }

Guard :: { (Raw, Raw) }
     : TypedExpr '=' TypedExpr %prec GUARD { ($1,$3) }
     | default File Line '=' TypedExpr %prec GUARD 
           { (RConst $2 $3 (Bo True), $5) }
						      

Globals :: { [RawDecl] }
       : { [] }
       | Global ';' Globals { $1:$3 }
       -- | error {% reportError "invalid global declaration" }

Global :: { RawDecl }
       : Type File Line name { GlobDecl $2 $3 ($4,$1) }
      -- | error {% reportError "invalid global declaration" }

DataDecl :: { [ConDecl] }
	: CDecl { [$1] }
        | CDecl '|' DataDecl { $1:$3 }

ExtDataDecl :: { [ConDecl] }
	   : ExtCDecl { [$1] }
           | ExtCDecl '|' ExtDataDecl { $1:$3 }
	   | { [] }
	-- | error {% reportError "invalid data declaration" }

DParams :: { [Type] }
       : { [] } -- No parameters
       | '<' DParamList '>' { $2 }
       -- | error {% reportError "invalid parameter list" }

DParamList :: { [Type] }
          : name { [TyVar $1] }
	  | name ',' DParamList { (TyVar $1):$3 }
	  -- | error {% reportError "invalid parameter list" }

-- We'll replace the "Prim Void" with the real type later.
CDecl :: { ConDecl }
     : name '(' CTypeList ')' Module 
--	 { Con $1 (Fn [] (fst $3) (Prim Void)) (snd $3) True }
	 { Con (NS $5 $1) (Fn [] (fst $3) (Prim Void)) (snd $3) True }
     | name Module 
--       { Con $1 (Fn [] [] (Prim Void)) [] True }
       { Con (NS $2 $1) (Fn [] [] (Prim Void)) [] True }
     | error {% reportError "Invalid constructor declaration" }

ExtCDecl :: { ConDecl }
        : Name '(' CTypeList ')' { Con $1 (Fn [] (fst $3) (Prim Void)) (snd $3) True }
        | Name { Con $1 (Fn [] [] (Prim Void)) [] True }

CTypeList :: { ([Type],[Name]) } 
CTypeList : Type { ([$1],[None]) }
	  | Type name { ([$1],[$2]) }
          | Type ',' CTypeList { (($1:(fst $3)), (None:(snd $3))) }
          | Type name ',' CTypeList { (($1:(fst $4)),($2:(snd $4))) }
	  | { ([], []) }
	  | error {% reportError "Invalid constructor type" }

PrimType :: { Type }
    : int { Prim Number }
    | char { Prim Character }
    | bool { Prim Boolean }
    | real { Prim RealNum }
    | stringtype { Prim StringType }
    | file { Prim File }
    | pointer { Prim Pointer } -- Foreign pointer
    | void { Prim Void }
    | exception { Prim Exception }
    -- | error {% reportError "invalid primitive type" }

Type :: { Type }
    : PrimType { $1 }
    | '[' Type ']' { Array $2 }
    | Type '(' TypeList ')' { Fn [] $3 $1 } -- Higher order type
    | Name { processTyName $1 }
--    | '<' Name '>' { TyVar $2 } -- Polymorphic type variable
--    | Name { Syn $1 } -- Type synonym
    | Name '<' TypeList '>' { User $1 $3 } -- User type application
--    | var { UnknownType }
    | TupleType { $1 }
    --| error {% reportError "Invalid type" }

TupleType :: { Type }
         : '(' Type ',' Type ')' { User (UN "Pair") [$2,$4] }
         | '(' Type ',' Type ',' Type ')' { User (UN "Triple") [$2,$4,$6] }
         | '(' Type ',' Type ',' Type ',' Type ')' 
	   { User (UN "Tuple4") [$2,$4,$6,$8] }
         | '(' Type ',' Type ',' Type ',' Type ',' Type ')' 
	   { User (UN "Tuple5") [$2,$4,$6,$8,$10] }
         | '(' Type ',' Type ',' Type ',' Type ',' Type ',' Type ')' 
	   { User (UN "Tuple6") [$2,$4,$6,$8,$10,$12] }

--Index: TypedExpr { [$1] }
--     | TypedExpr ',' Index { $1:$3 }

ArgList :: { [(Name, Type, Maybe Raw, ArgType)] }
        : Var Type name DefaultArg { [($3,$2,$4,$1)] }
	| Var Type name DefaultArg ',' ArgList { ($3,$2,$4,$1):$6 }
	| { [] }
	| error {% reportError "Invalid argument list" }

Var :: { ArgType }
    : var { Var }
    | { Copy }

ArgTypeList :: { [(Type, Maybe Raw)] }
            : Type DefaultArg { [($1,$2)] }
	    | Type DefaultArg ',' ArgTypeList { ($1,$2):$4 }
	    | { [] }

DefaultArg :: { Maybe Raw }
          : { Nothing }
	  | '=' TypedExpr { Just $2 }

TypeList :: { [Type] }
         : Type { [$1] }
         | Type ',' TypeList { $1:$3 }
	 | { [] }
	 -- | error {% reportError "invalid type list" }

Expr :: { Raw }
     : VoidExpr ';' { $1 }
     | ControlExp { $1 }
     | ControlExp Expr File Line { RSeq $3 $4 $1 $2 }
     | VoidExpr ';' Expr File Line { RSeq $4 $5 $1 $3 }
     | BindExp { $1 }
     --| '{' Expr '}' { $2 }
     -- | error {% reportError "invalid expression" }

TypedExpr :: { Raw }
         : Const { $1 }
         | '(' TypedExpr ')' { $2 }
	 | MathExp { $1 }
	 | if '(' TypedExpr ')' File Line TypedExpr else TypedExpr
	       { RIf $5 $6 $3 $7 $9 }
--	 | Name '(' ExprList File Line ')' { RApply $4 $5 (RQVar $4 $5 $1) $3 }
	 | lambda '(' LamArgList ')' File Line BlockExp { mkClosure $5 $6 $3 $7 }
	 | lambda '(' LamArgList ')' File Line arrow '{' TypedExpr '}'
	      { mkClosure $5 $6 $3 (RReturn $5 $6 $9) }
	 | exception '(' File Line TypedExpr ',' TypedExpr ')' { RExcept $3 $4 $5 $7 }
	 | vm File Line { RVMPtr $2 $3 }
--	 | foreign Type name '(' ExprList File Line ')'
--	     { RForeign $6 $7 $2 $3 $5 }
--	 | size '(' TypedExpr File Line ')' %prec LENGTH { RLength $4 $5 $3 }
--	 | inputnum File Line { RInputNum $2 $3}
--	 | inputstr File Line { RInputStr $2 $3 }
         | CoerceExp { $1 }
	 | '[' ArrayEntries File Line ']' { RArrayInit $3 $4 $2 }
	 | TypedCaseExp { $1 }
	 | throw TypedExpr File Line { RThrow $3 $4 $2 }
	 | TypedSugar { $1 }
	 | TypedExpr '`' Name '`' File Line TypedExpr 
	   {
	      RApply $5 $6 (RQVar $5 $6 $3) [$1,$7]
	   }
	 | VarExpr { $1 }
	 | metavar File Line { RMetavar $2 $3 $1 }
--	 | error {% reportError "invalid expression" }

LamArgList :: { [(Name,Type)] }
           : Type name { [($2,$1)] }
	   | Type name ',' LamArgList { ($2,$1):$4 }
	   | name { [($1,UnknownType)] }
	   | name ',' LamArgList { ($1,UnknownType):$3 }
	   | { [] }
	-- | error {% reportError "invalid argument list" }

VarExpr :: { Raw }
        : Name {% getLineNo `thenP` \ln -> 
		   getFileName `thenP` \fn ->
		   returnP (RVar fn ln $1) }
	| TypedExpr '[' TypedExpr ']' File Line { RIndex $5 $6 $1 $3 }
	| VarExpr '.' File Line Name { RField $3 $4 $1 $5 }
        | '@' Name File Line { RQVar $3 $4 $2 }
	| TypedExpr '(' ExprList File Line ')' { RApply $4 $5 (qhack $1) $3 }
	| TypedExpr atbrack ExprList File Line ')' 
               { RPartial $4 $5 (qhack $1) $3 }
	| '@' Name '(' ExprList File Line ')' { RApply $5 $6 (RQVar $5 $6 $2) $4 }


TypedSugar :: { Raw }
          : -- Tuples first 
	    '(' TypedExpr ',' TypedExpr ')' File Line 
	      { RApply $6 $7 (RQVar $6 $7 (UN "Pair")) [$2,$4] }
          | '(' TypedExpr ',' TypedExpr ',' TypedExpr ')' File Line 
	      { RApply $8 $9 (RQVar $8 $9 (UN "Triple")) [$2,$4,$6] }
          | '(' TypedExpr ',' TypedExpr ',' TypedExpr ',' TypedExpr ')' 
	    File Line 
	      { RApply $10 $11 (RQVar $10 $11 (UN "Tuple4")) [$2,$4,$6,$8] }
          | '(' TypedExpr ',' TypedExpr ',' TypedExpr ',' TypedExpr ','
	      TypedExpr ')' 
	    File Line 
      { RApply $12 $13 (RQVar $12 $13 (UN "Tuple5")) [$2,$4,$6,$8,$10] }
          | '(' TypedExpr ',' TypedExpr ',' TypedExpr ',' TypedExpr ','
	      TypedExpr ',' TypedExpr ')' 
	    File Line 
      { RApply $14 $15 (RQVar $14 $15 (UN "Tuple6")) [$2,$4,$6,$8,$10,$12] }
    -- Array shorthands
          | '[' TypedExpr dots TypedExpr File Line ']'
      { RApply $5 $6 (RQVar $5 $6 (UN "range")) [$2,$4,RConst $5 $6 (Num 1)] }
          | '[' TypedExpr ',' TypedExpr dots TypedExpr File Line ']'
      { RApply $7 $8 (RQVar $7 $8 (UN "range"))
	  [$2,$6,cfold $ RInfix $7 $8 Minus $4 $2] }

ArrayEntries :: { [Raw] }
            : { [] }
	    | TypedExpr { [$1] }
	    | TypedExpr ',' ArrayEntries { $1:$3 }
	    -- | error {% reportError "invalid array" }

VoidExpr :: { Raw }
        : AssignExp { $1 }
	| MathShorthand { $1 }
	| throw '(' TypedExpr ')' File Line { RThrow $5 $6 $3 }
        | return TypedExpr File Line { RReturn $3 $4 $2 }
	| return File Line { RVoidReturn $2 $3 }
	| IOExp { $1 }
--        | VarExpr '(' ExprList File Line ')' { RApply $4 $5 (qhack $1) $3 }
	| ApplyExp { $1 }
        | Name atbrack ExprList File Line ')' 
             { RPartial $4 $5 (RQVar $4 $5 $1) $3 }
--	| foreign Type name '(' ExprList ')' File Line 
--	    { RForeign $7 $8 $2 $3 $5 }
	| break File Line { RBreak $2 $3 }
	| pass File Line { RNoop $2 $3 }
	| File Line { RNoop $1 $2 }
	| metavar File Line { RMetavar $2 $3 $1 }
	| error {% reportError "Invalid void expression" }

--Func: Name File Line { RQVar $2 $3 $1 }
--    | VarExpr { $1 }

ApplyExp :: { Raw }
         : Name '(' ExprList File Line ')' 
	       { RApply $4 $5 (RQVar $4 $5 $1) $3 }
--	 | Name '(' VarList ')' '=' File Line TypedExpr 
--	       { mkPattAssign $6 $7 ($1,$3) $8 }

ExprList :: { [Raw] }
         : { [] }
	 | NonEmptyExprList { $1 }

NonEmptyExprList :: { [Raw] }
                 : TypedExpr { [$1] }
                 | TypedExpr ',' NonEmptyExprList { $1:$3 }
		 --| error {% reportError "invalid expression list" }

Const :: { Raw }
      : int_lit File Line { RConst $2 $3 (Num $1) }
      | char_lit File Line { RConst $2 $3 (Ch $1) }
      | bool_lit File Line { RConst $2 $3 (Bo $1) }
      | real_lit File Line { RConst $2 $3 (Re $1) }
      | string File Line { RConst $2 $3 (Str $1) }
      --| error {% reportError "invalid constant" }

MathExp :: { Raw }
        : TypedExpr '+' File Line TypedExpr { cfold $ RInfix $3 $4 Plus $1 $5 }
	| TypedExpr '-' File Line TypedExpr { cfold $ RInfix $3 $4 Minus $1 $5 }
	| TypedExpr '*' File Line TypedExpr { cfold $ RInfix $3 $4 Times $1 $5 }
	| TypedExpr '/' File Line TypedExpr { cfold $ RInfix $3 $4 Divide $1 $5 }
	| TypedExpr '%' File Line TypedExpr { cfold $ RInfix $3 $4 Modulo $1 $5 }
	| TypedExpr power File Line TypedExpr { cfold $ RInfix $3 $4 Power $1 $5 }
	| TypedExpr inteq File Line TypedExpr { cfold $ RInfix $3 $4 Equal $1 $5 }
	| TypedExpr intne File Line TypedExpr { cfold $ RInfix $3 $4 NEqual $1 $5 }
	| TypedExpr '<' File Line TypedExpr { cfold $ RInfix $3 $4 OpLT $1 $5 }
	| TypedExpr '>' File Line TypedExpr { cfold $ RInfix $3 $4 OpGT $1 $5 }
	| TypedExpr le File Line TypedExpr { cfold $ RInfix $3 $4 OpLE $1 $5 }
	| TypedExpr ge File Line TypedExpr { cfold $ RInfix $3 $4 OpGE $1 $5 }
	| TypedExpr '&' File Line TypedExpr { cfold $ RInfix $3 $4 OpAnd $1 $5 }
	| TypedExpr '|' File Line TypedExpr { cfold $ RInfix $3 $4 OpOr $1 $5 }
	| TypedExpr andbool File Line TypedExpr 
	        { cfold $ RInfix $3 $4 OpAndBool $1 $5 }
	| TypedExpr orbool File Line TypedExpr { cfold $ RInfix $3 $4 OpOrBool $1 $5 }
	| TypedExpr '^' File Line TypedExpr { cfold $ RInfix $3 $4 OpXOR $1 $5 }
	| TypedExpr shleft File Line TypedExpr { cfold $ RInfix $3 $4 OpShLeft $1 $5 }
	| TypedExpr shright File Line TypedExpr 
	       { cfold $ RInfix $3 $4 OpShRight $1 $5 }
	| '!' File Line TypedExpr { cfold $ RUnary $2 $3 Not $4 }
	| '-' File Line TypedExpr %prec NEG { cfold $ RUnary $2 $3 Neg $4 }
--	| TypedExpr app TypedExpr File Line { RAppend $4 $5 $1 $3 }
	| error {% reportError "Invalid expression" }

-- MathShorthand
--    : Lvalue inc File Line 
--     { RAssignOp $3 $4 Plus $1 (RConst $3 $4 (Num 1)) }
--    | Lvalue dec File Line 
--     { RAssignOp $3 $4 Minus $1 (RConst $3 $4 (Num 1)) }
--    | Lvalue incby File Line TypedExpr
--     { RAssignOp $3 $4 Plus $1 $5 }
--    | Lvalue decby File Line TypedExpr
--     { RAssignOp $3 $4 Minus $1 $5 }
--    | Lvalue multby File Line TypedExpr
--     { RAssignOp $3 $4 Times $1 $5 }
--    | Lvalue divby File Line TypedExpr
--     { RAssignOp $3 $4 Divide $1 $5 }

MathShorthand :: { Raw }
    : Lvalue inc File Line 
     { RAssign $3 $4 $1 (RInfix $3 $4 Plus (lvaltoexp $1) 
  		       (RConst $3 $4 (Num 1))) }
    | Lvalue dec File Line 
     { RAssign $3 $4 $1 (RInfix $3 $4 Minus (lvaltoexp $1) 
  		       (RConst $3 $4 (Num 1))) }
    | Lvalue incby File Line TypedExpr
     { RAssign $3 $4 $1 (RInfix $3 $4 Plus (lvaltoexp $1) $5) }
    | Lvalue decby File Line TypedExpr
     { RAssign $3 $4 $1 (RInfix $3 $4 Minus (lvaltoexp $1) $5) }
    | Lvalue multby File Line TypedExpr
     { RAssign $3 $4 $1 (RInfix $3 $4 Times (lvaltoexp $1) $5) }
    | Lvalue divby File Line TypedExpr
     { RAssign $3 $4 $1 (RInfix $3 $4 Divide (lvaltoexp $1) $5) }
	  

BindExp : Type name '=' TypedExpr ';' File Line Expr 
	     { RBind $6 $7 $2 $1 $4 $8 }
--	| Type name ';' Expr File Line { RDeclare $5 $6 ($2,True) $1 $4 }
--	| Type NameList ';' File Line Expr { declare $4 $5 $2 $1 $6 }
	-- | error {% reportError "invalid expression" }

--NameList : name { [$1] }
--	 | name ',' NameList { $1:$3 }
	 -- | error {% reportError "invalid name list" }

CoerceExp :: { Raw }
          : PrimType '(' TypedExpr File Line ')' { RCoerce $4 $5 $1 $3 }
         -- | error {% reportError "invalid coercion" }

AssignExp :: { Raw }
          : Lvalue File Line Rvalue { RAssign $2 $3 $1 $4 }
	  | PattVal File Line Rvalue { mkPattAssign $2 $3 $1 $4 }
	 -- | error {% reportError "invalid assignment" }

Lvalue :: { RAssign }
      : name File Line { RAName $2 $3 $1 }
      | Lvalue '[' TypedExpr File Line ']' { RAIndex $4 $5 $1 $3 }
      | Lvalue '.' name File Line { RAField $4 $5 $1 $3 }
      -- | error {% reportError "lvalue expected" }

ControlExp :: { Raw }
           : if '(' TypedExpr ')' File Line BlockExp else BlockExp 
	       { RIf $5 $6 $3 $7 $9 }
	   | if '(' TypedExpr ')' File Line BlockExp else ControlExp 
	       { RIf $5 $6 $3 $7 $9 }
	   | if '(' TypedExpr ')' File Line BlockExp 
	       { RIf $5 $6 $3 $7 (RNoop $5 $6) }
	   | while '(' TypedExpr ')' File Line BlockExp 
	       { RWhile $5 $6 $3 $7 }
	   | do BlockExp File Line while '(' TypedExpr ')'
	       { RDoWhile $3 $4 $2 $7 }
	   | for Lvalue Index in TypedExpr File Line BlockExp
	       { RFor $6 $7 $2 $3 $5 $8 }
	   | for '(' VoidExpr ';' TypedExpr ';' VoidExpr ')' File Line
	       BlockExp
	       { RSeq $9 $10 $3 (RWhile $9 $10 $5 (RSeq $9 $10 $11 $7)) }
	   | repeat File Line ControlExp { RWhile $2 $3 (RConst $2 $3 (Bo True)) $4 }
	   | try BlockExp catch '(' name ')' BlockExp Finally File Line 
		      { RTryCatch $9 $10 $2 $7 $5 $8 }
	   | CaseExp { $1 }
	   -- | error {% reportError "invalid control structure" }

Index :: { Maybe Name }
      : { Nothing }
      | '@' name { Just $2 }

Finally :: { Raw }
       : File Line { RNoop $1 $2 }
       | finally BlockExp { $2 }

TypedCaseExp :: { Raw }
       : case TypedExpr of File Line '{' TypedCaseAlts '}' { RCase $4 $5 $2 $7 }
	-- | error {% reportError "invalid case expression" }

TypedCaseAlts :: { [RCaseAlt] }
              : TypedCaseAlt { [$1] }
--	      | { [] }
	      | TypedCaseAlt '|' TypedCaseAlts { $1:$3 }
	 -- | error {% reportError "invalid case alternative" }

TypedCaseAlt :: { RCaseAlt }
        : Name '(' VarList ')' arrow File Line TypedExpr ';'
	    { RAlt $6 $7 $1 $3 $8 } 
	| Name arrow File Line TypedExpr ';'
	    { RAlt $3 $4 $1 [] $5 } 
        | Const arrow File Line Expr
            { RConstAlt $3 $4 (getConst $1) $5 }
        | default arrow File Line TypedExpr ';'
            { RDefault $3 $4 $5 }
	-- | error {% reportError "invalid case alternative" }

CaseExp :: { Raw }
        : case TypedExpr of File Line '{' CaseAlts '}' { RCase $4 $5 $2 $7 }
	-- | error {% reportError "invalid case expression" }

CaseAlts :: { [RCaseAlt] }
         : CaseAlt { [$1] }
--	 | { [] }
	 | CaseAlt '|' CaseAlts { $1:$3 }
	 -- | error {% reportError "invalid case alternative" }

CaseAlt :: { RCaseAlt }
        : PattVal arrow File Line Expr
	  { RAlt $3 $4 (fst $1) (snd $1) $5 }
	| Name '(' VarList ')' arrow File Line Expr 
	  { RAlt $6 $7 $1 $3 $8 }
	| Name arrow File Line Expr 
	  { RAlt $3 $4 $1 [] $5 }
        | Const arrow File Line Expr
            { RConstAlt $3 $4 (getConst $1) $5 }
        | default arrow File Line Expr
            { RDefault $3 $4 $5 }
	-- | error {% reportError "invalid case alternative" }

PattVal :: { (Name, [Name]) } 
	: '(' VarList File Line ')' {% mkTuplePat $3 $4 $2 }
--	| Name '(' VarList ')' Expr
--	  { ($1,$3) }

VarList :: { [Name] }
        : { [] } 
	| Name { [$1] }
	| Name ',' VarList { $1:$3 }
	-- | error {% reportError "invalid variable list" }

BlockExp :: { Raw }
         : '{' Expr '}' { $2 }
	 | '{' '}' File Line { RNoop $3 $4 }
	 | VoidExpr ';' { $1 }
	 -- | error {% reportError "invalid expression" }

IOExp :: { Raw }
      : print TypedExpr File Line { RPrint $3 $4 $2 }
      -- | error {% reportError "invalid expression" }

Line :: { LineNumber }
     : {- empty -}      {% getLineNo }

File :: { String } 
     : {- empty -} %prec NONE  {% getFileName }

Module :: { Name } 
     : {- empty -} %prec NONE  {% getModuleName }

Libdirs :: { [FilePath] }
     : {- empty -} %prec NONE  {% getLibdirs }

{

reportError :: String -> P a
reportError err = getFileName `thenP` \fn -> 
		  getLineNo `thenP` \line ->
		      getContent `thenP` \str ->
		      failP (fn ++ ":" ++ show line ++ ":" ++ err ++ " - before: " ++ take 80 str ++ " ...") 

happyError :: P a
happyError = reportError "Parse error"

mkType :: [(Name,Type,Maybe Raw,ArgType)] -> Type -> Type
mkType ts ret = Fn (map thrd ts) (map sndt ts) ret
   where sndt (x,y,z,w) = y
	 thrd (x,y,z,w) = z

mkType' :: [(Type,Maybe Raw)] -> Type -> Type
mkType' ts ret = Fn (map snd ts) (map fst ts) ret

mkBody :: String -> LineNumber -> [(Name,Type,Maybe Raw,ArgType)] -> 
	  Raw -> Raw
--mkBody f l _ [] r = r
mkBody f l ts r = RLambda f l (map fourth ts) (map fstsnd ts) r
  where fstsnd (x,y,z,w) = (x,y)
	fourth (x,y,z,w) = w

-- Probably pointless. Used to be needed...
mkClosure :: String -> LineNumber -> [(Name,Type)] -> Raw -> Raw
mkClosure f l ts r = RClosure f l ts r

-- Create a datatype declaration. Boolean flags whether this will
-- generate code (it won't need to if it is imported from another module).
mkRet :: Name -> [Type] -> [ConDecl] -> Bool -> [ConDecl]
mkRet n params [] cg = []
mkRet n params ((Con nm (Fn a b c) fs _):xs) cg
    = (Con nm (Fn a b (User n params)) fs cg):(mkRet n params xs cg)

mkIf :: String -> Int -> [(Raw,Raw)] -> Raw
mkIf f l [] = RThrow f l (RExcept f l (RConst f l (Str "Unguarded guard")) 
			              (RConst f l (Num 1)))
mkIf f l ((exp,res):xs) = RIf f l exp res (mkIf f l xs)

-- Turn foreign function bindings into real functions
mkForeign :: String -> [Foreign] -> [RawDecl]
mkForeign lib [] = []
mkForeign lib ((ForeignDecl f l opts ty args kname cname doc):fs) =
    (FB (f,l,kname,Fn (map (\x -> Nothing) args) (map snd args) ty,
	 (Inline:opts), 
	 Defined mkFcall) doc):
       (mkForeign lib fs)
    where mkFcall = RLambda f l (map (\x -> Var) args) args (body ty)
	  body (Prim Void) = RForeign f l (Prim Void) cname mkfargs
	  body ty = RReturn f l (RForeign f l ty cname mkfargs)
	  mkfargs = map (RVar f l) (map fst args)
--          argnames = map (\x -> MN ("c",x)) [1..]


-- Create a pattern matching assignment statement
-- case exp of n(ns) -> ;
mkPattAssign :: String -> Int -> (Name, [Name]) -> Raw -> Raw
mkPattAssign f l (n,ns) exp =
    RCase f l exp alts
  where alts = [ok,broke]
	ok = RAlt f l n ns (RNoop f l)
	broke = RDefault f l badpatt
	badpatt = RThrow f l (RExcept f l bmsg bcode)
	bmsg = RConst f l (Str "Bad pattern matching assignment")
	bcode = RConst f l (Num 0)

mkTuplePat :: String -> Int -> [Name] -> P (Name, [Name])
mkTuplePat f l xs
   | length xs < 2 || length xs > 6 = reportError "Funny sized tuple"
   | otherwise = returnP (UN (tnames!!((length xs)-2)), xs)
 where tnames = ["Pair","Triple","Tuple4","Tuple5","Tuple6"]

declare :: String -> LineNumber -> [Name] -> Type -> Raw -> Raw
declare f ln [x] ty r = RDeclare f ln (x,False) ty r
declare f ln (x:xs) ty r = RDeclare f ln (x,False) ty (declare f ln xs ty r)
			    
exportall :: Bool -> [RawDecl] -> [RawDecl]
exportall False xs = xs
exportall True xs = ea' xs
  where ea' [] = []
	ea' ((FB (f,l,n,t,opts,raw) s):xs) 
	    = ((FB (f,l,n,t,Export:opts,raw) s):(ea' xs))
	ea' ((DDecl f l dopts n ty cons doc):xs)
	    = (DDecl f l (DExport:dopts) n ty cons doc):(ea' xs)
	ea'((TSyn (f,l,n,ty,t,False)):xs) =
	    (TSyn (f,l,n,ty,t,True)):(ea' xs)
	ea' (x:xs) = x:(ea' xs)
         
processTyName n@(UN (c:cs)) | isUpper c = User n []
			    | otherwise = TyVar n
processTyName n = User n []

-- If a function is just a name, quote it.
qhack (RVar f l n) = RQVar f l n
qhack x = x

findFile :: [FilePath] -> FilePath -> IO (Maybe String)
findFile [] path
  = return Nothing
findFile (x:xs) path 
  = catch
         (do --putStrLn $ "Trying " ++ x ++ path
	     f <- readFile (x++path)
	     return (Just f))
         (\e -> findFile xs path)

parse modn libdirs s fn = mkparse s fn libdirs (UN modn) 1
parseprog modn libdirs s fn = mkparseprog s fn libdirs (UN modn) 1

--rreturn :: a -> Result a
--rreturn foo = return foo

}
