#ifndef KADU_STATUS_H
#define KADU_STATUS_H

#include <qobject.h>
#include <qpixmap.h>
#include <qstring.h>
#include <qstringlist.h>

/**
	\enum eUserStatus
	Opisuje stan statusu.
**/
enum eUserStatus
{
	Online, /*!< Dostpny */
	Busy, /*!< Zaraz wracam */
	Invisible, /*!< Ukryty */
	Offline, /*!< Niedostpny */
	Blocking /*!< Blokujcy */
};

/**
	Klasa ta zawiera informacje o statusie kontaktu lub uytkownika Kadu, oraz metody
	suce do zmiany tego statusu.
	\class UserStatus
	\brief Klasa opisujca status.
**/
class UserStatus : public QObject
{
	Q_OBJECT

	private:
		bool Changed;

	protected:
		/**
			\var eUserStatus Stat
			Zawiera informacj o aktualnym statusie w postaci enumerycznej.
		**/
		eUserStatus Stat;

		/**
			\var QString Description
			Zawiera aktualny opis.
		**/
		QString Description;

		/**
			\var bool FriendsOnly
			Zawiera informacj o trybie "tylko dla znajomych".
		**/
		bool FriendsOnly;

	public:
		/**
			\fn UserStatus()
			Standardowy konstruktor.
		**/
		UserStatus();

		/**
			\fn UserStatus(const UserStatus &copyMe)
			Konstruktor kopiujcy.
			\param copyMe obiekt ktry chcemy skopiowa.
		**/
		UserStatus(const UserStatus &copyMe);

		/**
			\fn void operator = (const UserStatus &copyMe)
			Operator kopiujcy.
			\param copyMe obiekt ktry chcemy skopiowa.
		**/
		UserStatus &operator = (const UserStatus &copyMe);

		/**
			\fn bool operator == (const UserStatus &compare) const
			Operator porwnujcy. Sprawdza tosamo obiektw.
			\param compare obiekt z ktrym chcemy porwna bierzcy obiekt.
		**/
		bool operator == (const UserStatus &compare) const;

		/**
			\fn bool operator != (const UserStatus &compare) const
			Operator porwnujcy. Sprawdza rnic pomidzy obiektami.
			\param compare obiekt z ktrym chcemy porwna bierzcy obiekt.
		**/
		bool operator != (const UserStatus &compare) const;

		/**
			\fn virtual UserStatus *copy() const
			Wykonuje kopi obiektu, ktr pniej naley samodzielnie zniszczy
		**/
		virtual UserStatus *copy() const;

		virtual ~UserStatus();

		/**
			\fn bool isOnline() const
			Sprawdza czy kontakt jest "dostpny".
			\return TRUE jeli jest "dostpny", FALSE w przeciwnym wypadku.
		**/
		bool isOnline() const;

		/**
			\fn bool isBusy() const
			Sprawdza czy kontakt ma status "zaraz wracam".
			\return TRUE jeli ma status "zaraz wracam", FALSE w przeciwnym wypadku.
		**/
		bool isBusy() const;

		/**
			\fn bool isInvisible() const
			Sprawdza czy kontakt jest "ukryty".
			\return TRUE jeli jest "ukryty", FALSE w przeciwnym wypadku.
		**/
		bool isInvisible() const;

		/**
			\fn bool isOffline() const
			Sprawdza czy kontakt jest "niedostpny".
			\return TRUE jeli jest "niedostpny", FALSE w przeciwnym wypadku.
		**/
		bool isOffline() const;

		/**
			\fn bool isBlocking() const
			Sprawdza czy kontakt jest "blokujcy".
			\return TRUE jeli jest "blokujcy", FALSE w przeciwnym wypadku.
		**/
		bool isBlocking() const;

		/**
			\fn static bool isOffline(int index)
			Sprawdza czy status dany w postaci indeksowej jest statusem "niedostpny" lub jego
			odpowiednikiem z opisem.
			\return TRUE jeli status oznacza "niedostpny", FALSE w przeciwnym wypadku.
		**/
		static bool isOffline(int index);

		/**
			\fn bool hasDescription() const
			Sprawdza czy status jest z opisem.
			\return TRUE gdy status jest z opisem, FALSE w przeciwnym wypadku.
		**/
		bool hasDescription() const;

		/**
			\fn bool isFriendsOnly() const
			Sprawdza czy status jest "tylko dla znajomych".
			\return TRUE jeli jest "tylko dla znajomych", FALSE w przeciwnym wypadku.
		**/
		bool isFriendsOnly() const;

		/**
			\fn QString description() const
			\return opis, o ile status jest z opisem, lub "" gdy opisu nie ma.
		**/
		QString description() const;

		/**
			\fn eUserStatus status() const
			\return status w postaci enumeracyjnej.
		**/
		eUserStatus status() const;

		/**
			\fn int index() const
			\return status w postaci indeksowej.
		**/
		int index() const;

		/**
			\fn static int index(eUserStatus stat, bool has_desc)
			Zamienia enumeracyjn posta statusu na indeksow.
			\param stat enumeracyjna posta statusu.
			\param has_desc informuje, czy jest to odmiana statusu z opisem.
			\return status w postaci indeksowej.
		**/
		static int index(eUserStatus stat, bool has_desc);

		/**
			\fn virtual QPixmap pixmap(bool mobile = false) const
			Zwraca ikon reprezentujc ten status.
			\param mobile informuje czy ikona ma zawiera element telefonu komrkowego. Domylnie FALSE.
			\return pixmap reprezentujc ikon.
		**/
		virtual QPixmap pixmap(bool mobile = false) const;

		/**
			\fn virtual QString pixmap(bool mobile = false) const
			Zwraca nazw ikonki reprezentujcej ten status.
			\param mobile informuje czy ikona ma zawiera element telefonu komrkowego. Domylnie FALSE.
			\return nazwa pixmapy reprezentujcej ikon.
		**/
		virtual QString pixmapName(bool mobile = false) const;

		/**
			\fn virtual QPixmap pixmap(const UserStatus &stat, bool mobile = false) const
			Zwraca ikon reprezentujc dany status.
			\param stat status ktrego ikon naley zwrci
			\param mobile informuje czy ikona ma zawiera element telefonu komrkowego. Domylnie FALSE.
			\return pixmap reprezentujc ikon.
		**/
		virtual QPixmap pixmap(const UserStatus &, bool mobile = false) const;

		/**
			\fn virtual QString pixmap(bool mobile = false) const
			Zwraca nazw ikonki reprezentujcej dany status.
			\param stat status ktrego ikon naley zwrci
			\param mobile informuje czy ikona ma zawiera element telefonu komrkowego. Domylnie FALSE.
			\return nazwa pixmapy reprezentujcej ikon.
		**/
		virtual QString pixmapName(const UserStatus &, bool mobile = false) const;

		/**
			\fn virtual QPixmap pixmap(eUserStatus stat, bool has_desc, bool mobile = false) const
			Zwraca ikon reprezentujc dany status.
			\param stat status w postaci enumeracyjnej ktrego ikon naley zwrci
			\param has_desc informuje, czy jest to odmiana statusu z opisem.
			\param mobile informuje czy ikona ma zawiera element telefonu komrkowego. Domylnie FALSE.
			\return pixmap reprezentujc ikon.
		**/
		virtual QPixmap pixmap(eUserStatus stat, bool has_desc, bool mobile = false) const;

		/**
			\fn virtual QString pixmap(bool mobile = false) const
			Zwraca nazw ikonki reprezentujc dany status.
			\param stat status w postaci enumeracyjnej ktrego ikon naley zwrci
			\param has_desc informuje, czy jest to odmiana statusu z opisem.
			\param mobile informuje czy ikona ma zawiera element telefonu komrkowego. Domylnie FALSE.
			\return nazwa pixmapy reprezentujcej ikon.
		**/
		virtual QString pixmapName(eUserStatus stat, bool has_desc, bool mobile = false) const;

		/**
			\fn static eUserStatus fromString(const QString& stat)
			Zamienia status z formy acucha znakw na posta enumeracyjn.
			\param stat acuch znakw reprezentujcy nazw statusu. Poprawne nazwy statusw to:
				\arg \c Online
				\arg \c Busy
				\arg \c Invisible
				\arg \c Blocking
			\return enumeracyjna posta statusu.
		**/
		static eUserStatus fromString(const QString& stat);

		/**
			\fn static QString toString(eUserStatus stat, bool has_desc)
			Zamienia status z postaci enumeracyjnej na form acucha znakw.
			\param stat posta enumeracyjna statusu
			\param has_desc informuje, czy jest to odmiana statusu z opisem.
			\return enumeracyjna posta statusu.
		**/
		static QString toString(eUserStatus stat, bool has_desc);

		/**
			\fn QString toString(eUserStatus stat, bool has_desc)
			Zamienia status na form acucha znakw.
			robi dokadnie tyle co UserStatus::toString(this->status(), this->hasDescription())
			\see toString(eUserStatus, bool)
			\return status w postaci napisu.
		**/
		QString toString() const;

		/**
			\fn static int count()
			\return liczb wszystkich moliwych statusw, cznie z "blokowany" oraz "niedostpny z opisem".
		**/
		static int count();

		/**
			\fn static int count()
			\return liczb wszystkich moliwych statusw, ktre moe zobaczy uytkownik Kadu po ustawieniu danego,
				a wic wyczajc "blokowany" oraz "niedostpny z opisem".
		**/
		static int initCount();

		/**
			\fn static QString name(int nr)
			Zamienia dany status z postaci indeksowej na form acucha znakw. Posta indeksowa musi mieci si
			w liczbie zwrconej przez UserStatus::count() (musi by przynajmniej o 1 mniejsza od tej liczby).
			\return status w postaci acucha znakw.
		**/
		static QString name(int nr);

		/**
			\fn QString name() const
			Zamienia aktualny status z postaci indeksowej na form acucha znakw.
			\return status w postaci acucha znakw.
		**/
		QString name() const;

		/**
			\fn void refresh()
			Odwiea aktualny status.
		**/
		void refresh();

		/**
			\fn virtual QString protocolName() const;
			Zwraca identyfikator protokou.
		**/
		virtual QString protocolName() const;

	public slots:
		/**
			\fn void setOnline(const QString& desc = "")
			Ustawia status na "dostpny" z ewentualnym opisem.
			\param desc opcjonalny opis. Domylnie brak.
		**/
		void setOnline(const QString& desc = QString::null);

		/**
			\fn void setBusy(const QString& desc = "")
			Ustawia status na "zaraz wracam" z ewentualnym opisem.
			\param desc opcjonalny opis. Domylnie brak.
		**/
		void setBusy(const QString& desc = QString::null);

		/**
			\fn void setInvisible(const QString& desc = "")
			Ustawia status na "ukryty" z ewentualnym opisem.
			\param desc opcjonalny opis. Domylnie brak.
		**/
		void setInvisible(const QString& desc = QString::null);

		/**
			\fn void setOffline(const QString& desc = "")
			Ustawia status na "niedostpny" z ewentualnym opisem.
			\param desc opcjonalny opis. Domylnie brak.
		**/
		void setOffline(const QString& desc = QString::null);

		/**
			\fn void setBlocking()
			Ustawia tryb blokujcy statusu.
		**/
		void setBlocking();

		/**
			\fn void setDescription(const QString& desc = "")
			Ustawia opis dla statusu.
			\param desc opcjonalny opis. Domylnie pusty.
		**/
		void setDescription(const QString& desc = QString::null);

		/**
			\fn void setStatus(const UserStatus& stat)
			Ustawia status na dany.
			\param stat status ktry chcemy ustawi
		**/
		void setStatus(const UserStatus& stat);

		/**
			\fn void setStatus(eUserStatus stat, const QString& desc = "")
			Ustawia status na dany.
			\param stat enumeracyjna posta statusu.
				\arg \c Online - Dostpny
				\arg \c Busy - Zaraz wracam
				\arg \c Invisible - Ukryty
				\arg \c Blocking - Blokujcy
				\arg \c Offline - Nidostpny
			\param desc opcjonalny opis. Domylnie brak.
		**/
		void setStatus(eUserStatus stat, const QString& desc = QString::null);

		/**
			\fn void setIndex(int index, const QString& desc = "")
			Ustawia status na dany.
			\param index indeksowa posta statusu.
			\note parzyste indeksy odpowiadaj statusom bez opisw.
			\param desc opcjonalny opis. Domylnie pusty.
		**/
		void setIndex(int index, const QString& desc = QString::null);

		/**
			\fn void setFriendsOnly(bool f)
			Ustawia tryb "tylko dla znajomych".
			\param f warto logiczna informujca o tym, czy status ma mie wczony ten tryb, czy nie.
		**/
		void setFriendsOnly(bool f);

	signals:
		/**
			\fn void goOnline(const QString& desc)
			Sygna emitowany, gdy status jest zmieniany na "dostpny".
			\param desc Opcjonalny opis. Moe mie warto "" jeli status jest bezopisowy.
		**/
		void goOnline(const QString& desc);

		/**
			\fn void goBusy(const QString& desc)
			Sygna emitowany, gdy status jest zmieniany na "zaraz wracam".
			\param desc Opcjonalny opis. Moe mie warto "" jeli status jest bezopisowy.
		**/
		void goBusy(const QString& desc);

		/**
			\fn void goInvisible(const QString& desc)
			Sygna emitowany, gdy status jest zmieniany na "ukryty".
			\param desc Opcjonalny opis. Moe mie warto "" jeli status jest bezopisowy.
		**/
		void goInvisible(const QString& desc);

		/**
			\fn void goOffline(const QString& desc)
			Sygna emitowany, gdy status jest zmieniany na "niedostpny".
			\param desc Opcjonalny opis. Moe mie warto "" jeli status jest bezopisowy.
		**/
		void goOffline(const QString& desc);

		/**
			\fn void goBlocking()
			Sygna jest emitowany, kiedy tryb blokujcy zostaje wczony w tym statusie.
		**/
		void goBlocking();

		/**
			\fn void changed(const UserStatus& status)
			Sygna jest emitowany, gdy status zosta zmieniony na ktrykolwiek.
			\param status status, na ktry dokonano zmiany
			\param oldStatus status, z ktrego dokonano zmiany
		**/
		void changed(const UserStatus &status, const UserStatus &oldStatus);
};

extern QStringList defaultdescriptions;

#endif
