// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the Material K-3D object, which controls the surface appearance of rendered geometry
		\author Tim Shead (tshead@k-3d.com)
*/

#include "utility.h"

#include <k3dsdk/classes.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/igl.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/node_change_signal.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/module.h>
#include <k3dsdk/vectors.h>

#include <sh/sh.hpp>
#include <shutil/shutil.hpp>
using namespace SH;
using namespace ShUtil;

namespace libk3dsh
{

/////////////////////////////////////////////////////////////////////////////
// constant

class constant :
	public k3d::node_change_signal<k3d::persistent<k3d::node> >,
	public k3d::imaterial,
	public k3d::gl::imaterial
{
	typedef k3d::node_change_signal<k3d::persistent<k3d::node> > base;

public:
	constant(k3d::idocument& Document) :
		base(Document),
		m_color(init_owner(*this) + init_name("color") + init_label(_("Color")) + init_description(_("Color")) + init_value(k3d::color(1, 1, 1)))
	{
		m_color_cache = convert(m_color.value());

		m_color.changed_signal().connect(sigc::mem_fun(*this, &constant::on_color_changed));

		try
		{
			ShMatrix4x4f mvp;
			mvp.meta("opengl:state", "state.matrix.mvp");
			mvp.meta("opengl:readonly", "true");

			vertex_shader = SH_BEGIN_PROGRAM("gpu:vertex")
			{
				ShInputPosition4f ipos;
				ShOutputPosition4f opos;

				opos = mvp | ipos;
			} SH_END;

			fragment_shader = SH_BEGIN_PROGRAM("gpu:fragment")
			{
				ShOutputColor3f Cs = m_color_cache;
			} SH_END;

		}
		catch(std::exception& e)
		{
			k3d::log() << error << k3d_file_reference << ": exception: " << e.what() << std::endl;
		}
		catch(...)
		{
			k3d::log() << error << k3d_file_reference << ": unknown exception" << std::endl;
		}
	}

	void on_color_changed(k3d::iunknown* Hint)
	{
		m_color_cache = convert(m_color.value());
		make_node_change_slot()(Hint);
	}

	k3d::gl::imaterial* gl_material()
	{
		return this;
	}

	k3d::ri::imaterial* ri_material()
	{
		return 0;
	}

	k3d::yafray::imaterial* yafray_material()
	{
		return 0;
	}

	void setup_gl_material()
	{
		try
		{
			shBind(vertex_shader);
			shBind(fragment_shader);
		}
		catch(std::exception& e)
		{
			k3d::log() << error << k3d_file_reference << ": exception: " << e.what() << std::endl;
		}
		catch(...)
		{
			k3d::log() << error << k3d_file_reference << ": unknown exception" << std::endl;
		}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<constant>,
				k3d::interface_list<k3d::imaterial,
				k3d::interface_list<k3d::gl::imaterial> > > factory(
			k3d::uuid(0xde0ae4d5, 0x1d8c4c89, 0xb7b6507b, 0x36138cad),
			"ConstantMaterial",
			_("Constant-color viewport surface shader"),
			"OpenGL Materials",
			k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_color;
	ShColor3f m_color_cache;
	
	ShProgram vertex_shader;
	ShProgram fragment_shader;
};

/////////////////////////////////////////////////////////////////////////////
// constant_factory

k3d::iplugin_factory& constant_factory()
{
	return constant::get_factory();
}

} // namespace libk3dsh

