// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the RenderManScript K-3D object, which can insert scripted data into RenderMan output
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/algebra.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/itime_sink.h>
#include <k3dsdk/itransform_sink.h>
#include <k3dsdk/itransform_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/scripted_node.h>

#define DEFAULT_SCRIPT "#python\n\nOutput = Input\n\n"

namespace libk3dscripting
{

/////////////////////////////////////////////////////////////////////////////
// transform_modifier_script

class transform_modifier_script :
	public k3d::scripted_node<k3d::persistent<k3d::node> >,
	public k3d::itransform_source,
	public k3d::itransform_sink
{
	typedef k3d::scripted_node<k3d::persistent<k3d::node> > base;

public:
	transform_modifier_script(k3d::idocument& Document) :
		base(Document),
		m_input(init_owner(*this) + init_name("input_matrix") + init_label(_("Input matrix")) + init_description(_("Input transformation matrix")) + init_value(k3d::identity3D())),
		m_output(init_owner(*this) + init_name("output_matrix") + init_label(_("Output matrix")) + init_description(_("Output matrix generated by the script")) + init_slot(sigc::mem_fun(*this, &transform_modifier_script::output_value)))
	{
		set_script(DEFAULT_SCRIPT);

		m_input.changed_signal().connect(sigc::mem_fun(*this, &transform_modifier_script::reset_output_cache));
		connect_script_changed_signal(sigc::mem_fun(*this, &transform_modifier_script::reset_output_cache));
	}

	k3d::iproperty& transform_source_output()
	{
		return m_output;
	}

	k3d::iproperty& transform_sink_input()
	{
		return m_input;
	}

	void reset_output_cache(k3d::iunknown* const Hint)
	{
		m_output.reset(Hint);
	}

	k3d::matrix4 output_value()
	{
		// Create a new output matrix, ready for modification by the script ...
		const k3d::matrix4 input = m_input.value();
		k3d::matrix4 output = input;

		k3d::iscript_engine::context_t context;
		context["Document"] = static_cast<k3d::iunknown*>(&document());
		context["Node"] = static_cast<k3d::iunknown*>(this);
		context["Input"] = input;
		context["Output"] = output;

		execute_script(context);

		return_val_if_fail(context["Output"].type() == typeid(k3d::matrix4), output);
		output = boost::any_cast<k3d::matrix4>(context["Output"]);

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<transform_modifier_script>,
			k3d::interface_list<k3d::itransform_sink, k3d::interface_list<k3d::itransform_source> > > factory(
			k3d::uuid(0xacafcc85, 0xa0bf4d69, 0x99592c4f, 0x7cf9b35c),
			"TransformModifierScript",
			_("Transform modifier that uses a script to modify a transformation matrix"),
			"Scripting Transform",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::matrix4, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_input;
	k3d_data(k3d::matrix4, immutable_name, change_signal, no_undo, computed_storage, no_constraint, read_only_property, no_serialization) m_output;
};

/////////////////////////////////////////////////////////////////////////////
// transform_modifier_script_factory

k3d::iplugin_factory& transform_modifier_script_factory()
{
	return transform_modifier_script::get_factory();
}

} // namespace libk3dscripting

