// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

#include <boost/multi_array.hpp>
#include <iomanip>
#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// bevel_faces_implementation

class bevel_faces_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >,
	public k3d::mouse_event_observer
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	bevel_faces_implementation(k3d::idocument& Document) :
		base(Document),
		mouse_event_observer("LMB Drag to bevel faces"),
		m_direction(k3d::init_name("direction") + k3d::init_description("Bevel direction [enumeration]") + k3d::init_value(INSIDE) + k3d::init_document(Document) + k3d::init_enumeration(direction_values())),
		m_height(k3d::init_name("height") + k3d::init_description("Height [number]") + k3d::init_value(0.2) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_inset(k3d::init_name("inset") + k3d::init_description("Inset [number]") + k3d::init_value(0.2) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_direction));
		enable_serialization(k3d::persistence::proxy(m_height));
		enable_serialization(k3d::persistence::proxy(m_inset));
		
		register_property(m_direction);
		register_property(m_height);
		register_property(m_inset);
		
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &bevel_faces_implementation::on_reset_geometry));
		m_direction.changed_signal().connect(SigC::slot(*this, &bevel_faces_implementation::on_reset_geometry));
		
		m_height.changed_signal().connect(SigC::slot(*this, &bevel_faces_implementation::on_reshape_geometry));
		m_inset.changed_signal().connect(SigC::slot(*this, &bevel_faces_implementation::on_reshape_geometry));
		
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &bevel_faces_implementation::on_create_geometry));
	}

	bool OnLButtonDrag(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current, const k3d::vector2& Last, const k3d::vector2& Start, const drag_type_t DragType)
	{
		const double sensitivity = 1.0;
		m_inset.set_value(m_inset.value() + sensitivity * (Current[0] - Last[0]));
		m_height.set_value(m_height.value() + sensitivity * (Last[1] - Current[1]));

		return true;
	}

	void on_reset_geometry()
	{
		m_new_points.clear();
		m_output_mesh.reset();
	}

	/// Returns true iff the given object is selected
	struct is_selected
	{
		bool operator()(k3d::face* const Face)
		{
			return Face->selected;
		}
	};

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		reshape_geometry();
	
		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		const direction_t direction = m_direction.property_value();

		// For each polyhedron ...
		for(k3d::mesh::polyhedra_t::iterator polyhedron = output->polyhedra.begin(); polyhedron != output->polyhedra.end(); ++polyhedron)
			{
				const bool was_solid = k3d::is_solid(**polyhedron);
				
				// Get the set of all selected faces ...
				k3d::polyhedron::faces_t selected_faces;
				k3d::copy_if((*polyhedron)->faces.begin(), (*polyhedron)->faces.end(), std::inserter(selected_faces, selected_faces.end()), is_selected());

				if(selected_faces.empty())
					selected_faces = (*polyhedron)->faces;
					
				// For each face ...
				for(k3d::polyhedron::faces_t::iterator selected_face = selected_faces.begin(); selected_face != selected_faces.end(); ++selected_face)
					{
						k3d::face& face = **selected_face;
						
						// Calculate the face normal ...
						k3d::vector3 face_normal = k3d::normal(face);
						if(0.0 == face_normal.Length2())
							continue;
						face_normal = k3d::normalize(face_normal);
			
						// Get the set of loops to be processed ...
						typedef std::vector<k3d::split_edge**> loops_t;
						loops_t loops;
						loops.push_back(&face.first_edge);
						for(k3d::face::holes_t::iterator hole = face.holes.begin(); hole != face.holes.end(); ++hole)
							loops.push_back(&(*hole));
						
						// For each loop ...
						for(loops_t::iterator loop = loops.begin(); loop != loops.end(); ++loop)
							{
								// Get the set of original points and edges ...
								k3d::mesh::points_t old_points;
								k3d::polyhedron::edges_t old_edges;
								for(k3d::split_edge* edge = **loop; edge && edge->face_clockwise; edge = edge->face_clockwise)
									{
										old_points.push_back(edge->vertex);
										old_edges.push_back(edge);
										if(edge->face_clockwise == **loop)
											break;
									}
								const unsigned long edge_count = old_edges.size();
			
								// Create new points and new edges for the old loop ...
								k3d::mesh::points_t new_points(edge_count);
								k3d::polyhedron::edges_t new_edges(edge_count);
								for(unsigned long i = 0; i != edge_count; ++i)
									{
										new_points[i] = new k3d::point(*old_points[i]);
										new_edges[i] = new k3d::split_edge(new_points[i]);
									}
								k3d::loop_edges(new_edges.begin(), new_edges.end());
								**loop = new_edges.front();

								// Create new edges for the bevel faces ...
								boost::multi_array<k3d::split_edge*, 2> bevel_edges(boost::extents[edge_count][3]);
								for(unsigned long i = 0; i != edge_count; ++i)
									{
										bevel_edges[i][0] = new k3d::split_edge(old_points[(i+1)%edge_count]);
										bevel_edges[i][1] = new k3d::split_edge(new_points[(i+1)%edge_count]);
										bevel_edges[i][2] = new k3d::split_edge(new_points[i]);
								
										old_edges[i]->face_clockwise = bevel_edges[i][0];
										bevel_edges[i][0]->face_clockwise = bevel_edges[i][1];
										bevel_edges[i][1]->face_clockwise = bevel_edges[i][2];
										bevel_edges[i][2]->face_clockwise = old_edges[i];
									}

								// Create bevel faces ...
								k3d::polyhedron::faces_t bevel_faces(edge_count);
								for(unsigned long i = 0; i != edge_count; ++i)
									bevel_faces[i] = new k3d::face(old_edges[i]);						

								// Cleanup adjacent edges ...
								for(unsigned long i = 0; i != edge_count; ++i)
									{
										k3d::join_edges(*bevel_edges[i][0], *bevel_edges[(i+1)%edge_count][2]);
										k3d::join_edges(*bevel_edges[i][1], *new_edges[i]);
									}

								k3d::mesh::points_t& source_points = INSIDE == direction ? old_points : new_points;
								k3d::mesh::points_t& target_points = INSIDE == direction ? new_points : old_points;

								// Offset new points ...
								const k3d::vector3 height_direction = (INSIDE == direction ? 1.0 : -1.0) * face_normal;
								for(unsigned long i = 0; i != edge_count; ++i)
									{
										const k3d::vector3 e1 = source_points[i]->position - source_points[(i+edge_count-1)%edge_count]->position;
										const k3d::vector3 e2 = source_points[(i+1)%edge_count]->position - source_points[i]->position;

										const k3d::vector3 e3 = k3d::normalize(e1 ^ face_normal);
										const k3d::vector3 e4 = k3d::normalize(e2 ^ face_normal);

										const k3d::vector3 inset_direction = (INSIDE == direction ? 1.0 : -1.0) * (-1 / cos(acos(e3 * e4) / 2)) * k3d::normalize(e3 + e4);

										m_new_points.push_back(new_point(source_points[i]->position, height_direction, inset_direction, *target_points[i]));
									}

								// Insert new geometry into the polyhedron ...
								output->points.insert(output->points.end(), new_points.begin(), new_points.end());
								(*polyhedron)->edges.insert((*polyhedron)->edges.end(), new_edges.begin(), new_edges.end());
								(*polyhedron)->faces.insert((*polyhedron)->faces.end(), bevel_faces.begin(), bevel_faces.end());

								for(unsigned long i = 0; i != edge_count; ++i)
									{
										(*polyhedron)->edges.push_back(bevel_edges[i][0]);
										(*polyhedron)->edges.push_back(bevel_edges[i][1]);
										(*polyhedron)->edges.push_back(bevel_edges[i][2]);
									}
							}
					}
					
				assert_warning(is_valid(**polyhedron));

				// If the polyhedron was solid when we started, it better be solid when we're done ...
				if(was_solid)
					assert_warning(is_solid(**polyhedron));
			}

		// Position all the new geometry we've created ...
		reshape_geometry();

		return output;
	}

	void reshape_geometry()
	{
		const double height = m_height.property_value();
		const double inset = m_inset.property_value();

		for(new_points_t::iterator new_point = m_new_points.begin(); new_point != m_new_points.end(); ++new_point)
			new_point->update(height, inset);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bevel_faces_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xb6fd5f3d, 0x84024b37, 0xa27ef081, 0x7ed644f7),
				"BevelFaces",
				"Bevels a surface by extruding each selected face",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	typedef enum
	{
		INSIDE,
		OUTSIDE
	} direction_t;

	friend std::ostream& operator << (std::ostream& Stream, const direction_t& Value)
	{
		switch(Value)
			{
				case INSIDE:
					Stream << "inside";
					break;
				case OUTSIDE:
					Stream << "outside";
					break;
			}
		return Stream;
	}

	friend std::istream& operator >> (std::istream& Stream, direction_t& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "inside")
			Value = INSIDE;
		else if(text == "outside")
			Value = OUTSIDE;
		else
			std::cerr << __PRETTY_FUNCTION__ << ": unknown enumeration [" << text << "]" << std::endl;

		return Stream;
	}

	const k3d::ienumeration_property::values_t& direction_values()
	{
		static k3d::ienumeration_property::values_t values;
		if(values.empty())
			{
				values.push_back(k3d::ienumeration_property::value_t("Inside", "inside", "Bevels inward from the original polygons"));
				values.push_back(k3d::ienumeration_property::value_t("Outside", "outside", "Bevels outward from the original polygons"));
			}
		
		return values;
	}

	k3d_enumeration_property(direction_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_direction;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_height;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_inset;

	/// Caches new geometry for better interactive performance
	class new_point
	{
	public:
		new_point(const k3d::vector3& OldPosition, const k3d::vector3& HeightDirection, const k3d::vector3& InsetDirection, k3d::point& Point) :
			old_position(OldPosition),
			height_direction(HeightDirection),
			inset_direction(InsetDirection),
			point(&Point)
		{
		}
	
		void update(const double Height, const double Inset)
		{
			point->position = old_position + (Height * height_direction) + (Inset * inset_direction);
		}
	
	private:
		k3d::vector3 old_position;
		k3d::vector3 height_direction;
		k3d::vector3 inset_direction;
		k3d::point* point;
	};
	
	/// Caches new geometry for better interactive performance
	typedef std::vector<new_point> new_points_t;
	/// Caches new geometry for better interactive performance
	new_points_t m_new_points;

};

/////////////////////////////////////////////////////////////////////////////
// bevel_faces_factory

k3d::iplugin_factory& bevel_faces_factory()
{
	return bevel_faces_implementation::get_factory();
}

} // namespace libk3dmesh

