/*
 *  Jajuk
 *  Copyright (C) 2003-2009 The Jajuk Team
 *  http://jajuk.info
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *  $Revision: 5924 $
 */

package org.jajuk.base;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.swing.JOptionPane;

import org.jajuk.events.JajukEvent;
import org.jajuk.events.JajukEvents;
import org.jajuk.events.ObservationManager;
import org.jajuk.services.core.ExitService;
import org.jajuk.services.players.QueueModel;
import org.jajuk.util.Conf;
import org.jajuk.util.Const;
import org.jajuk.util.Messages;
import org.jajuk.util.ReadOnlyIterator;
import org.jajuk.util.UpgradeManager;
import org.jajuk.util.UtilSystem;
import org.jajuk.util.log.Log;

/**
 * Convenient class to manage devices.
 */
public final class DeviceManager extends ItemManager {
  
  /** Supported device types names. */
  private final List<String> alDevicesTypes = new ArrayList<String>(10);

  /** Self instance. */
  private static DeviceManager singleton;

  /** Date last global refresh. */
  private long lDateLastGlobalRefresh = 0;

  /** List of deep-refresh devices after an upgrade. */
  private final Set<Device> devicesDeepRefreshed = new HashSet<Device>();
  
  /** DeviceTypes Identification strings  Note: this needs to correspond with the constants in @see org.jajuk.base.Device !! */
  public static final String[] DEVICE_TYPES = { "Device_type.directory", "Device_type.file_cd",
      "Device_type.network_drive", "Device_type.extdd", "Device_type.player" };


  /** Auto-refresh thread. */
  private final Thread tAutoRefresh = new Thread("Device Auto Refresh Thread") {
    @Override
    public void run() {
      while (!ExitService.isExiting()) {
        try {
          Thread.sleep(Const.AUTO_REFRESH_DELAY);
          refreshAllDevices();
        } catch (Exception e) {
          Log.error(e);
        }
      }
    }
  };

  /** DOCUMENT_ME. */
  private volatile boolean bGlobalRefreshing = false;

  /**
   * No constructor available, only static access.
   */
  private DeviceManager() {
    super();
    // register properties
    // ID
    registerProperty(new PropertyMetaInformation(Const.XML_ID, false, true, false, false, false,
        String.class, null));
    // Name
    registerProperty(new PropertyMetaInformation(Const.XML_NAME, false, true, true, false, false,
        String.class, null));
    // Type
    registerProperty(new PropertyMetaInformation(Const.XML_TYPE, false, true, true, false, false,
        Long.class, null));
    // URL
    registerProperty(new PropertyMetaInformation(Const.XML_URL, false, true, true, false, false,
        Long.class, null));
    // Auto-mount
    registerProperty(new PropertyMetaInformation(Const.XML_DEVICE_AUTO_MOUNT, false, true, true,
        false, false, Boolean.class, null));
    // Auto-refresh
    registerProperty(new PropertyMetaInformation(Const.XML_DEVICE_AUTO_REFRESH, false, true, true,
        false, false, Double.class, 0d));
    // Expand
    registerProperty(new PropertyMetaInformation(Const.XML_EXPANDED, false, false, false, false,
        true, Boolean.class, false));
    // Synchro source
    registerProperty(new PropertyMetaInformation(Const.XML_DEVICE_SYNCHRO_SOURCE, false, false,
        true, false, false, String.class, null));
    // Synchro mode
    registerProperty(new PropertyMetaInformation(Const.XML_DEVICE_SYNCHRO_MODE, false, false, true,
        false, false, String.class, null));
  }

  /**
   * Start auto refresh thread.
   * DOCUMENT_ME
   */
  public void startAutoRefreshThread() {
    if(!tAutoRefresh.isAlive()) {
      tAutoRefresh.setPriority(Thread.MIN_PRIORITY);
      tAutoRefresh.start();
    }
  }

  /**
   * Gets the instance.
   * 
   * @return singleton
   */
  public static DeviceManager getInstance() {
    if (singleton == null) {
      singleton = new DeviceManager();
    }
    return singleton;
  }

  /**
   * Register a device.
   * 
   * @param sName DOCUMENT_ME
   * @param lDeviceType DOCUMENT_ME
   * @param sUrl DOCUMENT_ME
   * 
   * @return device
   */
  public Device registerDevice(String sName, long lDeviceType, String sUrl) {
    String sId = createID(sName);
    return registerDevice(sId, sName, lDeviceType, sUrl);
  }

  /**
   * Register a device with a known id.
   * 
   * @param sName DOCUMENT_ME
   * @param sId DOCUMENT_ME
   * @param lDeviceType DOCUMENT_ME
   * @param sUrl DOCUMENT_ME
   * 
   * @return device
   */
  public synchronized Device registerDevice(String sId, String sName, long lDeviceType, String sUrl) {
    Device device = getDeviceByID(sId);
    if (device != null) {
      return device;
    }
    device = new Device(sId, sName);
    device.setProperty(Const.XML_TYPE, lDeviceType);
    device.setUrl(sUrl);
    registerItem(device);
    return device;
  }

  /**
   * Check none device already has this name or is a parent directory.
   * 
   * @param sName DOCUMENT_ME
   * @param iDeviceType DOCUMENT_ME
   * @param sUrl DOCUMENT_ME
   * @param bNew DOCUMENT_ME
   * 
   * @return 0:ok or error code
   */
  public int checkDeviceAvailablity(String sName, int iDeviceType, String sUrl, boolean bNew) {
    // don't check if it is a CD as all CDs may use the same mount point
    if (iDeviceType == Device.TYPE_CD) {
      return 0;
    }
    // check name and path
    for (Device deviceToCheck : DeviceManager.getInstance().getDevices()) {
      // If we check an existing device unchanged, just leave
      if (!bNew && sUrl.equals(deviceToCheck.getUrl())) {
        continue;
      }
      // check for a new device with an existing name
      if (bNew
          && (sName.equalsIgnoreCase(deviceToCheck.getName()))) {
        return 19;
      }
      String sUrlChecked = deviceToCheck.getUrl();
      // check it is not a sub-directory of an existing device
      File fNew = new File(sUrl);
      File fChecked = new File(sUrlChecked);
      if (fNew.equals(fChecked) || UtilSystem.isDescendant(fNew, fChecked)
          || UtilSystem.isAncestor(fNew, fChecked)) {
        return 29;
      }
    }
    // check availability
    if (iDeviceType != Device.TYPE_EXT_DD) { // not a remote device, TBI for remote
      // test directory is available
      File file = new File(sUrl);
      // check if the url exists and is readable
      if (!file.exists() || !file.canRead()) {
        return 143;
      }
    }
    return 0;
  }

  /**
   * Register a device type.
   * 
   * @param sDeviceType DOCUMENT_ME
   */
  public void registerDeviceType(String sDeviceType) {
    alDevicesTypes.add(sDeviceType);
  }

  /**
   * Gets the device types number.
   * 
   * @return number of registered devices
   */
  public int getDeviceTypesNumber() {
    return alDevicesTypes.size();
  }

  /**
   * Gets the device types.
   * 
   * @return Device types iteration
   */
  public Iterator<String> getDeviceTypes() {
    return alDevicesTypes.iterator();
  }

  /**
   * Get a device type name for a given index.
   * 
   * @param index DOCUMENT_ME
   * 
   * @return device name for a given index
   */
  public String getDeviceType(long index) {
    return alDevicesTypes.get((int) index);
  }

  /**
   * Remove a device.
   * 
   * @param device DOCUMENT_ME
   */
  public synchronized void removeDevice(Device device) {
    // show confirmation message if required
    if (Conf.getBoolean(Const.CONF_CONFIRMATIONS_REMOVE_DEVICE)) {
      int iResu = Messages.getChoice(Messages.getString("Confirmation_remove_device"),
          JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);
      if (iResu != JOptionPane.YES_OPTION) {
        return;
      }
    }
    // if device is refreshing or synchronizing, just leave
    if (device.isSynchronizing() || device.isRefreshing()) {
      Messages.showErrorMessage(13);
      return;
    }
    // check if device can be unmounted
    if (!QueueModel.canUnmount(device)) {
      Messages.showErrorMessage(121);
      return;
    }
    // if it is mounted, try to unmount it
    if (device.isMounted()) {
      try {
        device.unmount();
      } catch (Exception e) {
        Messages.showErrorMessage(13);
        return;
      }
    }
    removeItem(device);
    DirectoryManager.getInstance().cleanDevice(device.getID());
    FileManager.getInstance().cleanDevice(device.getID());
    PlaylistManager.getInstance().cleanDevice(device.getID());
    // Clean the collection up
    org.jajuk.base.Collection.cleanupLogical();
    // remove synchronization if another device was synchronized
    // with this device
    for (Device deviceToCheck : getDevices()) {
      if (deviceToCheck.containsProperty(Const.XML_DEVICE_SYNCHRO_SOURCE)) {
        String sSyncSource = deviceToCheck.getStringValue(Const.XML_DEVICE_SYNCHRO_SOURCE);
        if (sSyncSource.equals(device.getID())) {
          deviceToCheck.removeProperty(Const.XML_DEVICE_SYNCHRO_SOURCE);
        }
      }
    }
    // Force suggestion view refresh to avoid showing removed albums
    ObservationManager.notify(new JajukEvent(JajukEvents.SUGGESTIONS_REFRESH));
  }

  /**
   * Checks if is any device refreshing.
   * 
   * @return whether any device is currently refreshing
   */
  public boolean isAnyDeviceRefreshing() {
    boolean bOut = false;
    ReadOnlyIterator<Device> it = DeviceManager.getInstance().getDevicesIterator();
    while (it.hasNext()) {
      Device device = it.next();
      if (device.isRefreshing()) {
        bOut = true;
        break;
      }
    }
    return bOut;
  }

  /**
   * Clean all devices.
   */
  public synchronized void cleanAllDevices() {
    for (Device device : getDevices()) {
      // Do not auto-refresh CD as several CD may share the same mount
      // point
      if (device.getType() == Device.TYPE_CD) {
        continue;
      }
      FileManager.getInstance().cleanDevice(device.getName());
      DirectoryManager.getInstance().cleanDevice(device.getName());
      PlaylistManager.getInstance().cleanDevice(device.getName());
    }
    clear();
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.jajuk.base.ItemManager#getIdentifier()
   */
  @Override
  public String getLabel() {
    return Const.XML_DEVICES;
  }

  /**
   * Gets the date last global refresh.
   * 
   * @return the date last global refresh
   */
  public long getDateLastGlobalRefresh() {
    return lDateLastGlobalRefresh;
  }

  /**
   * Refresh of all devices with auto-refresh enabled (used in automatic mode)
   * Must be the shortest possible.
   */
  public void refreshAllDevices() {
    try {
      // check thread is not already refreshing
      if (bGlobalRefreshing) {
        return;
      }
      bGlobalRefreshing = true;
      lDateLastGlobalRefresh = System.currentTimeMillis();
      boolean bNeedUIRefresh = false;
      for (Device device : getDevices()) {
        // Do not auto-refresh CD as several CD may share the same mount
        // point
        if (device.getType() == Device.TYPE_CD) {
          continue;
        }
        double frequency = 60000 * device.getDoubleValue(Const.XML_DEVICE_AUTO_REFRESH);
        // check if this device needs auto-refresh
        if (frequency == 0d
            || device.getDateLastRefresh() > (System.currentTimeMillis() - frequency)) {
          continue;
        }
        /*
         * Check if devices contains files, otherwise it is not mounted we have
         * to check this because of the automatic cleaner thread musn't remove
         * all references
         */
        File[] files = new File(device.getUrl()).listFiles();
        if (!device.isRefreshing() && files != null && files.length > 0) {
          /*
           * Check if this device should be deep-refresh after an upgrade
           */
          boolean bNeedDeepAfterUpgrade = UpgradeManager.isUpgradeDetected()
              && !devicesDeepRefreshed.contains(device);
          if (bNeedDeepAfterUpgrade) {
            // Store this device to avoid duplicate deep refreshes
            devicesDeepRefreshed.add(device);
          }
          // cleanup device
          bNeedUIRefresh = bNeedUIRefresh | device.cleanRemovedFiles();
          // refresh the device (deep refresh forced after an upgrade)
          bNeedUIRefresh = bNeedUIRefresh | device.refreshCommand(bNeedDeepAfterUpgrade,false);

          // UI refresh if required
          if (bNeedUIRefresh) {
            // Cleanup logical items
            Collection.cleanupLogical();
            /*
             * Notify views to refresh once the device is refreshed, do not wait
             * all devices refreshing as it may be tool long
             */
            ObservationManager.notify(new JajukEvent(JajukEvents.DEVICE_REFRESH));
          }
        }
      }
      // Display end of refresh message with stats
    } catch (Exception e) {
      Log.error(e);
    } finally {
      bGlobalRefreshing = false;
    }
  }

  /**
   * Gets the device by id.
   * 
   * @param sID Item ID
   * 
   * @return Element
   */
  public Device getDeviceByID(String sID) {
    return (Device) getItemByID(sID);
  }

  /**
   * Gets the device by name.
   * 
   * @param sName device name
   * 
   * @return device by given name or null if no match
   */
  public Device getDeviceByName(String sName) {
    for (Device device : getDevices()) {
      if (device.getName().equals(sName)) {
        return device;
      }
    }
    return null;
  }

  /**
   * Gets the devices.
   * 
   * @return ordered devices list
   */
  @SuppressWarnings("unchecked")
  public synchronized List<Device> getDevices() {
    return (List<Device>) getItems();
  }

  /**
   * Gets the devices iterator.
   * 
   * @return devices iterator
   */
  @SuppressWarnings("unchecked")
  public synchronized ReadOnlyIterator<Device> getDevicesIterator() {
    return new ReadOnlyIterator<Device>((Iterator<Device>) getItemsIterator());
  }
}
