/****************************************************************************
 *                              WaveWidget.cc
 * Author: Matthew Ballance
 * Desc:   Implements a widget for displaying waveforms. The WaveWidget
 *         utilizes the SigDB to store the signal data...
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *****************************************************************************/
#include "WaveWidget.h"
#include "WidgetManager.h"
#include "WidgetCommon.h"
#include "SdbReader.h"
#include "SdbrCursor.h"
#include "CallbackMgr.h"
#include "PsPixmapObj.h"
#include "BitVector.h"
#include "BitVectorABVal.h"

#include "SigDB.h"
#include "SdbSignal.h"
#include "DFIO.h"
#include "DFIOTrace.h"

#include "LogInstance.h"
#include "LogDestination.h"
#include "LogMgr.h"

#include "TreeWidget.h"
#include "TreeWidgetNode.h"
#include "ConfigDB.h"

#undef  TRACE_REPAINT
#undef  NO_DRAWING
#undef  NO_TEXT
#undef  TRACE_WIN_REPAINT

#define FP stderr

#ifdef TRACE_REPAINT
#define DBG_MSG(x) fprintf x
#else
#define DBG_MSG(x)
#endif

#ifdef TRACE_WIN_REPAINT
#define WIN_MSG(x) fprintf x
#else
#define WIN_MSG(x)
#endif

#define BETWEEN(lb, v, ub) ((((lb) <= (v)) && ((ub) >= (v))))

/******************************************************************
 * WaveWidget::SDBRSigUpdateCB()
 *
 * Must update simTime and totalTime
 ******************************************************************/
int WaveWidget::SDBRSigUpdateCB(
        ClientData        clientData,
        Tcl_Interp       *d_interp,
        int               objc,
        Tcl_Obj          *const objv[])
{
    WaveWidget *ww = (WaveWidget *)clientData;

//    fprintf(stderr, "SDBRSigUpdateCB()\n");
    ww->d_runningMax = 0;

    ww->xscroll_valid = 0;
    ww->d_sigValNeedUpdate = true;
    ww->SetupRedraw(RedrawForce);

    return 0;
}

/******************************************************************
 * WaveWidget::SDBRSigDataUpdateCB()
 ******************************************************************/
int WaveWidget::SDBRSigDataUpdateCB(
        ClientData        clientData,
        Tcl_Interp       *d_interp,
        int               objc,
        Tcl_Obj          *const objv[])
{
    WaveWidget *ww = (WaveWidget *)clientData;

//    fprintf(stderr, "SDBRSigDataUpdateCB(running=%d , end=%d)\n",
//            ww->d_runningMax, ww->displayEndTime);

    if (ww->d_runningMax < ww->displayEndTime) {
        ww->SetupRedraw(RedrawForce);
    }

    ww->d_runningMax = ww->sdbr->getMaxTime();
}

/******************************************************************
 * WaveWidget()
 ******************************************************************/
WaveWidget::WaveWidget(
                       Tcl_Interp      *d_interp,
                       Uint32           objc,
                       Tcl_Obj         *const objv[]) :
    strVal(128), XObjWidgetBase(d_interp, objc, objv)
{
    Uint32             i;
    ClientData         cbId;

    if (!ok) {
        return;
    }
    ok = 0;

    this->button_down = 0;

    d_WinChangeCBList = CallbackMgr::GetCbList(CBTYPE_WAVE_WINDOW_CHANGE,
            Tcl_GetString(objv[1]));

    for (i=0; i<16; i++) {
        d_cbArgs[i] = Tcl_NewObj();
        Tcl_IncrRefCount(d_cbArgs[i]);
    }

    d_useXGC = true;

    memset(&d_config, 0, sizeof(WaveConfig));
    d_config.traceHeight      = 16;
    dy               = (d_config.traceHeight*7)/16;

    d_config.cursor = None;

    d_sigTree        = 0;

    d_backingPixmap.setWindow(getWin());
    d_displayPixmap.setWindow(getWin());

    widgetWidth      = 0;       
    widgetHeight     = 0;       
    sdbr             = 0;

    pixelsPerTimeUnit = 2.0;
    totalTime         = 0;
    displayStartTime  = 0;
    newDisplayStartTime = 0;
    newDisplayStartPos  = 0;
    displayStartPos     = 0;
    yTmpStorage         = 0;
    yTmpStorageSize     = 0; 
    pixmapStartPixPos   = 0;

    grabbedCursor       = 0;
    scratchBuf          = 0;
    scratchBufLen       = 0;
    d_cursorCB          = 0;
    d_cursorListCB      = 0;

    d_backingPixmapValid = 0;
    pixmapYReallocThresh = 0;
    pixmapXReallocThresh = 0;
    timeScaleValid      = 0;
    timeScale           = 0;
    SDBRSigUpdateCb     = 0;
    SDBRSigListCb       = 0;
    SDBRSigDataUpdateCb = 0;
    xscroll_valid       = 0;

    d_startTrace.low    = 0;
    d_startTrace.high   = 0;
    d_endTrace.low      = 0;
    d_endTrace.high     = 0;

    pixmapStartTime     = 0;
    pixmapEndTime       = 0;

    d_runningMax        = 0;

    Tk_SetClass(getWin(), "WaveWidget");

    if (Tk_InitOptions(d_interp, (char *)&d_config,
                d_OptTab, getWin()) != TCL_OK) {
        return;
    }

    if (Configure(objc-2, &objv[2]) != TCL_OK) {
        fprintf(stderr, "ERROR: WaveWidget configure failed: %s\n",
                Tcl_GetStringResult(d_interp));
        return;
    }

    /**** Now that we're configured, setup the basic graphics stuff... ****/
    SetupGraphics();

    SetupBindings();

    /**** blank the window... ****/
    d_WinPixmap.blank();

    ComputeGeometry();

    LogRegionType   *lrt;
    LogMgr::FindRegionType("WaveWindow", &lrt);
    log_inst = new LogInstance(lrt, Tcl_GetString(objv[1]));

    log_low(log_inst, LogMgr::DebugLevel_Low);
    log_med(log_inst, LogMgr::DebugLevel_Med);
    log_high(log_inst, LogMgr::DebugLevel_High);

    LogDestination *ld = new LogDestination();

    log_inst->AddDestination(ld);
    ld->setLogLevel(LogMgr::DebugLevel_Off);

    WidgetMgr_AddInst(d_interp, "wave_widget", 
            Tcl_GetString(objv[1]), this);

    Tcl_SetObjResult(d_interp, objv[1]);

    ok = 1;
}

/******************************************************************
 * ~WaveWidget()
 ******************************************************************/
WaveWidget::~WaveWidget(void)
{
    Uint32    i;

    Tk_FreeConfigOptions((char *)&d_config, d_OptTab, getWin());

    if (SDBRSigUpdateCb) {
        CallbackMgr_DeleteCb(SDBRSigUpdateCb);
    }

    if (SDBRSigListCb) {
        CallbackMgr_DeleteCb(SDBRSigListCb);
    }

    if (SDBRSigDataUpdateCb) {
        CallbackMgr_DeleteCb(SDBRSigDataUpdateCb);
    }

    if (d_cursorCB) {
        CallbackMgr_DeleteCb(d_cursorCB);
    }

    if (d_cursorListCB) {
        CallbackMgr_DeleteCb(d_cursorListCB);
    }
}

/******************************************************************
 * InstCmd()
 ******************************************************************/
Int32 WaveWidget::InstCmd(Uint32 objc, Tcl_Obj *const objv[])
{
    Uint32 tmp, idx, no_offset;

    if (strncmp(Tcl_GetString(objv[1]), "config", 6)) {
        switch(CmdSwitch(ww_cmds, Tcl_GetString(objv[1]))) {
            case WWC_Redraw:
                d_backingPixmapValid = 0;
                SetupRedraw(RedrawForce);
                UpdateScrollbars();
            break;

            case WWC_Xview:
                return Xview(objc, objv);
                break;

            case WWC_Yview:
                return Yview(objc, objv);
                break;

            case WWC_TimeBounds:
                return TimeBounds();
            break;

            case WWC_SetLimits:
                return SetLimits(objc, objv);
                break;

            case WWC_GetLimits:
            {
                Tcl_Obj *list = Tcl_NewListObj(0,0);
                Tcl_ListObjAppendElement(d_interp, list,
                    Tcl_NewIntObj(displayStartTime));
                Tcl_ListObjAppendElement(d_interp, list,
                    Tcl_NewIntObj(displayStartTime+PixelsToTime(widgetWidth)));

                Tcl_SetObjResult(d_interp, list);
            }
            break;

            /********************************************************
             * WWC_PixToTime
             ********************************************************/
            case WWC_PixToTime:
                if (objc < 3) {
                    Tcl_AppendResult(d_interp, "Too few arguments (3 expected)", 
                            0);
                    return TCL_ERROR;
                }
                idx = 2;
                if (!strcmp(Tcl_GetString(objv[idx]), "-no_offset")) {
                    no_offset = 1;
                    idx++;
                } else {
                    no_offset = 0;
                }

                tmp = strtoul(Tcl_GetString(objv[idx]), 0, 0);

                Tcl_SetObjResult(d_interp, 
                        Tcl_NewIntObj((PixelsToTime(tmp)+
                            ((no_offset)?0:displayStartTime))));
            break;

            /********************************************************
             * TimeToPix
             ********************************************************/
            case WWC_TimeToPix:
                if (objc < 3) {
                    Tcl_AppendResult(d_interp, "Too few arguments (3 expected)", 
                            0);
                    return TCL_ERROR;
                }
                tmp = strtoul(Tcl_GetString(objv[2]), 0, 0);
                Tcl_SetObjResult(d_interp, 
                        Tcl_NewIntObj(TimeToPixels(tmp)+displayStartTime));
            break;

            case WWC_Cget:
                /**** ensure that the value is valid... ****/
                d_config.cursorpane_height = getCursorPaneHeight();

                Tcl_SetObjResult(d_interp, 
                        Tk_GetOptionValue(d_interp, (char *)&d_config, d_OptTab,
                            objv[2], getWin()));
                break;

            case WWC_Print:
                return Print(objc-1, &objv[1]);
                break;

            case WWC_PageLayout:
                return PageLayout(objc-1, &objv[1]);
                break;

            case WWC_PixPerTime:
                if (objc == 2) {
                    /**** Default: Return current pixPerTime ****/
                    Tcl_SetObjResult(d_interp, 
                            Tcl_NewDoubleObj(pixelsPerTimeUnit));
                } else if (objc == 4) {
                    /**** We've been passed:  Pixmap time/page 
                     **** Based on this, we want to calculate pix/time
                     ****/
                    return CalcPagePixPerTime(objc-2, &objv[2]);
                } else {
                    Tcl_AppendResult(d_interp, "too few args", 0);
                    return TCL_ERROR;
                }
                break;

            /********************************************************
             * TimePerPage
             ********************************************************/
            case WWC_TimePerPage:
                if (objc == 4) {
                    return CalcTimePerPage(objc-2, &objv[2]);
                } else {
                    Tcl_AppendResult(d_interp, "too few args", 0);
                    return TCL_ERROR;
                }
                break;

            /********************************************************
             * update_sigvals
             ********************************************************/
            case WWC_UpdateSigVals:
                UpdateSigValTree(d_sigValTime);
                break;

            default:
                break;
        }
    } else { 
        return Configure(objc-2, &objv[2]);
    }

    return TCL_OK;
}

/******************************************************************
 * cursorEventCB()
 ******************************************************************/
int WaveWidget::cursorEventCB(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               ojbc,
                Tcl_Obj          *const objv[])
{
    WaveWidget *wavew = (WaveWidget *)clientData;

    wavew->SetupRedraw(RepaintCursor);
    return TCL_OK;
}

/******************************************************************
 * SetLimits
 ******************************************************************/
Int32 WaveWidget::SetLimits(Uint32 objc, Tcl_Obj *const objv[])
{
    Double    newPixTime;
    Int32     lb, ub, delta, start, end;

    log_low.enter("SetLimits()\n");

    if (objc < 4) {
        char buf[8];
        sprintf(buf, "%d", objc);
        Tcl_AppendResult(d_interp, "Too few args - expecting 4 got ", buf, 0);
        return TCL_ERROR;
    }

    timeScaleValid = 0;

    if (Tcl_GetIntFromObj(d_interp, objv[2], &lb) != TCL_OK) {
        log_low.leave("SetLimits(ERROR)\n");
        return TCL_ERROR;
    }
    if (Tcl_GetIntFromObj(d_interp, objv[3], &ub) != TCL_OK) {
        log_low.leave("SetLimits(ERROR)\n");
        return TCL_ERROR;
    }
    
    if (ub > lb) {
        delta = ub-lb;
        start = lb;
        end   = ub;
    } else {
        delta = lb-ub;
        start = ub;
        end   = lb;
    }
   
    /**** Can't zoom to time=time ****/
    if (!(end > start)) {
        log_low.leave("SetLimits(Ignore)\n");
        return TCL_OK;
    }

    /*** Want to calculate pixels/time unit 
     *** - Already know how many time units we wish to cover
     *** - Know how big the display is (widgetWidth)
     ***/
    newPixTime = (((Double)(widgetWidth))/((Double)delta));

    pixelsPerTimeUnit = newPixTime;
    newDisplayStartTime = displayStartTime = start;
    newDisplayEndTime   = displayEndTime   = end;

    if (end > totalTime) {
        totalTime = end;
    }

    UpdateScrollbars();

    log_low("SetLimits: lb = %d ; ub = %d ; start = %d ; zoom = %f\n",
            lb, ub, newDisplayStartTime, pixelsPerTimeUnit);

    setWindow();
    SetupRedraw(RedrawForce);

    log_low.leave("SetLimits()\n");
    return TCL_OK;
}


/******************************************************************
 * TimeBounds()
 ******************************************************************/
Int32 WaveWidget::TimeBounds(void)
{
    Uint32   time = 0;

    Tcl_Obj *list = Tcl_NewListObj(0,0);
    Tcl_ListObjAppendElement(d_interp, list, 
            Tcl_NewIntObj(0));

    if (sdbr) {
        time = sdbr->getMaxTime();
    }

    Tcl_ListObjAppendElement(d_interp, list, Tcl_NewIntObj(time));

    Tcl_SetObjResult(d_interp, list);

    return TCL_OK;
}

/******************************************************************
 * SetupGraphics()
 ******************************************************************/
void WaveWidget::SetupGraphics()
{
    static char           *colorNames[] = {
        "vector_exp_1_color", "vector_exp_0_color", "vector_exp_x_color",
        "vector_exp_z_color", "vector_exp_chg_color", "vector_bin_color", 
        "vector_chg_color", "vector_x_color", 
        "vector_z_color", "scalar_1_color", "scalar_0_color", 
        "scalar_x_color", "scalar_z_color", "scalar_chg_color",
        "cursor_unsel_color",
        "cursor_sel_color", "cursor_lock_unsel_color",
        "cursor_lock_sel_color", "scale_color", "timescale_text_color",
        "selbox_color", "cursor_measure_color", "separator_color"
    };

    GCObj::invalidate(getGCs(), TC_NumColors);
    GCObj *gcObjs = getGCs();

    /**** First, free objects... ****/
    if (d_config.background) {
        Tk_Free3DBorder(d_config.background);
    }

    for (Uint32 i=0; i<TC_NumColors; i++) {
        if (d_config.colors[i]) {
            Tk_FreeColor(d_config.colors[i]);
        }
    }

    /**** Get the background color... ****/
    d_config.background = Tk_Get3DBorder(d_interp, getWin(), 
            getSchemeValue("wavewin_bg"));
   
    /**** Need to get all colors... ****/
    for (Uint32 i=0; i<TC_NumColors; i++) {
        d_config.colors[i] = Tk_GetColor(d_interp, getWin(),
                getSchemeValue(colorNames[i]));
    }

    /**** Now, setup base attributes of all GCs ****/
    for (Uint32 i=0; i<TC_NumColors; i++) {
        gcObjs[i].setFont(d_config.tkfont);
        gcObjs[i].setBgColor(d_config.background);
        gcObjs[i].setFgColor(d_config.colors[i]);
    }

    /**** Set line-width for selected and normal cursors... ****/
    gcObjs[TC_CursorUnselColor].setLineWidth(d_config.cursorWidth);
    gcObjs[TC_CursorSelColor].setLineWidth(d_config.cursorSelWidth);
    gcObjs[TC_CursorLockUnselColor].setLineWidth(d_config.cursorWidth);
    gcObjs[TC_CursorLockSelColor].setLineWidth(d_config.cursorSelWidth);

    d_WinPixmap.setBgColor(d_config.background);
    d_backingPixmap.setBgColor(d_config.background);
    d_displayPixmap.setBgColor(d_config.background);

    SetupRedraw(RedrawForce);
}

/******************************************************************
 * getSchemeValue()
 ******************************************************************/
const char *WaveWidget::getSchemeValue(const char *val_name)
{
    static struct {
        const char             *value_name;
        const char             *value_default;
    } defaults[] = {
        { "wavewin_bg",                       "black" },
        { "vector_exp_1_color",               "green" },
        { "vector_exp_0_color",               "green" },
        { "vector_exp_x_color",               "red"   },
        { "vector_exp_z_color",               "blue"  },
        { "vector_exp_chg_color",             "green" },
        { "vector_bin_color",                 "green" },
        { "vector_x_color",                   "red"   },
        { "vector_z_color",                   "blue"  },
        { "scalar_1_color",                   "green" },
        { "scalar_0_color",                   "green" },
        { "scalar_x_color",                   "red"   },
        { "scalar_z_color",                   "cyan"  },
        { "scalar_chg_color",                 "green" },
        { "cursor_unsel_color",               "blue"  },
        { "cursor_sel_color",                 "blue"  },
        { "cursor_lock_unsel_color",          "red"   },
        { "cursor_lock_sel_color",            "red"   },
        { "scale_color",                      "white"  },
        { "timescale_text_color",             "white"  },
        { "selbox_color",                     "green"  },
        { "cursor_measure_color",             "grey95" },
        { "separator_color",                  "white"  },
        { 0,                                  0        }
    }; 
    int   idx = 0;

    /**** First, try to get a value from the config database... ****/
    char *optName = new char [strlen(d_config.colorScheme)+2+strlen(val_name)];
    strcpy(optName, d_config.colorScheme);
    strcat(optName, ".");
    strcat(optName, val_name);

    const char *val;

    if ((val = ConfigDB_GetCurrent(optName))) {
        delete [] optName;
        return val;
    }
    delete [] optName;

    while (defaults[idx].value_name) {
        if (!strcmp(defaults[idx].value_name, val_name)) {
            return defaults[idx].value_default;
        }
        idx++;
    }
       
    fprintf(stderr, "ERROR: unable to get value for \"%s\"\n", val_name);
    return 0;
}

/******************************************************************
 * Configure()
 ******************************************************************/
int WaveWidget::Configure(Uint32 objc, Tcl_Obj *const objv[])
{
    Tk_SavedOptions        savedOptions;
    int                    mask;

    log_med.enter("Configure(%d)\n", objc);

    for (Uint32 i=0; i<objc; i+=2) {
        log_med("%s %s\n", Tcl_GetString(objv[i]), Tcl_GetString(objv[i+1]));
    }

    if (Tk_SetOptions(d_interp, (char *)&d_config, d_OptTab,
                objc, objv, getWin(), &savedOptions, &mask) != TCL_OK) {
        Tk_RestoreSavedOptions(&savedOptions);
        log_med.leave("Configure(%d) (ERROR)\n", objc);
        return TCL_ERROR;
    }

    if (mask & Opt_Graphics) {
        SetupGraphics();
        GCObj::invalidate(getGCs(), TC_NumColors);
    }

    if (mask & Opt_ColorScheme) {
        SetupGraphics();
        GCObj::invalidate(getGCs(), TC_NumColors);
    }

    if ((mask & Opt_Sdbr) && d_config.sdbrName && d_config.sdbrName[0]) {
        SdbReader    *newSdbr;

        newSdbr = (SdbReader *)WidgetMgr_GetObjHandle(d_interp, "SDBR",
                d_config.sdbrName);

        if (!newSdbr) {
            Tcl_AppendResult(d_interp, "SDBR ", d_config.sdbrName, 
                    "doesn't exist", 0);
            log_med.leave("Configure(%d) (ERROR)\n", objc);
            return TCL_ERROR;
        }

        /**** Cleanup after previous sdbr ****/
        if (d_cursorCB) {
            CallbackMgr_DeleteCb(d_cursorCB);
        }

        CallbackMgr_AddCb(CBTYPE_CURSOR_LIST_UPDATE, d_config.sdbrName,
                WaveWidget::cursorEventCB, this, &d_cursorListCB);
        CallbackMgr_AddCb(CBTYPE_CURSOR_POS_CHANGE, d_config.sdbrName,
                WaveWidget::cursorEventCB, this, &d_cursorCB);

        CallbackMgr_AddCb(CBTYPE_SDBR_SIGLIST_UPDATE, d_config.sdbrName,
                WaveWidget::SDBRSigUpdateCB, this, &SDBRSigListCb);
        CallbackMgr_AddCb(CBTYPE_SDBR_SIGDATA_UPDATE, d_config.sdbrName,
                WaveWidget::SDBRSigDataUpdateCB, this, &SDBRSigDataUpdateCb);

        sdbr = newSdbr;
    }

    if ((mask & Opt_SigTree) && d_config.sigTreeName && 
            d_config.sigTreeName[0]) {
        TreeWidget    *sigTree;

        sigTree = (TreeWidget *)WidgetMgr_GetObjHandle(d_interp,
                "tree_widget", d_config.sigTreeName);

        if (!sigTree) {
            Tcl_AppendResult(d_interp, "Tree Widget ", d_config.sigTreeName,
                    " doesn't exist", 0);
            return TCL_ERROR;
        }
        d_sigTree = sigTree;
    }

    if ((mask & Opt_SigValTree) && d_config.sigValTreeName &&
            d_config.sigValTreeName[0]) {
        TreeWidget        *sigValTree;

        sigValTree = (TreeWidget *)WidgetMgr_GetObjHandle(d_interp,
                "tree_widget", d_config.sigValTreeName);

        if (!sigValTree) {
            Tcl_AppendResult(d_interp, "Tree Widget ", 
                    d_config.sigValTreeName, " doesn't exist", 0);
            return TCL_ERROR;
        }
        d_sigValTree = sigValTree;
    }

    Tk_FreeSavedOptions(&savedOptions);

    log_med.leave("Configure(%d)\n", objc);

    return TCL_OK;
}

/******************************************************************
 * UpdateXScroll()
 * Calculate the extents of the scrollbar. Return the 
 * right edge (%) of the scrollbar. 
 * Set the scrollbar pos if the 'set' var is true
 ******************************************************************/
void WaveWidget::UpdateXScroll(void)
{
    Double        end      = 1.0;
    Double        begin    = 0.0;
    Double        tmp;
    SdbSignal    *sig;
    Char          cmd[128];

    log_high.enter("UpdateXScroll()\n");

    if (d_config.xscrollcommand) { 

        if (totalTime) {
            log_high("displayStartTime=%d totalTime=%d widgetWidth=%d\n",
                    displayStartTime, totalTime, widgetWidth);

            begin = (Double)((Double)displayStartTime)/((Double)totalTime);
            tmp = displayStartTime + PixelsToTime(widgetWidth);
            end   = (Double)((Double)tmp)/((Double)totalTime);
        }
 
        sprintf(cmd, " %g %g ", begin, end);

        log_high("cmd is: %s %s\n", d_config.xscrollcommand, cmd);
        if (Tcl_VarEval(d_interp, d_config.xscrollcommand, cmd, 0) != TCL_OK) {
            fprintf(stderr, "ERROR: XScroll command failed\n");
        }
    } else {
        log_high("no xscrollcommand\n");
    }

    log_high.leave("UpdateXScroll()\n");
}

/******************************************************************
 * UpdateScrollbars()
 ******************************************************************/
void WaveWidget::UpdateScrollbars(void)
{
    Char          cmd[1024];
    SdbSignal    *sig;
    Double        begin = 0.0;
    Double        end   = 1.0;
    Double        beginTime, lastTime, fullTime;
    Uint32        tmp;

    if (d_config.yscrollcommand != 0) {
	    /*
        sprintf(cmd, yscrollcommand);
        
        Tcl_Eval(d_interp, yscrollcommand);
	     */
    }

    UpdateXScroll();
}

/******************************************************************
 * Xview
 *
 * Setup a request for a new displayed area and schedule a 
 * redraw...
 *
 * - total
 * - visible
 * - display_start
 ******************************************************************/
Int32 WaveWidget::Xview(Uint32 objc, Tcl_Obj *const objv[])
{
    Int32      type;
    Uint32     tmp;
    double     fraction;
    Int32      count;
    Uint32     redraw = 0;
    Int32      tmpX;
    Uint32     tmpStartTime;

    log_high.enter("Xview()\n");

    for (Uint32 i=0; i<objc; i++) {
        log_high("%s\n", Tcl_GetString(objv[i]));
    }

    type = Tk_GetScrollInfoObj(d_interp, objc, objv, &fraction, &count);

    switch (type) {
        default:
        case TK_SCROLL_ERROR:
            log_high.leave("Xview() (ERROR)\n");
            return TCL_ERROR;
            break;

        case TK_SCROLL_MOVETO:
            if (fraction >= 0.0 && fraction <= 1.0) {
                tmpStartTime = (Uint32)(fraction*totalTime);
            } else if (fraction < 0.0) {
                tmpStartTime = 0;
            } else {
                tmpStartTime = ((totalTime+50) - PixelsToTime(widgetWidth));
            }
            break;

        case TK_SCROLL_PAGES:
            if (count > 0) { 
                tmpX = (PixelsToTime(widgetWidth) * 8)/10;
                tmpX = displayStartTime + tmpX;
                if (tmpX < totalTime) {
                    tmpStartTime = tmpX;
                }
            } else {
                tmpX = (PixelsToTime(widgetWidth) * 8)/10;
                tmpX = displayStartTime - tmpX;
                if (tmpX >= 0) {
                    tmpStartTime = tmpX;
                } else {
                    tmpStartTime = 0;
                } 
            }
            break;

        case TK_SCROLL_UNITS:
            if ((displayStartTime + (5*count)) >= 0) {
                tmpX = displayStartTime + (5*count);
                if (tmpX < totalTime && tmpX >= 0) {
                    tmpStartTime = tmpX;
                }
            }
        break;
    }


    if ((tmpStartTime+PixelsToTime(widgetWidth)) < (totalTime+50)) {
        newDisplayStartTime = tmpStartTime;
    } else {
        newDisplayStartTime = (totalTime+50) - PixelsToTime(widgetWidth);
    }

    setWindow();

    log_high("newDisplayStartTime = %d\n", newDisplayStartTime);
    UpdateXScroll();
    SetupRedraw(RepaintScrollX);

    log_high.leave("Xview()\n");
    return TCL_OK; 
}

/******************************************************************
 * Yview
 *
 * Setup a request for a new displayed area and schedule a 
 * redraw...
 ******************************************************************/
Int32 WaveWidget::Yview(Uint32 objc, Tcl_Obj *const objv[])
{
    Int32  type;
    double fraction;
    Int32  count;
    Uint32 redraw = 1;
    Uint32 i, totalSize = 0;

    if (!sdbr) {
        return TCL_OK;
    }

    for (i=0; i<sdbr->d_signals.length(); i++) {
        totalSize += sdbr->d_signals.idx(i)->getVisTraces();
    }

    totalSize *= getTraceSep();

    type = Tk_GetScrollInfoObj(d_interp, objc, objv, &fraction, &count);

    switch (type) {
        default:
        case TK_SCROLL_ERROR:
            return TCL_ERROR;
        break;

        case TK_SCROLL_MOVETO:
            newDisplayStartPos = (Uint32)(totalSize * fraction);
            log_high("yview moveto %f\n", fraction);
        break;

        case TK_SCROLL_PAGES:
            log_high("yview scroll %d pages\n", count);
        break;

        case TK_SCROLL_UNITS:
            log_high("yview scroll %d units\n", count);
        break;
    }

    if (redraw) {
        SetupRedraw(RepaintScrollY);
    }
   
    return TCL_OK; 
}

/****************************************************************** 
 * IdleProc()
 *
 * Top-level display function.  This function is called for 
 * several reasons:
 *  - Size change
 *  - Expose event
 *  - Scrollbar change
 *  - Mouse drag (selection cursor)
 *  - Mouse draw (moving cursor)
 *  - Request from application
 *    - end of sim step
 *    - addition of cursor
 *
 * We may be required to do one of several things:
 *  - Just copy the display pixmap to the display
 *  - Move the display pixmap within the backing pixmap
 *  - Copy portions of the backing pixmap, update part of the
 *    backing pixmap, and copy to the display pixmap
 ******************************************************************/
void WaveWidget::IdleProc()
{
    Uint32 copyStartX, copyStartY;

    if (!winValid()) {
        return;
    }

    Tk_DefineCursor(getWin(), d_config.cursor);

    if (IS_FLAG_SET(RedrawForce)) {
        log_med("Invalidate pixmap\n");
        d_backingPixmapValid = 0;
    }

   /*** Check to be sure that the pixmaps are appropriately sized ***/
    if (IS_FLAG_SET(RepaintExpose) || !d_backingPixmapValid) {
        log_high("Updating Pixmaps\n");
        setWindow();
        UpdatePixmaps();
    }

    d_displayPixmap.blank();

   /*** Redraw the backing pixmaps if necessary ***/
    if (!d_backingPixmapValid || IS_FLAG_SET(RepaintScrollX|RepaintScrollY)) {
        Tk_DefineCursor(getWin(), d_config.busyCursor);
        RepaintBackingPixmaps();
        Tk_DefineCursor(getWin(), d_config.activeCursor);
    }

   /*** Copy "correct" area to display pixmap ***/
    displayEndTime = displayStartTime + PixelsToTime(widgetWidth);
    copyStartX = TimeToPixels(displayStartTime - pixmapStartTime);
    copyStartY = displayStartPos - pixmapStartPixPos;

    log_high("----> displayPixmap = backingPixmap\n");

    /**** Just copy up to the start of the CursorPane ****/
    d_displayPixmap = d_backingPixmap(copyStartX, 
            copyStartY, width(), height()-getCursorPaneHeight());
    log_high("<---- displayPixmap = backingPixmap\n");

   /*** Draw any overlay info on the display pixmap ***/
    if (IS_FLAG_SET(RepaintMouseDrag) && IS_FLAG_SET(BUTTONS_DOWN)) {
        log_high("Drawing selection box\n");
        DrawSelectionBox(d_displayPixmap);
    }

    if (d_sigValNeedUpdate) {
        UpdateSigValTree(d_sigValTime);
        d_sigValNeedUpdate = false;
    }

   /*** Finally, copy to the screen... ***/
    d_PreOverlay(d_displayPixmap, d_startTrace, d_endTrace); 

    if (sdbr) {
        Uint32 cwStart   = (d_displayPixmap.height()-getCursorPaneHeight());
        Uint32 tagCenter = cwStart+(getTraceSep()/2);
        DrawTimeScale(d_displayPixmap, displayStartTime, displayEndTime,
                height()-getCursorPaneHeight(), getCursorPaneHeight(), 1);
        DrawCursors(sdbr->cursorMgr, d_displayPixmap, 1, tagCenter, 1);
    }

    d_PostOverlay(d_displayPixmap, d_startTrace, d_endTrace);

    log_high("----> d_WinPixmap = displayPixmap\n");
    d_WinPixmap = d_displayPixmap();
    log_high("<---- d_WinPixmap = displayPixmap\n");

    CLR_FLAG(RepaintCursor|RepaintExpose);

    if (!xscroll_valid) {
        xscroll_valid = 1;
        UpdateXScroll();
    }
    Tk_DefineCursor(getWin(), d_config.cursor);
}

/******************************************************************
 * UpdateTimeScale()
 *
 * Based on the current timescale, find out the distance between
 * timescale bars...
 *
 * Inputs:
 * - pixmapStartTime
 *   Start time for the pixmap
 *
 * - pixmapEndTime
 *   End time for the pixmap
 *
 * - pixmapWidth
 *   Width in pixels of the pixmap
 *
 * Return:
 * - Returns the distance between bars
 ******************************************************************/
Uint32 WaveWidget::UpdateTimeScale(
        Uint32        pixmapStartTime,
        Uint32        pixmapEndTime,
        Uint32        pixmapWidth)
{
    Tk_TextLayout     layout;
    Char              buf[32];
    int               t_width, t_height;
    Double            tPerPix, dt;
    Uint32            i, majorTs, minorTs, tmp;
    static            Uint32   timeScaleConsts[] = {
        1,  5, 10, 100, 1000, 10000, 100000, 1000000,
        10000000, 100000000};
    Uint32            numScaleConsts = sizeof(timeScaleConsts)/sizeof(Uint32);

    tPerPix = (((Double)(pixmapEndTime-pixmapStartTime))/(Double)pixmapWidth);

    if (!sdbr) {
        return 1000000;
    }

    /**** First, find out how wide the max timestamp will be
     ****/
    sprintf(buf, "%d%s", pixmapEndTime, sdbr->get_resString());
    layout = Tk_ComputeTextLayout(d_config.tkfont, buf, strlen(buf), 0,
            TK_JUSTIFY_CENTER, 0, &t_width, &t_height);
    Tk_FreeTextLayout(layout);

    /**** Find out how much time a maximally-sized label takes
     ****/
    dt = (tPerPix*(Double)t_width);

    /**** Okay, find the smallest timescale that contains dt
     ****/
    for (i=0; i<numScaleConsts; i++) {
        if (timeScaleConsts[i] > dt) {
            majorTs = timeScaleConsts[i];
            break;
        }
    }

    /**** Now, we want to divide this timescale step a bit more...
     ****/
    tmp = minorTs = majorTs;
    while (minorTs > 1) {
        tmp = tmp / 2;

        if (majorTs > 100) {
            if ((dt > tmp) || (tmp % 10)) {
                break;
            } else {
                minorTs = tmp;
            }
        } else if (majorTs > 10) {
            if ((dt > tmp) || (tmp % 5)) {
                break;
            } else {
                minorTs = tmp;
            }
        } else if (majorTs > 5) {
            if ((dt > tmp) || (tmp % 2)) {
                break;
            } else {
                minorTs = tmp;
            }
        } else if (dt < tmp) {
            minorTs = tmp;
        } else {
            break;
        }
    }

    return minorTs;
}


/******************************************************************
 * ComputeGeometry()
 ******************************************************************/
void WaveWidget::ComputeGeometry(void)
{
    Tk_GeometryRequest(getWin(), 100, 100);
    Tk_SetInternalBorder(getWin(), 10);
}

/******************************************************************
 * SendSelBoxCmd()
 ******************************************************************/
void WaveWidget::SendSelBoxCmd(
        Uint32 xi, 
        Uint32 yi,
        Uint32 xf, 
        Uint32 yf)
{
    Uint32 ti, tf, tmp;

    if (xi > xf) {
        tmp = xi;
        xi = xf;
        xf = tmp;
    }

    if (yi > xf) {
        tmp = yi;
        yi = yf;
        yf = tmp;
    }

    if (scratchBufLen <= (strlen(d_config.selectBoxCmd)+128)) {
        if (scratchBuf) {
            delete[] scratchBuf;
        }
        scratchBuf = new Char[strlen(d_config.selectBoxCmd)+128];
        scratchBufLen = strlen(d_config.selectBoxCmd)+128;
    }

    sprintf(scratchBuf, "%s %d %d %d %d", d_config.selectBoxCmd, 
            xi, yi, xf, yf);

    if (Tcl_Eval(d_interp, scratchBuf) != TCL_OK) {
        fprintf(stderr, "ERROR: cmd failed - %s\n",
            Tcl_GetStringResult(d_interp));
    }
}

/********************************************************************
 * setWindow()
 *
 * Args are:
 * - time_start
 * - time_end
 * - pix_per_time
 ********************************************************************/
void WaveWidget::setWindow()
{
    Tcl_SetIntObj(d_cbArgs[0], displayStartTime);
    Tcl_SetIntObj(d_cbArgs[1], (displayStartTime+PixelsToTime(width())));
    Tcl_SetDoubleObj(d_cbArgs[2], pixelsPerTimeUnit);

    d_WinChangeCBList->Invoke(3, d_cbArgs);
}

/********************************************************************
 * getOptionSpec()
 ********************************************************************/
Tk_OptionSpec *WaveWidget::getOptionSpec()
{
    static Tk_OptionSpec optSpecs[] = {
        {TK_OPTION_STRING, "-color_scheme", "ColorScheme", "ColorScheme",
            "WaveWidget.ColorSchemes.default", 
            -1, Tk_Offset(WaveConfig, colorScheme), 0, 0, Opt_ColorScheme},

        {TK_OPTION_FONT, "-font", "font", "Font", "Fixed 18",
            -1, Tk_Offset(WaveConfig, tkfont), 0, 0, Opt_Graphics},

        /**** 
         ****/
        {TK_OPTION_INT, "-trace_height", "trace_height", "TraceHeight",
            "16", -1, Tk_Offset(WaveConfig, traceHeight), 0, 0, 0},
        {TK_OPTION_INT, "-cursorwidth", "cursorwidth", "CursorWidth",
            "1", -1, Tk_Offset(WaveConfig, cursorWidth), 0, 0, 0},
        {TK_OPTION_INT, "-cursorselwidth", "cursorselwidth", "CursorSelWidth",
            "2", -1, Tk_Offset(WaveConfig, cursorSelWidth), 0, 0, 0},

        {TK_OPTION_STRING, "-sdbr", "sdbr", "SdbReader",
            (char *)NULL, -1, Tk_Offset(WaveConfig, sdbrName), 
            0, 0, Opt_Sdbr},
        {TK_OPTION_STRING, "-sig_tree", "sig_tree", "SigTree",
            (char *)NULL, -1, Tk_Offset(WaveConfig, sigTreeName), 
            0, 0, Opt_SigTree},
        {TK_OPTION_STRING, "-sigval_tree", "sigval_tree", "SigValTree",
            (char *)NULL, -1, Tk_Offset(WaveConfig, sigValTreeName), 
            0, 0, Opt_SigValTree},

        /**** Cursor mode - "select" or "zoom"
         ****/
        {TK_OPTION_STRING, "-cursor_mode", "cursor_mode",
            "CursorMode", "select", -1,
            Tk_Offset(WaveConfig, cursorMode), 0, 0, 0},

        {TK_OPTION_STRING, "-yscrollcommand", "yscrollcommand", 
            "YScrollCommand", (char *)NULL, -1,
            Tk_Offset(WaveConfig, yscrollcommand), 0, 0, 0},
        {TK_OPTION_STRING, "-xscrollcommand", "xscrollcommand", 
            "XScrollCommand", (char *)NULL, -1,
            Tk_Offset(WaveConfig, xscrollcommand), 0, 0, 0},
        {TK_OPTION_STRING, "-progresscommand", "progresscommand",
            "ProgressCommand", (char *)NULL, -1,
            Tk_Offset(WaveConfig, progressCommand), 0, 0, 0},
        {TK_OPTION_STRING, "-cursor_move_cmd", "cursor_move_cmd",
            "CursorMoveCmd", (char *)NULL, -1,
            Tk_Offset(WaveConfig, cursorMoveCmd), 0, 0, 0},

        {TK_OPTION_STRING, "-select_box_cmd", "select_box_cmd", "SelBoxCmd",
            (char *)NULL, -1, Tk_Offset(WaveConfig, selectBoxCmd), 0, 0, 0},

        {TK_OPTION_CURSOR, "-active_cursor", "active_cursor", "ActiveCursor",
            "arrow", -1, Tk_Offset(WaveConfig, activeCursor), 0, 0, 0},
        {TK_OPTION_CURSOR, "-busy_cursor", "busy_cursor", "BusyCursor",
            "watch", -1, Tk_Offset(WaveConfig, busyCursor), 0, 0, 0},

        {TK_OPTION_INT, "-pady", "pady", "PadY",
            "2", -1, Tk_Offset(WaveConfig, trace_pady), 0, 0, 0},

        {TK_OPTION_INT, "-cursorpane_height", "cursorpane_height", "CursorPane",
            "18", -1, Tk_Offset(WaveConfig, cursorpane_height), 0, 0, 0},

        {TK_OPTION_CURSOR, "-cursor2", "cursor2", "Cursor",
            "", -1, Tk_Offset(WaveConfig, cursor), 
            TK_OPTION_NULL_OK, 0, 0},

        {TK_OPTION_END, (char *)NULL, (char *)NULL,
            (char *)NULL, (char *)NULL, -1, 0, 0, 0, 0}
    };

    return optSpecs;
}

/********************************************************************
 * CreateInst()
 ********************************************************************/
int WaveWidget::CreateInst(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               objc,
                Tcl_Obj          *const objv[])
{
    WaveWidget *wave_w = new WaveWidget(interp, objc, objv);

    return (wave_w->ok)?TCL_OK:TCL_ERROR;
}

Tk_OptionTable WaveWidget::d_OptTab = 0;

/********************************************************************
 * Wave_widget_Init()
 *
 * Procedure is called by the ModuleLoader...
 ********************************************************************/
extern "C" int Wave_widget_Init(Tcl_Interp *d_interp)
{
    LogRegionType   *lr;
    
    Tcl_PkgProvide(d_interp, "wave_widget", "8.1");

    lr = new LogRegionType("WaveWindow", 
            "Wave Window",
            "",
            "",
            "");

    LogMgr::AddRegionType(lr);

    WaveWidget::d_OptTab = Tk_CreateOptionTable(d_interp,
            WaveWidget::getOptionSpec());

    CallbackMgr_AddCbType(CBTYPE_WAVE_WINDOW_CHANGE);

    return ObjWidgetBase::WidgetInit(
            d_interp,
            "wave_widget",
            "1.0",
            "wave_widget",
            &WaveWidget::CreateInst);
}

extern "C" int Wave_widget_SafeInit(Tcl_Interp *d_interp)
{
    return Wave_widget_Init(d_interp);
}

