/*
 bans.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gboolean bantype;

static void ban_free(GList **list, BAN_REC *rec)
{
    g_return_if_fail(list != NULL);
    g_return_if_fail(rec != NULL);

    g_free(rec->ban);
    if (rec->setby != NULL) g_free(rec->setby);
    g_free(rec);

    *list = g_list_remove(*list, rec);
}

void banlist_free(GList *banlist)
{
    while (banlist != NULL)
        ban_free(&banlist, banlist->data);
}

BAN_REC *ban_add(CHANNEL_REC *channel, gchar *ban, gchar *nick, time_t time)
{
    BAN_REC *rec;

    g_return_val_if_fail(channel != NULL, NULL);
    g_return_val_if_fail(ban != NULL, NULL);
    g_return_val_if_fail(nick != NULL, NULL);

    rec = g_new(BAN_REC, 1);
    rec->ban = g_strdup(ban);
    rec->setby = g_strdup(nick);
    rec->time = time;

    channel->banlist = g_list_append(channel->banlist, rec);

    signal_emit("ban new", 1, rec);
    return rec;
}

void ban_remove(CHANNEL_REC *channel, gchar *ban)
{
    GList *tmp;

    g_return_if_fail(ban != NULL);

    for (tmp = g_list_first(channel->banlist); tmp != NULL; tmp = tmp->next)
    {
        BAN_REC *rec = tmp->data;

        if (g_strcasecmp(rec->ban, ban) == 0)
        {
            signal_emit("ban remove", 1, rec);
            ban_free(&channel->banlist, rec);
            break;
        }
    }
}

BAN_REC *ban_exception_add(CHANNEL_REC *channel, gchar *ban, gchar *nick, time_t time)
{
    BAN_REC *rec;

    g_return_val_if_fail(channel != NULL, NULL);
    g_return_val_if_fail(ban != NULL, NULL);
    g_return_val_if_fail(nick != NULL, NULL);

    rec = g_new(BAN_REC, 1);
    rec->ban = g_strdup(ban);
    rec->setby = g_strdup(nick);
    rec->time = time;

    channel->ebanlist = g_list_append(channel->ebanlist, rec);

    signal_emit("ban exception new", 1, rec);
    return rec;
}

void ban_exception_remove(CHANNEL_REC *channel, gchar *ban)
{
    GList *tmp;

    g_return_if_fail(ban != NULL);

    for (tmp = g_list_first(channel->ebanlist); tmp != NULL; tmp = tmp->next)
    {
        BAN_REC *rec = tmp->data;

        if (g_strcasecmp(rec->ban, ban) == 0)
        {
            signal_emit("ban exception remove", 1, rec);
            ban_free(&channel->ebanlist, rec);
            break;
        }
    }
}

/* Get ban mask */
gchar *ban_get_mask(CHANNEL_REC *channel, gchar *nick)
{
    NICK_REC *rec;
    gchar *str, *p, *l;
    gint count;

    g_return_val_if_fail(channel != NULL, NULL);
    g_return_val_if_fail(nick != NULL, NULL);

    rec = nicklist_find(channel, nick);
    if (rec == NULL || rec->host == NULL) return NULL;

    str = irc_get_mask(nick, rec->host, bantype);

    /* there's a limit of 10 characters in user mask. so, banning a guy with
       user mask of 10 characters gives us "*1234567890", which is one too
       much.. so, replace the 10th character with '*' */
    p = strchr(str, '!');
    if (p == NULL) return str;

    count = 0; l = NULL;
    while (*p != '@' && *p != '\0')
    {
	if (count++ == 10)
	    l = p+1;
	p++;
    }

    if (l != NULL && l != p && *p == '@')
    {
	l[-1] = '*';
	g_memmove(l, p, strlen(p)+1);
    }
    return str;
}

static gboolean bans_channel_destroyed(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    banlist_free(channel->banlist);
    banlist_free(channel->ebanlist);
    return TRUE;
}

static gboolean cmd_bantype(gchar *data, SERVER_REC *server, CHANNEL_REC *curchan)
{
    GList *list, *tmp;
    gchar bantypestr[5];

    g_return_val_if_fail(data != NULL, FALSE);

    if (toupper(data[0]) == 'N')
    {
	bantype = IRC_MASK_USER | IRC_MASK_DOMAIN;
        strcpy(bantypestr, "UD");
    }
    else if (toupper(data[0]) == 'H')
    {
	bantype = IRC_MASK_HOST | IRC_MASK_DOMAIN;
        strcpy(bantypestr, "HD");
    }
    else if (toupper(data[0]) == 'D')
    {
	bantype = IRC_MASK_DOMAIN;
        strcpy(bantypestr, "D");
    }
    else if (toupper(data[0]) == 'C')
    {
        list = str2list(data, ' ');
        if (list->next != NULL)
        {
	    bantype = 0;
            for (tmp = list->next; tmp != NULL; tmp = tmp->next)
            {
                gchar *str = tmp->data;

                if (toupper(*str) == 'N')
                    bantype |= IRC_MASK_NICK;
                else if (toupper(*str) == 'U')
                    bantype |= IRC_MASK_USER;
                else if (toupper(*str) == 'H')
                    bantype |= IRC_MASK_HOST;
                else if (toupper(*str) == 'D')
                    bantype |= IRC_MASK_DOMAIN;
            }
	}
	if (list != NULL)
	{
	    g_free(list->data);
	    g_list_free(list);
	}

        bantypestr[0] = '\0';
        if (bantype & IRC_MASK_NICK) strcat(bantypestr, "N");
        if (bantype & IRC_MASK_USER) strcat(bantypestr, "U");
        if (bantype & IRC_MASK_HOST) strcat(bantypestr, "H");
        if (bantype & IRC_MASK_DOMAIN) strcat(bantypestr, "D");
    }

    signal_emit("ban type changed", 1, bantypestr);

    return TRUE;
}

static gboolean cmd_ban(gchar *data, SERVER_REC *server, CHANNEL_REC *curchan)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel, *nicks, *nick, *ptr;
    GString *str;
    gboolean banhost;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_OPTCHAN | PARAM_FLAG_GETREST,
                            curchan, &channel, &nicks);
    if (!ischannel(*channel)) cmd_param_error(CMDERR_NOT_JOINED);
    if (*nicks == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    chanrec = channel_find(server, channel);
    if (chanrec == NULL) cmd_param_error(CMDERR_CHAN_NOT_FOUND);
    if (!chanrec->wholist) cmd_param_error(CMDERR_CHAN_NOT_SYNCED);

    str = g_string_new(channel);
    while (*nicks != '\0')
    {
        nick = cmd_get_param(&nicks);

        banhost = FALSE;
        for (ptr = nick; *ptr != '\0'; ptr++)
        {
            if (!banhost && *ptr == '!')
                banhost = TRUE;
            else if (banhost && *ptr == '@')
                break;
        }
        if (*ptr == '\0') banhost = FALSE;

        if (banhost)
            ptr = nick;
        else
        {
            ptr = ban_get_mask(chanrec, data);
            if (ptr == NULL) continue;
        }

        g_string_sprintfa(str, " %s", ptr);
        if (!banhost) g_free(ptr);
    }

    modes_set(str->str, "+b", server, chanrec);

    g_string_free(str, TRUE);
    g_free(params);

    return TRUE;
}

static gboolean cmd_unban(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    GString *str;
    GList *tmp;

    str = g_string_new(NULL);
    for (tmp = g_list_first(channel->banlist); tmp != NULL; tmp = tmp->next)
    {
	BAN_REC *rec = tmp->data;

	if (match_wildcards(data, rec->ban))
	{
	    if (str->len == 0)
		g_string_assign(str, rec->ban);
	    else
		g_string_sprintfa(str, " %s", rec->ban);
	}
    }

    modes_set(str->str, "-b", server, channel);
    g_string_free(str, TRUE);

    return TRUE;
}

void bans_init(void)
{
    /* default to bantype */
    bantype = IRC_MASK_USER | IRC_MASK_DOMAIN;

    signal_add("channel destroyed", (SIGNAL_FUNC) bans_channel_destroyed);
    command_bind("bantype", NULL, (SIGNAL_FUNC) cmd_bantype);
    command_bind("ban", NULL, (SIGNAL_FUNC) cmd_ban);
    command_bind("unban", NULL, (SIGNAL_FUNC) cmd_unban);
}

void bans_deinit(void)
{
    signal_remove("channel destroyed", (SIGNAL_FUNC) bans_channel_destroyed);
    command_unbind("bantype", (SIGNAL_FUNC) cmd_bantype);
    command_unbind("ban", (SIGNAL_FUNC) cmd_ban);
    command_unbind("unban", (SIGNAL_FUNC) cmd_unban);
}
