/*
 setup-txt.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "../ui-common/txt.h"

#include "setup-int.h"
#include "setup-themes.h"
#include "setup-colors.h"

static gint linepos[TEXTS+1];

static GtkWidget *setup_dialog;
static CHANNEL_REC *txt_channel;

static gboolean sig_setup_apply(GtkWidget *dialog)
{
    proplist_t tprop, fprop, pkey, fname;
    GList *tmp;
    gint num;

    g_return_val_if_fail(dialog != NULL, FALSE);

    for (tmp = g_list_first(themes); tmp != NULL; tmp = tmp->next)
    {
        THEME_REC *rec = tmp->data;

	/* try to open theme.. */
	tprop = PLGetProplistWithPath(rec->path);
	if (tprop == NULL)
	{
	    /* not found, create new theme file */
	    tprop = PLMakeDictionaryFromEntries(NULL, NULL);
	    fname = PLMakeString(rec->path);
	    tprop = PLSetFilename(tprop, fname);
	    PLRelease(fname);
	}

	/* First clear the formats.. */
	pkey = PLMakeString("formats");
	PLRemoveDictionaryEntry(tprop, pkey);
	PLRelease(pkey);
	tprop = config_make_dict(tprop, "formats");
	fprop = config_get_prop(tprop, "formats");

	/* write list of theme formats */
        for (num = 0; num < TEXTS; num++)
	{
            if (rec->format[num] != NULL)
	    {
		if (strcmp(rec->format[num], default_formats[num].def) != 0)
		{
		    /* create dictionary for channel and add it to array */
		    config_set_str(fprop, default_formats[num].tag, rec->format[num]);
		}
                else
                {
                    g_free(rec->format[num]);
                    rec->format[num] = NULL;
                }
            }
	}
	if (!readonly) PLSave(tprop, TRUE);
	PLRelease(tprop);
    }

    return TRUE;
}

static void set_window_bg(WINDOW_VIEW_REC *view)
{
    GtkStyle *style;

    g_return_if_fail(view != NULL);

    style = gtk_style_copy(view->text->style);
    gtk_style_ref(style);
    style->base[0] = THEME_GUI(temp_current_theme)->colors[COLOR_BACKGROUND];
    gtk_widget_set_style(view->text, style);
    gtk_style_unref(style);

}
void setup_drawtext_set(CHANNEL_REC *channel, gint *linepos, gboolean init)
{
    static gfloat oldval;
    static gint oldlines, len;
    static THEME_REC *old_theme;

    WINDOW_VIEW_REC *view;
    WINDOW_REC *window;
    gint n;

    g_return_if_fail(channel != NULL);

    window = CHANNEL_PARENT(channel);
    view = WINDOW_VIEW(window);

    if (init)
    {
	old_theme = current_theme;
	current_theme = temp_current_theme;

        set_window_bg(view);

        gui_printtext_configure();

        oldval = GTK_TEXT(view->text)->vadj->value;
        oldlines = window->lines;
        len = gtk_text_get_length(GTK_TEXT(view->text));

        gtk_text_freeze(GTK_TEXT(view->text));
        gtk_text_set_point(GTK_TEXT(view->text), len);
    }
    else
    {
        for (n = 0; linepos[n] != -1; n++)
            linepos[n] -= len;
        window->lines = oldlines;

        /*FIXME: this doesn't remove the lines from linelen list*/
        gtk_text_set_point(GTK_TEXT(view->text), 0);
        gtk_text_forward_delete(GTK_TEXT(view->text), len);
        gtk_text_thaw(GTK_TEXT(view->text));
        gtk_adjustment_set_value(GTK_TEXT(view->text)->vadj, oldval);

        current_theme = old_theme;

        gui_printtext_configure();
    }
}

static gboolean setup_theme_redraw(THEME_REC *theme)
{
    GtkEntry *entry;
    GtkText *text;
    gint num;

    setup_drawtext_set(txt_channel, linepos, TRUE);

    text = GTK_TEXT(WINDOW_VIEW(CHANNEL_PARENT(txt_channel))->text);
    for (num = 0; num < TEXTS; num++)
    {
	if (default_formats[num].tag == NULL)
	{
	    printtext(NULL, txt_channel->name, MSGLEVEL_NEVER, "\n - %s\n", default_formats[num].def);
	}
	else
	{
	    printtext(NULL, txt_channel->name, MSGLEVEL_NEVER,
		      theme->format[num] == NULL ? default_formats[num].def :
		      theme->format[num]);
	}
        linepos[num] = gtk_text_get_point(text);
    }
    linepos[num] = -1;

    setup_drawtext_set(txt_channel, linepos, FALSE);

    /* redraw prompt */
    entry = gtk_object_get_data(GTK_OBJECT(setup_dialog), "txtentry");
    num = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(entry), "linenum"));
    gtk_entry_set_text(entry, temp_current_theme->format[num] == NULL ?
                       default_formats[num].def : temp_current_theme->format[num]);
    return TRUE;
}

static void sig_text_click(GtkWidget *textwidget, GdkEventButton *event, GtkWidget *entry)
{
    gboolean ignore_word;
    guint pos, len, line;

    g_return_if_fail(textwidget != NULL);
    g_return_if_fail(entry != NULL);

    len = gtk_text_get_length(GTK_TEXT(textwidget));
    pos = gtk_editable_get_position(GTK_EDITABLE(textwidget));

    ignore_word = pos >= len;
    if (!ignore_word)
    {
        for (line = 0; line < TEXTS; line++)
            if (linepos[line] >= pos) break;

        if (line < TEXTS)
        {
            gtk_object_set_data(GTK_OBJECT(entry), "line", &temp_current_theme->format[line]);
            gtk_object_set_data(GTK_OBJECT(entry), "linenum", GINT_TO_POINTER(line));
            gtk_entry_set_text(GTK_ENTRY(entry), temp_current_theme->format[line] == NULL ?
                               default_formats[line].def : temp_current_theme->format[line]);
            gtk_widget_grab_focus(entry);
        }
    }
}

static void sig_line_changed(GtkWidget *entry)
{
    gchar **str;

    g_return_if_fail(entry != NULL);

    str = gtk_object_get_data(GTK_OBJECT(entry), "line");
    if (*str != NULL) g_free(*str);
    *str = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));

    setup_theme_redraw(temp_current_theme);
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static gboolean signal_false(void)
{
    return FALSE;
}

static gboolean sig_setup_destroyed(void)
{
    WINDOW_VIEW_REC *view;
    WINDOW_REC *window;

    /* destroy format window */
    window = CHANNEL_PARENT(txt_channel);
    g_free(WINDOW_GUI(window));

    view = WINDOW_VIEW(window);
    if (view->pixmap != NULL) g_free(view->pixmap);
    g_free(view);

    g_free(txt_channel->gui_data);
    signal_add("channel destroyed", (SIGNAL_FUNC) signal_false);
    channel_destroy(txt_channel);
    signal_remove("channel destroyed", (SIGNAL_FUNC) signal_false);
    window->channels = NULL;

    signal_add("window destroyed", (SIGNAL_FUNC) signal_false);
    ui_window_destroy(window);
    signal_remove("window destroyed", (SIGNAL_FUNC) signal_false);

    setup_dialog = NULL;

    return TRUE;
}

static void sig_change_page(GtkNotebook *notebook, GtkNotebookPage *page, gint pagenum)
{
    if (setup_dialog != NULL && pagenum == 2)
	setup_theme_redraw(temp_current_theme);
}

static gboolean setup_txt(GtkWidget *dialog, GtkWidget *notebook)
{
    GtkWidget *mainbox, *scrollbox, *entry, *separator;
    WINDOW_VIEW_REC *view;
    WINDOW_REC *window;

    g_return_val_if_fail(dialog != NULL, FALSE);

    setup_dialog = dialog;

    mainbox = gtk_vbox_new(FALSE, 0);

    /* text widget */
    separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(mainbox), separator, FALSE, FALSE, 5);

    signal_add_first("channel created", (SIGNAL_FUNC) signal_false);
    txt_channel = channel_create(NULL, _("<formats>"), CHANNEL_TYPE_SETUP, TRUE);
    signal_remove("channel created", (SIGNAL_FUNC) signal_false);

    signal_add_first("window created", (SIGNAL_FUNC) signal_false);
    window = ui_window_create(txt_channel, FALSE);
    windows = g_list_remove(windows, window);
    signal_remove("window created", (SIGNAL_FUNC) signal_false);

    txt_channel->ui_data = window;
    window->channels = g_list_append(window->channels, txt_channel);

    view = g_new0(WINDOW_VIEW_REC, 1);
    window->gui_data = view;

    view->gui = g_new0(GUI_WINDOW_REC, 1);
    WINDOW_GUI(window)->views = g_list_append(NULL, view);

    txt_channel->gui_data = g_new0(GUI_CHANNEL_REC, 1);
    CHANNEL_PARENT(txt_channel) = window;
    CHANNEL_GUI(txt_channel)->font_normal = font_normal;
    CHANNEL_GUI(txt_channel)->font_bold = font_bold;

    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(mainbox), scrollbox, TRUE, TRUE, 0);

    view->text = gtk_text_new(NULL, NULL);
    gtk_container_add(GTK_CONTAINER(scrollbox), view->text);

    window->channels = g_list_append(NULL, txt_channel);
    gui_window_set_background(window);

    entry = gtk_entry_new();
    gtk_object_set_data(GTK_OBJECT(dialog), "txtentry", entry);
    gtk_box_pack_start(GTK_BOX(mainbox), entry, FALSE, FALSE, 0);

    gtk_signal_connect(GTK_OBJECT(entry), "activate",
                       GTK_SIGNAL_FUNC(sig_line_changed), NULL);
    gtk_signal_connect_after(GTK_OBJECT(view->text), "button_press_event",
                             GTK_SIGNAL_FUNC(sig_text_click), entry);

    gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
                       GTK_SIGNAL_FUNC(sig_change_page), NULL);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
			     mainbox, gtk_label_new(_("Text Formats")));
    return TRUE;
}

void init_txt(void)
{
    signal_add("setup themes apply again", (SIGNAL_FUNC) sig_setup_apply);
    signal_add("setup themes destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_add("setup themes show", (SIGNAL_FUNC) setup_txt);
    signal_add("setup themes redraw", (SIGNAL_FUNC) setup_theme_redraw);
}

void deinit_txt(void)
{
    signal_remove("setup themes apply again", (SIGNAL_FUNC) sig_setup_apply);
    signal_remove("setup themes destroyed", (SIGNAL_FUNC) sig_setup_destroyed);
    signal_remove("setup themes show", (SIGNAL_FUNC) setup_txt);
    signal_remove("setup themes redraw", (SIGNAL_FUNC) setup_theme_redraw);
}
