/*
 setup-lists.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "setup-int.h"

static GList *temp_aliases, *temp_ignores, *temp_completions, *temp_notifies, *temp_hilights, *temp_replaces, *temp_popups;

static GtkWidget *setup_dialog;

typedef LIST_REC *(LIST_FUNC) (GList **, gchar *, gchar *);
typedef void (LISTREMOVE_FUNC) (GList **, LIST_REC *);

LIST_REC *setup_ignore_add(GList **list, gchar *key, gchar *value)
{
    return ignore_add(key, value);
}

LIST_REC *setup_notifylist_add(GList **list, gchar *key, gchar *value)
{
    return notifylist_add(key, value);
}

void setup_ignore_remove(GList **list, LIST_REC *rec)
{
    ignore_remove_rec(rec, "");
}

void setup_notifylist_remove(GList **list, LIST_REC *rec)
{
    notifylist_remove(rec);
}

/* add new item to list */
static LIST_REC *list_append(GList **list, gchar *key, gchar *value)
{
    LIST_REC *rec;

    g_return_val_if_fail(list != NULL, NULL);
    g_return_val_if_fail(key != NULL, NULL);
    g_return_val_if_fail(value != NULL, NULL);

    rec = g_new(LIST_REC, 1);
    rec->key = g_strdup(key);
    rec->value = g_strdup(value);

    *list = g_list_append(*list, rec);
    return rec;
}

static void apply_list(GList **list, GList **templist, gchar *section, gchar *key, gchar *value, LIST_FUNC addfunc, LISTREMOVE_FUNC removefunc)
{
    proplist_t lprop, iprop, pkey;
    GList *tmp;

    g_return_if_fail(list != NULL);
    g_return_if_fail(templist != NULL);
    g_return_if_fail(section != NULL);

    while (*list != NULL)
        removefunc(list, (*list)->data);

    /* First clear the list.. */
    pkey = PLMakeString(section);
    PLRemoveDictionaryEntry(cprop, pkey);

    lprop = NULL;
    if (g_list_length(*templist) > 0)
    {
	/* Create the array for list */
	lprop = PLMakeArrayFromElements(NULL);
	cprop = PLInsertDictionaryEntry(cprop, pkey, lprop);
	PLRelease(pkey);
    }

    for (tmp = g_list_first(*templist); tmp != NULL; tmp = tmp->next)
    {
        LIST_REC *rec = tmp->data;

	/* create dictionary for list record and add it to array */
	iprop = PLMakeDictionaryFromEntries(NULL, NULL);
	lprop = PLAppendArrayElement(lprop, iprop);

        addfunc(list, rec->key, rec->value);
        config_set_str(iprop, key, rec->key);
        config_set_str(iprop, value, rec->value);
    }
    if (!readonly) PLSave(cprop, TRUE);
}

static void sig_setup_apply(void)
{
    apply_list(&ignores, &temp_ignores, "ignores", "mask", "level", setup_ignore_add, setup_ignore_remove);
    apply_list(&aliases, &temp_aliases, "aliases", "alias", "command", list_add, list_remove);
    apply_list(&completions, &temp_completions, "completions", "short", "long", list_add, list_remove);
    apply_list(&notifies, &temp_notifies, "notifies", "mask", "ircnets", setup_notifylist_add, setup_notifylist_remove);
    apply_list(&hilights, &temp_hilights, "hilights", "text", "channels", list_add, list_remove);
    apply_list(&replaces, &temp_replaces, "replaces", "text", "replace", list_add, list_remove);
    apply_list(&popups, &temp_popups, "popups", "label", "command", list_append, list_remove);
}

static void sig_setup_destroyed(void)
{
    while (temp_aliases != NULL)
        list_remove(&temp_aliases, temp_aliases->data);

    while (temp_ignores != NULL)
        list_remove(&temp_ignores, temp_ignores->data);

    while (temp_completions != NULL)
        list_remove(&temp_completions, temp_completions->data);

    while (temp_notifies != NULL)
        list_remove(&temp_notifies, temp_notifies->data);

    while (temp_hilights != NULL)
        list_remove(&temp_hilights, temp_hilights->data);

    while (temp_replaces != NULL)
        list_remove(&temp_replaces, temp_replaces->data);

    while (temp_popups != NULL)
        list_remove(&temp_popups, temp_popups->data);

    setup_dialog = NULL;
}

static void sig_list_new(GtkWidget *button, GtkWidget *clist)
{
    GList **temp_list;
    LIST_REC *rec;
    gchar *data[2];
    gint row;

    g_return_if_fail(clist != NULL);

    data[0] = gtk_object_get_data(GTK_OBJECT(clist), "newkey");
    data[1] = gtk_object_get_data(GTK_OBJECT(clist), "newvalue");
    temp_list = gtk_object_get_data(GTK_OBJECT(clist), "templist");

    g_return_if_fail(data[0] != NULL);
    g_return_if_fail(data[1] != NULL);
    g_return_if_fail(temp_list != NULL);

    rec = list_append(temp_list, data[0], data[1]);

    data[0] = rec->key;
    data[1] = rec->value;
    row = gtk_clist_append(GTK_CLIST(clist), data);
    gtk_clist_set_row_data(GTK_CLIST(clist), row, rec);
    gtk_clist_select_row(GTK_CLIST(clist), row, 0);

    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void sig_list_delete(GtkWidget *button, GtkWidget *clist)
{
    GList **temp_list;
    LIST_REC *rec;
    gint row;

    g_return_if_fail(clist != NULL);

    row = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (row == -1) return;

    temp_list = gtk_object_get_data(GTK_OBJECT(clist), "templist");
    g_return_if_fail(temp_list != NULL);
    rec = gtk_clist_get_row_data(GTK_CLIST(clist), row);
    list_remove(temp_list, rec);

    gtk_clist_remove(GTK_CLIST(clist), row);
    gtk_clist_select_row(GTK_CLIST(clist), row == 0 ? row : row-1, 0);

    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void sig_move_row(GtkWidget *clist, gint source, gint dest, GList **list)
{
    gpointer data;

    data = g_list_nth(*list, source)->data;
    *list = g_list_remove(*list, data);
    *list = g_list_insert(*list, data, dest);
    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static void sig_unselect_row(GtkWidget *clist, gint row)
{
    GtkWidget *keyentry, *valueentry;

    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
    keyentry = gtk_object_get_data(GTK_OBJECT(clist), "keyentry");
    valueentry = gtk_object_get_data(GTK_OBJECT(clist), "valueentry");
    g_return_if_fail(keyentry != NULL);
    g_return_if_fail(valueentry != NULL);

    gtk_entry_set_text(GTK_ENTRY(keyentry), "");
    gtk_entry_set_text(GTK_ENTRY(valueentry), "");
    gtk_widget_set_sensitive(keyentry, FALSE);
    gtk_widget_set_sensitive(valueentry, FALSE);
}

static void sig_select_row(GtkWidget *clist, gint row)
{
    LIST_REC *rec;
    GtkWidget *keyentry, *valueentry;

    g_return_if_fail(clist != NULL);

    if (row < 0)
    {
        sig_unselect_row(clist, row);
        return;
    }

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(row));
    keyentry = gtk_object_get_data(GTK_OBJECT(clist), "keyentry");
    valueentry = gtk_object_get_data(GTK_OBJECT(clist), "valueentry");
    g_return_if_fail(keyentry != NULL);
    g_return_if_fail(valueentry != NULL);

    rec = gtk_clist_get_row_data(GTK_CLIST(clist), row);
    if (rec != NULL)
    {
        gtk_entry_set_text(GTK_ENTRY(keyentry), rec->key);
        gtk_entry_set_text(GTK_ENTRY(valueentry), rec->value);

        gtk_widget_set_sensitive(keyentry, TRUE);
        gtk_widget_set_sensitive(valueentry, TRUE);
    }
}

static void sig_entry_changed(GtkWidget *entry, GtkWidget *clist)
{
    LIST_REC *rec;
    gchar *text;
    gint row;

    g_return_if_fail(entry != NULL);
    g_return_if_fail(clist != NULL);

    row = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (row == -1) return;

    text = gtk_entry_get_text(GTK_ENTRY(entry));
    rec = gtk_clist_get_row_data(GTK_CLIST(clist), row);

    if (entry == gtk_object_get_data(GTK_OBJECT(clist), "keyentry"))
    {
        if (strcmp(rec->key, text) == 0) return;
        g_free(rec->key);
        rec->key = g_strdup(text);
        if (gtk_object_get_data(GTK_OBJECT(clist), "key_strup"))
            g_strup(rec->key);
        gtk_clist_set_text (GTK_CLIST(clist), row, 0, rec->key);
    }
    else
    {
        if (strcmp(rec->value, text) == 0) return;
        g_free(rec->value);
        rec->value = g_strdup(text);
        gtk_clist_set_text (GTK_CLIST(clist), row, 1, rec->value);
    }

    gnome_property_box_changed(GNOME_PROPERTY_BOX(setup_dialog));
}

static GtkWidget *setup_list(gchar *titles[], GList **list, GList **templist, gboolean key_strup, gchar *newkey, gchar *newvalue)
{
    gchar *data[2];
    GtkWidget *vbox, *hbox;
    GtkWidget *scrollbox;
    GtkWidget *entry, *clist, *button;
    GList *tmp;

    g_return_val_if_fail(titles != NULL, NULL);
    g_return_val_if_fail(list != NULL, NULL);
    g_return_val_if_fail(templist != NULL, NULL);
    g_return_val_if_fail(newkey != NULL, NULL);
    g_return_val_if_fail(newvalue != NULL, NULL);

    vbox = gtk_vbox_new(FALSE, 0);

    gtk_object_set_data(GTK_OBJECT(setup_dialog), "list", list);

    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(vbox), scrollbox, TRUE, TRUE, 0);

    clist = gtk_clist_new_with_titles(2, titles);
    gtk_clist_set_reorderable(GTK_CLIST(clist), TRUE);
    gtk_signal_connect(GTK_OBJECT(clist), "row_move",
		       GTK_SIGNAL_FUNC(sig_move_row), list);
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
                       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);
    gtk_object_set_data(GTK_OBJECT(clist), "newkey", newkey);
    gtk_object_set_data(GTK_OBJECT(clist), "newvalue", newvalue);
    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
    gtk_object_set_data(GTK_OBJECT(clist), "key_strup", GINT_TO_POINTER((gint) key_strup));
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_clist_set_column_width(GTK_CLIST(clist), 0, 200);
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    entry = gtk_entry_new();
    gtk_widget_set_sensitive(entry, FALSE);
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
                       GTK_SIGNAL_FUNC(sig_entry_changed), clist);
    gtk_object_set_data(GTK_OBJECT(clist), "keyentry", entry);
    gtk_widget_set_usize(entry, 200, -1);
    gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);

    entry = gtk_entry_new();
    gtk_widget_set_sensitive(entry, FALSE);
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
                       GTK_SIGNAL_FUNC(sig_entry_changed), clist);
    gtk_object_set_data(GTK_OBJECT(clist), "valueentry", entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

    /* buttons */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 10);

    button = gtk_button_new_with_label(_("New"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_list_new), clist);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);

    button = gtk_button_new_with_label(_("Delete"));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(sig_list_delete), clist);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 10);

    /* draw the list */
    gtk_clist_freeze(GTK_CLIST(clist));
    *templist = NULL;
    for (tmp = g_list_first(*list); tmp != NULL; tmp = tmp->next)
    {
        LIST_REC *rec = tmp->data;
        gint row;

        data[0] = rec->key;
        data[1] = rec->value;
        row = gtk_clist_append(GTK_CLIST(clist), data);

        rec = list_append(templist, rec->key, rec->value);
        gtk_clist_set_row_data(GTK_CLIST(clist), row, rec);
    }
    gtk_object_set_data(GTK_OBJECT(clist), "templist", templist);
    gtk_clist_thaw(GTK_CLIST(clist));

    gtk_clist_select_row(GTK_CLIST(clist), 0, 0);

    return vbox;
}

static void setup_list_dialog(gint page)
{
    gchar *titles[2];

    if (setup_dialog != NULL)
    {
        /* Setup already open */
#ifdef HAVE_GNOME
        gtk_notebook_set_page(GTK_NOTEBOOK(GNOME_PROPERTY_BOX(setup_dialog)->notebook), page);
#else
        gtk_notebook_set_page(gnome_property_box_get_notebook(setup_dialog), page);
#endif
        gdk_window_raise(setup_dialog->window);
        return;
    }

    setup_dialog = gnome_property_box_new();
    gtk_window_set_policy(GTK_WINDOW(setup_dialog), FALSE, TRUE, FALSE);
#ifdef HAVE_GNOME
    gtk_widget_set_usize(setup_dialog, -1, 350);
#else
    gtk_widget_set_usize(setup_dialog, 400, 350);
#endif

    titles[0] = _("Alias");
    titles[1] = _("Command");
    gnome_property_box_append_page(GNOME_PROPERTY_BOX(setup_dialog),
                                   setup_list(titles, &aliases, &temp_aliases, TRUE, "NEW", ""),
                                   gtk_label_new(_("Aliases")));

    titles[0] = _("Mask");
    titles[1] = _("What to ignore");
    gnome_property_box_append_page(GNOME_PROPERTY_BOX(setup_dialog),
                                   setup_list(titles, &ignores, &temp_ignores, FALSE, "nobody", "ALL"),
                                   gtk_label_new(_("Ignores")));

    titles[0] = _("Tag");
    titles[1] = _("Completed text");
    gnome_property_box_append_page(GNOME_PROPERTY_BOX(setup_dialog),
                                   setup_list(titles, &completions, &temp_completions, TRUE, "NEW", ""),
                                   gtk_label_new(_("Completions")));

    titles[0] = _("Mask");
    titles[1] = _("IRC network(s)");
    gnome_property_box_append_page(GNOME_PROPERTY_BOX(setup_dialog),
                                   setup_list(titles, &notifies, &temp_notifies, FALSE, "new", ""),
                                   gtk_label_new(_("Notify List")));

    titles[0] = _("Text");
    titles[1] = _("Color# (optional) and Channels");
    gnome_property_box_append_page(GNOME_PROPERTY_BOX(setup_dialog),
                                   setup_list(titles, &hilights, &temp_hilights, FALSE, "new", ""),
                                   gtk_label_new(_("Hilight Texts")));

    titles[0] = _("Text");
    titles[1] = _("Replace with");
    gnome_property_box_append_page(GNOME_PROPERTY_BOX(setup_dialog),
                                   setup_list(titles, &replaces, &temp_replaces, FALSE, "new", ""),
                                   gtk_label_new(_("Replaces")));

    titles[0] = _("Label");
    titles[1] = _("Command");
    gnome_property_box_append_page(GNOME_PROPERTY_BOX(setup_dialog),
                                   setup_list(titles, &popups, &temp_popups, FALSE, "new", ""),
                                   gtk_label_new(_("Popup Menu")));

#ifdef HAVE_GNOME
    gtk_signal_connect(GTK_OBJECT(setup_dialog), "apply",
                       GTK_SIGNAL_FUNC(sig_setup_apply), NULL);
#else
    gnome_property_box_set_apply_func(setup_dialog, (GtkFunction) sig_setup_apply, setup_dialog);
#endif
    gtk_signal_connect(GTK_OBJECT(setup_dialog), "destroy",
                       GTK_SIGNAL_FUNC(sig_setup_destroyed), &setup_dialog);

    gtk_widget_show_all(setup_dialog);
#ifdef HAVE_GNOME
    gtk_notebook_set_page(GTK_NOTEBOOK(GNOME_PROPERTY_BOX(setup_dialog)->notebook), page);
#else
    gtk_notebook_set_page(gnome_property_box_get_notebook(setup_dialog), page);
#endif
}

void setup_aliases(void)
{
    setup_list_dialog(0);
}

void setup_ignores(void)
{
    setup_list_dialog(1);
}

void setup_completions(void)
{
    setup_list_dialog(2);
}

void setup_notifies(void)
{
    setup_list_dialog(3);
}

void setup_hilights(void)
{
    setup_list_dialog(4);
}

void setup_replaces(void)
{
    setup_list_dialog(5);
}

void setup_popups(void)
{
    setup_list_dialog(6);
}

void init_lists(void)
{
    setup_dialog = NULL;
}

void deinit_lists(void)
{
    if (setup_dialog != NULL)
        gtk_widget_destroy(setup_dialog);
}
