/*
 gui-statusbar.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

/* how often to redraw lagging time */
#define LAG_REFRESH_TIME 10
/* If we haven't been able to check lag for this long, "(??)" is added after
   the lag */
#define MAX_LAG_UNKNOWN_TIME 30

static gboolean signals_added;
static gint timeout_tag;
static time_t last_statusbar_redraw;

void gui_statusbar_destroy(MAIN_WINDOW_REC *window, GUI_STATUSBAR_REC *rec)
{
    g_free(rec->text);
    g_free(rec);
    window->statusbar_texts = g_list_remove(window->statusbar_texts, rec);
}

gint gui_statusbar_push(MAIN_WINDOW_REC *window, gchar *text)
{
    GUI_STATUSBAR_REC *rec;

    g_return_val_if_fail(window != NULL, -1);
    g_return_val_if_fail(text != NULL, -1);

    rec = g_new(GUI_STATUSBAR_REC, 1);
    rec->id = ++window->statusbar_id;
    rec->text = g_strdup(text);
    window->statusbar_texts = g_list_append(window->statusbar_texts, rec);

    gtk_statusbar_pop(GTK_STATUSBAR(window->statusbar), 1);
    gtk_statusbar_push(GTK_STATUSBAR(window->statusbar), 1, text);
    return rec->id;
}

void gui_statusbar_pop(MAIN_WINDOW_REC *window, gint id)
{
    GUI_STATUSBAR_REC *rec;
    GList *link;
    gboolean last;

    g_return_if_fail(window != NULL);

    for (link = g_list_first(window->statusbar_texts); link != NULL; link = link->next)
    {
        rec = link->data;

        if (rec->id == id) break;
    }
    g_return_if_fail (link != NULL);
    last = link->next == NULL;

    gui_statusbar_destroy(window, rec);
    if (last)
    {
        /* latest in stack, pop it. */
        gtk_statusbar_pop(GTK_STATUSBAR(window->statusbar), 1);

        if (window->statusbar_texts != NULL)
        {
            rec = g_list_last(window->statusbar_texts)->data;
            gtk_statusbar_push(GTK_STATUSBAR(window->statusbar), 1, rec->text);
        }
    }
}

void gui_statusbar_change(MAIN_WINDOW_REC *window, gint id, gchar *text)
{
    GUI_STATUSBAR_REC *rec;
    GList *link;

    g_return_if_fail(window != NULL);
    g_return_if_fail(text != NULL);

    for (link = g_list_first(window->statusbar_texts); link != NULL; link = link->next)
    {
        rec = link->data;

        if (rec->id == id) break;
    }
    g_return_if_fail (link != NULL);

    g_free(rec->text);
    rec->text = g_strdup(text);

    if (g_list_last(window->statusbar_texts) == link)
    {
        /* latest in stack, redraw it. */
        gtk_statusbar_pop(GTK_STATUSBAR(window->statusbar), 1);
        gtk_statusbar_push(GTK_STATUSBAR(window->statusbar), 1, text);
    }
}

static void gui_statusbar_redraw(MAIN_WINDOW_REC *window)
{
    CHANNEL_REC *channel;
    SERVER_REC *server;
    NICK_REC *nickrec;
    GString *str;
    time_t now;

    g_return_if_fail(window != NULL);

    channel = window->active->active;
    server = channel->server;

    str = g_string_new(" ");

    now = time(NULL);
    last_statusbar_redraw = now;

    if (server != NULL)
    {
        /* server */
	nickrec = nicklist_find(channel, server->nick);

        g_string_sprintfa(str, "[%s:%d", server->address, server->port);
        if (server->ircnet != NULL) g_string_sprintfa(str, " (%s)", server->ircnet);
        g_string_append(str, "] ");

	/* nick */
        g_string_append_c(str, '[');
	if (nickrec != NULL && (nickrec->op || nickrec->voice))
	{
	    /* op / voice in channel */
	    g_string_append_c(str, nickrec->op ? '@' : '+');
	}
	g_string_append(str, server->nick);
	if (server->usermode != NULL)
	{
	    /* user mode */
	    g_string_sprintfa(str, "(+%s", server->usermode);
	    if (server->usermode_away) g_string_append(str, ") (zZzZ");
	    g_string_append_c(str, ')');
	}
	g_string_append(str, "] ");

	if (server->lag_last_check != 0)
	{
	    /* lag */
	    if (server->lag_sent == 0 || now-server->lag_sent < 5)
	    {
		g_string_sprintfa(str, _("[lag %d.%02d"), server->lag/1000, (server->lag % 1000)/10);
                if (now-server->lag_last_check > MAX_LAG_UNKNOWN_TIME)
		    g_string_append(str, "(??) ");
		g_string_append(str, "] ");
	    }
	    else
	    {
		/* big lag, still waiting .. */
		g_string_sprintfa(str, _("[lag %ld (??)] "), now-server->lag_sent);
	    }
	}
    }

    gui_statusbar_change(window, window->main_id, str->str);
    g_string_free(str, TRUE);
}

static gboolean sig_redraw_server(SERVER_REC *server)
{
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    for (tmp = g_list_first(mainwindows); tmp != NULL; tmp = tmp->next)
    {
        MAIN_WINDOW_REC *rec = tmp->data;

        if (rec->active->active->server == server)
            gui_statusbar_redraw(rec);
    }

    return TRUE;
}

static gboolean sig_redraw_channel(CHANNEL_REC *channel)
{
    MAIN_WINDOW_REC *window;

    if (channel == NULL) return TRUE;

    window = WINDOW_GUI(CHANNEL_PARENT(channel))->parent;
    if (window->active->active == channel)
        gui_statusbar_redraw(window);

    return TRUE;
}

static gboolean sig_server_looking(SERVER_REC *server)
{
    GUI_SERVER_REC *gui;
    gchar *str;

    g_return_val_if_fail(server != NULL, FALSE);

    gui = SERVER_GUI(server);
    gui->window = WINDOW_GUI(CHANNEL_PARENT(cur_channel))->parent;
    gtk_widget_show(gui->window->cancel);

    gtk_object_set_data(GTK_OBJECT(gui->window->cancel), "server", server);

    str = g_strdup_printf(_("Looking up server %s"), server->address);
    gui->statusbar_id = gui_statusbar_push(gui->window, str);
    g_free(str);

    return TRUE;
}

static gboolean sig_server_connecting(SERVER_REC *server)
{
    gchar *str;

    g_return_val_if_fail(server != NULL, FALSE);

    str = g_strdup_printf(_("Connecting to %s:%d"), server->address, server->port);
    gui_statusbar_change(SERVER_GUI(server)->window, SERVER_GUI(server)->statusbar_id, str);
    g_free(str);

    return TRUE;
}

static gboolean sig_server_finished(SERVER_REC *server)
{
    GUI_SERVER_REC *gui;

    g_return_val_if_fail(server != NULL, FALSE);

    gui = SERVER_GUI(server);
    if (gui->window != NULL)
    {
	gtk_widget_hide(gui->window->cancel);
	gui_statusbar_pop(gui->window, gui->statusbar_id);

	gui->window = NULL;
	gui->statusbar_id = -1;
    }

    return TRUE;
}

static void sig_cancel_clicked(GtkWidget *button, MAIN_WINDOW_REC *window)
{
    SERVER_REC *server;

    server = gtk_object_get_data(GTK_OBJECT(window->cancel), "server");
    if (server != NULL)
	server_disconnect(server);
}

static gboolean sig_mainwin_created(MAIN_WINDOW_REC *window)
{
    g_return_val_if_fail(window != NULL, FALSE);

    gtk_signal_connect(GTK_OBJECT(window->cancel), "clicked",
                       GTK_SIGNAL_FUNC(sig_cancel_clicked), window);
    return TRUE;
}

static gint sig_redraw_statusbar(void)
{
    GList *tmp;

    /* refresh statusbar every 10 seconds */
    if (time(NULL)-last_statusbar_redraw < LAG_REFRESH_TIME)
	return 1;

    for (tmp = g_list_first(mainwindows); tmp != NULL; tmp = tmp->next)
	gui_statusbar_redraw(tmp->data);
    return 1;
}

static gboolean sig_setup_changed(void)
{
    GList *tmp;

    gui_statusbar_deinit();
    gui_statusbar_init();

    /* hide/show statusbars */
    for (tmp = g_list_first(mainwindows); tmp != NULL; tmp = tmp->next)
    {
	MAIN_WINDOW_REC *window = tmp->data;

	if (!setup_get_bool("toggle_show_statusbar"))
	    gtk_widget_hide(window->statusbar);
	else
	{
	    gui_statusbar_redraw(window);
	    gtk_widget_show(window->statusbar);
	}
    }

    return TRUE;
}

void gui_statusbar_init(void)
{
    last_statusbar_redraw = 0;

    signal_add("setup changed", (SIGNAL_FUNC) sig_setup_changed);
    if (!setup_get_bool("toggle_show_statusbar"))
	return;

    timeout_tag = gui_timeout_add(LAG_REFRESH_TIME*1000, (GUITimeoutFunction) sig_redraw_statusbar, NULL);

    signals_added = TRUE;
    signal_add("server connected", (SIGNAL_FUNC) sig_redraw_server);
    signal_add("server disconnected", (SIGNAL_FUNC) sig_redraw_server);
    signal_add("server lag", (SIGNAL_FUNC) sig_redraw_server);
    signal_add("user mode changed", (SIGNAL_FUNC) sig_redraw_server);
    signal_add("server nick changed", (SIGNAL_FUNC) sig_redraw_server);
    signal_add("channel server changed", (SIGNAL_FUNC) sig_redraw_channel);
    signal_add("channel changed", (SIGNAL_FUNC) sig_redraw_channel);
    signal_add("server looking", (SIGNAL_FUNC) sig_server_looking);
    signal_add("server connecting", (SIGNAL_FUNC) sig_server_connecting);
    signal_add("server connected", (SIGNAL_FUNC) sig_server_finished);
    signal_add("server connect failed", (SIGNAL_FUNC) sig_server_finished);
    signal_add("server disconnected", (SIGNAL_FUNC) sig_server_finished);
    signal_add("gui mainwindow created", (SIGNAL_FUNC) sig_mainwin_created);
}

void gui_statusbar_deinit(void)
{
    signal_remove("setup changed", (SIGNAL_FUNC) sig_setup_changed);

    if (!signals_added)
	return;

    signals_added = FALSE;
    gui_timeout_remove(timeout_tag);
    signal_remove("server connected", (SIGNAL_FUNC) sig_redraw_server);
    signal_remove("server disconnected", (SIGNAL_FUNC) sig_redraw_server);
    signal_remove("server lag", (SIGNAL_FUNC) sig_redraw_server);
    signal_remove("user mode changed", (SIGNAL_FUNC) sig_redraw_server);
    signal_remove("server nick changed", (SIGNAL_FUNC) sig_redraw_server);
    signal_remove("channel server changed", (SIGNAL_FUNC) sig_redraw_channel);
    signal_remove("channel changed", (SIGNAL_FUNC) sig_redraw_channel);
    signal_remove("server looking", (SIGNAL_FUNC) sig_server_looking);
    signal_remove("server connecting", (SIGNAL_FUNC) sig_server_connecting);
    signal_remove("server connected", (SIGNAL_FUNC) sig_server_finished);
    signal_remove("server connect failed", (SIGNAL_FUNC) sig_server_finished);
    signal_remove("server disconnected", (SIGNAL_FUNC) sig_server_finished);
    signal_remove("gui mainwindow created", (SIGNAL_FUNC) sig_mainwin_created);
}
