/*
 dialog-textwidget.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gint window_find_line(WINDOW_REC *window, gint pos)
{
    GList *tmp;
    gint linepos, line;

    linepos = 0; line = 0;
    for (tmp = WINDOW_VIEW(window)->linepos; tmp != NULL; tmp = tmp->next, line++)
    {
        linepos += GPOINTER_TO_INT(tmp->data)+1;
        if (linepos > pos)
            return line;
    }

    return line;
}

static void sig_find_prev(GtkWidget *dialog)
{
    WINDOW_REC *window;
    GtkEditable *text;
    GtkEntry *entry;
    gchar *find, *str, *ptr, *tmp;
    gint start, n, block, findlen;

    text = gtk_object_get_data(GTK_OBJECT(dialog), "text");
    window = gtk_object_get_data(GTK_OBJECT(dialog), "window");
    entry = gtk_object_get_data(GTK_OBJECT(dialog), "entry");

    find = gtk_entry_get_text(entry);
    g_strup(find);
    findlen = strlen(find);

    start = text->has_selection ? text->selection_start_pos : gtk_text_get_length(GTK_TEXT(text));
    if (*find == '\0' || findlen > start) return;

    block = findlen > 1024 ? findlen*2 : 2048;
    for (n = 0; n < start; n += block-findlen)
    {
        str = gtk_editable_get_chars(text, start-n-block < 0 ? 0 : start-n-block, start-n);
        if (str == NULL) break;

        g_strup(str);
        for (ptr = NULL, tmp = str; tmp != NULL; )
        {
            tmp = strstr(tmp, find);
            if (tmp == NULL) break;

            ptr = tmp;
            tmp += findlen;
        }
        if (ptr != NULL)
        {
            /* found! */
            start = (start-n-block < 0 ? 0 : start-n-block) + (gint) (ptr-str);
            gtk_adjustment_set_value(GTK_TEXT(text)->vadj, window_find_line(window, start) * (GTK_TEXT(text)->vadj->upper/window->lines));
            gtk_editable_select_region(text, start, start+findlen);
            break;
        }
        g_free(str);
    }
}

static void sig_find_next(GtkWidget *dialog)
{
    WINDOW_REC *window;
    GtkEditable *text;
    GtkEntry *entry;
    gchar *find, *str, *ptr;
    gint start, len, n, block, findlen;

    text = gtk_object_get_data(GTK_OBJECT(dialog), "text");
    window = gtk_object_get_data(GTK_OBJECT(dialog), "window");
    entry = gtk_object_get_data(GTK_OBJECT(dialog), "entry");

    find = gtk_entry_get_text(entry);
    g_strup(find);
    findlen = strlen(find);

    start = text->has_selection ? text->selection_end_pos : 0;
    len = gtk_text_get_length(GTK_TEXT(text))-start;
    if (*find == '\0' || findlen > len) return;

    block = findlen > 1024 ? findlen*2 : 2048;
    for (n = 0; n < len; n += block-findlen)
    {
        str = gtk_editable_get_chars(text, start+n,
                                     n+block < len ? start+n+block : start+len);
        if (str == NULL) break;

        g_strup(str);
        ptr = strstr(str, find);
        if (ptr != NULL)
        {
            /* found! */
            start += n+(gint) (ptr-str);
            gtk_adjustment_set_value(GTK_TEXT(text)->vadj, window_find_line(window, start) * (GTK_TEXT(text)->vadj->upper/window->lines));
            gtk_editable_select_region(text, start, start+findlen);
            break;
        }
        g_free(str);
    }
}

static void sig_destroyed(GtkWidget *widget, GtkWidget **dialog)
{
    *dialog = NULL;
}

void dialog_textwidget_find(GtkWidget *widget, MAIN_WINDOW_REC *window)
{
    static GtkWidget *find_dialog = NULL;
    GtkWidget *dialog, *hbox, *label, *entry;

    if (find_dialog != NULL)
    {
        gdk_window_raise(find_dialog->window);
        return;
    }

    if (WINDOW_VIEW(window->active)->itext)
    {
        gui_dialog(DIALOG_ERROR, _("Find doesn't work with IText windows yet"));
        return;
    }

    find_dialog = dialog =
        gnome_dialog_new(PACKAGE, _("Find previous"), _("Find next"),
                         GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                       GTK_SIGNAL_FUNC(sig_destroyed), &find_dialog);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
    gui_widget_depends_data(dialog, "window destroyed", window->active);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

    label = gtk_label_new(_("Find text: "));
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    entry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, sig_find_prev, GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, sig_find_next, GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 2, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_object_set_data(GTK_OBJECT(dialog), "text", WINDOW_VIEW(window->active)->text);
    gtk_object_set_data(GTK_OBJECT(dialog), "window", window->active);
    gtk_object_set_data(GTK_OBJECT(dialog), "entry", entry);

    gtk_widget_show_all(dialog);
}

static gboolean append_to_file(gchar *fname, gchar *text, gint len)
{
    gchar *path;
    gint f;

    path = convert_home(fname);
    f = open(path, O_WRONLY | O_APPEND | O_CREAT, LOG_FILE_CREATE_MODE);
    if (f == -1)
    {
        gui_dialog(DIALOG_ERROR, _("Couldn't create file:\n%s"), path);
	g_free(path);
        return FALSE;
    }
    g_free(path);

    write(f, text, len);
    write(f, "\n", 1);
    close(f);
    return TRUE;
}

static void sig_save_ok(GtkWidget *dialog)
{
    GtkWidget *selbut, *entry;
    GtkEditable *text;
    gint start, end;
    gboolean usesel;
    gchar *str;

    text = gtk_object_get_data(GTK_OBJECT(dialog), "text");
    selbut = gtk_object_get_data(GTK_OBJECT(dialog), "selbut");
    entry = gtk_object_get_data(GTK_OBJECT(dialog), "entry");

    usesel = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(selbut));

    start = usesel ? text->selection_start_pos : 0;
    end = usesel ? text->selection_end_pos : gtk_text_get_length(GTK_TEXT(text));
    if (end > start)
    {
        str = gtk_editable_get_chars(text, start, end);
        append_to_file(gtk_entry_get_text(GTK_ENTRY(entry)), str, end-start+1);
        g_free(str);
    }

    gtk_widget_destroy(dialog);
}

void dialog_textwidget_save(GtkWidget *widget, MAIN_WINDOW_REC *window)
{
    static GtkWidget *save_dialog = NULL;
    GtkWidget *dialog, *hbox, *entry, *label, *button;
    WINDOW_VIEW_REC *view;

    if (save_dialog != NULL)
    {
        gdk_window_raise(save_dialog->window);
        return;
    }

    view = WINDOW_VIEW(window->active);
    if (view->itext)
    {
        gui_dialog(DIALOG_ERROR, _("Save doesn't work with IText windows yet"));
        return;
    }

    if (gtk_text_get_length(GTK_TEXT(view->text)) == 0)
    {
        gui_dialog(DIALOG_OK, _("Window doesn't have any text"));
        return;
    }

    save_dialog = dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK,
                                            GNOME_STOCK_BUTTON_CANCEL, NULL);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                       GTK_SIGNAL_FUNC(sig_destroyed), &save_dialog);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
    gui_widget_depends_data(dialog, "window destroyed", window->active);

    button = gtk_check_button_new_with_label(_("Save only selected text"));
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), button, FALSE, FALSE, 0);

    if (GTK_EDITABLE(view->text)->has_selection)
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
    else
        gtk_widget_set_sensitive(button, FALSE);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

    label = gtk_label_new(_("Save window buffer to: "));
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    entry = gnome_file_entry_new("Window buffer", _("Save window buffer"));
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, sig_save_ok, GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_object_set_data(GTK_OBJECT(dialog), "text", view->text);
    gtk_object_set_data(GTK_OBJECT(dialog), "selbut", button);
    gtk_object_set_data(GTK_OBJECT(dialog), "entry", gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(entry)));

    gtk_widget_show_all(dialog);
}

void dialog_textwidget_init(void)
{
}

void dialog_textwidget_deinit(void)
{
}
