/*
 * Configurable ps-like program.
 * Global definitions.
 *
 * Copyright (c) 2008 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#ifndef	IPS_H
#define	IPS_H

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>


#ifndef	SYSTEM_INIT_FILE
#define	SYSTEM_INIT_FILE	"/usr/local/lib/ips.init"
#endif

#ifndef	USER_INIT_FILE
#define	USER_INIT_FILE		".ipsrc"
#endif

#ifndef	SYSTEM_INIT_MACRO
#define	SYSTEM_INIT_MACRO	"SysInit"
#endif

#ifndef	USER_INIT_MACRO
#define	USER_INIT_MACRO		"UserInit"
#endif

#define	FIRST_LINE		"#ips#"		/* first line of init files */


/*
 * Some constants related to the system.
 */
#define	TICKS		100		/* ticks in a second */
#define	CPU_SCALE	10000		/* scaling factor for cpu usage */
#define	MEMORY_SCALE	1000		/* scaling factor for memory usage */
#define	CLICK_SIZE	4096		/* size of memory allocation chunk */
#define	BASE_USER_UID	100		/* first "user" uid */

/*
 * Define widths of numbers which depend on the size of a long.
 */
#if	2147483647 + 1 > 0
#define	LONG_HEX_DIGITS		16
#define	LONG_DIGITS		21
#define	LONG_HEX_FORMAT		"%016lx"
#else
#define	LONG_HEX_DIGITS		8
#define	LONG_DIGITS		11
#define	LONG_HEX_FORMAT		"%08lx"
#endif

/*
 * Some default values.
 */
#define	DEFAULT_INIT_SEC	1	/* default initialization time */
#define	DEFAULT_SLEEP_SEC	10	/* default sleep between loops */
#define	DEFAULT_SCROLL_SEC	30	/* default time between autoscrolls */
#define	DEFAULT_OVERLAP_LINES	1	/* default overlapping lines */
#define	DEFAULT_SYNC_SEC	300	/* default time to force data sync */
#define	DEFAULT_ACTIVE_SEC	60	/* default time to consider active */
#define	DEFAULT_DEATH_SEC	30	/* default time to preserve death */
#define	DEFAULT_WIDTH		80	/* default width if not known */
#define	DEFAULT_GEOMETRY	"150x50"	/* geometry for X11 */
#define	DEFAULT_GEOMETRY_ROWS	50	/* row part of geometry */
#define	DEFAULT_GEOMETRY_COLS	150	/* col part of geometry */
#define	DEFAULT_FONT		"fixed"	/* font name for X11 */
#define	DEFAULT_FOREGROUND	"black"	/* foreground color for X11 */
#define	DEFAULT_BACKGROUND	"white"	/* background color for X11 */


/*
 * File descriptors.
 */
#define	STDIN	0
#define	STDOUT	1
#define	STDERR	2


/*
 * Some limits on the program.
 */
#define	MAX_PIDS	100
#define	MAX_USERS	100
#define	MAX_GROUPS	100
#define	MAX_PROGRAMS	20
#define	MAX_WORDS	1000
#define	MAX_COLUMNS	50
#define	MAX_SEPARATION	20
#define	MAX_MACRO_LEN	16
#define	MAX_OPTION_DEPTH 20
#define	MAX_EXPR_DEPTH	20
#define	MAX_WIDTH	(1024 * 31)
#define	MAX_COMMAND_LEN	(1024 * 10)
#define	MAX_ENVIRON_LEN	(1024 * 20)
#define	MAX_INFO_LEN	256
#define	MAX_PATH_LEN	1024
#define	BUF_COMMAND_LEN	128
#define	MAX_PROGRAM_LEN	32
#define	MAX_WCHAN_LEN	32


/*
 * Macros to help parse strings.
 */
#define	isblank(ch)	(((ch) == ' ') || ((ch) == '\t'))
#define	isdigit(ch)	(((ch) >= '0') && ((ch) <= '9'))
#define	isupper(ch)	(((ch) >= 'A') && ((ch) <= 'Z'))
#define	islower(ch)	(((ch) >= 'a') && ((ch) <= 'z'))
#define	ismacro(ch)	isupper(ch)


/*
 * Boolean values.
 */
typedef	int	BOOL;

#define	FALSE	((BOOL) 0)
#define	TRUE	((BOOL) 1)


/*
 * Some other typedefs.
 */
typedef	unsigned long	ULONG;
typedef	struct	COLUMN	COLUMN;


/*
 * Special values for unknown user, group, or device ids.
 */
#define	BAD_UID		((uid_t) -1)
#define	BAD_GID		((gid_t) -1)
#define	BAD_DEVID	((dev_t) -1)


/*
 * Argument structure.
 * This is for parsing command line arguments.
 */
typedef	struct
{
	int	count;			/* number of arguments */
	char **	table;			/* table of arguments */
} ARGS;


/*
 * Structure which holds information about a process.
 */
typedef	struct	PROC	PROC;

struct	PROC
{
	PROC *	next;			/* next process in list */
	BOOL	isvalid;		/* the process is existent */
	BOOL	isancient;		/* the process is before we started */
	BOOL	isnew;			/* the process is just created */
	BOOL	isactive;		/* the process is active */
	BOOL	changed;		/* the process state has changed */
	BOOL	isshown;		/* the process is to be shown */
	BOOL	hascommand;		/* there is a real command line */
	ULONG	live_counter;		/* counter to mark existant procs */
	ULONG	run_order;		/* counter value when last active */
	time_t	last_saved_time;	/* time status was copied to old */
	time_t	last_active_time;	/* time process last was active */
	time_t	last_sync_time;		/* time status was last synchonized */
	time_t	deathTime;		/* time process was seen to be dead */
	uid_t	uid;			/* effective user id */
	gid_t	gid;			/* effective group id */
	pid_t	pid;			/* process id */
	pid_t	parent_pid;		/* parent pid */
	pid_t	process_group;		/* process group */
	pid_t	session_id;		/* session id */
	dev_t	tty_dev;		/* controlling terminal's device */
	pid_t	tty_pgrp;		/* process group of terminal */
	int	state;			/* process state character */
	int	processor;		/* processor process is on */
	int	exitsignal;		/* signal causing exit */
	ULONG	flags;			/* kernel flags */
	long	min_flt;		/* minor page faults */
	long	maj_flt;		/* major page faults */
	long	cmin_flt;		/* child minor page faults */
	long	cmaj_flt;		/* child major page faults */
	long	utime;			/* user runtime in jiffies */
	long	stime;			/* system runtime in jiffies */
	long	cutime;			/* child user runtime */
	long	cstime;			/* child system runtime */
	long	nice;			/* nice value */
	long	priority;		/* scheduling priority */
	long	realTimePriority;	/* real time priority */
	long	timeout;		/* timeout */
	long	it_real_value;		/* jiffies sleeping for */
	long	vsize;			/* virtual size of process in bytes */
	long	rss;			/* resident size in clicks */
	long	rss_limit;		/* resident size limit */
	long	policy;			/* policy */
	ULONG	start_time_ticks;	/* jiffies uptime when started */
	ULONG	start_time_clock;	/* time when started */
	ULONG	start_code;		/* beginning address of code */
	ULONG	end_code;		/* ending address of code */
	ULONG	start_stack;		/* starting address of stack */
	ULONG	esp;			/* spack pointer */
	ULONG	eip;			/* instruction pointer */
	ULONG	wchan;			/* wait channel address */
	ULONG	signal;			/* current signal */
	ULONG	sigblock;		/* signals to block */
	ULONG	sigignore;		/* signals to ignore */
	ULONG	sigcatch;		/* signals to catch */
	ULONG	nswap;			/* pages swapped out */
	ULONG	cnswap;			/* child pages swapped out */
	int	percent_cpu;		/* percentage of cpu used */
	int	percent_memory;		/* percentage of memory used */
	int	openfiles;		/* number of open files */
	int	threadCount;		/* number of threads */
	int	command_length;		/* length of command line */
	int	environment_length;	/* length of environment */
	int	cwd_path_length;	/* length of current working dir path */
	int	root_path_length;	/* length of root directory path */
	int	exec_path_length;	/* length of executable path */
	char *	command;		/* command line */
	char *	environment;		/* environment */
	char *	cwd_path;		/* path of current working directory */
	char *	root_path;		/* path of root directory */
	char *	exec_path;		/* path of executable */
	char *	stdio_paths[3];		/* paths of stdin, stdout, stderr */
	char	program[MAX_PROGRAM_LEN + 2];	/* program name */
	char	command_buffer[BUF_COMMAND_LEN + 2];	/* command buffer */
	char	wchan_symbol[MAX_WCHAN_LEN + 2];	/* wait channel */

	/*
	 * Status which is saved in order to determine active processes
	 * even if they are currently sleeping for this snapshot.
	 * A process is active if any of this differs from the current
	 * value.
	 */
	int	old_state;
	int	old_flags;
	long	old_min_flt;
	long	old_maj_flt;
	long	old_utime;
	long	old_stime;
	ULONG	old_start_time_ticks;
	ULONG	old_end_code;	
	ULONG	old_esp;
	ULONG	old_eip;
	ULONG	old_wchan;
};

#define	NULL_PROC	((PROC *) 0)


/*
 * Column justification definitions.
 */
typedef	int	JUSTIFY;

#define	LEFT	((JUSTIFY) 0)
#define	RIGHT	((JUSTIFY) 1)
#define	CENTER	((JUSTIFY) 2)


/*
 * Values returned by expressions.
 */
typedef	struct
{
	int		type;		/* type of value */
	long		intval;		/* integer value */
	const char *	strval;		/* string value */
	COLUMN *	column;		/* column value */
} VALUE;


#define	VALUE_BAD	0
#define	VALUE_NONE	1
#define	VALUE_NUMBER	2
#define	VALUE_STRING	3
#define	VALUE_COLUMN	4
#define	VALUE_BOOLEAN	5

#define	TWOVAL(first, second)	(((first) * 10) + (second))


/*
 * Flags for what a column requires to be collected.
 * These flags are OR'd together.
 */
typedef	unsigned int	USEFLAG;

#define	USE_NONE	((USEFLAG) 0x0000)
#define	USE_INIT	((USEFLAG) 0x0001)
#define	USE_DEV_NAME	((USEFLAG) 0x0002)
#define	USE_OPEN_FILE	((USEFLAG) 0x0004)
#define	USE_CURR_DIR	((USEFLAG) 0x0008)
#define	USE_COMMAND	((USEFLAG) 0x0010)
#define	USE_SELF	((USEFLAG) 0x0020)
#define	USE_STDIN	((USEFLAG) 0x0040)
#define	USE_STDOUT	((USEFLAG) 0x0080)
#define	USE_STDERR	((USEFLAG) 0x0100)
#define	USE_ENVIRON	((USEFLAG) 0x0200)
#define	USE_ROOT_DIR	((USEFLAG) 0x0400)
#define	USE_EXEC_INODE	((USEFLAG) 0x0800)
#define	USE_USER_NAME	((USEFLAG) 0x1000)
#define	USE_GROUP_NAME	((USEFLAG) 0x2000)
#define	USE_WCHAN	((USEFLAG) 0x4000)


/*
 * Structure for one column that can be displayed.
 */
struct	COLUMN
{
	char *		name;		/* column name for commands */
	char *		heading;	/* heading string for column */
	int		minwidth;	/* absolute minimum width of column */
	int		initwidth;	/* initial minimum width of column */
	int		width;		/* actual minimum width of column */
	JUSTIFY		justify;	/* how value is justified in column */
	USEFLAG		useflag;	/* what data column uses */
	const char *	(*showfunc)(const PROC * proc);
	int		(*sortfunc)(const PROC * proc1, const PROC * proc2);
	void		(*evalfunc)(const PROC * proc, VALUE * retval);
	BOOL		(*testfunc)(const PROC * proc);
};


/*
 * Identifiers for the different types of macros.
 * Some features of these macro types are built-in.
 * These definitions cannot be changed in isolation.
 */
typedef	int	MACRO_TYPE;

#define	MACRO_TYPE_NONE		((MACRO_TYPE) -1)
#define	MACRO_TYPE_OPTION	((MACRO_TYPE) 0)
#define	MACRO_TYPE_COLUMN	((MACRO_TYPE) 1)
#define	MACRO_TYPE_EXPRESSION	((MACRO_TYPE) 2)

      
/*
 * Interface to communicate with an input/output display device.
 * This can be, for example, a dumb terminal, a curses terminal,
 * or an X11 window.
 */
typedef	struct	DISPLAY	DISPLAY;

struct	DISPLAY
{
	BOOL	(*open)(DISPLAY *);		/* open display */
	void	(*close)(DISPLAY *);		/* close display */
	void	(*refresh)(DISPLAY *);		/* refresh display */
	void	(*beginPage)(DISPLAY *);	/* begin output of page */
	void	(*putChar)(DISPLAY *, int);	/* output character */
	void	(*putString)(DISPLAY *, const char *);	/* output string */
	void	(*putBuffer)(DISPLAY *, const char *, int);	/* buffer */
	void	(*endPage)(DISPLAY *);		/* end output of page */
	BOOL	(*eventWait)(DISPLAY *, int);	/* handle events and wait */
	BOOL	(*inputReady)(DISPLAY *);	/* check if input is ready */
	int	(*readChar)(DISPLAY *);		/* read input character */
	void	(*ringBell)(DISPLAY *);		/* ring the bell */
	int	(*getRows)(DISPLAY *);		/* get number of rows */
	int	(*getCols)(DISPLAY *);		/* get number of columns */
	BOOL	(*doesScroll)(DISPLAY *);	/* whether display scrolls */
};


#define	DISPLAY_TYPE_TTY	"tty"
#define	DISPLAY_TYPE_CURSES	"curses"
#define	DISPLAY_TYPE_X11	"x11"


/*
 * List of columns being shown.
 */
extern	int	show_count;
extern	COLUMN *show_list[MAX_COLUMNS];


/*
 * Other global variables.
 */
extern	ULONG		start_uptime;		/* uptime jiffies at start */
extern	time_t		start_time;		/* clock time at start */
extern	time_t		current_time;		/* current clock time */
extern	long		total_memory_clicks;	/* amount of total memory */
extern	ULONG		live_counter;		/* counter for live procs */
extern	BOOL		ancient_flag;		/* seeing pre-existing procs */
extern	BOOL		no_self;		/* don't show myself */
extern	BOOL		no_root;		/* don't show root procs */
extern	BOOL		no_header;		/* don't show column header */
extern	BOOL		info;			/* show info line at top */
extern	BOOL		my_procs;		/* only show my procs */
extern	BOOL		active_only;		/* only show active procs */
extern	BOOL		clear_screen;		/* clear screen each loop */
extern	BOOL		loop;			/* loop showing status */
extern	BOOL		running;		/* we still want to run */
extern	BOOL		frozen;			/* data collection is frozen */
extern	BOOL		update;			/* update once even if frozen */
extern	BOOL		needRefresh;		/* need to refresh display */
extern	BOOL		vertical;		/* vertical output format */
extern	BOOL		top_set;		/* top option was used */
extern	BOOL		top_auto;		/* autosize height for top */
extern	BOOL		use_open_files;		/* using open file info */
extern	BOOL		use_current_directory;	/* using current dir info */
extern	BOOL		use_root_directory;	/* using root dir info */
extern	BOOL		use_exec_inode;		/* using executable info */
extern	BOOL		use_device_names;	/* using device name info */
extern	BOOL		use_user_names;		/* using user name info */
extern	BOOL		use_group_names;	/* using group name info */
extern	BOOL		use_init;		/* using initial sleep */
extern	BOOL		use_command;		/* using command line info */
extern	BOOL		use_self;		/* using my own proc info */
extern	BOOL		use_environment;	/* using environment info */
extern	BOOL		use_wchan;		/* using wait channel symbol */
extern	BOOL		use_stdio[3];		/* using various stdio info */
extern	pid_t		my_pid;			/* my pid */
extern	uid_t		my_uid;			/* my real user id */
extern	gid_t		my_gid;			/* my real group id */
extern	dev_t		null_dev;		/* device of /dev/null */
extern	ino_t		null_inode;		/* inode of /dev/null */
extern	int		proc_alloc_count;	/* allocated proc structures */
extern	int		deathTime;		/* seconds for dead processes */
extern	int		active_time;		/* seconds for active procs */
extern	int		pid_count;		/* pids in pid_list */
extern	int		user_count;		/* users in user_list */
extern	int		group_count;		/* groups in group_list */
extern	int		program_count;		/* programs in program_list */
extern	int		width;			/* width of output */
extern	int		height;			/* height of output */
extern	int		separation;		/* blanks between columns */
extern	int		sleep_time_ms;		/* milliseconds between loops */
extern	int		sync_time;		/* seconds between syncs */
extern	int		init_time;		/* seconds for initial sleep */
extern	int		top_count;		/* number of procs for top */
extern	int		scrollSeconds;		/* seconds between scrolling */
extern	int		overlapLines;		/* lines of overlap */
extern	int		skipCount;		/* lines to skip in display */
extern	int		procShowCount;		/* processes wanting showing */
extern	int		procTotalCount;		/* count of all processes */
extern	PROC *		process_list;		/* list of existent procs */
extern	PROC *		free_process_list;	/* free proc structure list */
extern	char *		geometry;		/* window geometry string */
extern	char *		fontName;		/* font name */
extern	char *		foregroundName;		/* foreground color name */
extern	char *		backgroundName;		/* background color name */
extern	char *		displayName;		/* display name */
extern	const char *	displayType;		/* display type */
extern	char		empty_string[4];	/* empty string */
extern	char		root_string[4];		/* root path string */
extern	pid_t		pid_list[MAX_PIDS];	/* pids to be shown */
extern	uid_t		user_list[MAX_USERS];	/* user ids to be shown */
extern	gid_t		group_list[MAX_GROUPS];	/* group ids to be shown */
extern	char		program_list[MAX_PROGRAMS][MAX_PROGRAM_LEN + 2];


/*
 * Global procedures.
 */
extern	BOOL	PatternMatch(const char * text, const char * pattern);
extern	char *	AllocateSharedString(const char * str, int len);
extern	void	FreeSharedString(char * str);
extern	char *	AllocTempString(int len);
extern	char *	CopyTempString(const char * oldcp);
extern	void	FreeTempStrings(void);
extern	void *	AllocMemory(int len);
extern	void *	ReallocMemory(void * oldBuffer, int len);
extern	char *	CopyString(const char *);
extern	void	ReplaceString(char **, const char *);
extern	void	DefaultAllOptions(void);
extern	void	DefaultColumns(void);
extern	void	DefaultColumnWidths(void);
extern	ULONG	GetUptime(void);
extern	BOOL	ParseSystemInitFile(void);
extern	BOOL	ParseUserInitFile(void);
extern	BOOL	ParseFile(const char * name, BOOL isOptional);
extern	BOOL	ParseOptions(ARGS * ap, int depth);
extern	BOOL	ExpandOptionName(const char * name, int depth);
extern	long	GetDecimalNumber(char ** cpp);
extern	double	GetFloatingNumber(char ** cpp);
extern	void	ClearCondition(void);
extern	BOOL	ParseCondition(char * str);
extern	USEFLAG	GetConditionUseFlags(void);
extern	USEFLAG	GetSortingUseFlags(void);
extern	void	ClearSorting(void);
extern	void	SortProcesses(void);
extern	void	UpdateProcessCounts(void);
extern	BOOL	AppendColumnSort(ARGS * ap, BOOL reverse);
extern	BOOL	AppendExpressionSort(ARGS * ap, BOOL reverse);
extern	PROC *	FindProcess(pid_t pid);
extern	BOOL	InitializeProcessData(void);
extern	BOOL	InitializeDisplay(void);
extern	void	TopPage(void);
extern	void	BottomPage(void);
extern	void	NextPage(void);
extern	void	PreviousPage(void);
extern	void	ScanProcesses(void);
extern	void	CheckActiveProcess(PROC * proc);
extern	void	RemoveDeadProcesses(void);
extern	void	InitialProcessScan(void);
extern	BOOL	IsShownProcess(const PROC * proc);
extern	void	ShowSelectedProcesses(void);
extern	void	ListMacros(void);
extern	COLUMN *FindColumn(const char * name);
extern	void	ListColumns(void);
extern	void	CollectUserNames(void);
extern	void	CollectGroupNames(void);
extern	void	CollectDeviceNames(void);
extern	void	CollectStaticSystemInfo(void);
extern	uid_t	FindUserId(const char * name);
extern	gid_t	FindGroupId(const char * name);
extern	const char *	FindUserName(uid_t uid);
extern	const char *	FindGroupName(gid_t gid);
extern	const char *	FindDeviceName(dev_t devid);
extern	const char *	FindDeviceFromInode(dev_t dev, ino_t inode);
extern	void	UpdateTimes(void);
extern	void	ResetScrollTime(void);
extern	void	WaitForCommands(long milliSeconds);
extern	BOOL	ReadCommands(void);
extern	int	ExpandArguments(ARGS * ap, char ** table, int tablelen);
extern	BOOL	MacroExists(MACRO_TYPE id, const char * name);
extern	BOOL	DefineMacro(MACRO_TYPE id, const char * name, const char * str);
extern	BOOL	ExpandMacro(MACRO_TYPE id, const char * name, ARGS * retargs);
extern	void	MakePrintable(char * cp, int len);
extern	void	GetTimeOfDay(struct timeval * retTimeVal);

extern	long	ElapsedMilliSeconds(const struct timeval * oldTime,
			const struct timeval * newTime);

extern	BOOL	CompareValues(const VALUE leftval,
			const VALUE rightval, int * result);

extern	BOOL	DpySetDisplay(const char * type);
extern	BOOL	DpyOpen(void);
extern	void	DpyClose(void);
extern	void	DpyRefresh(void);
extern	void	DpyBeginPage(void);
extern	void	DpyChar(int ch);
extern	void	DpyString(const char * str);
extern	void	DpyBuffer(const char * buffer, int length);
extern	void	DpyEndPage(void);
extern	BOOL	DpyInputReady(void);
extern	int	DpyReadChar(void);
extern	void	DpyRingBell(void);
extern	int	DpyGetRows(void);
extern	int	DpyGetCols(void);
extern	BOOL	DpyDoesScroll(void);
extern	BOOL	DpyEventWait(int milliSeconds);

extern	DISPLAY *	GetTtyDisplay(void);
extern	DISPLAY *	GetCursesDisplay(void);
extern	DISPLAY *	GetX11Display(void);

#endif
