/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the IIWU program. 
 * IIWU is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */
#ifndef _IIWUSYNTH_H
#define _IIWUSYNTH_H

#ifdef __cplusplus
extern "C" {
#endif

#if defined(WIN32)
#if defined(IIWUSYNTH_DLL_EXPORTS)
#define IIWU_SYNTH_API __declspec(dllexport)
#else
#define IIWU_SYNTH_API __declspec(dllimport)
#endif
#else
#define IIWU_SYNTH_API
#endif

/*
 *    Synth
 */
typedef struct _iiwu_synth_t iiwu_synth_t;

/** Creates a new syntehsizer object. As soon as the synthesizer is
 * created, it will start playing. the driver and the device can be
 * NULL in which case the defaults will be used. */
IIWU_SYNTH_API iiwu_synth_t* new_iiwu_synth(int poly, int auport, char* driver, char* dev);

/** Deletes the synthesizer previously created with new_iiwu_synth.*/
IIWU_SYNTH_API int delete_iiwu_synth(iiwu_synth_t* synth);


/* MIDI channel messages */

/** Send a noteon message */
IIWU_SYNTH_API int iiwu_synth_noteon(iiwu_synth_t* synth, int chan, int key, int vel);

/** Send a noteoff message */
IIWU_SYNTH_API int iiwu_synth_noteoff(iiwu_synth_t* synth, int chan, int key);

/** Send a program change message */
IIWU_SYNTH_API int iiwu_synth_program_change(iiwu_synth_t* synth, int chan, int program);

/** Send a control change message */
IIWU_SYNTH_API int iiwu_synth_cc(iiwu_synth_t* synth, int chan, int ctrl, int val);

/** Send a pitch bend message */
IIWU_SYNTH_API int iiwu_synth_pitch_bend(iiwu_synth_t* synth, int chan, int val);

/** Short-hand functions */
IIWU_SYNTH_API int iiwu_synth_bank_select(iiwu_synth_t* synth, int chan, unsigned int bank);


/** Load a SoundFont. The newly loaded SoundFont will be put on top
 * of the SoundFont stack. Presets are searched starting from the
 * SoundFont on the top of the stack, working the way down the stack
 * until a preset is found.  */
IIWU_SYNTH_API int iiwu_synth_sfload(iiwu_synth_t* synth, const char* filename);

/** Returns the number of loaded SoundFonts */
IIWU_SYNTH_API int iiwu_synth_sfcount(iiwu_synth_t* synth);

/** Returns the name of the SoundFont */
IIWU_SYNTH_API char* iiwu_synth_sfname(iiwu_synth_t* synth, int num);

/** Reload a SoundFont */
IIWU_SYNTH_API char* iiwu_synth_sfreload(iiwu_synth_t* synth, int num);

/** Remove a SoundFont for the stack */
IIWU_SYNTH_API char* iiwu_synth_sfunload(iiwu_synth_t* synth, int num);


/* Get information about the presets */

/** Get the name of first preset of a SoundFont */
IIWU_SYNTH_API char* iiwu_synth_first_preset(iiwu_synth_t* synth, int font);

/** Get the name of the next preset of a SoundFont */
IIWU_SYNTH_API char* iiwu_synth_next_preset(iiwu_synth_t* synth, int font);

/** Get the name of the preset selected of a channel */
IIWU_SYNTH_API char* iiwu_synth_get_channel_preset(iiwu_synth_t* synth, int chan);

  /** Get a textual representation of the last error */
IIWU_SYNTH_API char* iiwu_synth_error(iiwu_synth_t* synth);



/* Low level function for plugin hosts */

IIWU_SYNTH_API int iiwu_synth_write(void* synth, void* buf, int len);
IIWU_SYNTH_API int iiwu_synth_write_lr(void* synth, int len, 
				       void* left, int loff, int lincr, 
				       void* right, int roff, int rincr);

#ifdef WIN32
/** Set the handle to the instance of the application on the Windows
    platform. The handle is needed to open DirectSound. */
IIWU_SYNTH_API void iiwu_set_hinstance(void* hinstance);
#endif



/* 

  Synthesizer settings

  the synth_settings are NOT used nor implemented, yet 
  possible settings are:
      polyphony:     maximum polyphony
      audio:         output the audio
      driver:        driver name
      device:        device name
      buffersize:    the synthesis buffer size
      queuesize:     the audio queue size
      hardware:      try to use hardware acceleration
      interpolation: type of interpolation {truncate,linear,cubic}
*/

/** returns the number of settings */
IIWU_SYNTH_API int iiwu_count_settings(void);

/** returns the name of setting i */
IIWU_SYNTH_API char* iiwu_setting_name(int i);

/** returns a description of setting i */
IIWU_SYNTH_API char* iiwu_setting_description(int i);

/* returns the type of the settings: 
     'b' = boolean {yes,no} or {true,false}
     'i' = integer
     'f' = floating point value
     's' = string
 */
IIWU_SYNTH_API char iiwu_setting_type(int i);

typedef struct _iiwu_synth_settings_t iiwu_synth_settings_t;

IIWU_SYNTH_API iiwu_synth_settings_t* new_iiwu_synth_settings(void);
IIWU_SYNTH_API int delete_iiwu_synth_settings(iiwu_synth_settings_t* params);
IIWU_SYNTH_API int iiwu_synth_settings_set(iiwu_synth_settings_t* params, char* name, char* val);
IIWU_SYNTH_API char* iiwu_synth_settings_get(iiwu_synth_settings_t* params, char* name);


/*
 *       MIDI
 */
typedef struct _iiwu_player_t iiwu_player_t;

IIWU_SYNTH_API iiwu_player_t* new_iiwu_player(iiwu_synth_t* synth);
IIWU_SYNTH_API int delete_iiwu_player(iiwu_player_t* player);
IIWU_SYNTH_API int iiwu_player_load(iiwu_player_t* player, char* midifile);
IIWU_SYNTH_API int iiwu_player_play(iiwu_player_t* player);
IIWU_SYNTH_API int iiwu_player_stop(iiwu_player_t* player);
IIWU_SYNTH_API int iiwu_player_join(iiwu_player_t* player);
IIWU_SYNTH_API int iiwu_player_set_loop(iiwu_player_t* player, int loop);
IIWU_SYNTH_API int iiwu_player_set_midi_tempo(iiwu_player_t* player, int tempo);
IIWU_SYNTH_API int iiwu_player_set_bpm(iiwu_player_t* player, int bpm);


typedef struct _iiwu_midi_handler_t iiwu_midi_handler_t;


/** Create a new midi handler. A midi handler connects to a midi input
    device and forwards incoming midi events to the synthesizer. */
iiwu_midi_handler_t* new_iiwu_midi_handler(iiwu_synth_t* synth, char* driver, char* device); 

/** Delete the midi handler.  */
int delete_iiwu_midi_handler(iiwu_midi_handler_t* dev); 

/** Wait until the midi handler has finished. */
int iiwu_midi_handler_join(iiwu_midi_handler_t* dev);


/* The following set of functions are meant as a means to monitor the
   midi input */

/** Get the name of the currently used midi driver */
char* iiwu_midi_handler_get_driver_name(iiwu_midi_handler_t* handler);

/** Get the name of the currently used midi device */
char* iiwu_midi_handler_get_device_name(iiwu_midi_handler_t* handler);

/** Get the status midi handler */
char* iiwu_midi_handler_get_status(iiwu_midi_handler_t* handler);

/** Get the event count for a particular midi event type. The type is
    specified by the midi opcode. */
int iiwu_midi_handler_get_event_count(iiwu_midi_handler_t* handler, int type);



/*
 *
 *      Low level functions
 * */

/*
 *      Auport
 */
typedef enum {
  IIWU_SAMPLE_DOUBLE,
  IIWU_SAMPLE_FLOAT,
  IIWU_SAMPLE_S8,
  IIWU_SAMPLE_U8,
  IIWU_SAMPLE_S16HE,
  IIWU_SAMPLE_S16BE,
  IIWU_SAMPLE_S16LE,
  IIWU_SAMPLE_U16HE,
  IIWU_SAMPLE_U16BE,
  IIWU_SAMPLE_U16LE
} iiwu_sample_format;

/** Audio port object. */
typedef struct _iiwu_auport_t iiwu_auport_t;

/** Calback function for the application used in the sound output
 *  pull function. It requests the application to synthesize 1
 *  buffer of samples. If the application does not return zero, the
 *  pull function stops and returns normally. */
typedef int (*iiwu_auport_callback_t)(void* data, void* out, int outlen);

/** new_iiwu_auport
 *
 * Open a new audio out port that pulls the samples from the
 * application.
 *
 * @param device The name of the audio device to be opened.
 * @param app_format The sample format used in the application. 
 * @param dev_format The sample format used in the device. 
 * @param qs The queue size. The queue size is the size of the
 * internal buffer used to store the samples. The size is measured in
 * frames (not in bytes nor in samples).
 * @param bs The buffer size. The buffer size is the number of sample
 * frames that will be written to the port. The size is measured in
 * frames (not in bytes nor in samples).
 * @param callBack The application's callback function.
 * @param appData Pointer to the application data.
 *
 * @return A pointer to the audio port or NULL if error.  */
IIWU_SYNTH_API iiwu_auport_t* new_iiwu_auport(char* driver, char* device, 
					      int app_format, int dev_format,
					      int sr, int ch, int bs, int qs, 
					      iiwu_auport_callback_t callback, 
					      void* data);

/** delete_iiwu_auport
 *
 * Close an audio port previously opened with new_iiwu_auport.
 *
 * @param port The audio port.
 */
IIWU_SYNTH_API int delete_iiwu_auport(iiwu_auport_t* p);


#ifdef __cplusplus
}
#endif

#endif /* _IIWUSYNTH_H */
