/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the IIWU program. 
 * IIWU is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include "iiwusynth.h"

#if !defined(WIN32) && !defined(MACINTOSH)
#define _GNU_SOURCE
#include <getopt.h>
#endif

void print_usage(void);
void print_help(void);

/*
 * the globals
 */
char* appname = NULL;
char* audio_driver = NULL;
char* audio_device = NULL;

/*
 * support for the getopt function
 */
#if !defined(WIN32) && !defined(MACINTOSH)
int getopt(int argc, char * const argv[], const char *optstring);
extern char *optarg;
extern int optind, opterr, optopt;
#endif

int main(int argc, char** argv) 
{
  iiwu_synth_t* synth;
  iiwu_player_t* player; 
  int arg1 = 1;
  int i, c;

  appname = argv[0];

#if !defined(WIN32) && !defined(MACINTOSH)
  /* handle command options. on posix machines only */
  opterr = 0;
  while (1) {
    int option_index = 0;
    static struct option long_options[] = {
      {"audio-driver", 1, 0, 'a'},
      {"audio-device", 1, 0, 'A'},
      {"help", 0, 0, 'h'},
      {0, 0, 0, 0}
    };
    c = getopt_long(argc, argv, "ha:A:", long_options, &option_index);
    if (c == -1) {
      break;
    }
    switch (c) {
    case 0:
      printf ("option %s", long_options[option_index].name);
      if (optarg) {
	printf (" with arg %s", optarg);
      }
      printf ("\n");
      break;
    case 'a':
      audio_driver = optarg;
      break;
    case 'A':
      audio_device = optarg;
      break;
    case 'h':
      print_help();
      exit(0);
    case '?':
      printf ("Unknown option %c\n", optopt);
      print_usage();
      exit(0);
      break;
    default:
      printf ("?? getopt returned character code 0%o ??\n", c);
    }
  }
  arg1 = optind;
#endif

  if (arg1 >= argc - 1) {
    print_usage();    
  }

  synth = new_iiwu_synth(128, 1, audio_driver, audio_device);
  if (synth == NULL) {
    printf("Failed to create the synthesizer\n");
    exit(-1);
  }

  for (i = arg1 + 1; i < argc; i++) {
    if (iiwu_synth_sfload(synth, argv[i]) != 0) {
      printf("Failed to load the SoundFont (%s)\n", argv[argc]);
      exit(-1);
    }
  }

  player = new_iiwu_player(synth);
  if (player == NULL) {
    printf("Failed to create the player\n");
    exit(-1);
  }
  if (iiwu_player_load(player, argv[arg1]) != 0) {
    printf("Failed to load the midifile (%s)\n", argv[argc]);
    exit(-1);
  }
  if (iiwu_player_play(player) != 0) {
    printf("Failed to play the midifile\n");
    exit(-1);
  }
  if (iiwu_player_join(player) != 0) {
    printf("Failed to play the midifile till the end\n");
    exit(-1);
  }
  delete_iiwu_player(player);
  delete_iiwu_synth(synth);
  return 0;
}

void print_usage() 
{
  printf("Usage: %s [options] [soundfonts]\n", appname);
  printf("Try -h for help.\n");
  exit(0);
}

void print_help() 
{
  printf("Usage: \n");
  printf("  iiwusynth [options] midifile soundfont1 soundfont2 ... \n");
  printf("Possible options:\n");
  printf(" -a, --audio-driver=[label]\n    the audio driver [oss,alsa,winmidi,...]\n\n");
  printf(" -A, --audio-device=[device]\n    the audio device\n\n");
  printf(" -h, --help\n    print out this help summary\n\n");
  exit(0);
}
