/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  A double-valued position
//

#ifndef IPOSITION_H
#define IPOSITION_H


#include "ipointermacro.h"

class iViewModule;


class iPosition
{

	friend class iView;

	IPOINTER_AS_PART(ViewModule);

public:

	iPosition(iViewModule *vm);
	~iPosition();

	void SetBoxPosition(const double *x);
	void SetBoxPosition(double x1, double x2, double x3);
	void SetBoxPosition(int i, double x);

	inline const double* BoxPosition() const { this->UpdateBoxPosition(); return mBoxX; }
	inline double BoxPosition(int i) const { this->UpdateBoxPosition(); return mBoxX[i]; }

	inline const double* OpenGLPosition() const { return mX; }
	inline double OpenGLPosition(int i) const { return mX[i]; }

	iPosition& operator=(iPosition&);
	iPosition& operator=(double *);
	const iPosition& operator=(const double *);

	inline operator double*() { return mX; } 
	inline operator const double*() const { return mX; } 

	inline double operator[](int i) const { return mX[i]; }
	inline double& operator[](int i) { return mX[i]; }

	void PeriodicWrap();
	void CutToBounds();

private:
	
	void UpdateOpenGLPosition();
	void UpdateBoxPosition() const;

	mutable double mBoxX[3];
	double mX[3];

	iPosition(const iPosition&); // not implemented
};


class iDistance
{

	friend class iView;

	IPOINTER_AS_PART(ViewModule);

public:

	iDistance(iViewModule *vm, bool size);
	~iDistance();

	void SetBoxDistance(double x);

	inline double BoxDistance() const { this->UpdateBoxDistance(); return mBoxX; }

	inline double OpenGLDistance() const { return mX; }

	iDistance& operator=(iDistance&);
	iDistance& operator=(double);

	inline bool operator<(const iDistance &p) const { return mX < p.mX; }
	inline bool operator>(const iDistance &p) const { return mX > p.mX; }
	inline bool operator<(double x) const { return mX < x; }
	inline bool operator>(double x) const { return mX > x; }

	inline operator double() const { return mX; } 

	void PeriodicWrap();
	void CutToBounds();

private:
	
	void UpdateOpenGLDistance();
	void UpdateBoxDistance() const;

	const bool mIsSize;
	mutable double mBoxX;
	double mX;

	iDistance(const iDistance&); // not implemented
};


//
//  iPosition functions
//
inline void iPosition::SetBoxPosition(const double *x)
{
	int i;
	for(i=0; i<3; i++) mBoxX[i] = x[i];
	this->UpdateOpenGLPosition();
}


inline void iPosition::SetBoxPosition(double x1, double x2, double x3)
{
	mBoxX[0] = x1;
	mBoxX[1] = x2;
	mBoxX[2] = x3;
	this->UpdateOpenGLPosition();
}


inline void iPosition::SetBoxPosition(int i, double x)
{
	if(i>=0 && i<3)
	{
		mBoxX[i] = x;
		this->UpdateOpenGLPosition();
	}
}


inline iPosition& iPosition::operator=(iPosition &p)
{
	int i;
	for(i=0; i<3; i++) mX[i] = p.mX[i];
	return *this;
}


inline iPosition& iPosition::operator=(double *x)
{
	int i;
	for(i=0; i<3; i++) mX[i] = x[i];
	return *this;
}


inline const iPosition& iPosition::operator=(const double *x)
{
	int i;
	for(i=0; i<3; i++) mX[i] = x[i];
	return *this;
}

//
//  iDistance functions
//
inline void iDistance::SetBoxDistance(double x)
{
	if(!mIsSize || x>0.0)
	{
		mBoxX = x;
		this->UpdateOpenGLDistance();
	}
}


inline iDistance& iDistance::operator=(iDistance &p)
{
	if(!mIsSize || p.mX>0.0) mX = p.mX; // assignment does not overwrite the Relative flag
	return *this;
}


inline iDistance& iDistance::operator=(double x)
{
	if(!mIsSize || x>0.0) mX = x;
	return *this;
}

#endif  // IPOSITION_H

