/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggdatatypeprovider.h"


#include "icontrolmodule.h"
#include "idata.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "ierror.h"
#include "ishell.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "iviewsubject.h"

#include "iggmainwindow.h"
#include "iggpageobject.h"


//
//  Base class
//
iggDataTypeProvider::iggDataTypeProvider(iShell *s) : iShellComponent(s)
{
}


iggDataTypeProvider::~iggDataTypeProvider()
{
}


const iObjectKey& iggDataTypeProvider::Translate(const iObjectKey &key) const
{
	if(key.Type() == iDataSubject::Type())
	{
		iDataSubject *subject = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetSubject(this->GetActiveDataType());
		if(subject == 0) return key;

		const iObjectKey *p = iObjectKeyRegistry::FindKey(key.RetypedKey(subject->RealType()));
		if(p == 0) return key; else return *p;
	}
	else
	{
		return key;
	}
}


//
//  DataTypeProvider for a given fixed DataType
//
iggFixedDataTypeProvider::iggFixedDataTypeProvider(const iDataType &type, iggWidget *parent) : iggDataTypeProvider((parent==0)?0:parent->GetShell())
{
	mInfoPointer = new iDataInfo; IERROR_ASSERT(mInfoPointer);
	*mInfoPointer = type;
}


iggFixedDataTypeProvider::~iggFixedDataTypeProvider()
{
	delete mInfoPointer;
}


int iggFixedDataTypeProvider::GetActiveDataTypeIndex() const
{
	return 0;
}


void iggFixedDataTypeProvider::SetActiveDataTypeIndex(int)
{
}


const iDataInfo& iggFixedDataTypeProvider::GetDataInfo() const
{
	return *mInfoPointer;
}


const iDataType& iggFixedDataTypeProvider::GetActiveDataType() const
{
	return mInfoPointer->Type(0);
}


//
//  DataTypeProvider for a given keyword
//
iggKeywordDataTypeProvider::iggKeywordDataTypeProvider(const iString &keywords, iggWidget *parent) : iggDataTypeProvider((parent==0)?0:parent->GetShell())
{
	mParent = parent;

	mActiveIndex = 0;
	mInfoPointer = new iDataInfo; IERROR_ASSERT(mInfoPointer);

	iDataType::FindAllTypesByKeywords(*mInfoPointer,keywords);
}


iggKeywordDataTypeProvider::~iggKeywordDataTypeProvider()
{
	delete mInfoPointer;
}


bool iggKeywordDataTypeProvider::IsThereDataType() const
{
	return (iData::None() != this->GetActiveDataType());
}


void iggKeywordDataTypeProvider::FindAvailableData()
{
	if(!this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(this->GetActiveDataType()))
	{
		//
		//  Switch to the first available type
		//
		int i;
		for(i=0; i<mInfoPointer->Count(); i++)
		{
			if(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(mInfoPointer->Type(i))) break;
		}
		if(i < mInfoPointer->Count()) this->SetActiveDataTypeIndex(i);
	}
}


int iggKeywordDataTypeProvider::GetActiveDataTypeIndex() const
{
	return mActiveIndex;
}


void iggKeywordDataTypeProvider::SetActiveDataTypeIndex(int v)
{
	if(mActiveIndex!=v && v>=0 && v<mInfoPointer->Count())
	{
		mActiveIndex = v;
		mParent->UpdateWidget();
	}
}


const iDataInfo& iggKeywordDataTypeProvider::GetDataInfo() const
{
	return *mInfoPointer;
}


const iDataType& iggKeywordDataTypeProvider::GetActiveDataType() const
{
	if(mActiveIndex>=0 && mActiveIndex<mInfoPointer->Count())
	{
		return mInfoPointer->Type(mActiveIndex);
	}
	else
	{
		return iData::None();
	}
}


//
//  DataTypeProvider for a ViewSubject
//
iggViewSubjectDataTypeProvider::iggViewSubjectDataTypeProvider(iggPageObject *page) : iggDataTypeProvider((page==0)?0:page->GetShell())
{
	mPage = page; IERROR_ASSERT(page);
}


iggViewSubjectDataTypeProvider::~iggViewSubjectDataTypeProvider()
{
}


int iggViewSubjectDataTypeProvider::GetActiveDataTypeIndex() const
{
	return mPage->GetViewObject()->GetActiveDataTypeIndex();
}


void iggViewSubjectDataTypeProvider::SetActiveDataTypeIndex(int v)
{
	if(v != mPage->GetViewObject()->GetActiveDataTypeIndex())
	{
		mPage->GetViewObject()->SetActiveDataTypeIndex(v);
		mPage->GetMainWindow()->UpdateAll();
	}
}


const iDataInfo& iggViewSubjectDataTypeProvider::GetDataInfo() const
{
	return mPage->GetViewObject()->GetPrimaryDataInfo();
}


const iDataType& iggViewSubjectDataTypeProvider::GetActiveDataType() const
{
	return mPage->GetViewObject()->GetDataType();
}


//
//  DataTypeProvider that returns scalar data
//
iggScalarFieldDataTypeProvider::iggScalarFieldDataTypeProvider(iggPageObject *page) : iggViewSubjectDataTypeProvider(page)
{
}


iggScalarFieldDataTypeProvider::~iggScalarFieldDataTypeProvider()
{
}


const iDataInfo& iggScalarFieldDataTypeProvider::GetDataInfo() const
{
#ifdef I_CHECK1
	IERROR_REPORT_BUG;
#endif
	return mPage->GetViewObject()->GetPrimaryDataInfo();
}


const iDataType& iggScalarFieldDataTypeProvider::GetActiveDataType() const
{
	return iDataType::ReSpecifyType(mPage->GetViewObject()->GetDataType(),"ScalarField");
}

