/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iimagecomposer.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "imath.h"
#include "ishellfactory.h"
#include "istereoimagearray.h"
#include "iviewmodule.h"

//
//  Templates
//
#include "iarraytemplate.h"


//
// Helper classes
//
void iImageComposerWindow::SetViewModule(iViewModule *vm)
{
	mViewModule = vm;
	mComposer->ClearCache();
}


iImageComposerBackgroundWindow* iImageComposerBackgroundWindow::New(int tx, int ty, iImageComposer *ic)
{
	return new iImageComposerBackgroundWindow(tx,ty,ic);
}


iImageComposerBackgroundWindow::iImageComposerBackgroundWindow(int tx, int ty, iImageComposer *ic) : iImageComposerWindow(0,ic)
{
	mType = BACKGROUND;
	mTileX = tx;
	mTileY = ty;
}


int iImageComposerBackgroundWindow::GetWidth() const
{
	if(this->GetViewModule() != 0) return this->GetViewModule()->GetThisImageWidth(); else if(!mWallpaperImage.IsEmpty()) return mWallpaperImage.Width(); else return 0;
}


int iImageComposerBackgroundWindow::GetHeight() const
{
	if(this->GetViewModule() != 0) return this->GetViewModule()->GetThisImageHeight(); else if(!mWallpaperImage.IsEmpty()) return mWallpaperImage.Height(); else return 0;
}


bool iImageComposerBackgroundWindow::LoadWallpaperImage(const iString &s)
{
	bool ret = mWallpaperImage.LoadFromFile(s);
	if(ret) 
	{
		mWallpaperFile = s;
		mComposer->UpdateSize();
		mComposer->ClearCache();
	}
	return ret;
}


int iImageComposerForegroundWindow::mPosQuantum = 10;

iImageComposerForegroundWindow* iImageComposerForegroundWindow::New(iViewModule *vm, iImageComposer *ic)
{
	return new iImageComposerForegroundWindow(vm,ic);
}


iImageComposerForegroundWindow::iImageComposerForegroundWindow(iViewModule *vm, iImageComposer *ic) : iImageComposerWindow(vm,ic)
{
	mType = FOREGROUND;
	mBorderWidth = 1;
	mScale = 0.5;
	mPosX = mPosY = 0;
}


int iImageComposerForegroundWindow::GetWidth() const
{
	if(this->GetViewModule() != 0) return this->GetViewModule()->GetThisImageWidth(); else return 0;
}


int iImageComposerForegroundWindow::GetHeight() const
{
	if(this->GetViewModule() != 0) return this->GetViewModule()->GetThisImageHeight(); else return 0;
}


int iImageComposerForegroundWindow::GetScaledWidth() const
{
	return round(mScale*this->GetWidth());
}


int iImageComposerForegroundWindow::GetScaledHeight() const
{
	return round(mScale*this->GetHeight());
}


void iImageComposerForegroundWindow::SetScale(float s)
{
	if(s>0.0 && s<=1.0 && fabs(s-mScale)>1.0e-4) 
	{
		mScale = s;
		this->CorrectPosition();
		mComposer->ClearCache();
	}
}


void iImageComposerForegroundWindow::SetPosition(int x, int y)
{
	//
	//  Quantize the position but not if we are against the outer wall
	//
	if(x+this->GetImageWidth() < mComposer->GetImageWidth()) mPosX = mPosQuantum*(x/mPosQuantum); else mPosX = x;
	if(y+this->GetImageHeight() < mComposer->GetImageHeight()) mPosY = mPosQuantum*(y/mPosQuantum); else mPosY = y;
	this->CorrectPosition();
	mComposer->ClearCache();
}


void iImageComposerForegroundWindow::SetBorderWidth(int w)
{
	if(w>=0 && w!=mBorderWidth)
	{
		mBorderWidth = w;
		this->CorrectPosition();
		mComposer->ClearCache();
	}
}


void iImageComposerForegroundWindow::SetBorderColor(const iColor& c)
{
	if(c != mBorderColor)
	{
		mBorderColor = c;
		mComposer->ClearCache();
	}
}


void iImageComposerForegroundWindow::CorrectPosition()
{
	if(mPosX < 0) mPosX = 0;
	if(mPosY < 0) mPosY = 0;

	if(mBorderWidth > this->GetWidth()/4) mBorderWidth = this->GetWidth()/4;
	if(mBorderWidth > this->GetHeight()/4) mBorderWidth = this->GetHeight()/4;

	//
	//  Is scale small enough?
	//
	if(this->GetImageWidth() > mComposer->GetImageWidth())
	{
		mPosX = 0;
		mScale = (float)(mComposer->GetImageWidth()-2*mBorderWidth)/this->GetWidth();
	}
		
	if(this->GetImageHeight() > mComposer->GetImageHeight()) 
	{
		mPosY = 0;
		mScale = (float)(mComposer->GetImageHeight()-2*mBorderWidth)/this->GetHeight();
	}		
	//
	//  Now we are guaranteed to fit into the image. But are we fully inside?
	//
	if(mPosX+this->GetImageWidth() > mComposer->GetImageWidth()) mPosX = mComposer->GetImageWidth() - this->GetImageWidth();
	if(mPosY+this->GetImageHeight() > mComposer->GetImageHeight()) mPosY = mComposer->GetImageHeight() - this->GetImageHeight();
}


//
//  Main class
//
IOBJECT_DEFINE_TYPE(iImageComposer,ImageComposer,ic,iObjectType::_Module);

IOBJECT_DEFINE_KEY(iImageComposer,BorderColor,bc,Color,1);
IOBJECT_DEFINE_KEY(iImageComposer,BackgroundWindowViewModule,bgv,Int,0);
IOBJECT_DEFINE_KEY(iImageComposer,BackgroundWindowWallpaperFile,bgw,String,0);
IOBJECT_DEFINE_KEY(iImageComposer,BorderWidth,bw,Int,1);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowBorderColor,fgc,Color,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowScale,fgs,Float,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowViewModule,fgv,Int,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowBorderWidth,fgw,Int,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowPositionX,fgx,Int,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowPositionY,fgy,Int,0);
IOBJECT_DEFINE_KEY(iImageComposer,ImageHeight,h,Int,1);
IOBJECT_DEFINE_KEY(iImageComposer,InnerBorder,ib,Bool,1);
IOBJECT_DEFINE_KEY(iImageComposer,NumForegroundWindows,nfg,Int,1);
IOBJECT_DEFINE_KEY(iImageComposer,NumTiles,nt,Int,2);
IOBJECT_DEFINE_KEY(iImageComposer,ScaleBackground,sb,Bool,1);
IOBJECT_DEFINE_KEY(iImageComposer,ImageWidth,w,Int,1);


iImageComposer* iImageComposer::New(iControlModule *cm, iShell *s)
{
	return new iImageComposer(cm,s);
}


iImageComposer::iImageComposer(iControlModule *cm, iShell *s) : iControlModuleComponent(cm), iShellComponent(s), mForegroundWindow(0)
{
	mBlockUpdate = false;

	mNumTilesX = mNumTilesY = 1;
	mBorderWidth = 0;
	mBorderColor = iColor(0,0,0);
	mInnerBorder = false;
	mFullWidth = mTileWidth = 640;
	mFullHeight = mTileHeight = 480;

	mBackgroundWindow = new iImageComposerBackgroundWindow*[1]; IERROR_ASSERT(mBackgroundWindow);
	mBackgroundWindow[0] = iImageComposerBackgroundWindow::New(0,0,this);
}


iImageComposer::~iImageComposer()
{
	int i;

	for(i=0; i<mForegroundWindow.Size(); i++)
	{
		mForegroundWindow[i]->Delete();
	}

	for(i=0; i<mNumTilesX*mNumTilesY; i++)
	{
		mBackgroundWindow[i]->Delete();
	}
	delete [] mBackgroundWindow;
}


void iImageComposer::SetNumTiles(int nx, int ny)
{ 
	int i, j;

	if(nx <= 0) nx = mNumTilesX;
	if(ny <= 0) ny = mNumTilesY;

	if(nx==mNumTilesX && ny==mNumTilesY) return;

	//
	//  CreateBackgroundImage checks for out-of-boundary requests, so we need to have
	//  mNumTiles[X,Y] Set before creating new windows
	//
	int oldNumTilesX = mNumTilesX;
	int oldNumTilesY = mNumTilesY;
	mNumTilesX = nx; 
	mNumTilesY = ny; 

	iImageComposerBackgroundWindow **tmp = new iImageComposerBackgroundWindow*[nx*ny]; IERROR_ASSERT(tmp);
	
	for(j=0; j<ny && j<oldNumTilesY; j++)
	{
		for(i=0; i<nx && i<oldNumTilesX; i++)
		{
			tmp[i+nx*j] = mBackgroundWindow[i+oldNumTilesX*j];
		}
	}

	for(j=ny; j<oldNumTilesY; j++)
	{
		for(i=0; i<oldNumTilesX; i++)
		{
			mBackgroundWindow[i+oldNumTilesX*j]->Delete();
		}
	}

	for(j=0; j<ny; j++)
	{
		for(i=nx; i<oldNumTilesX; i++)
		{
			mBackgroundWindow[i+oldNumTilesX*j]->Delete();
		}
	}

	for(j=oldNumTilesY; j<ny; j++)
	{
		for(i=0; i<nx; i++)
		{
			tmp[i+nx*j] = iImageComposerBackgroundWindow::New(i,j,this);
		}
	}

	for(j=0; j<oldNumTilesY; j++)
	{
		for(i=oldNumTilesX; i<nx; i++)
		{
			tmp[i+nx*j] = iImageComposerBackgroundWindow::New(i,j,this);
		}
	}

	delete [] mBackgroundWindow;
	mBackgroundWindow = tmp;

	this->UpdateSize();
	this->ClearCache();
}


void iImageComposer::AddForegroundWindow(iViewModule *v)
{
	if(v == 0) return;

	mForegroundWindow.Add(iImageComposerForegroundWindow::New(v,this));
	this->ClearCache();
}


void iImageComposer::RemoveForegroundWindow(int n)
{
	if(n<0 || n>=mForegroundWindow.Size()) return;

	mForegroundWindow[n]->Delete();
	mForegroundWindow.Remove(n);
	this->ClearCache();
}


void iImageComposer::MoveToBack(int n)
{
	if(n<0 || n>=mForegroundWindow.MaxIndex()) return;

	int i;
	iImageComposerForegroundWindow *tmp = mForegroundWindow[n];
	for(i=n; i<mForegroundWindow.MaxIndex(); i++) mForegroundWindow[i] = mForegroundWindow[i+1];
	mForegroundWindow.Last() = tmp;
	this->ClearCache();
}


void iImageComposer::SetScaleBackground(bool s)
{
	if(s != mScaleBackground)
	{
		mScaleBackground = s;
		this->UpdateSize();
		this->ClearCache();
	}
}


void iImageComposer::SetInnerBorder(bool s)
{
	if(s != mInnerBorder)
	{
		mInnerBorder = s;
		this->UpdateSize(); 
		this->ClearCache();
	}
}


void iImageComposer::SetBorderWidth(int s) 
{ 
	if(s >= 0) 
	{ 
		mBorderWidth = s; 
		this->UpdateSize(); 
		this->ClearCache();
	} 
}


void iImageComposer::SetBorderColor(const iColor &c) 
{ 
	if(c != mBorderColor) 
	{ 
		mBorderColor = c; 
		this->ClearCache();
	} 
}


void iImageComposer::Update()
{
	this->UpdateWindowList();
	this->UpdateSize();
}


void iImageComposer::UpdateSize()
{
	if(mBlockUpdate) return;
	
	int i;
	//
	//  Compute real size
	//
	mTileWidth = mTileHeight = 0;
	for(i=0; i<mNumTilesX*mNumTilesY; i++)
	{
		if(mTileWidth < mBackgroundWindow[i]->GetWidth()) mTileWidth = mBackgroundWindow[i]->GetWidth();
		if(mTileHeight < mBackgroundWindow[i]->GetHeight()) mTileHeight = mBackgroundWindow[i]->GetHeight();
	}
	if(mTileWidth==0 || mTileHeight==0) 
	{
		mTileWidth = this->GetControlModule()->GetViewModule()->GetThisImageWidth(); 
		mTileHeight = this->GetControlModule()->GetViewModule()->GetThisImageHeight();
	}

	mFullWidth = mNumTilesX*mTileWidth + 2*mBorderWidth;
	mFullHeight = mNumTilesY*mTileHeight + 2*mBorderWidth;
	if(mInnerBorder)
	{
		mFullWidth += (mNumTilesX-1)*mBorderWidth;
		mFullHeight += (mNumTilesY-1)*mBorderWidth;
	}

	//
	//  Make sure no foreground window moves out of image area
	//
	for(i=0; i<mForegroundWindow.Size(); i++)
	{
		mForegroundWindow[i]->CorrectPosition();
	}
}


void iImageComposer::UpdateWindowList()
{
	iViewModule *vm;
	int i, k;
	bool present;

	for(i=0; i<mNumTilesX*mNumTilesY; i++) if((vm=mBackgroundWindow[i]->GetViewModule()) != 0)
	{
		present = false;
		for(k=0; !present && k<this->GetControlModule()->GetNumberOfViewModules(); k++)
		{
			if(vm == this->GetControlModule()->GetViewModule(k)) present = true;
		}
		if(!present) mBackgroundWindow[i]->SetViewModule(0);
	}

	for(i=0; i<mForegroundWindow.Size(); i++) 
	{
		vm = mForegroundWindow[i]->GetViewModule();
		present = false;
		for(k=0; !present && k<this->GetControlModule()->GetNumberOfViewModules(); k++)
		{
			if(vm == this->GetControlModule()->GetViewModule(k)) present = true;
		}
		if(!present) this->RemoveForegroundWindow(i);
	}
}


iImageComposerBackgroundWindow* iImageComposer::GetBackgroundWindow(int i) const
{
	if(i>=0 && i<mNumTilesX*mNumTilesY) return mBackgroundWindow[i]; else return 0;
}


iImageComposerForegroundWindow* iImageComposer::GetForegroundWindow(int i) const
{
	if(i>=0 && i<mForegroundWindow.Size()) return mForegroundWindow[i]; else return 0;
}


void iImageComposer::PackStateBody(iString &s) const
{
	int i, iv[2];

	this->PackValue(s,KeyScaleBackground(),mScaleBackground);
	this->PackValue(s,KeyInnerBorder(),mInnerBorder);
	this->PackValue(s,KeyBorderColor(),mBorderColor);
	this->PackValue(s,KeyBorderWidth(),mBorderWidth);
	this->PackValue(s,KeyImageWidth(),mFullWidth);
	this->PackValue(s,KeyImageHeight(),mFullHeight);
	
	iv[0] = mNumTilesX; iv[1] = mNumTilesY;
	this->PackValue(s,KeyNumTiles(),iv,2);
	this->PackValue(s,KeyNumForegroundWindows(),mForegroundWindow.Size());

	int *itmp = new int[mNumTilesX*mNumTilesY]; IERROR_ASSERT(itmp);
	for(i=0; i<mNumTilesX*mNumTilesY; i++) itmp[i] = (mBackgroundWindow[i]->GetViewModule() != 0) ? mBackgroundWindow[i]->GetViewModule()->GetWindowNumber() : -1;
	this->PackValue(s,KeyBackgroundWindowViewModule(),itmp,mNumTilesX*mNumTilesY);

	iString *stmp = new iString[mNumTilesX*mNumTilesY]; IERROR_ASSERT(stmp);
	for(i=0; i<mNumTilesX*mNumTilesY; i++) stmp[i] = mBackgroundWindow[i]->GetWallpaperFile();
	this->PackValue(s,KeyBackgroundWindowWallpaperFile(),stmp,mNumTilesX*mNumTilesY);
	delete [] stmp;

	//
	//  We need the keys to be present even if there are no windows, so that QueryValue functions
	//  do not fail
	//
	int n = mForegroundWindow.Size();
	if(n == 0) n++;

	if(n > mNumTilesX*mNumTilesY)
	{
		delete [] itmp;
		itmp = new int[n]; IERROR_ASSERT(itmp);
	}

	if(mForegroundWindow.Size() == 0) itmp[0] = 0;
	for(i=0; i<mForegroundWindow.Size(); i++) itmp[i] = mForegroundWindow[i]->GetViewModule()->GetWindowNumber();
	this->PackValue(s,KeyForegroundWindowViewModule(),itmp,n);

	for(i=0; i<mForegroundWindow.Size(); i++) itmp[i] = mForegroundWindow[i]->GetPositionX();
	this->PackValue(s,KeyForegroundWindowPositionX(),itmp,n);
	for(i=0; i<mForegroundWindow.Size(); i++) itmp[i] = mForegroundWindow[i]->GetPositionY();
	this->PackValue(s,KeyForegroundWindowPositionY(),itmp,n);

	for(i=0; i<mForegroundWindow.Size(); i++) itmp[i] = mForegroundWindow[i]->GetBorderWidth();
	this->PackValue(s,KeyForegroundWindowBorderWidth(),itmp,n);

	iColor *ctmp = new iColor[n]; IERROR_ASSERT(ctmp);
	for(i=0; i<mForegroundWindow.Size(); i++) ctmp[i] = mForegroundWindow[i]->GetBorderColor();
	this->PackValue(s,KeyForegroundWindowBorderColor(),ctmp,n);
	delete [] ctmp;

	float *ftmp = new float[n]; IERROR_ASSERT(ftmp);
	if(mForegroundWindow.Size() == 0) ftmp[0] = 1.0;
	for(i=0; i<mForegroundWindow.Size(); i++) ftmp[i] = mForegroundWindow[i]->GetScale();
	this->PackValue(s,KeyForegroundWindowScale(),ftmp,n);
	delete [] ftmp;

	delete [] itmp;
}


void iImageComposer::UnPackStateBody(const iString &s)
{
	int i, k, iv[2]; bool b; iColor c;
	
	mBlockUpdate = true;

	if(this->UnPackValue(s,KeyScaleBackground(),b)) this->SetScaleBackground(b);
	if(this->UnPackValue(s,KeyInnerBorder(),b)) this->SetInnerBorder(b);
	if(this->UnPackValue(s,KeyBorderColor(),c)) this->SetBorderColor(c);
	if(this->UnPackValue(s,KeyBorderWidth(),i)) this->SetBorderWidth(i);
	
	iv[0] = mNumTilesX; iv[1] = mNumTilesY;
	if(this->UnPackValue(s,KeyNumTiles(),iv,2)) this->SetNumTiles(iv[0],iv[1]);

	if(this->UnPackValue(s,KeyNumForegroundWindows(),i))
	{
		while(i < mForegroundWindow.Size()) this->RemoveForegroundWindow(i);
		while(i > mForegroundWindow.Size()) this->AddForegroundWindow(this->GetControlModule()->GetViewModule());
	}

	int *itmp = new int[mNumTilesX*mNumTilesY]; IERROR_ASSERT(itmp);
	for(i=0; i<mNumTilesX*mNumTilesY; i++) itmp[i] = (mBackgroundWindow[i]->GetViewModule() != 0) ? mBackgroundWindow[i]->GetViewModule()->GetWindowNumber() : -1;
	if(this->UnPackValue(s,KeyBackgroundWindowViewModule(),itmp,mNumTilesX*mNumTilesY)) 
	{
		for(i=0; i<mNumTilesX*mNumTilesY; i++) if(itmp[i]>=0 && (mBackgroundWindow[i]->GetViewModule()==0 || itmp[i]!=mBackgroundWindow[i]->GetViewModule()->GetWindowNumber()))
		{
			for(k=0; k<this->GetControlModule()->GetNumberOfViewModules() && itmp[i]!=this->GetControlModule()->GetViewModule(k)->GetWindowNumber(); k++);
			if(k < this->GetControlModule()->GetNumberOfViewModules()) mBackgroundWindow[i]->SetViewModule(this->GetControlModule()->GetViewModule(k));
		}
	}

	iString *stmp = new iString[mNumTilesX*mNumTilesY]; IERROR_ASSERT(stmp);
	for(i=0; i<mNumTilesX*mNumTilesY; i++) stmp[i] = mBackgroundWindow[i]->GetWallpaperFile();
	if(this->UnPackValue(s,KeyBackgroundWindowWallpaperFile(),stmp,mNumTilesX*mNumTilesY))
	{
		for(i=0; i<mNumTilesX*mNumTilesY; i++) if(!stmp[i].IsEmpty() && stmp[i]!=mBackgroundWindow[i]->GetWallpaperFile())
		{
			mBackgroundWindow[i]->LoadWallpaperImage(stmp[i]);
		}
	}
	delete [] stmp;

	//
	//  All background windows are loaded. We can update  now so that
	//  position setting works.
	//
	mBlockUpdate = false;
	this->Update();

	if(mForegroundWindow.Size() > 0)
	{
		if(mForegroundWindow.Size() > mNumTilesX*mNumTilesY)
		{
			delete [] itmp;
			itmp = new int[mForegroundWindow.Size()]; IERROR_ASSERT(itmp);
		}
		
		for(i=0; i<mForegroundWindow.Size(); i++) itmp[i] = mForegroundWindow[i]->GetViewModule()->GetWindowNumber();
		if(this->UnPackValue(s,KeyForegroundWindowViewModule(),itmp,mForegroundWindow.Size()))
		{
			for(i=0; i<mForegroundWindow.Size(); i++) if(itmp[i] != mForegroundWindow[i]->GetViewModule()->GetWindowNumber())
			{
				for(k=0; k<this->GetControlModule()->GetNumberOfViewModules() && itmp[i]!=this->GetControlModule()->GetViewModule(k)->GetWindowNumber(); k++);
				if(k < this->GetControlModule()->GetNumberOfViewModules()) mForegroundWindow[i]->SetViewModule(this->GetControlModule()->GetViewModule(k));
			}
		}

		for(i=0; i<mForegroundWindow.Size(); i++) itmp[i] = mForegroundWindow[i]->GetPositionX();
		if(this->UnPackValue(s,KeyForegroundWindowPositionX(),itmp,mForegroundWindow.Size()))
		{
			for(i=0; i<mForegroundWindow.Size(); i++) if(itmp[i] != mForegroundWindow[i]->GetPositionX())
			{
				mForegroundWindow[i]->SetPositionX(itmp[i]);
			}
		}
		
		for(i=0; i<mForegroundWindow.Size(); i++) itmp[i] = mForegroundWindow[i]->GetPositionY();
		if(this->UnPackValue(s,KeyForegroundWindowPositionY(),itmp,mForegroundWindow.Size()))
		{
			for(i=0; i<mForegroundWindow.Size(); i++) if(itmp[i] != mForegroundWindow[i]->GetPositionY())
			{
				mForegroundWindow[i]->SetPositionY(itmp[i]);
			}
		}

		for(i=0; i<mForegroundWindow.Size(); i++) itmp[i] = mForegroundWindow[i]->GetBorderWidth();
		if(this->UnPackValue(s,KeyForegroundWindowBorderWidth(),itmp,mForegroundWindow.Size()))
		{
			for(i=0; i<mForegroundWindow.Size(); i++) if(itmp[i] != mForegroundWindow[i]->GetBorderWidth())
			{
				mForegroundWindow[i]->SetBorderWidth(itmp[i]);
			}
		}

		iColor *ctmp = new iColor[mForegroundWindow.Size()]; IERROR_ASSERT(ctmp);
		for(i=0; i<mForegroundWindow.Size(); i++) ctmp[i] = mForegroundWindow[i]->GetBorderColor();
		if(this->UnPackValue(s,KeyForegroundWindowBorderColor(),ctmp,mForegroundWindow.Size()))
		{
			for(i=0; i<mForegroundWindow.Size(); i++) if(ctmp[i] != mForegroundWindow[i]->GetBorderColor())
			{
				mForegroundWindow[i]->SetBorderColor(ctmp[i]);
			}
		}
		delete [] ctmp;

		float *ftmp = new float[mForegroundWindow.Size()]; IERROR_ASSERT(ftmp);
		for(i=0; i<mForegroundWindow.Size(); i++) ftmp[i] = mForegroundWindow[i]->GetScale();
		if(this->UnPackValue(s,KeyForegroundWindowScale(),ftmp,mForegroundWindow.Size()))
		{
			for(i=0; i<mForegroundWindow.Size(); i++) if(fabs(ftmp[i]-mForegroundWindow[i]->GetScale()) > 1.0e-4)
			{
				mForegroundWindow[i]->SetScale(ftmp[i]);
			}
		}
		delete [] ftmp;
	}

	delete [] itmp;
}


bool iImageComposer::IsActive() 
{
	int i;
	
	//
	//  Is there an active background window?
	//
	for(i=0; i<mNumTilesX*mNumTilesY; i++)
	{
		if(!mBackgroundWindow[i]->IsEmpty()) return true;
	}

	//
	//  Is there an active foreground window?
	//
	if(mForegroundWindow.Size() > 0) return true;

	//
	// At the very least, is there a border?
	//
	return mBorderWidth > 0;
}

										  
void iImageComposer::Compose(iViewModule *vm, iStereoImageArray &images) 
{ 
	char rgb[3];
	int i, j, k, l, w1, h1, w0, h0, w2, xoff, yoff, ioff, joff;

	if(vm == 0)
	{
		this->GetErrorStatus()->Set("No ViewModule is specified");
		return;
	}

	//
	//  We received a request to compose the images and put the data into images.
	//  First, make sure that Composer is up to date
	//
	this->UpdateWindowList();

	int eye;
	iStereoImage winIm, outIm;
	unsigned char *dPtr[2], *dPtr1, *wPtr, *wPtr1;
	
	int d = winIm.Depth();

	//
	//  Prepare the left eye
	//
	bool work = false;

	outIm.Scale(mFullWidth,mFullHeight);
	dPtr[0] = outIm.LeftEye().DataPointer();
	for(j=0; j<mFullHeight; j++)  // fill with white
	{
		for(i=0; i<mFullWidth; i++)
		{
			dPtr1 = dPtr[0] + d*(i+mFullWidth*j);
			for(k=0; k<d; k++) dPtr1[k] = char(255); // fastest assignment!!
		}
	}

	if(mBorderWidth > 0)
	{
		work = true;
		//
		//  Create the border
		//
		mBorderColor.GetRGB(rgb);
		//
		//  Vertical lines
		//
		for(j=0; j<mFullHeight; j++)  // sort of a waste, but it is much simpler this way
		{
			for(i=0; i<mBorderWidth; i++)
			{
				dPtr1 = dPtr[0] + d*(i+mFullWidth*j);
				for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
			for(i=mFullWidth-mBorderWidth; i<mFullWidth; i++)
			{
				dPtr1 = dPtr[0] + d*(i+mFullWidth*j);
				for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}

			if(mInnerBorder)
			{
				for(k=1; k<mNumTilesX; k++)
				{
					xoff = k*(mBorderWidth+mTileWidth);
					for(i=xoff; i<xoff+mBorderWidth; i++)
					{
						dPtr1 = dPtr[0] + d*(i+mFullWidth*j);
						for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}
				}
			}
		}
		//
		//  Horizontal lines
		//
		for(j=0; j<mBorderWidth; j++)  // sort of a waste, but it is much simpler this way
		{
			for(i=0; i<mFullWidth; i++)
			{
				dPtr1 = dPtr[0] + d*(i+mFullWidth*j);
				for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
		}

		for(j=mFullHeight-mBorderWidth; j<mFullHeight; j++)
		{
			for(i=0; i<mFullWidth; i++)
			{
				dPtr1 = dPtr[0] + d*(i+mFullWidth*j);
				for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
		}

		if(mInnerBorder)
		{
			for(k=1; k<mNumTilesY; k++)
			{
				xoff = k*(mBorderWidth+mTileHeight);
				for(j=xoff; j<xoff+mBorderWidth; j++)
				{
					for(i=0; i<mFullWidth; i++)
					{
						dPtr1 = dPtr[0] + d*(i+mFullWidth*j);
						for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}
				}
			}
		}
	}

	//
	//  Fill in background tiles with their background colors 
	//
	for(l=0; l<mNumTilesX*mNumTilesY; l++)
	{
		if(!mBackgroundWindow[l]->IsEmpty())  
		{
			ioff = mBorderWidth + mTileWidth*mBackgroundWindow[l]->GetTileX();
			joff = mBorderWidth + mTileHeight*mBackgroundWindow[l]->GetTileY(); 
			if(mInnerBorder)
			{
				ioff += mBorderWidth*mBackgroundWindow[l]->GetTileX();
				joff += mBorderWidth*mBackgroundWindow[l]->GetTileY();
			}

			if(mBackgroundWindow[l]->GetViewModule() != 0)
			{
				//
				// fill the tile with the background color
				//
				mBackgroundWindow[l]->GetViewModule()->GetBackgroundColor().GetRGB(rgb);
				for(j=0; j<mTileHeight; j++)  // sort of a waste, but it is much simpler this way
				{
					for(i=0; i<mTileWidth; i++)
					{
						dPtr1 = dPtr[0] + d*(i+ioff+mFullWidth*(j+joff));
						for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}
				}
			}
		}
	}

	//
	//  Begin creating a non-stereo image
	//
	int numEyes = 1;
	dPtr[1] = 0; // for debugging - will crash if tries to write into the right eye image

	//
	//  Background windows 
	//
	for(l=0; l<mNumTilesX*mNumTilesY; l++)
	{
		if(!mBackgroundWindow[l]->IsEmpty())  
		{
			work = true;

			ioff = mBorderWidth + mTileWidth*mBackgroundWindow[l]->GetTileX();
			joff = mBorderWidth + mTileHeight*mBackgroundWindow[l]->GetTileY(); 
			if(mInnerBorder)
			{
				ioff += mBorderWidth*mBackgroundWindow[l]->GetTileX();
				joff += mBorderWidth*mBackgroundWindow[l]->GetTileY();
			}

			if(mBackgroundWindow[l]->GetViewModule() != 0)
			{
				//
				//  Render the image
				//
				this->GetErrorStatus()->Monitor(mBackgroundWindow[l]->GetViewModule(),"ImageComposer: ");
				mBackgroundWindow[l]->GetViewModule()->RenderStereoImage(winIm);
				if(this->GetErrorStatus()->IsError()) return;
				//
				//  Check that dimensions are ok.
				//
				if(mScaleBackground && (winIm.Width()!=mTileWidth || winIm.Height()!=mTileHeight))
				{
					winIm.Scale(mTileWidth,mTileHeight);
				}
			}
			else if(!mBackgroundWindow[l]->GetWallpaperImage().IsEmpty())
			{
				winIm = mBackgroundWindow[l]->GetWallpaperImage();
			}
			else
			{
				this->GetErrorStatus()->Set("Missing contents of an non-empty window.");
				return;
			}

			//
			//  Check whether we need to switch to a stereo mode
			//
			if(winIm.IsStereo())
			{
				//
				//  Prepare the right eye
				//
				numEyes = 2;
				outIm.CopyLeftEyeToRightEye();
				dPtr[1] = outIm.RightEye().DataPointer();
			}

			w1 = winIm.Width();
			h1 = winIm.Height();
			xoff = ioff + (mTileWidth-w1)/2;
			yoff = joff + (mTileHeight-h1)/2;
			for(eye=0; eye<numEyes; eye++)
			{
				wPtr = winIm.Eye(eye).DataPointer();
				for(j=0; j<h1; j++)  // sort of a waste, but it is much simpler this way
				{
					for(i=0; i<w1; i++)
					{
						dPtr1 = dPtr[eye] + d*(i+xoff+mFullWidth*(j+yoff));
						for(k=0; k<d; k++) dPtr1[k] = wPtr[k];
						wPtr += d;
					}
				}
			}
		}
	}

	//
	//  Foreground windows 
	//
	for(l=0; l<mForegroundWindow.Size(); l++)
	{
		work = true;

		mForegroundWindow[l]->CorrectPosition();
		w0 = mForegroundWindow[l]->GetScaledWidth();
		h0 = mForegroundWindow[l]->GetScaledHeight();
		w1 = mForegroundWindow[l]->GetImageWidth();
		h1 = mForegroundWindow[l]->GetImageHeight();
		xoff = mForegroundWindow[l]->GetPositionX();
		yoff = mForegroundWindow[l]->GetPositionY();
		//
		//  Render the image
		//
		if(mForegroundWindow[l]->GetViewModule() != 0)
		{
			this->GetErrorStatus()->Monitor(mForegroundWindow[l]->GetViewModule(),"Image Composer: ");
			mForegroundWindow[l]->GetViewModule()->RenderStereoImage(winIm);
			if(this->GetErrorStatus()->IsError()) return;
			//
			//  Scale image down if needed
			//
			if(mForegroundWindow[l]->GetScale() < 1.0)
			{
				winIm.Scale(w0,h0);
			}
		}
		else
		{
			this->GetErrorStatus()->Set("Missing ViewModule for foreground window #"+iString::FromNumber(l));
			return;
		}

		//
		//  Check whether we need to switch to a stereo mode
		//
		if(winIm.IsStereo())
		{
			//
			//  Prepare the right eye
			//
			numEyes = 2;
			outIm.CopyLeftEyeToRightEye();
			dPtr[1] = outIm.RightEye().DataPointer();
		}

		//
		// fill the tile with the border color
		//
		if(mForegroundWindow[l]->GetBorderWidth() > 0)
		{
			mForegroundWindow[l]->GetBorderColor().GetRGB(rgb);
			for(eye=0; eye<numEyes; eye++)
			{
				for(j=0; j<h1; j++)  // sort of a waste, but it is much simpler this way
				{
					for(i=0; i<w1; i++)
					{
						dPtr1 = dPtr[eye] + d*(i+xoff+mFullWidth*(j+yoff));
						for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}
				}
			}
		}

		xoff += mForegroundWindow[l]->GetBorderWidth();
		yoff += mForegroundWindow[l]->GetBorderWidth();

		//
		//  Chop off too large a foreground
		//
		w2 = w1 = winIm.Width();
		h1 = winIm.Height();
		if(xoff+w1 > mFullWidth) w1 = mFullWidth - xoff;
		if(yoff+h1 > mFullHeight) h1 = mFullHeight - yoff;

		for(eye=0; eye<numEyes; eye++)
		{
			wPtr = winIm.Eye(eye).DataPointer();
			for(j=0; j<h1; j++)  // sort of a waste, but it is much simpler this way
			{
				for(i=0; i<w1; i++)
				{
					dPtr1 = dPtr[eye] + d*(i+xoff+mFullWidth*(j+yoff));
					wPtr1 = wPtr + d*(i+w2*j);  // Must be w2 to have the right stride even if the foreground is too large
					for(k=0; k<d; k++) dPtr1[k] = wPtr1[k];
				}
			}
		}
	}

	if(work)
	{
		images.Clear();
		images.Add(outIm);
	}
	else
	{
		this->GetErrorStatus()->Monitor(vm);
		vm->RenderImages(images);
	}
}

