/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  Base classes for shell observers
//

#ifndef IEVENTOBSERVER_H
#define IEVENTOBSERVER_H


#include <vtkCommand.h>
#include "icontrolmodulecomponent.h"
#include "iviewmodulecomponent.h"


class iPicker;
class iShell;

class vtkCriticalSection;
class vtkObject;
class vtkProcessObject;
class vtkTimerLog;


class iEventObserver : public vtkCommand
{

public:

	static void BlockEventObservers(bool s) { mBlock = s; }

protected:

	iEventObserver();
	virtual ~iEventObserver();

	static bool mBlock;
	vtkCriticalSection *mMutex;
};


class iProgressEventObserver : public iEventObserver, protected iViewModuleComponent
{

public:
	
	enum ProgressMode
	{
		_Default = 0,
		_Reading = 1,
		_Projecting = 2,
		_Rendering = 3,
		_Writing = 4,
		_Operating = 5,
		_Formatting = 6,
		_Shifting = 7
	};

	static iProgressEventObserver* New(iViewModule *vm);

	virtual void Started(ProgressMode mode);
	virtual void Execute(vtkObject *caller, unsigned long, void *callData);
	virtual void SetProgress(float fraction);
	virtual void Finished();

	void SetMode(ProgressMode m);

	inline bool IsAborted() const { return mAborted; }

protected:

	iProgressEventObserver(iViewModule *vm);

	virtual void ChildStarted() = 0;
	virtual void ChildFinished() = 0;
	virtual bool CheckAbort() = 0;
	virtual void SetProgressBody(float fraction) = 0;

	ProgressMode mMode;
	bool mAborted;
};


class iAbortRenderEventObserver : public iEventObserver, public iViewModuleComponent
{
	
public:
	
	static iAbortRenderEventObserver* New(iViewModule *vm);

	virtual void Execute(vtkObject *caller, unsigned long, void *callData);

	virtual void Started() = 0;
	virtual void Finished() = 0;
	virtual bool CheckAbort() = 0;
	virtual void PostFinished() = 0;

	inline bool IsCancelled() const { return mCancelled; }
	inline float GetRenderTime() const { return mRenderTime; }

	static void BlockAbortRenderEventObservers(bool s) { mBlockThis = s; }
	static bool AbortRenderEventObserversBlocked(){ return mBlockThis; }

protected:

	iAbortRenderEventObserver(iViewModule *vm);
	~iAbortRenderEventObserver();
	
	vtkTimerLog *mRenderTimer;
	float mRenderTime;

private:

	static bool mBlockThis;
	bool mCancelled, mInProgress;
};


class iSlaveAbortRenderEventObserver : public iEventObserver, protected iViewModuleComponent
{
	
public:
	
	static iSlaveAbortRenderEventObserver* New(iAbortRenderEventObserver *);

	virtual void Execute(vtkObject *caller, unsigned long, void *callData);

protected:

	iSlaveAbortRenderEventObserver(iAbortRenderEventObserver *m);
	virtual ~iSlaveAbortRenderEventObserver();
	
	iAbortRenderEventObserver *mMaster;
};


class iPickEventObserver : public iEventObserver, protected iViewModuleComponent
{
	
public:
	
	static iPickEventObserver* New(iViewModule *vm);

	virtual void Execute(vtkObject *caller, unsigned long, void *callData);

	virtual void Started() = 0;
	virtual void Finished(const iPicker *) = 0;

protected:

	iPickEventObserver(iViewModule *vm);
};


class iRecordEventObserver : public iEventObserver, protected iViewModuleComponent
{
	
public:
	
	static iRecordEventObserver* New(iViewModule *vm);

	virtual void Execute(vtkObject *caller, unsigned long, void *callData);

	virtual void UpdateInteraction() = 0;

protected:

	iRecordEventObserver(iViewModule *vm);
};


class iParallelUpdateEventObserver : public iEventObserver, protected iControlModuleComponent
{
	
public:
	
	static iParallelUpdateEventObserver* New(iControlModule *cm);

	virtual void Execute(vtkObject *caller, unsigned long, void *callData);

	virtual void UpdateInformation() = 0;

protected:

	iParallelUpdateEventObserver(iControlModule *cm);
};


class iAnimatorEventObserver : public iEventObserver, protected iViewModuleComponent
{
	
public:
	
	static iAnimatorEventObserver* New(iViewModule *vm);

	virtual void Execute(vtkObject *caller, unsigned long, void *callData);

	virtual void OnCameraPath(int step) = 0;

protected:

	iAnimatorEventObserver(iViewModule *vm);
};


class iMarkerEventObserver : public iEventObserver, protected iViewModuleComponent
{
	
public:
	
	static iMarkerEventObserver* New(iViewModule *vm);

	virtual void Execute(vtkObject *caller, unsigned long, void *callData);

protected:

	iMarkerEventObserver(iViewModule *vm);
};

#endif // IEVENTOBSERVER_H

