/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idata.h"

#include <limits.h>

//
//  Templates
//
#include "iarraytemplate.h"


namespace iData_Private
{
	bool Initialized = false;
	iSearchableArray<iString> TypeQualifiers("");
	iSearchableArray<iString> TypeSpecifiers("");
	iDataInfo AllData;

	void InitPrivateNamespace()
	{
		if(!Initialized)
		{
			Initialized = true;
			TypeSpecifiers.Add("ScalarField");
 			TypeSpecifiers.Add("VectorField");
			TypeSpecifiers.Add("TensorField");
			TypeSpecifiers.Add("ParticleSet");
		}
	}

	bool IsValidKeyword(const iString &kw, iString &qual, iString &spec)
	{
		if(kw.Contains(",") != 0) return false;

		iString q, s;
		switch(kw.Contains(":"))
		{
		case 0:
			{
				q = qual;
				s = kw;  // remember previous qualifier
				break;
			}
		case 1:
			{
				q = kw.Section(":",0,0) + ":";
				s = kw.Section(":",1,1);
				break;
			}
		default:
			{
				return false;
			}
		}
	
		if((!q.IsEmpty() && TypeQualifiers.Find(q)==-1) || TypeSpecifiers.Find(s)==-1) return false;

		qual = q;
		spec = s;
		return true;
	}
};


using namespace iData_Private;


//
//  DataKind class
//
iDataClass::iDataClass(const iString& tname)
{
	InitPrivateNamespace();

	mTextName = tname;
	mName = tname.Substitute(" ","");

	//
	//  Check that the kind is unique
	//
	if(TypeSpecifiers.Find(mName) == -1)
	{
		IERROR_FINISH("Incorrectly configured DataKind object.");
	}
}


iDataClass::~iDataClass()
{
}


//
//  DataType class
//
iDataType::iDataType(int id, const iString& tname, const iString& sname, int priority, const iString &keywords, bool primary, unsigned int rank) : mId(id), mPriority(priority), mIsPrimary(primary), mRank(rank), mTextName(tname), mShortName(sname), mKeywords(keywords)
{
	int i, n;

	InitPrivateNamespace();

	mName = tname.Substitute(" ","");

	for(i=0; i<AllData.Count(); i++)
	{
		if(mId==AllData.Type(i).GetId() || mName==AllData.Type(i).GetName() || mShortName==AllData.Type(i).GetShortName() || (primary && mKeywords==AllData.Type(i).mKeywords))
		{
			IERROR_FINISH("Incorrectly configured DataType object.");
		}
	}

	//
	//  Analyse keywords
	//
	if(!mKeywords.IsEmpty())
	{
		iString q = mKeywords.Section(":",0,0);
		iString s = mKeywords.Section(":",1,1);
		if(q.IsEmpty() || s.IsEmpty())
		{
			IERROR_FINISH("Incorrectly configured DataType object.");
		}

		q += ":";
		if(TypeQualifiers.Find(q) == -1) // this qualifier has not been registered yet
		{
			TypeQualifiers.Add(q);
		}

		n = s.Contains(",");
		for(i=0; i<=n; i++) if(TypeSpecifiers.Find(s.Section(",",i,i)) == -1)
		{
			IERROR_FINISH("Incorrectly configured DataType object.");
		}
	}

	AllData += *this;
}


iDataType::~iDataType()
{
	AllData -= *this;
}


bool iDataType::IsOfClass(const iDataClass &c) const
{
	//
	//  Analyse keywords
	//
	if(!mKeywords.IsEmpty())
	{
		iString s = mKeywords.Section(":",1,1);
		int i, n = s.Contains(",");
		for(i=0; i<=n; i++) if(s.Section(",",i,i) == c.GetName())
		{
			return true;
		}
	}
	return false;
}


const iDataType& iDataType::FindTypeById(int id)
{
	int i;
	for(i=0; i<AllData.Count(); i++) if(AllData.Type(i).GetId() == id) return AllData.Type(i);
	return iData::None();
}


const iDataType& iDataType::FindTypeByName(const iString &name)
{
	int i;
	for(i=0; i<AllData.Count(); i++) if(AllData.Type(i).IsOfType(name)) return AllData.Type(i);
	return iData::None();
}


const iDataType& iDataType::ReSpecifyType(const iDataType &type, const iString &specifier)
{
	iString q, s;
	
	if(!IsValidKeyword(type.mKeywords.Section(",",0,0),q,s))
	{
		IERROR_REPORT("Inorrectly configured data types.");
		return iData::None();
	}

	return FindTypeByKeyword(q+specifier);
}


const iDataType& iDataType::FindTypeByKeyword(const iString &keyword)
{
	int i;
	iString q, s;
	
	if(!IsValidKeyword(keyword,q,s)) return iData::None();

	for(i=0; i<AllData.Count(); i++) if(AllData.Type(i).mKeywords.Find(q)!=-1 && AllData.Type(i).mKeywords.Find(s)!=-1) return AllData.Type(i);
	
	IERROR_REPORT("Inorrectly configured data types.");
	return iData::None();
}


void iDataType::FindAllTypesByKeywords(iDataInfo &set, const iString &keywords)
{
	int i, j, n = keywords.Contains(",");
	iString k, s, q;
	for(i=0; i<=n; i++)
	{
		k = keywords.Section(",",i,i);
		if(IsValidKeyword(k,q,s))
		{
			for(j=0; j<AllData.Count(); j++) if(AllData.Type(j).mKeywords.Find(s)!=-1 && (q.IsEmpty() || AllData.Type(j).mKeywords.Find(s)!=-1)) set += AllData.Type(j);
		}
	}
}


//
//  DataInfo class
//
iDataInfo::iDataInfo() : mArr(&iData::None())
{
}


iDataInfo::iDataInfo(const iDataInfo &set) : mArr(&iData::None())
{
	int i;
	for(i=0; i<set.Count(); i++) mArr.AddUnique(&(set.Type(i)));
}


iDataInfo::iDataInfo(const iDataType &type) : mArr(&iData::None())
{
	if(!type.IsNull()) mArr.Add(&type);
}


iDataInfo::~iDataInfo()
{
}


void iDataInfo::Erase()
{
	mArr.Clear();
}


iDataInfo& iDataInfo::operator=(const iDataInfo &info)
{
	int i;

	mArr.Clear();
	for(i=0; i<info.Count(); i++) mArr.Add(&(info.Type(i)));

	return *this;
}


iDataInfo& iDataInfo::operator+=(const iDataType &type)
{
	if(!type.IsNull()) mArr.AddUnique(&type);
	return *this;
}


iDataInfo& iDataInfo::operator+=(const iDataInfo &info)
{
	int i;
	const iSearchableArray<const iDataType*> &arr(info.mArr); // cache for speed

	for(i=0; i<arr.Size(); i++) if(!arr[i]->IsNull()) mArr.AddUnique(arr[i]);
	return *this;
}


iDataInfo& iDataInfo::operator-=(const iDataType &type)
{
	if(!type.IsNull()) mArr.Remove(&type);
	return *this;
}


bool iDataInfo::Includes(const iDataType &type) const
{
	return mArr.Find(&type) >= 0;
}


int iDataInfo::Index(const iDataType &type) const
{
	int i;
	for(i=0; i<mArr.Size(); i++) if(*mArr[i] == type) return i;
	return -1;
}


const iDataType& iDataInfo::Type(int i) const
{
	if(i>=0 && i<mArr.Size()) return *(mArr[i]); else return iData::None();
}


//
//  Defined data kinds
//
const iDataClass& iData::ScalarField()
{
	static const iDataClass tmp("Scalar Field");
	return tmp;
}


const iDataClass& iData::VectorField()
{
	static const iDataClass tmp("Vector Field");
	return tmp;
}


const iDataClass& iData::TensorField()
{
	static const iDataClass tmp("Tensor Field");
	return tmp;
}


const iDataClass& iData::ParticleSet()
{
	static const iDataClass tmp("Particle Set");
	return tmp;
}


//
//  Defined data types
//
const iDataType& iData::None()
{
	static const iDataType tmp(0,"Null","",iMath::_IntMin,"");
	return tmp;
}


const iDataType& iData::UniformScalars()
{
	static const iDataType tmp(1,"Uniform Scalars","us",0,"basic:ScalarField");
	return tmp;
}


const iDataType& iData::UniformVectors()
{
	static const iDataType tmp(2,"Uniform Vectors","uv",100,"basic:VectorField",true,1U);
	return tmp;
}


const iDataType& iData::UniformTensors()
{
	static const iDataType tmp(3,"Uniform Tensors","ut",200,"basic:TensorField",true,2U);
	return tmp;
}


const iDataType& iData::BasicParticles()
{
	static const iDataType tmp(4,"Basic Particles","bp",300,"basic:ParticleSet");
	return tmp;
}


//
//  Defined data infos
//
const iDataInfo& iData::Any()
{
	return AllData;
}

