/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icrosssectionviewsubject.h"


#include "ibasicdatadistributor.h"
#include "icolorbars.h"
#include "icrosssectionpipeline.h"
#include "idataexplorer.h"
#include "idatalimits.h"
#include "ierror.h"
#include "ilookuptable.h"
#include "iorthoslicer.h"
#include "ireplicatedactor.h"
#include "iviewmodule.h"
#include "iviewsubjectparallelpipeline.h"

#include <vtkImageData.h>
#include <vtkPolyData.h>
#include <vtkProperty.h>
#include <vtkTexture.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWSUBJECT_DEFINE_TYPE(iCrossSectionViewSubject,CrossSection,x);

IOBJECT_DEFINE_KEY(iCrossSectionViewSubject,Dir,d,Int,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewSubject,InterpolateData,id,Bool,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewSubject,SpecialLocation,loc,Int,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewSubject,Method,m,Int,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewSubject,OverTheEdgeFlag,oe,Bool,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewSubject,SampleRate,-sr,Int,1);
IOBJECT_DEFINE_KEY(iCrossSectionViewSubject,Var,v,OffsetInt,1);

IOBJECT_DEFINE_DISTANCE_KEY(iCrossSectionViewSubject,Pos,x);

//
//  Inherited keys
//
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COMMON(iCrossSectionViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_SHADING(iCrossSectionViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_PALETTE(iCrossSectionViewSubject,1);


//
// iCrossSectionViewSubject class
//
iCrossSectionViewSubject::iCrossSectionViewSubject(iViewModule *vm, const iDataType &type, const iString &name) : iSolidViewSubject(vm,type,name,1,_ViewSubjectNoColor|_ViewSubjectNoOpacity|_ViewSubjectNoReplicating), mPos(vm,false), mPosMax(vm,false), mVar(0)
{
	mObjectType = _ObjectTypeCrossSection;

	mVar = 0;
	mDir = 2;
	mPos = -1.0;
	mPosMax = 1.0;
	
	mMethod = 1;
	mInterpolateData = true;

	mOverTheEdgeFlag = false;

	mSampleRate = 1;
	mForcePolygonalMethod = false;

	//
	//  Do VTK stuff
	//	
	mTexture = vtkTexture::New(); IERROR_ASSERT(mTexture);
	mTextureData = vtkImageData::New(); IERROR_ASSERT(mTextureData);

	mActors[0]->SetScalarVisibility(true);
	mActors[0]->ColorByArrayComponent(0,0);
	mActors[0]->GetProperty()->SetColor(1.0,1.0,1.0);
	mActors[0]->GetProperty()->SetOpacity(1.0);
	this->SetShading(false);

	mTexture->SetLookupTable(mActors[0]->GetLookupTable());
	mTexture->SetQualityTo32Bit();

	mTextureData->SetScalarTypeToFloat();
	mTexture->SetInput(mTextureData);
}


iCrossSectionViewSubject::~iCrossSectionViewSubject()
{
	this->ShowColorBars(false);

	mTexture->Delete();
	mTextureData->Delete();
}


void iCrossSectionViewSubject::ConfigureBody()
{
	//
	//  Create pipeline (must be created after the object is fully created)
	//	
	this->AddMainPipeline(1);
	mActors[0]->SetInput(this->Pipeline()->GetOutput(0));
}


void iCrossSectionViewSubject::FinishInitialization()
{
	this->SetMethod(mMethod);
	this->SetInterpolateData(mInterpolateData);
	this->SetDir(mDir);
}


void iCrossSectionViewSubject::ResetInput()
{
	this->Pipeline()->SetNthInput(0,this->GetData());
	if(mMethod == 1) this->UpdateTextureSize();
}


void iCrossSectionViewSubject::ResetPipeline()
{
}


void iCrossSectionViewSubject::SetMethod(int m)
{
	switch(m) 
	{
	case 0:
		{
			mActors[0]->SetTexture(0); 
			break;
		}
	case 1: 
		{
			mActors[0]->SetTexture(mTexture); 
			this->UpdateTextureSize();
			break;
		}
	default: return;
	}
	mMethod = m;
	this->Pipeline()->UpdateContents(iCrossSectionPipeline::_Method);
	this->ClearCache();
}


void iCrossSectionViewSubject::ForcePolygonalMethod(bool s)
{
	mForcePolygonalMethod = s;
	this->Pipeline()->UpdateContents(iCrossSectionPipeline::_Method);
}


void iCrossSectionViewSubject::ShowColorBarsBody(bool show)
{
	this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPriorityCrossSection,this->GetVar(),this->GetDataType(),mPalettes[0],show);
}


void iCrossSectionViewSubject::SetDir(int d)
{ 
	if(d>=0 && d<3)
	{
		mDir = d;
		this->Pipeline()->UpdateContents(iCrossSectionPipeline::_Dir);
		if(mMethod == 1) this->UpdateTextureSize();
		this->ClearCache();
	}
}


void iCrossSectionViewSubject::SetVar(int v)
{ 
	if(v>=0 && v<this->GetLimits()->GetNumVars() && v!=this->GetVar())
	{
		this->ShowColorBars(false);
		mVar = v;
		this->ShowColorBars(true);
		this->Pipeline()->UpdateContents(iCrossSectionPipeline::_Var);
		this->SyncWithData(this->Request(mVar));
		this->ClearCache();
	}
}


bool iCrossSectionViewSubject::SetPos(double p, bool checkPosMax)
{
	iDistance tmp(this->GetViewModule(),false);
	tmp = p;
	return this->SetPos(tmp,checkPosMax);
}


bool iCrossSectionViewSubject::SetPos(const iDistance &p, bool checkPosMax)
{ 
	bool ret = false;
	
	double spa[3], org[3];
	org[0] = org[1] = org[2] = 0.0;
	if(mDir == 0) mPosMax = 1.0; else 
	{
		int dims[3];
		vtkImageData *input = iRequiredCast<vtkImageData>(this->GetData());
		input->GetDimensions(dims);
		if(dims[0]>0 && dims[1]>0 && dims[2]>0)
		{
			input->GetSpacing(spa);
			input->GetOrigin(org);
			mPosMax = org[mDir-1] + spa[mDir-1]*(dims[mDir-1]-1);
		}
		else mPosMax = 1.0;
	}
	
	mPos = p;
	if(mPos>mPosMax && checkPosMax)
	{
		mPos = mPosMax;
		ret = true;
	}
	if(mPos < -1.0) mPos = -1.0;

	this->Pipeline()->UpdateContents(iCrossSectionPipeline::_Pos);
	this->ClearCache();
	
	return ret;
}


void iCrossSectionViewSubject::SetInterpolateData(bool s)
{
	mInterpolateData = s;
	if(mInterpolateData) 
	{
		mTexture->InterpolateOn(); 
		mActors[0]->GetProperty()->SetInterpolationToGouraud();
	}
	else 
	{
		mTexture->InterpolateOff(); 
		mActors[0]->GetProperty()->SetInterpolationToFlat();
	}
	this->Pipeline()->UpdateContents(iCrossSectionPipeline::_InterpolateData);
	this->ClearCache();
}


void iCrossSectionViewSubject::SetSampleRate(int p)
{
	mSampleRate = p;
	this->Pipeline()->UpdateContents(iCrossSectionPipeline::_SampleRate);
	this->ClearCache();
}


void iCrossSectionViewSubject::ShowBody(bool show)
{
	if(show)
	{
		this->UpdateTextureSize();
		mActors[0]->VisibilityOn();
		this->ShowColorBars(true);
	} 
	else 
	{
		this->ShowColorBars(false);
		mActors[0]->VisibilityOff();
	}
}


float iCrossSectionViewSubject::GetExtraMemorySize() const
{
	return mTextureData->GetActualMemorySize();
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iCrossSectionViewSubject::SolidViewSubjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyInterpolateData(),mInterpolateData);

	this->PackValue(s,KeyVar(),this->GetVar());
	this->PackValue(s,KeyDir(),mDir);
	this->PackValue(s,KeyMethod(),mMethod);

	this->PackValue(s,KeySampleRate(),this->GetSampleRate());
	this->PackValue(s,KeyOverTheEdgeFlag(),mOverTheEdgeFlag);

	this->PackValueDistance(s,KeyPos(),mPos);
}


void iCrossSectionViewSubject::SolidViewSubjectUnPackStateBody(const iString &s)
{
	int i; bool b;

	if(this->UnPackValue(s,KeyInterpolateData(),b)) this->SetInterpolateData(b);

	if(this->UnPackValue(s,KeyVar(),i)) this->SetVar(i);
	if(this->UnPackValue(s,KeyDir(),i)) this->SetDir(i);
	if(this->UnPackValue(s,KeyMethod(),i)) this->SetMethod(i);
	
	if(this->UnPackValue(s,KeySampleRate(),i)) this->SetSampleRate(i);

	if(this->UnPackValueDistance(s,KeyPos(),mPos)) this->SetPos(mPos,false);

	//
	//  Special "action" keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states

	if(this->UnPackValue(s,KeySpecialLocation(),i)) this->PlaceAtSpecialLocation(i); 

	iObject::ReportMissingKeys(true);
}


void iCrossSectionViewSubject::PlaceAtSpecialLocation(int n)
{
	switch(n)
	{
	case 0:
	case 1:
		{
			if(this->IsDataPresent())
			{
				iDataExplorer *de = iDataExplorer::New(this->GetViewModule());
				if(de != 0)
				{
					de->SetActiveDataType(this->GetDataType());
					de->SetInputComponent(this->GetVar());
					const iDataExplorer::Info &di = de->GetInfo(true);
					if(n == 0)
					{
						if(di.CellMin >= 0) this->SetPos(di.PosMin->OpenGLPosition(mDir));
					}
					else
					{
						if(di.CellMax >= 0) this->SetPos(di.PosMax->OpenGLPosition(mDir));
					}
					de->Delete();
				}
			}
			break;
		}
	case 2:
	case 3:
	case 4:
		{
			this->SetPos(0.5*(n-3));
			break;
		}
	}
}


bool iCrossSectionViewSubject::CanBeShown() const
{
	return (this->GetVar() < this->GetLimits()->GetNumVars());
}


void iCrossSectionViewSubject::ViewSubjectSyncWithData(const iDataSyncRequest &r)
{
	if(mMethod == 1) this->UpdateTextureSize();

	if(r.Var()==-1 || r.Var()==this->GetVar())
	{
		// this replaces SetPal
		if(this->GetVar() < this->GetLimits()->GetNumVars())
		{
			mActors[0]->SyncWithLimits(this->GetLimits(),this->GetVar());
		}
	}
}


void iCrossSectionViewSubject::UpdateTextureSize()
{
	int dims[3], dimsOut[3], oldDimsOut[3];
	int Uidx, Vidx;
	iOrthoSlicer::GetUV(mDir,Uidx,Vidx);

	iRequiredCast<vtkImageData>(this->GetData())->GetDimensions(dims);
	//
	//  Texture dimensions
	//
	dimsOut[0] = dims[Uidx] - 1;
	dimsOut[1] = dims[Vidx] - 1;
	dimsOut[2] = 1;

	mTextureData->GetDimensions(oldDimsOut);
	if(oldDimsOut[0]!=dimsOut[0] || oldDimsOut[1]!=dimsOut[1] || oldDimsOut[2]!=dimsOut[2])
	{
		mTextureData->Initialize();
		mTextureData->SetDimensions(dimsOut);
		mTextureData->AllocateScalars();
	}
}


void iCrossSectionViewSubject::RemoveInternalDataForExtraComponents()
{
	mTextureData->Initialize();
}


iViewSubjectPipeline* iCrossSectionViewSubject::CreatePipeline(int)
{
	return new iCrossSectionPipeline(this);
}


void iCrossSectionViewSubject::ConfigureMainPipeline(iViewSubjectPipeline *p, int)
{
	iViewSubjectParallelPipeline *pp = iPointerCast<iViewSubjectParallelPipeline,iViewSubjectPipeline>(p);

	iImageDataDistributor *idd = new iImageDataDistributor(pp->GetDataManager());
	pp->GetDataManager()->AddDistributor(idd);
	pp->GetDataManager()->AddCollector(new iPolyDataCollector(pp->GetDataManager(),idd));
}

