/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icontourfilter.h"


#include "ivtk.h"

#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkToolkits.h>

#if defined(VTK_USE_PATENTED) || !defined(IVTK_4)
#include <vtkMarchingContourFilter.h>
#define IVTK_CONTOURFILTER		vtkMarchingContourFilter
#else
#include <vtkContourFilter.h>
#define IVTK_CONTOURFILTER		vtkContourFilter
#endif


//
//  Templates
//
#include "igenericfiltertemplate.h"


iContourFilter::iContourFilter(iViewSubject *vo) : iGenericFilter<vtkDataSetToPolyDataFilter,vtkDataSet,vtkPolyData>(vo,1,true,false)
{
	mCurVar = 0;

	IVTK_CONTOURFILTER *w = IVTK_CONTOURFILTER::New(); IERROR_ASSERT(w);
	w->UseScalarTreeOn();
	w->ComputeNormalsOn();
	w->ComputeScalarsOff();
	w->ComputeGradientsOff();

	if(vo->IsCreatingMainPipeline())
	{
		w->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetAbortRenderEventObserver());
	}
	else
	{
		w->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetSlaveAbortRenderEventObserver());
	}

	mWorker = w;
}


iContourFilter::~iContourFilter()
{
	mWorker->Delete();
}


void iContourFilter::SetCurrentVar(int n)
{ 
	if(n>=0 && n!=mCurVar)
	{
		mCurVar = n;
		this->Modified();
	}
}


float iContourFilter::GetLevel() const
{
	return iRequiredCast<IVTK_CONTOURFILTER>(mWorker)->GetValue(0);
}


void iContourFilter::SetLevel(float v)
{
	iRequiredCast<IVTK_CONTOURFILTER>(mWorker)->SetValue(0,v);
	this->Modified();
}


void iContourFilter::ProduceOutput()
{
	vtkDataSet *input = this->GetInput();
	vtkPolyData *output = this->GetOutput();

	//
	//  VTK contouring filters always make the isosurface of the first component. We simple shift the data by mCurVar values
	//  so that the current value is always the 0 component.
	//
	int numComp;
	if(input->GetPointData()==0 || input->GetPointData()->GetScalars()==0 || (numComp=input->GetPointData()->GetScalars()->GetNumberOfComponents())<1 || mCurVar<0 || mCurVar>=numComp)
	{
		output->Initialize();
		return; 
    }

	vtkDataSet *newInput = input->NewInstance(); IERROR_ASSERT(newInput);
	newInput->ShallowCopy(input);

	vtkFloatArray *scalars = iRequiredCast<vtkFloatArray>(input->GetPointData()->GetScalars());
	vtkFloatArray *newScalars = vtkFloatArray::New(); IERROR_ASSERT(newScalars);
	newScalars->SetNumberOfComponents(numComp);
	newScalars->SetArray(scalars->GetPointer(mCurVar),scalars->GetSize(),1);
	newInput->GetPointData()->SetScalars(newScalars);
	newScalars->Delete();

	iRequiredCast<IVTK_CONTOURFILTER>(mWorker)->SetInput(newInput);
	newInput->Delete();

	iRequiredCast<IVTK_CONTOURFILTER>(mWorker)->Update();

	output->ShallowCopy(iRequiredCast<IVTK_CONTOURFILTER>(mWorker)->GetOutput());
}

