/*
 * Program to control ICOM radios
 * 
 * Subroutine library
 */
#include "icom.h"

/*
 * Local function prototypes
 */
static void doublefreq(double, u_char *, int); /* double to frequency */
static double freqdouble(u_char *, int); /* frequency to double */

/*
 * Initialize
 */
void
init()
{
	initpkt();		/* initialize (device dependent) */
}

/*
 * select_radio(rad, ident) - select/initalize radio structure
 */
struct icom *
select_radio(			/* returns structure pointer or NULL */
	int ident		/* radio identifier */
	)
{
	struct icom *radio;	
	struct cmd1msg cmdband = {V_RBAND, FI};
	struct readbandmsg rspband;
	struct setbankmsg cmdbank = {V_SMEM, 0xa0, 0, FI}; 
	double dtemp1;
	int i, temp1, temp2;
	char s1[NAMMAX];
	u_char rsp[BMAX];

	/*
	 * Find prototype. Bad tables if not found.
	 */
	for (i = 0; name[i].name[0] != '\0'; i++)
		if (name[i].ident == ident)
			break;
	if (name[i].ident == 0)
		return (NULL);

	/*
	 * If already initialized, return structure pointer. Otherwise,
	 * allocate and initialized structure and read band edges.
	 */
	if (name[i].radio != NULL)
		return (name[i].radio);
	radio = (struct icom *)malloc(sizeof(struct icom));
	memset((char *)radio, 0, sizeof(struct icom));
	strcpy(radio->name, name[i].name);
	radio->ident = name[i].ident;
	radio->maxch = name[i].maxch;
	radio->probe = name[i].probe;
	radio->flags = name[i].flags;
 	radio->mchan = 1;
	radio->mode = 0xff;
	radio->modetab = name[i].modetab;
	retry = 1;
	temp1 = setcmda(radio, (u_char *)&cmdband, (u_char *)&rspband);
	if (temp1 < sizeof(cmdband) || rspband.cmd != cmdband.cmd) {
		retry = RETRY;
		free(radio);
		return (NULL);
	}
	retry = RETRY;
	radio->lband = freqdouble(rspband.lband, 5) / 1e6;
	radio->uband = freqdouble(rspband.uband, 5) / 1e6;
	if (radio->lband > radio->uband) {
		dtemp1 = radio->lband;
		radio->lband = radio->uband;
		radio->uband = dtemp1;
	}
	radio->lstep = radio->lband;
	radio->ustep = radio->uband;

	/*
	 * If the V_SMEM, 0xa0 command works, the radio supports
	 * multiple memory banks. We have to set the bank to 0 so the
	 * subsequent read_chan() works.
	 */
	if (setcmda(radio, (u_char *)&cmdbank, rsp))
		radio->flags |= F_BANK;

	/*
	 * If the V_VFOM (memory -> vfo) command works, we have to use
	 * it every time the channel is changed. We have to do this
	 * first so the subsequent readchan() works.
	 */
	if (setcmd(radio, V_VFOM, FI))
		radio->flags |= F_VFO;

	/*
	 * If the V_ROFFS command works, the radio supports transmit
	 * duplex. We need to do this before the first readchan() so the
	 * F_OFFSET bit gets noticed.
	 */
	if (setcmd(radio, V_ROFFS, FI))
		radio->flags |= F_OFFSET;

	/*
	 * Determine the frequency resolution. This depends on the radio
	 * and TS button. We first read the frequency and mode of
	 * channel 1, then we write it with a frequency near the lower
	 * band edge. The value is chosen with '1' in each significant
	 * digit ending in the units digit. Then, we read the frequency
	 * back and determine the resolution corresponding to the first
	 * significant digit which contains '1'.
	 */
	loadfreq(radio, radio->lband + 11111e-6);
	readfreq(radio);
	sprintf(s1, "%8.6lf", radio->vfo - radio->lband);
	radio->minstep = 6;
	if (s1[7] == '1')
		radio->minstep = 0;
	else if (s1[6] == '1')
		radio->minstep = 3;
	else if (s1[5] == '1')
		radio->minstep = 6;
	else if (s1[4] != '1')
		printf("*** invalid tuning step\n");
	radio->rate = radio->minstep;
	radio->step = logtab[radio->rate];

	/*
	 * Determine whether the frequency needs to be reset after a
	 * mode change. We first write the channel with mode USB, then
	 * read the frequency back and remember it. Then we write the
	 * channel with mode LSB, read back the frequency and see if it
	 * changed. If so, we have to write the frequency every time the
	 * mode is changed. What a drag.
	 */
	if (*modetoa(M_USB, radio->modetab) != '\0' || *modetoa(M_LSB,
	    radio->modetab) != '\0') {
		loadmode(radio, M_USB);
		readfreq(radio);
		dtemp1 = radio->vfo;
		loadmode(radio, M_LSB);
		readfreq(radio);
		if (radio->vfo != dtemp1)
			radio->flags |= F_RELD;
	}

	/*
	 * Restore contents of channel 1.
	 */
	readchan(radio, 1);
	name[i].radio = radio;
	return (radio);
}

/*
 * setbank(radio, bank) - select bank number (R8500)
 *
 * In the virtual radio semantics, this routine selects the memory
 * bank, but does not initialize the scratch registers holding the
 * current frequency, mode and related values.
 */
setbank(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int bank		/* bank number */
	)
{
	struct setbankmsg cmd = {V_SMEM, 0xa0, 0, FI}; 
	u_char rsp[BMAX];
	char chr[5];

	sprintf(chr, "%02d", bank);
	cmd.bank = ((chr[0] & 0x0f) << 4) | (chr[1] & 0x0f);
	if (!setcmda(radio, (u_char *)&cmd, rsp)) {
		printf("*** invalid bank number\n");
		return (1);
	}
	radio->bank = bank;
	return (0);
}

/*
 * setchan(radio, chan) - select channel number
 *
 * In the virtual radio semantics, this routine selects the memory
 * channel, but does not initialize the scratch registers holding the
 * current frequency, mode and related values. Only the R7000 conforms
 * to this model. For other radios, we have to simulate this.
 */
int
setchan(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int chan		/* memory subcommand */
	)
{
	struct setchanmsg cmd = {V_SMEM, 0, 0, FI};
	u_char rsp[BMAX];
	char chr[NAMMAX];

	sprintf(chr, "%04d", chan);
	if (radio->flags & F_BANK) {
		cmd.mchan[0] = ((chr[0] & 0x0f) << 4) | (chr[1] & 0x0f);
		cmd.mchan[1] = ((chr[2] & 0x0f) << 4) | (chr[3] & 0x0f);
	} else {
		cmd.mchan[0] = ((chr[2] & 0x0f) << 4) | (chr[3] & 0x0f);
		cmd.mchan[1] = FI;
	}
	if (!setcmda(radio, (u_char *)&cmd, rsp)) {
		printf("*** invalid channel number\n");
		return (1);
	}
	radio->mchan = chan;
	return (0);
}

/*
 * readchan(radio, chan) - read channel freq, mode and related data
 *
 * In the virtual radio semantics, this command selects the memory
 * channel, then copies the contents to the scratch registers holding
 * the current frequency and mode.
 */
int
readchan(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int mchan		/* channel number */
	)
{
	struct cmd1msg cmdmode = {V_RMODE, FI};
	struct modemsg rspmode;
	struct cmd1msg cmdoff = {V_ROFFS, FI};
	struct offsetmsg rspoff;
	int temp;

	radio->mode = 0xff;
	if (setchan(radio, mchan))
		return (1);
	if (radio->flags & F_VFO)
		setcmd(radio, V_VFOM, FI);

	/*
	 * For R8500, read the channel data
	 */
	if (radio->flags & F_BANK) {
		if (read_chan(radio, radio->bank, mchan, &radio->chan))
		    {
			printf("*** invalid read channel\n");
			return (1);
		}
		radio->vfo = radio->chan.freq;
		radio->mode = radio->chan.mode;
		return (0);
	}

	/*
	 * Read mode and frequency
	 */
	temp = setcmda(radio, (u_char *)&cmdmode, (u_char *)&rspmode);
	if (temp < 3 || cmdmode.cmd != rspmode.cmd) {
		printf("*** invalid packet format %02x\n", cmdmode.cmd);
		return (1);
	}
	radio->mode = rspmode.mode[0];
	if (temp > 3)
		radio->mode |= (rspmode.mode[1] << 8) | 0x8000;
	if (radio->mode == 0xff)
		return (0);
	if (readfreq(radio))
		return (1);

	/*
	 * Read offset (VHF transceivers only)
	 */
	if (!(radio->flags & F_OFFSET))
		return (0);
	temp = setcmda(radio, (u_char *)&cmdoff, (u_char *)&rspoff);
	if (temp < sizeof(rspoff) || cmdoff.cmd != rspoff.cmd) {
		printf("*** invalid packet format %02x\n", cmdoff.cmd);
		return (1);
	}
	radio->duplex = freqdouble(rspoff.offset, 5) / 10;
	return (0);
}

/*
 * readfreq(radio, freq) - read frequency
 */
int
readfreq(			/* returns 0 (ok), 1 (error) */
	struct icom *radio	/* radio structure */
	)
{
	struct cmd1msg cmd = {V_RFREQ, FI};
	struct freqmsg rsp;
	int temp;
	double dtemp;

	temp = setcmda(radio, (u_char *)&cmd, (u_char *)&rsp);
	if (temp < sizeof(rsp) || cmd.cmd != rsp.cmd) {
		printf("*** invalid packet format  %02x\n", cmd.cmd);
		return (1);
	}
	dtemp = (freqdouble(rsp.freq, 5) - radio->bfo[radio->mode &
	    0x7]) / (1. + radio->vfo_comp);
	radio->vfo = dtemp / 1e6;
	return (0);
}

/*
 * loadfreq(radio, freq) - load frequency
 */
int
loadfreq(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	double freq		/* frequency (MHz) */
	)
{
	struct freqmsg cmd = {V_SFREQ, 0, 0, 0, 0, FI, FI};
	u_char rsp[BMAX];
	double dtemp;
	int temp;

	dtemp = freq * (1. + radio->vfo_comp) + radio->bfo[radio->mode &
	    0x7];
	if (radio->flags & F_735)
		temp = 4;
	else
		temp = 5;
	doublefreq(dtemp * 1e6, cmd.freq, temp);
	if (!setcmda(radio, (u_char *)&cmd, rsp)) {
		printf("*** invalid frequency\n");
		return (1);
	}
	radio->vfo = dtemp;
	return (0);
}

/*
 * loadmode(radio, mode) - load mode
 */
int
loadmode(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int mode		/* mode */
	)
{
	struct modemsg cmd = {V_SMODE, 0, FI, FI};
	u_char rsp[BMAX];

	cmd.mode[0] = mode & 0xff;
	if ((mode & 0x8000) != 0)
		cmd.mode[1] = (mode >> 8) & 0x7f;
	if (!setcmda(radio, (u_char *)&cmd, rsp)) {
		printf("*** invalid mode\n");
		return (1);
	}
	radio->mode = mode;

	/*
	 * Some radios are too lazy to compensate fpr the BFO shift.
	 */
	if (radio->flags & F_RELD)
		return (loadfreq(radio, radio->vfo));
	return (0);
}

/*
 * loadoffset(radio, offset) - load transmit offset
 */
int
loadoffset(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	double freq		/* tx offset (kHz) */
	)
{
	struct offsetmsg cmd = {V_SOFFS, 0, 0, 0, FI};
	u_char rsp[BMAX];

	doublefreq(freq * 10, cmd.offset, sizeof(cmd.offset));
	if (!setcmda(radio, (u_char *)&cmd, rsp)) {
		printf("*** invalid duplex offset\n");
		return (1);
	}
	radio->duplex = freq;
	return (0);
}

/*
 * sendcw(radio, string) - send CW message (775)
 */
int
sendcw(				/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	char *string		/* ASCII string */
	)
{
	u_char cmd[BMAX], *ptr;
	struct cmd1msg rsp;
	int temp, i;

	ptr = cmd; *ptr++ = V_ASCII;
	for (i = 0; string[i] != '\0'; i++)
		*ptr++ = string[i];
	*ptr++ = FI;
	temp = setcmda(radio, (u_char *)&cmd, (u_char *)&rsp);
	if (temp < sizeof(rsp) || rsp.cmd != ACK) {
		printf("*** no CW capability\n");
		return (1);
	}
	return (0);
}

/*
 * loaddial(radio, step) - load dial tuning step (775, R8500)
 */
int
loaddial(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	double freq		/* dial tuning step (kHz) */
	)
{
	struct dialmsg cmd = {V_DIAL, 0, FI};
	u_char rsp[BMAX];

	doublefreq(freq, &cmd.dial, 1);
	if (!setcmda(radio, (u_char *)&cmd, rsp)) {
		printf("*** invalid dial tuning step\n");
		return (1);
	}
	radio->tune = freq;
	return (0);
}

/*
 * loadbank(radio, bank, name) - write bank name (R8500)
 *
 * This routine writes the selected bank name, but does not affect the
 * scratch registers containing the frequency, mode and related data.
 */
int
loadbank(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int bank,		/* bank number */
	char *name		/* channel name */
	)
{
	struct bankmsg cmd = {V_SETW, S_WBNK, 0, 0, 0, 0, 0, 0, FI};
	u_char rsp[BMAX];
	char s1[BMAX];

	sprintf(s1, "%02d", bank);
	cmd.bank = ((s1[0] & 0xf) << 4) | (s1[1] & 0xf);
	strncpy(cmd.name, name, sizeof(cmd.name));
	if (!setcmda(radio, (u_char *)&cmd, rsp)) {
		printf("*** invalid packet format %02x\n", cmd.cmd);
		return (1);
	}
	return (0);
}

/*
 * readbank(radio, bank, name) - read bank name (R8500)
 *
 * This routine reads the selected bank name, but does not affect the
 * scratch registers containing the frequency, mode and related data.
 */
int
readbank(			/* returns 0 (ok), 1 (error) */
	struct icom *radio,	/* radio structure */
	int bank,		/* bank number */
	char *name		/* channel name */
	)
{
	struct setbankmsg cmd = {V_SETW, S_RBNK, 0, FI};
	struct bankmsg rsp;
	int temp;
	char s1[BMAX];

	sprintf(s1, "%02d", bank);
	cmd.bank = ((s1[0] & 0xf) << 4) | (s1[1] & 0xf);
	temp = setcmda(radio, (u_char *)&cmd, (u_char *)&rsp);
	if (temp < sizeof(rsp) || cmd.cmd != rsp.cmd) {
		printf("*** invalid packet format %02x\n", cmd.cmd);
		return (1);
	}
	strncpy(name, rsp.name, sizeof(rsp.name)); /* bank name */
	name[sizeof(rsp.name)] = '\0';
}

/*
 * Write channel data (R8500)
 *
 * This routine writes the selected channel data, but does not affect
 * the scratch registers containing the frequency, mode and related data.
 */
int
write_chan(
	struct icom *radio,	/* radio structure */
	int bank,		/* bank number */
	int mchan,		/* channel number */
	struct chan *chan	/* channel data */
	)
{
	struct chanmsg cmd = {V_SETW, S_WCHN};
	struct cmd1msg rsp;
	int temp;
	double dtemp;
	char s1[BMAX];

	sprintf(s1, "%02d", bank);
	cmd.bank = ((s1[0] & 0xf) << 4) | (s1[1] & 0xf);
	sprintf(s1, "%04d", mchan);
	cmd.mchan[0] = ((s1[0] & 0xf) << 4) | (s1[1] & 0xf);
	cmd.mchan[1] = ((s1[2] & 0xf) << 4) | (s1[3] & 0xf);
	doublefreq(chan->freq * 1e6, cmd.freq, 5);
	cmd.mode[0] = (chan->mode >> 8) & 0x7f;
	cmd.mode[1] = chan->mode & 0xff;
	cmd.step = chan->step;
	doublefreq(chan->pstep * 10., cmd.pstep, 2);
	cmd.atten = chan->atten;
	cmd.scan = chan->scan;
	strncpy(cmd.name, chan->name, sizeof(cmd.name));
	cmd.fd = FI;
	temp = setcmda(radio, (u_char *)&cmd, (u_char *)&rsp);
	if (temp < sizeof(rsp)) {
		printf("*** invalid packet format %02x\n", cmd.cmd);
		return (1);
	}
	return (0);
}

/*
 * Read channel data (R8500)
 *
 * This routine reads the selected channel data, but does not affect the
 * scratch registers containing the frequency, mode and related data.
 */
int
read_chan(
	struct icom *radio,	/* radio structure */
	int bank,		/* bank number */
	int mchan,		/* channel number */
	struct chan *chan	/* channel data */
	)
{
	struct chanrqmsg cmd = {V_SETW, S_RCHN, 0, 0, 0, FI};
	struct chanmsg rsp;
	int temp;
	char s1[BMAX];

	sprintf(s1, "%02d", bank);
	cmd.bank = ((s1[0] & 0xf) << 4) | (s1[1] & 0xf);
	sprintf(s1, "%04d", mchan);
	cmd.mchan[0] = ((s1[0] & 0xf) << 4) | (s1[1] & 0xf);
	cmd.mchan[1] = ((s1[2] & 0xf) << 4) | (s1[3] & 0xf);
	temp = setcmda(radio, (u_char *)&cmd, (u_char *)&rsp);
	if (temp < sizeof(rsp) || cmd.cmd != rsp.cmd) {
		printf("*** invalid packet format %02x\n", cmd.cmd);
		return (1);
	}
	chan->bank = bank;
	chan->mchan = mchan;
	chan->freq = freqdouble(rsp.freq, 5) / 1e6;
	chan->duplex = 0;
	chan->mode = (rsp.mode[0] << 8) | rsp.mode[1]  | 0x8000;
	chan->step = rsp.step;
	chan->pstep = freqdouble(rsp.pstep, 2) / 10.;
	chan->atten = rsp.atten;
	chan->scan = rsp.scan;
	strncpy(chan->name, rsp.name, sizeof(rsp.name));
	chan->name[sizeof(rsp.name)] = '\0';
	return (0);
}

/*
 * These low level routines send a command to the radio and receive the
 * reply. They return the length of the reply or zero if error.
 */
/*
 * setcmd(radio, cmd, subcmd) - utility control
 */
int
setcmd(				/* returns length or 0 (error) */
	struct icom *radio,	/* radio structure */
	int cmmd,		/* command */
	int subcmd		/* subcommand */
	)
{
	struct cmd2msg cmd = {0, FI, FI};
	u_char rsp[BMAX];

	cmd.cmd = cmmd;
	cmd.subcmd = subcmd;
	return (setcmda(radio, (u_char *)&cmd, rsp));
}

/*
 * setcmda(radio, cmd, rsp) - utility control
 */
int
setcmda(			/* returns length or 0 (error) */
	struct icom *radio,	/* radio structure */
	u_char *cmd,		/* command list */
	u_char *rsp		/* return list */
	)
{
	int temp;

	temp = sndpkt(radio->ident, cmd, rsp);
	if (temp < 1 || rsp[0] == NAK)
		return (0);
	return (temp);
}

/*
 * BCD conversion routines
 *
 * These routines convert between internal (floating point) format and
 * radio (BCD) format. BCD data are two digits per octet in big normal
 * or reversed order, depending on data type. These routines convert
 * between internal and reversed data types.
 *
 * 	data type	BCD	units	order
 * 	====================================================
 * 	bank		2	1	normal		
 * 	channel		2/4	1	normal (4 for R8500)
 * 	frequency	8/10	1/10 Hz	reversed (8 for 735)
 * 	transmit offset	6	100 Hz	reversed
 * 	dial tune step	4	100 Hz	reversed
 */
/*
 * doublefreq(freq, y, len) - double to ICOM frequency with padding
 */
void
doublefreq(			/* returns void */
	double freq,		/* frequency */
	u_char *x,		/* radio frequency */
	int len			/* length (octets) */
	)
{
	int i;
	char s1[11];
	char *y;

	sprintf(s1, " %10.0lf", freq);
	y = s1 + 10;
	i = 0;
	while (*y != ' ') {
		x[i] = *y-- & 0x0f;
		x[i] = x[i] | ((*y-- & 0x0f) << 4);
		i++;
	}
	for (; i < len; i++)
		x[i] = 0;
	x[i] = FI;
}

/*
 * freqdouble(x, len) - ICOM frequency to double
 */
double
freqdouble(			/* returns frequency */
	u_char *x,		/* radio frequency */
	int len			/* length (octets) */
	)
{
	int i;
	char s[11];
	char *y;
	double dtemp;

	y = s + 2 * len;
	*y-- = '\0';
	for (i = 0; i < len && x[i] != FI; i++) {
		*y-- = (char)((x[i] & 0x0f) + '0');
		*y-- = (char)(((x[i] >> 4) & 0x0f) + '0');
	}
	sscanf(s, "%lf", &dtemp);
	return (dtemp);
}

/* end program */
