/* ptal-printd -- PTAL print daemon */

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <pwd.h>	/* TODO: getpwuid() */
#include <grp.h>	/* TODO: getgrgid() */
#include <signal.h>
#include <syslog.h>
#include <ptal.h>

int waitForData(int fd,int timeoutInSeconds) {
	fd_set rset;
	struct timeval timeout,*pTimeout=&timeout;
	FD_ZERO(&rset);
	FD_SET(fd,&rset);
	if (timeoutInSeconds<0) {
		pTimeout=0;
	} else {
		timeout.tv_sec=timeoutInSeconds;
		timeout.tv_usec=0;
	}
	return (select(fd+1,&rset,0,0,pTimeout)-1);
}

#define LEN_BUFFER 4096
char uel1[]="\033%-12345X";
char uel2[]="\033E\033%-12345X";

#define _PTAL_LOG_ERROR(args...) \
	do { \
		PTAL_LOG_ERROR(args); \
		syslog(LOG_LPR|LOG_ERR,args); \
	} while(0)

#define ASSERT(condition,string,recovery) \
	if (!(condition)) { \
		_PTAL_LOG_ERROR("ptal-printd(%s): %s failed!\n", \
			ptalName?ptalName:"",string); \
		recovery; \
	}
#define ASSERTA(condition,string) ASSERT(condition,string,goto abort)

void signalHandler(int signum) {
	signal(signum,signalHandler);
	PTAL_LOG_DEBUG("Caught signal %d!\n",signum);
}

int main(int argc,char **argv) {
	char *ptalName=0,*pipeName=0,*pipePath="/dev/ptal-printd";
	char colonReplacement='_';
	mode_t pipeMode=S_IRUSR|S_IWUSR|S_IWGRP|S_IWOTH;
	uid_t pipeOwner=getuid();
	gid_t pipeGroup=getgid();
	int nofork=0;
	int retryDelay=30;
	int timeout=-1;
	int sendUel=1;

	struct stat statbuf;
	ptalDevice_t dev;
	ptalChannel_t chan;
	int i,fd,len,r,w,wsize,bitBucket;
	char buffer[LEN_BUFFER+1],*oldPipeName;
	pid_t pid;

	/* Standard I/O file descriptors may be missing when invoked from
	 * a Linux USB hotplug script.  Let /dev/null take their place. */
	while (42) {
		fd=open("/dev/null",O_RDWR);
		if (fd<0) break;
		if (fd>2) {	/* 2=standard error */
			close(fd);
			break;
		}
	}
	/* Set up syslog. */
	openlog("ptal-printd",LOG_NDELAY,LOG_LPR);

	ASSERTA(ptalInit()!=PTAL_ERROR,"ptalInit");

	while (42) {
		argc--; argv++; if (argc<=0) break;
		PTAL_LOG_DEBUG("ptal-printd: token=<%s>\n",*argv);

		if (**argv!='-') {
			if (ptalName) goto syntaxError;
			ptalName=*argv;

		} else if (!strcmp(*argv,"-pipename")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipeName=*argv;

		} else if (!strcmp(*argv,"-pipepath")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipePath=*argv;

		} else if (!strcmp(*argv,"-colon")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			colonReplacement=**argv;

		} else if (!strcmp(*argv,"-mode")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipeMode=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-owner")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipeOwner=atoi(*argv);

		} else if (!strcmp(*argv,"-group")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pipeGroup=atoi(*argv);

		} else if (!strcmp(*argv,"-like")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			if (stat(*argv,&statbuf)<0) {
				_PTAL_LOG_ERROR("ptal-printd: "
					"stat(%s) failed!\n",*argv);
				goto abort;
			}
			pipeMode=statbuf.st_mode;
			pipeOwner=statbuf.st_uid;
			pipeGroup=statbuf.st_gid;

		} else if (!strcmp(*argv,"-nofork")) {
			nofork=1;

		} else if (!strcmp(*argv,"-retrydelay")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			retryDelay=atoi(*argv);

		} else if (!strcmp(*argv,"-timeout")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			timeout=atoi(*argv);

		} else if (!strcmp(*argv,"-nouel")) {
			sendUel=0;

		} else {
syntaxError:
			PTAL_LOG_ERROR(
"Syntax: ptal-printd <provider>:<device> [options...]\n"
"Supported options:\n"
"-pipename <filename> -- sets name of pipe (default based on device)\n"
"-pipepath <path> -- sets path of pipe if name contains no slashes\n"
"-colon <c> -- sets character to replace colon when using default pipename\n"
"-mode 0<octal> -- sets pipe permissions (use leading 0 for octal)\n"
"-owner <uid> -- sets pipe owner (numeric uid only)\n"
"-group <gid> -- sets pipe group (numeric gid only)\n"
"-like </dev/lpX> -- copies mode, owner and group from another device\n"
"-nofork -- runs in foreground (default is background daemon)\n"
"-retrydelay <seconds> -- sets retry delay if device open failed\n"
"-timeout <seconds> -- sets idle timeout (default is no timeout)\n"
"-nouel -- prevents wrapping the job with UELs\n"
"\n"
"Typical invocations (substitute as appropriate for <suffix>):\n"
"ptal-printd mlc:par:<suffix> -like /dev/lp0\n"
"ptal-printd mlc:usb:<suffix> -like /dev/lp0\n"
				);
			goto abort;
		}
	}
	PTAL_LOG_DEBUG("ptal-printd: no more tokens.\n");

	if (!ptalName) goto syntaxError;
	/* Create default pipeName suffix based on ptalName. */
	if (!pipeName) {
		len=strlen(ptalName);
		pipeName=malloc(len+1);
		ASSERTA(pipeName,"malloc(pipeName #1)");
		strcpy(pipeName,ptalName);
		for (i=0;i<len;i++) {
			if (pipeName[i]==':') pipeName[i]=colonReplacement;
		}
	}
	PTAL_LOG_DEBUG("ptal-printd: pipeName=<%s>\n",pipeName);
	/* If pipeName didn't include a path, then prepend default path. */
	if (!strchr(pipeName,'/')) {
		oldPipeName=pipeName;
		pipeName=malloc(strlen(pipePath)+strlen(pipeName)+2);
		ASSERTA(pipeName,"malloc(pipeName #2)");
		strcpy(pipeName,pipePath);
		strcat(pipeName,"/");
		strcat(pipeName,oldPipeName);
		/* TODO: Deallocate the old pipeName if it was previously
		 * malloced, as opposed to referenced from argv. */
	}

	pipeMode&=0777;
	PTAL_LOG_DEBUG("ptal-printd: ptalName=<%s>, pipeName=<%s>, "
		"timeout=%d, mode=0%3.3o, owner=%d, group=%d.\n",
		ptalName,pipeName,timeout,pipeMode,pipeOwner,pipeGroup);

	dev=ptalDeviceOpen(ptalName);
	ASSERTA(dev,"ptalDeviceOpen");
	chan=ptalChannelAllocate(dev);
	ASSERTA(chan,"ptalChannelAllocate");
	ptalChannelSetRemoteService(chan,PTAL_STYPE_PRINT,0,0);

	unlink(pipeName);
	ASSERTA(mknod(pipeName,S_IFIFO|pipeMode,0)>=0,"mknod");
	ASSERTA(chmod(pipeName,pipeMode)>=0,"chmod");
	ASSERTA(chown(pipeName,pipeOwner,pipeGroup)>=0,"chown");

	if (!nofork) {
		pid=fork();
		ASSERTA(pid>=0,"fork");
		if (pid) {
			return 0;
		}
	}

	signal(SIGPIPE,signalHandler);

	syslog(LOG_LPR|LOG_NOTICE,
		"ptal-printd(%s) successfully initialized.\n",ptalName);

	while (42) {
		fd=open(pipeName,O_RDONLY);
		ASSERTA(fd>=0,"open");

		ASSERTA(waitForData(fd,-1)>=0,"waitForData");

		while (ptalChannelOpen(chan)==PTAL_ERROR) {
			_PTAL_LOG_ERROR("ptal-printd(%s): ptalChannelOpen "
				"failed!  Will delay and retry.\n",ptalName);
			sleep(retryDelay);
		}

		wsize=LEN_BUFFER;
		ptalChannelAdjustPacketSizes(chan,&wsize,0);

		if (sendUel) {
			len=strlen(uel1);
			r=ptalChannelWrite(chan,uel1,len);
			if (r!=len) {
				_PTAL_LOG_ERROR("ptal-printd(%s): "
					"writing uel1 returns %d!\n",
					ptalName,r);
				goto closeAndRetry;
			}
		}

		PTAL_LOG_DEBUG("\n\nptal-printd(%s): beginning of job.\n\n",
			ptalName);
		bitBucket=0;
		while (42) {
			if (waitForData(fd,timeout)<0) {
				PTAL_LOG_DEBUG("ptal-printd(%s): timed out, "
					"assuming end of job.\n",ptalName);
				break;
			}

			r=read(fd,buffer,wsize);
			PTAL_LOG_DEBUG("ptal-printd(%s): read() returns %d.\n",
				ptalName,r);
			if (r<=0) break;
			if (bitBucket) continue;

			w=ptalChannelWrite(chan,buffer,r);
			if (w!=r) {
				_PTAL_LOG_ERROR("ptal-printd(%s): "
					"ptalChannelWrite "
					"returns %d, expected=%d!  "
					"Bit-bucketing rest of print job.\n",
					ptalName,w,r);
				bitBucket=1;
			}
		}
		PTAL_LOG_DEBUG("\n\nptal-printd(%s): end of job.\n\n",
			ptalName);

		if (sendUel && !bitBucket) {
			len=strlen(uel2);
			r=ptalChannelWrite(chan,uel2,len);
			if (r!=len) {
				_PTAL_LOG_ERROR("ptal-printd(%s): "
					"writing uel2 returns %d!\n",
					ptalName,r);
				/* goto closeAndRetry; */
			}
		}

closeAndRetry:
		ptalChannelClose(chan);

		close(fd);
	}

abort:
	ptalDone();
	unlink(pipeName);
	return 1;
}
