/* ptal-connect -- PTAL command-line stream connection client */

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include "ptal.h"

#define LEN_BUFFER 4096

#define STDIN	0
#define STDOUT	1

int ptalConnectProviderCallback(char *name,void *cbd) {
	PTAL_LOG_ERROR("\t%s\n",name);
	return 1;
}

int ptalConnectDeviceCallback(ptalDevice_t dev,void *cbd) {
	PTAL_LOG_ERROR("\t%s:%s\n",
		ptalDeviceGetProviderName(dev),
		ptalDeviceGetName(dev));
	return 1;
}

int main(int argc,char **argv) {
	char *argv0=*argv,*serviceName=0;
	int retcode=1,dump=0,serviceType=PTAL_STYPE_NONE,socketID=0;
	char buffer[LEN_BUFFER];
	int wsize=LEN_BUFFER,rsize=LEN_BUFFER;
	int maxForwardDatalen=wsize,maxReverseDatalen=rsize;
	ptalDevice_t dev=0;
	ptalChannel_t chan=0;
	int r,w,n,fd,ateof=0;
	fd_set rset;
	struct timeval timeout,*pTimeout=0;

	timeout.tv_sec=0;
	timeout.tv_usec=0;

	if (ptalInit()==PTAL_ERROR) goto abort;

	while (42) {
		argc--; argv++; if (argc<=0) break;

		if (**argv!='-') {
			if (dev) goto syntaxError;

			dev=ptalDeviceOpen(*argv);
			if (!dev) {
				goto abort;
			}

			chan=ptalChannelAllocate(dev);
			if (!chan) {
				goto abort;
			}

		} else if (!strcmp(*argv,"-dump")) {
			dump=1;

		} else if (!strcmp(*argv,"-eoftimeout")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			timeout.tv_sec=atoi(*argv);

		} else if (!strcmp(*argv,"-fwdlen")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			maxForwardDatalen=atoi(*argv);

		} else if (!strcmp(*argv,"-revlen")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			maxReverseDatalen=atoi(*argv);

		} else if (!strcmp(*argv,"-print")) {
			serviceType=PTAL_STYPE_PRINT;

		} else if (!strcmp(*argv,"-scan")) {
			serviceType=PTAL_STYPE_SCAN;

		} else if (!strcmp(*argv,"-socket")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			serviceType=PTAL_STYPE_GENERIC;
			socketID=atoi(*argv);

		} else if (!strcmp(*argv,"-service")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			serviceType=PTAL_STYPE_GENERIC;
			serviceName=*argv;

		} else {
syntaxError:
			PTAL_LOG_ERROR("Syntax: ptal-connect "
				"<provider>:<devname> <action> [<options>]\n"
				"Where <provider> is one of:\n");
			ptalProviderEnumerate(ptalConnectProviderCallback,0);
#if 0
			PTAL_LOG_ERROR("Predefined devices:\n");
			ptalDeviceEnumerate(0,ptalConnectDeviceCallback,0);
#endif
			PTAL_LOG_ERROR("<action> is one of:\n"
				"\t-print   (default for ptal-print)\n"
				"\t-scan\n"
				"\t-socket <socketID>\n"
				"\t-service <serviceName>\n"
				"<options> is zero or more of:\n"
				"\t-dump\n"
				"\t-eoftimeout <seconds>\n"
				"\t-fwdlen <maxForwardDatalen>\n"
				"\t-revlen <maxReverseDatalen>\n"
				);
			goto abort;
		}
	}

	if (!dev || !chan) {
		goto syntaxError;
	}

	if (serviceType==PTAL_STYPE_NONE) {
		if (!strstr(argv0,"ptal-print")) goto syntaxError;
		serviceType=PTAL_STYPE_PRINT;
	}
	ptalChannelSetRemoteService(chan,serviceType,socketID,serviceName);

	ptalChannelSetPacketSizes(chan,maxForwardDatalen,maxReverseDatalen);

	if (ptalChannelOpen(chan)==PTAL_ERROR) {
		goto abort;
	}

	ptalChannelAdjustPacketSizes(chan,&wsize,&rsize);

	if (dump) {
		ptalDump(0);
	}

	while (42) {
		n=0;
		FD_ZERO(&rset);

		if (!ateof) ptalFdPrepareForSelect(STDIN,&n,&rset,0,0);
		ptalChannelPrepareForSelect(chan,&fd,&n,&rset,0,0);

		r=select(n,&rset,0,0,pTimeout);
		if (r<=0) {
			PTAL_LOG_DEBUG("ptal-connect: select returns %d.\n",r);
			break;
		}

		if (FD_ISSET(STDIN,&rset)) {
			r=read(STDIN,buffer,wsize);
			if (r<=0) {
				PTAL_LOG_DEBUG("ptal-connect: "
					"read returns %d.\n",r);
				if (!timeout.tv_sec) break;
				ateof=1;
				FD_CLR(STDIN,&rset);
				pTimeout=&timeout;
			} else {
				w=ptalChannelWrite(chan,buffer,r);
				if (w!=r) {
					PTAL_LOG_ERROR("ptal-connect: "
						"ptalChannelWrite "
						"returns %d, expected=%d!\n",
						w,r);
					break;
				}
			}
		}

		if (FD_ISSET(fd,&rset)) {
			r=ptalChannelRead(chan,buffer,rsize);
			if (r<=0) {
				PTAL_LOG_DEBUG("ptal-connect: "
					"ptalChannelRead returns %d.\n",r);
				break;
			}
			w=write(STDOUT,buffer,r);
			if (w!=r) {
				PTAL_LOG_ERROR("ptal-connect: write "
					"returns %d, expected=%d!\n",w,r);
				break;
			}
		}
	}

	retcode=0;
abort:
	ptalDone();
	return retcode;
}
