/* 
 * File: gmlplaylist.c
 *
 * Medialib search and playlist editor
 * Copyright (c) 2005-2009 Johannes Heimansberg
 *
 * requires GTK+ 2.6 or better and XMMS2
 *
 * Released under the GNU General Public License v2
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkdialog.h>
#include <gdk/gdkkeysyms.h>
#include "xmmsclient/xmmsclient.h"
#include "gmedialib.h"
#include "gtrackinfo/trackinfo.h"
#include "gtrackinfo/gtrackinfo.h"
#include "xmms2ctrl/xmms2ctrl.h"
#include "sdecode/sdecode.h"
#include "addrnd.xpm"
#include "shuffle.xpm"
#include "rank0.xpm"
#include "rank1.xpm"
#include "rank2.xpm"
#include "rank3.xpm"
#include "rank4.xpm"

extern xmmsc_connection_t *connection; /* available from gxmms2.c */
static GdkAtom drag_atom;

static void update_pl_entry(xmmsv_t     *val,  void   *gmlptr, 
                            GtkTreeIter *iter, gint32  pos)
{
	gint32        id;
	static gint32 last_pos = -2;
	GMedialib    *gml = (GMedialib *)gmlptr;
	const gchar  *url, *artist_tmp = NULL, *title_tmp = NULL, *channel;
	const gchar  *album = "[Unknown Album]";
	gchar        *filename;
	gchar        *artist = NULL, *title = NULL, *prefix = "", *url_utf8;
	gchar         album_with_tracknr[256];
	gint          tracknr = -1, rating = -1;
	static gchar  last_title[256] = "", last_artist[256] = "";

	xmmsv_dict_entry_get_string(val, "artist",  &artist_tmp);
	xmmsv_dict_entry_get_string(val, "title",   &title_tmp);
	xmmsv_dict_entry_get_string(val, "album",   &album);
	xmmsv_dict_entry_get_int(val,    "id",      &id);
	xmmsv_dict_entry_get_string(val, "url",     &url);
	xmmsv_dict_entry_get_int(val,    "tracknr", &tracknr);
	xmmsv_dict_entry_get_int(val,    "rating",  &rating);
	if (xmmsv_dict_entry_get_string(val, "channel", &channel)) {
		prefix = "[Stream] ";
		album = channel;
	}

	rating--;

	if (tracknr > 0 && gml->album_tracknr_digits > 0)
		if (gml->album_tracknr_digits >= 3)
			snprintf(album_with_tracknr, 256, "%s [%03d]", album, tracknr);
		else
			snprintf(album_with_tracknr, 256, "%s [%02d]", album, tracknr);
	else
		snprintf(album_with_tracknr, 256, "%s", album);

	if (url) {
		url_utf8 = decode_url(url);
		filename = g_path_get_basename(url_utf8);
		g_free(url_utf8);
	} else {
		filename = "[Unknown]";
	}

	if (artist_tmp == NULL && channel == NULL) {
		artist = g_malloc(sizeof(gchar) * strlen(filename) + 1);
		g_strlcpy(artist, filename, sizeof(gchar) * strlen(filename) + 1);
	} else if (artist_tmp != NULL) {
		artist = g_malloc(sizeof(gchar) * (strlen(artist_tmp) + strlen(prefix) + 1));
		g_snprintf(artist, strlen(artist_tmp) + strlen(prefix) + 1, 
		           "%s%s", prefix, artist_tmp);
	} else if (channel != NULL) {
		artist = g_malloc(sizeof(gchar) * (strlen(prefix) + 1));
		g_snprintf(artist, strlen(prefix) + 1, 
		           "%s", prefix);
	}
	if (title_tmp == NULL) {			
		title = g_malloc(sizeof(gchar) * strlen(filename) + 1);
		g_strlcpy(title, filename, sizeof(gchar) * strlen(filename) + 1);
	} else {
		title = g_malloc(sizeof(gchar) * strlen(title_tmp) + 1);
		g_strlcpy(title, title_tmp, sizeof(gchar) * strlen(title_tmp) + 1);		
	}

	if (gml->playlist.ls != NULL &&
	    ( ( ((strcmp(title, last_title) != 0 || strcmp(artist, last_artist) != 0)) && 
	        (title[0] != '\0' || artist[0] != '\0') ) || pos != last_pos) ) {
		last_pos = pos;
		g_strlcpy(last_title, title, sizeof(last_title));
		g_strlcpy(last_artist, artist, sizeof(last_artist));
		if (gtk_list_store_iter_is_valid(gml->playlist.ls, iter)) {
			gtk_list_store_set(gml->playlist.ls, iter, 
			                   1, id,
			                   2, artist, 3, title,
			                   4, album_with_tracknr, -1);
			if (rating >= 0 && rating <= 4) {
				gtk_list_store_set(gml->playlist.ls, iter, 
				                   5, gml->playlist.icon_rating[rating], -1);
			} else {
				gtk_list_store_set(gml->playlist.ls, iter, 5, NULL, -1);
			}
		}
	}
	g_free(artist);
	g_free(title);

	if (url)
		g_free(filename);
}

int gml_playlist_n_add_track_with_info(xmmsv_t *val, void *gmlptr)
{
	static const char *pref[] = { "client/generic", "server", "plugins/*",
	                              "client/*", "*", NULL };
	guint               pos;
	gint32              id;
	static GtkTreeIter  iter;
	GMedialib          *gml = (GMedialib *)gmlptr;
	xmmsv_t            *info;

	pos = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(gml->playlist.ls), NULL);

	info = xmmsv_propdict_to_dict(val, pref);

	xmmsv_dict_entry_get_int(info, "id", &id);

	if (gml->playlist.ls != NULL) {
		gtk_list_store_append(gml->playlist.ls, &iter);
		gtk_list_store_set(gml->playlist.ls, &iter, 
		                   0, pos, 1, id, -1);
	}

	update_pl_entry(info, gml, &iter, pos);
	if (gml->playlist.refresh == TRUE) {
		gml->playlist.refresh_counter--;
		if (gml->playlist.refresh_counter == 0) gml->playlist.refresh = FALSE;
	}
	xmmsv_unref(info);
	return FALSE;
}

static int n_pl_playlist_list(xmmsv_t *val, void *userdata)
{
	GMedialib   *gml = (GMedialib *)userdata;

	if (gml->playlist.refresh_counter == 0) {
		xmmsv_list_iter_t *it;
		gml->playlist.refresh = TRUE;
		xmmsv_get_list_iter(val, &it);
		for (; xmmsv_list_iter_valid(it); xmmsv_list_iter_next(it)) {
			xmmsv_t        *entry;
			xmmsc_result_t *res2;
			gint            i;

			xmmsv_list_iter_entry(it, &entry);
			if (!xmmsv_get_int(entry, &i))
				printf("Broken result\n");

			gml->playlist.refresh_counter++;
			res2 = xmmsc_medialib_get_info(connection, i);
			xmmsc_result_notifier_set(res2, gml_playlist_n_add_track_with_info, gml);
			xmmsc_result_unref(res2);
		}
	}
	gml_set_statusbar_text(gml, "Ready.");
	return FALSE;
}

void gml_pl_refresh_playlist(GMedialib *gml)
{
	xmmsc_result_t *res;

	if (gml->playlist.refresh_counter == 0) {
		gml_set_statusbar_text(gml, "Loading playlist...");

		if (gml->playlist.ls != NULL) {
			gtk_list_store_clear(gml->playlist.ls);
		}

		res = xmmsc_playlist_list_entries(connection, "_active");

		xmmsc_result_notifier_set(res, n_pl_playlist_list, gml);
		xmmsc_result_unref(res);
	}
}

static int n_pl_update_current_pos(xmmsv_t *val, void *arg)
{
	gint       pos = 0;
	GMedialib *gml = (GMedialib *)arg;

	xmmsv_dict_entry_get_int(val, "position", &pos);
	gml->playlist.current_pos = pos;
	if (gml->playlist.move_finished)
		gml_playlist_update_entry(gml, gml->playlist.current_pos);
	gtk_widget_queue_draw(gml->playlist.list);
	return FALSE;
}

static int bc_handle_current_id(xmmsv_t *val, void *userdata)
{
	xmmsc_result_t *res2;

	res2 = xmmsc_playlist_current_pos(connection, NULL);
	xmmsc_result_notifier_set(res2, n_pl_update_current_pos, userdata);
	xmmsc_result_unref(res2);
	return TRUE;
}

static int bc_handle_medialib_entry_changed(xmmsv_t *val, void *gmlptr)
{
	GMedialib *gml = GMEDIALIB(gmlptr);

	if (gml->playlist.move_finished)
		gml_playlist_update_entry(gml, gml->playlist.current_pos);
	return TRUE;
}

static int bc_handle_playlist_changed(xmmsv_t *val, void *gmlptr)
{
	GMedialib    *gml = GMEDIALIB(gmlptr);
	gint          id, signal, pl_length, pos_del = -1, pos_ins = -1, pos = -1;
	guint         counter;
	GtkTreeIter   iter;
	GtkTreeModel *model;

	/* get playlist length */
	pl_length = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(gml->playlist.ls),
		                                       NULL);

	xmmsv_dict_entry_get_int(val, "type", &signal);
	switch (signal) {
		case XMMS_PLAYLIST_CHANGED_ADD:
			xmmsv_dict_entry_get_int(val, "id", &id);

			/* add item to gtk list */
			if (gml->playlist.ls != NULL) {
				gtk_list_store_append(gml->playlist.ls, &iter);
				gtk_list_store_set(gml->playlist.ls, &iter, 
				                   0, pl_length, 1, id, -1);
				gml_playlist_update_entry(gml, pl_length);
			}
			break;
		case XMMS_PLAYLIST_CHANGED_INSERT:
			xmmsv_dict_entry_get_int(val, "id", &id);
			xmmsv_dict_entry_get_int(val, "position", &pos_ins);
			if (gml->playlist.ls != NULL) {
				gtk_list_store_insert(gml->playlist.ls, &iter, pos_ins);
				gtk_list_store_set(gml->playlist.ls, &iter, 
				                   0, pos_ins, 1, id, -1);
				gml_playlist_update_entry(gml, pos_ins);
				counter = pos_ins;
				model = GTK_TREE_MODEL(gml->playlist.ls);
				do {
					gtk_tree_model_get(model, &iter, 0, &pos, -1);
					gtk_list_store_set(gml->playlist.ls, &iter, 
					   	               0, counter, -1);
					counter++;
				} while (gtk_tree_model_iter_next(model, &iter));
			}
			break;
		case XMMS_PLAYLIST_CHANGED_SHUFFLE:
		case XMMS_PLAYLIST_CHANGED_SORT:
			gml_pl_refresh_playlist(gml);
			break;
		case XMMS_PLAYLIST_CHANGED_REMOVE:
			xmmsv_dict_entry_get_int(val, "position", &pos_del);
			if (gml->playlist.ls != NULL) {
				model = GTK_TREE_MODEL(gml->playlist.ls);

				if (gtk_tree_model_get_iter_first(model, &iter)) {
					gboolean flag = FALSE;
					counter = 0;
					do {
						gtk_tree_model_get(model, &iter, 0, &pos, -1);
						if (pos == pos_del) {
							flag = TRUE;
							gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
							counter = pos;
							if (pos == pl_length - 1) break;
						}
						if (flag) {
							gtk_list_store_set(gml->playlist.ls, &iter, 
							                   0, counter, -1);
							counter++;
						}
					} while (gtk_tree_model_iter_next(model, &iter));
				}
			}
			break;
		case XMMS_PLAYLIST_CHANGED_CLEAR:
			if (gml->playlist.ls != NULL)
				gtk_list_store_clear(gml->playlist.ls);
			break;
		case XMMS_PLAYLIST_CHANGED_MOVE:
			break;
	}
	return TRUE;
}


static void view_selected_foreach_func(GtkTreeModel *model,
                                       GtkTreePath  *path,
                                       GtkTreeIter  *iter,
                                       gpointer      userdata)
{
	gint pos;

	gtk_tree_model_get(model, iter, 0, &pos, -1);
	*(gint *)userdata = pos;
}

/* returns the playlist position of the selected row 
   (or -1 if more or less than one row is selected) */
gint gml_pl_get_selected_pos(GMedialib *gml)
{
	GtkTreeSelection *selection;
	gint              pos = -1;

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));

	/* exactly 1 row selected */
	if (gtk_tree_selection_count_selected_rows(selection) == 1) {
		gtk_tree_selection_selected_foreach(selection, view_selected_foreach_func, &pos);
	}
	return pos;
}


static void update_current_pl_pos(GMedialib *gml)
{
	xmmsc_result_t *res;

	res = xmmsc_playlist_current_pos(connection, NULL);
	xmmsc_result_notifier_set(res, n_pl_update_current_pos, gml);
	xmmsc_result_unref(res);
}

static void pl_cell_data_function(GtkTreeViewColumn *col,
                                  GtkCellRenderer   *renderer,
                                  GtkTreeModel      *model,
                                  GtkTreeIter       *iter,
                                  gpointer           userdata)
{
	GMedialib *gml = GMEDIALIB(userdata);
	guint      pos = 0;
	guint      cur_pos = (guint)gml->playlist.current_pos;

	gtk_tree_model_get(model, iter, 0, &pos, -1);

	if (pos == cur_pos)
		g_object_set(renderer, "weight", PANGO_WEIGHT_BOLD, 
		             "weight-set", TRUE, NULL);
	else
		g_object_set(renderer, "weight-set", FALSE, NULL);
}

static int n_tickle(xmmsv_t *val, void *userdata)
{
	GMedialib  *gml = (GMedialib *)userdata;
	char const *xerr;

	if (xmmsv_get_error(val, &xerr)) {
		gchar buffer[128];
		snprintf(buffer, 128, "ERROR: Couldn't jump to that song: %s", xerr);
		gml_set_statusbar_text(gml, buffer);
	} else {
		xmms2ctrl_play(connection);
		gml_set_statusbar_text(gml, "Ready.");
	}
	return FALSE;
}

static int n_jump_to_track(xmmsv_t *val, void *userdata)
{
	GMedialib  *gml = (GMedialib *)userdata;
	char const *xerr;

	if (xmmsv_get_error(val, &xerr)) {
		gchar buffer[128];
		snprintf(buffer, 128, "ERROR: Couldn't jump to that song: %s",
		         xerr);
		gml_set_statusbar_text(gml, buffer);
	} else {
		xmmsc_result_t *res2;
		res2 = xmmsc_playback_tickle(connection);
		xmmsc_result_notifier_set(res2, n_tickle, userdata);
		xmmsc_result_unref(res2);
	}
	return FALSE;
}

static void cb_pl_on_tree_view_row_activated(GtkTreeView       *view, 
                                             GtkTreePath       *path,
                                             GtkTreeViewColumn *col,
                                             gpointer           userdata)
{
	GtkTreeIter     iter;
	GtkTreeModel   *model;
	xmmsc_result_t *res;

	model = gtk_tree_view_get_model(view);

	if (gtk_tree_model_get_iter(model, &iter, path)) {
		guint pos;

		gtk_tree_model_get(model, &iter, 0, &pos, -1);

		/* jump to track */
		res = xmmsc_playlist_set_next(connection, pos);
		xmmsc_result_notifier_set(res, n_jump_to_track, userdata);
		xmmsc_result_unref(res);
	}
}

enum { UP = -1, DOWN = 1 };

struct params_2
{
	GMedialib *gml;
	guint     direction;
};

static int n_playlist_move(xmmsv_t *val, void *arguments)
{
	struct params_2  *p = (struct params_2 *)arguments;
	GtkTreeSelection *sel = 
		gtk_tree_view_get_selection(GTK_TREE_VIEW(p->gml->playlist.list));
	GtkTreeModel     *model = GTK_TREE_MODEL(p->gml->playlist.ls);
	GList            *list = gtk_tree_selection_get_selected_rows(sel, &model);
	GtkTreePath      *path = list->data;

	GtkTreeIter       selected_row, other_row;

	guint             pos, id, other_pos, other_id;
	gchar            *artist = NULL,       *track = NULL, *album = NULL;
	gchar            *other_artist = NULL, *other_track = NULL;
	gchar            *other_album = NULL;
	GdkPixbuf        *icon_rank, *icon_other_rank;
	char const       *xerr;

	gtk_tree_model_get_iter(GTK_TREE_MODEL(p->gml->playlist.ls),
	                        &selected_row, path);
	gtk_tree_model_get(model, &selected_row, 0, &pos,    1, &id, 
	                                         2, &artist, 3, &track, 
	                                         4, &album,  5, &icon_rank, -1);

	if (xmmsv_get_error(val, &xerr)) {
		gchar buf[128];
		snprintf(buf, 128, "Unable to move playlist entry: %s",
		         xerr);
		gml_set_statusbar_text(p->gml, buf);
	} else {
		if (p->direction == DOWN)
			gtk_tree_path_next(path);
		else
			gtk_tree_path_prev(path);
		gtk_tree_model_get_iter(GTK_TREE_MODEL(p->gml->playlist.ls),
		                        &other_row, path);
		if (p->direction == DOWN)
			gtk_tree_path_prev(path);
		else
			gtk_tree_path_next(path);
		gtk_tree_model_get_iter(GTK_TREE_MODEL(p->gml->playlist.ls),
		                        &selected_row, path);
		gtk_tree_model_get(model, &other_row, 
		                   0, &other_pos, 
		                   1, &other_id, 
		                   2, &other_artist, 
		                   3, &other_track, 
		                   4, &other_album, 
		                   5, &icon_other_rank,
		                   -1);
		gtk_list_store_set(p->gml->playlist.ls,
		                   &selected_row,
		                   0, pos,
	                       1, other_id,
	                       2, other_artist,
	                       3, other_track,
	                       4, other_album,
	                       5, icon_other_rank,
	                       -1);

		gtk_list_store_set(p->gml->playlist.ls,
		                   &other_row,
		                   0, other_pos,
		                   1, id,
		                   2, artist,
		                   3, track,
		                   4, album,
		                   5, icon_rank,
		                   -1);
		                   
		if (p->direction == DOWN)
			gtk_tree_path_next(path);
		else
			gtk_tree_path_prev(path);
		gtk_tree_view_set_cursor(GTK_TREE_VIEW(p->gml->playlist.list),
		                         path, NULL, FALSE);
		gtk_widget_grab_focus(GTK_WIDGET(p->gml->playlist.list));
		gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(p->gml->playlist.list),
		                             path, NULL, TRUE, 0.5, 0.5);
		g_free(artist);
		g_free(track);
		gml_set_statusbar_text(p->gml, "Ready.");
	}

	p->gml->playlist.move_finished = TRUE;

	g_list_foreach(list, (GFunc)gtk_tree_path_free, NULL);
	g_list_free(list);
	update_current_pl_pos(p->gml);
	g_free(p);
	return FALSE;
}

static void pl_move_item(GMedialib *gml, guint direction)
{
	xmmsc_result_t   *res;
	struct params_2  *p = g_malloc(sizeof(struct params_2));
	GtkTreeSelection *sel = 
		gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	GtkTreeModel     *model = GTK_TREE_MODEL(gml->playlist.ls);
	GtkTreeIter       selected_row;
	GList            *list = gtk_tree_selection_get_selected_rows(sel, &model);
	GtkTreePath      *path = list->data;
	guint             cnt = gtk_tree_selection_count_selected_rows(sel);
	guint             pos;

	if (cnt == 1 && gml->playlist.move_finished) { /* exactly one row selected */
		gml->playlist.move_finished = FALSE;
		p->gml = gml;
		p->direction = direction;

		gtk_tree_model_get_iter(GTK_TREE_MODEL(p->gml->playlist.ls),
		                        &selected_row, path);
		gtk_tree_model_get(model, &selected_row, 0, &pos, -1);

		g_list_foreach(list, (GFunc)gtk_tree_path_free, NULL);
		g_list_free(list);

		res = xmmsc_playlist_move_entry(connection, NULL, pos, pos + direction);
		xmmsc_result_notifier_set(res, n_playlist_move, p);
		xmmsc_result_unref(res);
	}
}

static int n_shuffle_or_sort_ready(xmmsv_t *val, void *userdata)
{
	GMedialib *gml = (GMedialib *)userdata;

	update_current_pl_pos(gml);
	return FALSE;
}

static void cb_pl_shuffle_button_pressed(GtkWidget *widget, gpointer userdata)
{
	GMedialib      *gml = (GMedialib *)userdata;
	xmmsc_result_t *res;

	if (gml->playlist.refresh_counter == 0) {
		res = xmmsc_playlist_shuffle(connection, NULL);
		xmmsc_result_notifier_set(res, n_shuffle_or_sort_ready, userdata);
		xmmsc_result_unref(res);
	}
}

static void pl_sort_menu_pos_func(GtkMenu  *menu,
                                  gint     *x,
                                  gint     *y,
                                  gboolean *push_in,
                                  gpointer  userdata)
{
	GMedialib     *gml = (GMedialib *)userdata;
	GtkAllocation  allocation = gml->playlist.button_sort_pl->allocation;

	gdk_window_get_origin(gml->playlist.button_sort_pl->window, x, y);
	*x += allocation.x;
	*y += allocation.y + allocation.height;
}

static gboolean cb_pl_sort_button_raw_pressed(GtkWidget      *widget,
                                              GdkEventButton *event,
                                              gpointer        userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gboolean   res = FALSE;

	if (event->button == 1) {
		gtk_menu_popup(
			GTK_MENU(gml->playlist.menu_sort_pl),
			NULL,
			NULL,
			(GtkMenuPositionFunc) pl_sort_menu_pos_func,
			userdata,
			3,
			event->time);
		res = TRUE;
	}
	return res;
}

static void cb_pl_sort_menu_activate(GtkMenuItem *menuitem, gpointer userdata)
{
	xmmsc_result_t *res;
	gchar          *sortby;
	xmmsv_t        *sortprop;

	sortby = g_utf8_strdown(
		gtk_label_get_label(GTK_LABEL(
		gtk_bin_get_child(GTK_BIN(menuitem)))), -1);
	sortprop = xmmsv_make_stringlist(&sortby, 1);

	res = xmmsc_playlist_sort(connection, NULL, sortprop);
	xmmsv_unref(sortprop);
	g_free(sortby);
	xmmsc_result_notifier_set(res, n_shuffle_or_sort_ready, userdata);
	xmmsc_result_unref(res);
}

struct params_entry_update {
	GMedialib   *gml;
	GtkTreeIter  iter;
	gint32       pos;
};

static int n_update_playlist_entry(xmmsv_t *val, void *arg)
{
	struct params_entry_update *p = (struct params_entry_update *)arg;
	GtkTreeIter                 iter;
	char const                 *xerr;

	iter = p->iter;

	if (xmmsv_get_error(val, &xerr))
		printf("%s\n", xerr);
	else {
		xmmsv_t *info = xmmsv_propdict_to_dict(val, NULL);
		update_pl_entry(info, p->gml, &iter, p->pos);
		xmmsv_unref(info);
	}

	g_free(p);
	return FALSE;
}

void gml_playlist_update_entry(GMedialib *gml, guint pos)
{
	struct params_entry_update *p = g_malloc(sizeof(struct params_entry_update));
	GtkTreeModel      *model = GTK_TREE_MODEL(gml->playlist.ls);
	GtkTreeIter        iter;
	guint              poscnt = 0;

	p->gml = gml;

	if (gtk_tree_model_get_iter_first(model, &iter)) {
		do {
			gint            id;
			xmmsc_result_t *res;

			gtk_tree_model_get(model, &iter, 1, &id, -1);
			if (pos == poscnt) {
				p->iter = iter;
				p->pos  = pos;
				res = xmmsc_medialib_get_info(connection, id);
				xmmsc_result_notifier_set(res, n_update_playlist_entry, p);
				xmmsc_result_unref(res);
			}
			poscnt++;
		} while (gtk_tree_model_iter_next(model, &iter));
	}
}

static int n_idlist_ready(xmmsv_t *val, void *pt)
{
	xmmsc_result_t *res2;
	int             i, j, cnt_item = 0, length, rnd[10];
	xmmsv_list_iter_t *it;

	length = xmmsv_list_get_size(val);
	/* create ten random numbers between 0 and length: */
	srand(time(NULL));
	for (i = 0; i < 10; i++)
		rnd[i] = rand() % length;
	/* sort the random numbers: */
	for (i = 0; i < 10; i++)
		for (j = 0; j < 10; j++)
			if (rnd[i] < rnd[j]) {
				int b = rnd[i];
				rnd[i] = rnd[j];
				rnd[j] = b;
			}

	i = 0;
	xmmsv_get_list_iter(val, &it);
	for (; xmmsv_list_iter_valid(it); xmmsv_list_iter_next(it)) {
		xmmsv_t *entry;
		gint     id;

		xmmsv_list_iter_entry(it, &entry);
		if (!xmmsv_get_int(entry, &id)) {
			printf("gxmms2: Broken resultset.\n");
		}
		if (cnt_item < 10 && i == rnd[cnt_item]) {
			res2 = xmmsc_playlist_add_id(connection, NULL, id);
			xmmsc_result_unref(res2);
			cnt_item++;
		}
		i++;
	}
	return FALSE;
}

static void cb_pl_add_rnd_button_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t    *res;
	GMedialib         *gml = (GMedialib *)userdata;
	xmmsc_coll_t      *universe;

	gml_set_statusbar_text(gml, "Adding random songs...");

	universe = xmmsc_coll_universe();
	res = xmmsc_coll_query_ids(connection, universe, NULL, 0, 0);

	xmmsc_result_notifier_set(res, n_idlist_ready, userdata);
	xmmsc_result_unref(res);
}

static void cb_pl_reload_button_pressed(GtkWidget *widget, gpointer pt);

static GList *selection_list = NULL;

static void pl_delete_selection(GtkTreeModel *model,
                                GtkTreePath  *path,
                                GtkTreeIter  *iter,
                                gpointer      userdata)
{
	guint pos, *value;

	gtk_tree_model_get(model, iter, 0, &pos, -1);

	value = g_malloc(sizeof(guint));
	*value = pos;
	selection_list = g_list_append(selection_list, (gpointer)value);
}

static gint comp_func(gconstpointer a, gconstpointer b)
{
	return (*(guint *)a > *(guint *)b ? -1 : 1);
}

static int n_playlist_remove(xmmsv_t *val, void *arguments)
{
	char const *xerr;

	if (xmmsv_get_error(val, &xerr)) {
		fprintf(stderr, "Couldn't remove item (%s)\n", xerr);
	}
	return FALSE;
}

static void cb_pl_button_delete_pressed(GtkWidget *widget, gpointer userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	GList            *element;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_delete_selection, NULL);

	selection_list = g_list_sort(selection_list, comp_func);
	element = selection_list;
	while (element != NULL) {
		xmmsc_result_t *res;
		guint           pos = *(guint *)(element->data);

		res = xmmsc_playlist_remove_entry(connection, NULL, pos);
		xmmsc_result_notifier_set(res, n_playlist_remove, NULL);
		xmmsc_result_unref(res);

		g_free(element->data);
		element = element->next;
	}
	g_list_free(selection_list);
	selection_list = NULL;

	update_current_pl_pos(gml);
}

static gboolean cb_pl_selection_key_press(GtkWidget   *treeview,
                                          GdkEventKey *event,
                                          gpointer     userdata)
{
	gboolean result = FALSE;

	if (event->type == GDK_KEY_PRESS && 
	    (event->keyval == GDK_Delete || event->keyval == GDK_KP_Delete)) {
		cb_pl_button_delete_pressed(treeview, userdata);
		result = TRUE;
	}
	if (event->type == GDK_KEY_PRESS && event->state == GDK_MOD1_MASK) {
	    switch (event->keyval) {
	    	case GDK_Up:
	    		pl_move_item((GMedialib *)userdata, UP);
	    		result = TRUE;
	    		break;
	    	case GDK_Down:
	    		pl_move_item((GMedialib *)userdata, DOWN);
	    		result = TRUE;
	    		break;
	    }
	}
	return result;
}

static void pl_play_selection(GtkTreeModel  *model,
                              GtkTreePath   *path,
                              GtkTreeIter   *iter,
                              gpointer       userdata)
{
	guint           pos;
	xmmsc_result_t *res;

	gtk_tree_model_get(model, iter, 0, &pos, -1);

	/* jump to track */
	res = xmmsc_playlist_set_next(connection, pos);
	xmmsc_result_notifier_set(res, n_jump_to_track, userdata);
	xmmsc_result_unref(res);
}

static int n_media_lib_get_info(xmmsv_t *val, void *arg)
{
	trackinfo  track;
	GtkWidget *dialog;

	trackinfo_update(val, &track);
	dialog = gtrackinfo_new();
	gtrackinfo_set_info(GTRACKINFO(dialog), &track);
	gtk_widget_show(dialog);
	return FALSE;
}

static void pl_info_about_selection(GtkTreeModel  *model,
                                    GtkTreePath   *path,
                                    GtkTreeIter   *iter,
                                    gpointer       userdata)
{
	guint           id;
	xmmsc_result_t *res;

	gtk_tree_model_get(model, iter, 1, &id, -1);
	res = xmmsc_medialib_get_info(connection, id);
	xmmsc_result_notifier_set(res, n_media_lib_get_info, NULL);
	xmmsc_result_unref(res);
}

struct params
{
	GMedialib   *gml;
	GtkTreeIter  iter;
};

static int n_playlist_current_pos(xmmsv_t *val, void *arguments)
{
	struct params *pm = (struct params *)arguments;
	gint           pos = 0, p = 0;
	GtkTreePath   *path;

	xmmsv_dict_entry_get_int(val, "position", &pos);
	pm->gml->playlist.current_pos = pos;

	gtk_tree_model_get(GTK_TREE_MODEL(pm->gml->playlist.ls), 
	                   &(pm->iter), 0, &p, -1);
	while (p != pos && 
	       gtk_tree_model_iter_next(GTK_TREE_MODEL(pm->gml->playlist.ls), 
	                                &(pm->iter))) {
		gtk_tree_model_get(GTK_TREE_MODEL(pm->gml->playlist.ls), 
		                   &(pm->iter), 0, &p, -1);
	}
	path = gtk_tree_model_get_path(GTK_TREE_MODEL(pm->gml->playlist.ls), 
	                               &(pm->iter));
	gtk_tree_view_set_cursor(GTK_TREE_VIEW(pm->gml->playlist.list),
	                         path, NULL, FALSE);
	gtk_widget_grab_focus(GTK_WIDGET(pm->gml->playlist.list));
	gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(pm->gml->playlist.list),
	                             path, NULL, TRUE, 0.5, 0.5);
	gtk_tree_path_free(path);
	g_free(pm);
	return FALSE;
}

static void cb_pl_button_track_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t *res;
	struct params  *p;

	p = g_malloc(sizeof(struct params));
	p->gml = (GMedialib *)userdata;
	
	if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(p->gml->playlist.ls),
	                                  &(p->iter))) {
		res = xmmsc_playlist_current_pos(connection, NULL);
		xmmsc_result_notifier_set(res, n_playlist_current_pos, p);
		xmmsc_result_unref(res);
	}
}

static void cb_pl_save_pl_button_pressed(GtkWidget *widget, gpointer userdata);

static int n_playlist_loaded(xmmsv_t *val, void *userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_refresh_playlist(gml);
	gml_set_statusbar_text(gml, "Ready.");
	return FALSE;
}

static int n_playlist_clear(xmmsv_t *val, void *userdata)
{
	xmmsc_result_t *res2;
	res2 = xmmsc_playlist_load(connection, "_gxmms2_tmp_list");
	xmmsc_result_notifier_set(res2, n_playlist_loaded, userdata);
	xmmsc_result_unref(res2);
	return FALSE;
}

static void cb_pl_clear_button_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t *res;
	GMedialib      *gml = (GMedialib *)userdata;
	gint            result = GTK_RESPONSE_NO;
	GtkWidget      *checkbutton;
	GtkWidget      *dialog;
	
	if (gml->playlist.new_playlist_confirm) {
		dialog = gtk_message_dialog_new(GTK_WINDOW(gml),
		                                GTK_DIALOG_DESTROY_WITH_PARENT,
		                                GTK_MESSAGE_QUESTION,
		                                GTK_BUTTONS_NONE,
		                                "Do you want to save the current playlist?");
		gtk_dialog_add_buttons(GTK_DIALOG(dialog), 
		                       GTK_STOCK_YES,    GTK_RESPONSE_YES,
		                       GTK_STOCK_NO,     GTK_RESPONSE_NO,
		                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
		                       NULL);

		checkbutton = gtk_check_button_new_with_mnemonic("_Don't ask me again");
		gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox),
		                  checkbutton);
		gtk_widget_show_all(dialog);

		result = gtk_dialog_run(GTK_DIALOG(dialog));

		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbutton))) {
			gml->playlist.new_playlist_confirm = FALSE;
		}

		if (result == GTK_RESPONSE_YES) {
			/* save the playlist */
			gtk_widget_destroy(dialog);
			cb_pl_save_pl_button_pressed(widget, gml);
		} else {
			gtk_widget_destroy(dialog);
		}
	}
	if (result != GTK_RESPONSE_REJECT) {
		res = xmmsc_playlist_create(connection, "_gxmms2_tmp_list");
		xmmsc_result_notifier_set(res, n_playlist_clear, userdata);
		xmmsc_result_unref(res);
	}
}

static int n_playlist_save(xmmsv_t *val, void *userdata)
{
	GMedialib  *gml = (GMedialib *)userdata;
	char const *xerr;

	if (xmmsv_get_error(val, &xerr)) {
		gml_set_statusbar_text(gml, (gchar *) xerr);
	} else {
		gml_playlists_update_playlists(gml);
		gml_set_statusbar_text(gml, "Playlist saved successfully.");
	}
	return FALSE;
}

struct params_save_pl
{
	GMedialib *gml;
	gchar     *pl_name_str;
};

static int n_coll_get(xmmsv_t *val, void *userdata)
{
	GMedialib  *gml = ((struct params_save_pl *)userdata)->gml;;
	char const *xerr;

	if (xmmsv_get_error(val, &xerr)) {
		gml_set_statusbar_text(gml, (gchar *) xerr);
	} else {
		xmmsc_coll_t   *playlist_coll;
		xmmsc_result_t *res2;

		xmmsv_get_coll(val, &playlist_coll);
		res2 = xmmsc_coll_save(connection, playlist_coll,
		                       ((struct params_save_pl *)userdata)->pl_name_str,
		                       XMMS_COLLECTION_NS_PLAYLISTS);
		xmmsc_result_notifier_set(res2, n_playlist_save, gml);
		xmmsc_coll_unref(playlist_coll);
	}
	g_free(((struct params_save_pl *)userdata)->pl_name_str);
	return FALSE;
}

static void cb_pl_save_pl_button_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t *res;
	gint            result;
	GtkWidget      *dialog = gtk_dialog_new_with_buttons("Save playlist...",
	                                                     GTK_WINDOW(userdata),
	                                                     GTK_DIALOG_MODAL | 
	                                                     GTK_DIALOG_DESTROY_WITH_PARENT,
	                                                     GTK_STOCK_OK,
	                                                     GTK_RESPONSE_ACCEPT,
	                                                     GTK_STOCK_CANCEL,
	                                                     GTK_RESPONSE_REJECT,
	                                                     NULL);
	GtkWidget      *label = gtk_label_new("Save as:");
	GtkWidget      *entry = gtk_entry_new();
	static struct   params_save_pl p;

	gtk_widget_set_size_request(GTK_WIDGET(dialog), 300, 100);
	gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), label);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), entry);
	gtk_widget_show_all(dialog);

	result = gtk_dialog_run(GTK_DIALOG(dialog));
	if (result == GTK_RESPONSE_ACCEPT) {
		gint   len = strlen(gtk_entry_get_text(GTK_ENTRY(entry)));
		gchar *pl_name_str = g_malloc(len);
		strncpy(pl_name_str, gtk_entry_get_text(GTK_ENTRY(entry)), len);
		/* save the playlist */
		res = xmmsc_coll_get(connection, "_active", "Playlists");
		p.gml = (GMedialib *)userdata;
		p.pl_name_str = pl_name_str;
		xmmsc_result_notifier_set(res, n_coll_get, &p);
		xmmsc_result_unref(res);
	}
	gtk_widget_destroy(dialog);
}

static void cb_pl_drag_data_get(GtkWidget        *widget,
                                GdkDragContext   *drag_context,
                                GtkSelectionData *selection,
                                guint             info,
                                guint             time,
                                gpointer          pt)
{
	GtkTreeSelection *tree_sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(widget));
	GtkTreeModel     *model = gtk_tree_view_get_model(GTK_TREE_VIEW(widget));
	GList            *sel = gtk_tree_selection_get_selected_rows(tree_sel, &model);
	gchar            *rows = NULL;

	for (; sel; sel = g_list_next(sel)) {
		gchar *path_string = gtk_tree_path_to_string((GtkTreePath*) sel->data);

		if (rows == NULL) {
			rows = path_string;
		} else {
			gchar *str = g_strconcat(rows, ";", path_string, NULL);
			g_free(rows);
			g_free(path_string);
			rows = str;
		}
	}
	g_list_free(sel);

	gtk_selection_data_set(selection, drag_atom, sizeof(guchar), 
	                       (guchar *)rows, strlen(rows) + 1);
}

static void cb_pl_drag_data_received(GtkWidget        *widget,
                                     GdkDragContext   *drag_context,
                                     gint              x,
                                     gint              y,
                                     GtkSelectionData *selection,
                                     guint             info,
                                     guint             time,
                                     gpointer          userdata)
{
	guchar                   *data_text = selection->data;
	gchar                   **rows = g_strsplit((gchar *)data_text, ";", -1);
	gchar                   **cur_row;
	GtkTreePath              *path;
	GtkTreeViewDropPosition   position;
	GtkTreeModel             *model;
	GtkTreeIter               iter;
	guint                     pos;
	guint                     i;
	GMedialib                *gml = (GMedialib *)userdata;

	g_return_if_fail(data_text != NULL);

	if (!gtk_tree_view_get_dest_row_at_pos(
		GTK_TREE_VIEW(widget), x, y, &path, &position))
		return;

	model = gtk_tree_view_get_model(GTK_TREE_VIEW(widget));
	gtk_tree_model_get_iter(model, &iter, path);
	gtk_tree_model_get(model, &iter, 0, &pos, -1);

	for (cur_row = rows; *cur_row; cur_row++) {
		GtkTreeIter     old_iter;
		guint           old_pos;
		xmmsc_result_t *res = NULL;

		gtk_tree_model_get_iter_from_string(model, &old_iter, *cur_row);
		gtk_tree_model_get(model, &old_iter, 0, &old_pos, -1);

		if ((gint)pos - (gint)old_pos < 0) {
			res = xmmsc_playlist_move_entry(connection, NULL, old_pos, pos);
			gtk_list_store_move_before(GTK_LIST_STORE(model),
			                           &old_iter, &iter);
		} else {
			res = xmmsc_playlist_move_entry(connection, NULL, old_pos, pos-1);
			gtk_list_store_move_before(GTK_LIST_STORE(model),
			                           &old_iter, &iter);
		}
		xmmsc_result_unref(res);
	}
	g_strfreev(rows);

	gtk_tree_model_get_iter_first(model, &iter);
	i = 0;
	do {
		gtk_list_store_set(GTK_LIST_STORE(model), &iter, 0, i, -1);
		i++;
	} while (gtk_tree_model_iter_next(model, &iter));
	update_current_pl_pos(gml);
}

static void cb_pl_reload_button_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib *gml = (GMedialib *)pt;
	gml_pl_refresh_playlist(gml);
}

static void search_for_artist_foreach_func(GtkTreeModel  *model,
                                           GtkTreePath   *path,
                                           GtkTreeIter   *selected_row,
                                           gpointer       userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gchar     *artist;

	gtk_tree_model_get(model, selected_row, 2, &artist, -1);

	if (gml_notebook_is_mlib_search_visible(GMEDIALIB(gml))) {
		gml_search_do_search(gml, SEARCH_ARTIST, artist);
		gtk_notebook_set_current_page(GTK_NOTEBOOK(gml->notebook), 0);
	}
	g_free(artist);
}

static void cb_pl_popup_search_artist_pressed(GtkWidget *menuitem,
                                              gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, search_for_artist_foreach_func,
	                                    (gpointer)gml);
}

static void search_for_album_foreach_func(GtkTreeModel  *model,
                                          GtkTreePath   *path,
                                          GtkTreeIter   *selected_row,
                                          gpointer       userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gchar     *album;

	gtk_tree_model_get(model, selected_row, 4, &album, -1);

	if (gml_notebook_is_mlib_search_visible(GMEDIALIB(gml))) {
		/* remove tracknr [xxx] from album name: */
		if (album[strlen(album) - 1] == ']' &&
		    album[strlen(album) - 2 - gml->album_tracknr_digits] == '[')
		    album[strlen(album) - 3 - gml->album_tracknr_digits] = '\0';

		gml_search_do_search(gml, SEARCH_ALBUM, album);
		gtk_notebook_set_current_page(GTK_NOTEBOOK(gml->notebook), 0);
	}
	g_free(album);
}

static void cb_pl_popup_search_album_pressed(GtkWidget *menuitem,
                                             gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, search_for_album_foreach_func,
	                                    (gpointer)gml);
}

static void cb_pl_popup_play_pressed(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_play_selection,
	                                    (gpointer)gml);
}

static void cb_pl_popup_info_pressed(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_info_about_selection, NULL);
}

struct params_3
{
	GMedialib *gml;
	gint       rating;
};

static void pl_rate_selected_tracks(GtkTreeModel  *model,
                                    GtkTreePath   *path,
                                    GtkTreeIter   *iter,
                                    gpointer       userdata)
{
	guint           id;
	xmmsc_result_t *res;
	GMedialib      *gml    = ((struct params_3 *)userdata)->gml;
	gint            rating = ((struct params_3 *)userdata)->rating;

	gtk_tree_model_get(model, iter, 1, &id, -1);

	if (rating >= 0 && rating <= 4) {
		gtk_list_store_set(gml->playlist.ls, iter, 
		                   5, gml->playlist.icon_rating[rating], -1);
	} else {
		gtk_list_store_set(gml->playlist.ls, iter, 
		                   5, NULL, -1);
	}
	res = xmmsc_medialib_entry_property_set_int_with_source(connection, id, 
	                                                        "client/generic",
	                                                        "rating", rating+1);
	xmmsc_result_unref(res);
}

static void cb_pl_popup_rate_track_4(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rating = 4;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_pl_popup_rate_track_3(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rating = 3;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_pl_popup_rate_track_2(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rating = 2;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_pl_popup_rate_track_1(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rating = 1;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_pl_popup_rate_track_0(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rating = 0;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_pl_popup_do_not_rate(GtkWidget *menuitem,
                                    gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rating = -1;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_column_pos_visible(GtkWidget *menuitem,
                                  gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_pos_visible(gml, !gml_pl_get_column_pos_visible(gml));
}

static void cb_column_id_visible(GtkWidget *menuitem,
                                 gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_id_visible(gml, !gml_pl_get_column_id_visible(gml));
}

static void cb_column_artist_visible(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_artist_visible(gml, !gml_pl_get_column_artist_visible(gml));
}

static void cb_column_track_visible(GtkWidget *menuitem,
                                    gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_track_visible(gml, !gml_pl_get_column_track_visible(gml));
}

static void cb_column_album_visible(GtkWidget *menuitem,
                                    gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_album_visible(gml, !gml_pl_get_column_album_visible(gml));
}

static void cb_column_rating_visible(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_rating_visible(gml, !gml_pl_get_column_rating_visible(gml));
}

static void pl_view_popup_menu(GtkWidget      *treeview,
                               GdkEventButton *event,
                               gpointer        userdata)
{
	GtkWidget        *menu, *menu_rating, *menu_columns, *menuitem;
	GtkTreeSelection *selection;
	GMedialib        *gml = (GMedialib *)userdata;
	gboolean          multiple = FALSE, none = FALSE;
	int               i;

	struct submenu
	{
		gchar    *name;
		gboolean (*get_fn)();
		void     (*callback_fn)();
	};

	struct submenu sm_columns[] = 
		{ {"#",      gml_pl_get_column_pos_visible,    cb_column_pos_visible},
		  {"ID",     gml_pl_get_column_id_visible,     cb_column_id_visible},
		  {"Artist", gml_pl_get_column_artist_visible, cb_column_artist_visible},
		  {"Track",  gml_pl_get_column_track_visible,  cb_column_track_visible},
		  {"Album",  gml_pl_get_column_album_visible,  cb_column_album_visible},
		  {"Rating", gml_pl_get_column_rating_visible, cb_column_rating_visible},
		  { NULL, NULL, NULL } };

	struct submenu sm_rating[] =
		{ {"4 (Best)",  NULL, cb_pl_popup_rate_track_4},
		  {"3",         NULL, cb_pl_popup_rate_track_3},
		  {"2",         NULL, cb_pl_popup_rate_track_2},
		  {"1",         NULL, cb_pl_popup_rate_track_1},
		  {"0 (Worst)", NULL, cb_pl_popup_rate_track_0},
		  {"No rating", NULL, cb_pl_popup_do_not_rate},
		  { NULL, NULL, NULL } };

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	if (gtk_tree_selection_count_selected_rows(selection) > 1)
		multiple = TRUE;
	if (gtk_tree_selection_count_selected_rows(selection) == 0)
		none = TRUE;

	/* Submenu: Track rating */
	menu_rating = gtk_menu_new();

	i = 0;
	while (sm_rating[i].name != NULL) {
		menuitem = gtk_menu_item_new_with_mnemonic(sm_rating[i].name);
		if (none) gtk_widget_set_sensitive(menuitem, FALSE);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu_rating), menuitem);
		g_signal_connect(menuitem, "activate",
		                 G_CALLBACK(sm_rating[i].callback_fn), gml);
		i++;
	}

	/* Submenu: Visible columns */
	menu_columns = gtk_menu_new();

	i = 0;
	while (sm_columns[i].name != NULL) {
		menuitem = gtk_check_menu_item_new_with_mnemonic(sm_columns[i].name);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem),
		                               sm_columns[i].get_fn(gml));
		gtk_menu_shell_append(GTK_MENU_SHELL(menu_columns), menuitem);
		g_signal_connect(menuitem, "activate",
		                 G_CALLBACK(sm_columns[i].callback_fn), gml);
		i++;
	}

	/* Context menu */
	menu = gtk_menu_new();

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_MEDIA_PLAY, NULL);
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_popup_play_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_DIALOG_INFO, NULL);
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_popup_info_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_REMOVE, NULL);
	if (none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_button_delete_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_with_mnemonic("Search for _Artist");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), 
	                              gtk_image_new_from_stock(GTK_STOCK_FIND, 
	                                                       GTK_ICON_SIZE_SMALL_TOOLBAR));
	if (multiple || none || !gml_notebook_is_mlib_search_visible(GMEDIALIB(gml)))
		gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_popup_search_artist_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_with_mnemonic("Search for Al_bum");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), 
	                              gtk_image_new_from_stock(GTK_STOCK_FIND, 
	                                                       GTK_ICON_SIZE_SMALL_TOOLBAR));
	if (multiple || none || !gml_notebook_is_mlib_search_visible(GMEDIALIB(gml)))
		gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_popup_search_album_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_menu_item_new_with_mnemonic("_Rate track(s)");
	if (none) gtk_widget_set_sensitive(menuitem, FALSE);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu_rating);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_menu_item_new_with_mnemonic("_Columns");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu_columns);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	gtk_widget_show_all(menu);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
	               (event != NULL) ? event->button : 0,
	               gdk_event_get_time((GdkEvent*)event));
}

static gboolean cb_pl_selection_button_press(GtkWidget      *treeview,
                                             GdkEventButton *event,
                                             gpointer        userdata)
{
	gboolean result = FALSE;
	if (event->type == GDK_BUTTON_PRESS  &&  event->button == 3) {
		pl_view_popup_menu(treeview, event, userdata);
		result = TRUE;
	}
	return result;
}

static gboolean cb_pl_selection_popup_menu(GtkWidget *treeview, 
                                           gpointer   userdata)
{
	pl_view_popup_menu(treeview, NULL, userdata);
	return TRUE;
}


gboolean gml_pl_get_column_pos_visible(GMedialib *gml)
{
	return gml->playlist.column_pos_visible;
}

gboolean gml_pl_get_column_id_visible(GMedialib *gml)
{
	return gml->playlist.column_id_visible;
}

gboolean gml_pl_get_column_artist_visible(GMedialib *gml)
{
	return gml->playlist.column_artist_visible;
}

gboolean gml_pl_get_column_track_visible(GMedialib *gml)
{
	return gml->playlist.column_track_visible;
}

gboolean gml_pl_get_column_album_visible(GMedialib *gml)
{
	return gml->playlist.column_album_visible;
}

gboolean gml_pl_get_column_rating_visible(GMedialib *gml)
{
	return gml->playlist.column_rating_visible;
}

gboolean gml_pl_get_new_playlist_confirm(GMedialib *gml)
{
	return gml->playlist.new_playlist_confirm;
}

void gml_pl_set_new_playlist_confirm(GMedialib *gml, gboolean confirm)
{
	gml->playlist.new_playlist_confirm = confirm;
}

void gml_pl_set_column_pos_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_pos_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_pos, visible);
}

void gml_pl_set_column_id_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_id_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_id, visible);
}

void gml_pl_set_column_artist_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_artist_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_artist, visible);
}

void gml_pl_set_column_track_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_track_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_track, visible);
}

void gml_pl_set_column_album_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_album_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_album, visible);
}

void gml_pl_set_column_rating_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_rating_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_rating, visible);
}

void gml_pl_setup_xmms_callbacks(GMedialib *gml)
{
	XMMS_CALLBACK_SET(connection, xmmsc_broadcast_playback_current_id,
 	                  bc_handle_current_id, gml);
	XMMS_CALLBACK_SET(connection, xmmsc_broadcast_medialib_entry_changed,
		              bc_handle_medialib_entry_changed, gml);
	XMMS_CALLBACK_SET(connection, xmmsc_broadcast_playlist_changed,
		              bc_handle_playlist_changed, gml);
}

void gml_create_playlist(GMedialib *gml)
{
	GtkCellRenderer  *renderer_pixbuf;
	GtkTreeSelection *selection;
	GtkTooltips      *tooltips;
	GdkPixbuf        *icon;
	GtkWidget        *album_item, *artist_item, *title_item, *tracknr_item;
	GtkTargetEntry    row_targets[1];

	gml->playlist.frame = gtk_frame_new("Playlist");
	gtk_container_set_border_width(GTK_CONTAINER(gml->playlist.frame), 2);

	/* table playlist */
	gml->playlist.table = gtk_table_new(2, 7, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(gml->playlist.table), 3);
	gtk_table_set_row_spacings(GTK_TABLE(gml->playlist.table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(gml->playlist.table), 4);
	gtk_container_add(GTK_CONTAINER(gml->playlist.frame), gml->playlist.table);

	/* clear playlist button */
	gml->playlist.button_new_pl = gtk_button_new();
	gml->playlist.image_new = gtk_image_new_from_stock(GTK_STOCK_NEW,
	                                     GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_new_pl), 
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_new_pl), 
	                  gml->playlist.image_new);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_new_pl,
		             0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* save playlist button */
	gml->playlist.button_save_pl = gtk_button_new();
	gml->playlist.image_save = gtk_image_new_from_stock(GTK_STOCK_SAVE,
	                                         GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_save_pl), 
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_save_pl), 
	                  gml->playlist.image_save);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_save_pl,
		             1, 2, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);


	/* add 10 random songs button */
	gml->playlist.button_add_rnd = gtk_button_new();
	icon = gdk_pixbuf_new_from_xpm_data((const char **)addrnd_xpm);
	gml->playlist.image_add_rnd = gtk_image_new_from_pixbuf(icon);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_add_rnd), 
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_add_rnd), 
	                  gml->playlist.image_add_rnd);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_add_rnd,
		             2, 3, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* shuffle playlist button */
	gml->playlist.button_shuffle_pl = gtk_button_new();
	icon = gdk_pixbuf_new_from_xpm_data((const char **)shuffle_xpm);
	gml->playlist.image_shuffle_pl = gtk_image_new_from_pixbuf(icon);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_shuffle_pl),
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_shuffle_pl), 
	                  gml->playlist.image_shuffle_pl);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_shuffle_pl,
		             3, 4, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);


	/* sort playlist button */
	gml->playlist.button_sort_pl = gtk_button_new();
	gml->playlist.image_sort_pl = gtk_image_new_from_stock(
	                              	GTK_STOCK_SORT_ASCENDING, 
	                              	GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_sort_pl),
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_sort_pl), 
	                  gml->playlist.image_sort_pl);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_sort_pl,
		             4, 5, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);
	gml->playlist.menu_sort_pl = gtk_menu_new();

	/* sort playlist menu */
	album_item   = gtk_menu_item_new_with_label("Album");
	artist_item  = gtk_menu_item_new_with_label("Artist");
	title_item   = gtk_menu_item_new_with_label("Title");
	tracknr_item = gtk_menu_item_new_with_label("Tracknr");
	gtk_menu_shell_append(GTK_MENU_SHELL(gml->playlist.menu_sort_pl), album_item);
	gtk_menu_shell_append(GTK_MENU_SHELL(gml->playlist.menu_sort_pl), artist_item);
	gtk_menu_shell_append(GTK_MENU_SHELL(gml->playlist.menu_sort_pl), title_item);
	gtk_menu_shell_append(GTK_MENU_SHELL(gml->playlist.menu_sort_pl), tracknr_item);
	gtk_widget_show(album_item);
	gtk_widget_show(artist_item);
	gtk_widget_show(title_item);
	gtk_widget_show(tracknr_item);
	gtk_signal_connect(GTK_OBJECT(album_item),   "activate",
	                   G_CALLBACK(cb_pl_sort_menu_activate),
					   (gpointer)gml);
	gtk_signal_connect(GTK_OBJECT(artist_item),  "activate",
                       G_CALLBACK(cb_pl_sort_menu_activate),
					   (gpointer)gml);
	gtk_signal_connect(GTK_OBJECT(title_item),   "activate",
	                   G_CALLBACK(cb_pl_sort_menu_activate),
					   (gpointer)gml);
	gtk_signal_connect(GTK_OBJECT(tracknr_item), "activate",
	                   G_CALLBACK(cb_pl_sort_menu_activate),
					   (gpointer)gml);

	/* reload playlist button */
	gml->playlist.button_reload_pl = gtk_button_new();
	gml->playlist.image_reload = gtk_image_new_from_stock(GTK_STOCK_REFRESH, 
	                                            GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_reload_pl),
	                                 GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_reload_pl),
	                  gml->playlist.image_reload);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_reload_pl,
		             5, 6, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* show-current-track button */
	gml->playlist.button_track = gtk_button_new();
	gml->playlist.image_track = gtk_image_new_from_stock(GTK_STOCK_JUMP_TO,
	                                           GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_track),
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_track),
	                  gml->playlist.image_track);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_track,
		             6, 7, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);
	gtk_widget_set_sensitive(gml->playlist.button_track, TRUE);

	/* list playlist */
	gml->playlist.ls = gtk_list_store_new(6, G_TYPE_INT, G_TYPE_INT, 
	                                      G_TYPE_STRING, G_TYPE_STRING,
	                                      G_TYPE_STRING, GDK_TYPE_PIXBUF);

	gml->playlist.list = 
		gtk_tree_view_new_with_model(GTK_TREE_MODEL(gml->playlist.ls));
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(gml->playlist.list), TRUE);

	row_targets[0].target = "PLAYLIST_MODEL_ROW";
	row_targets[0].flags = GTK_TARGET_SAME_WIDGET;
	row_targets[0].info = 0;
	drag_atom = gdk_atom_intern(row_targets[0].target, FALSE);
	gtk_tree_view_enable_model_drag_source(
		GTK_TREE_VIEW(gml->playlist.list),
		GDK_BUTTON1_MASK,
		row_targets,
		G_N_ELEMENTS(row_targets),
		GDK_ACTION_MOVE);
	gtk_tree_view_enable_model_drag_dest(
		GTK_TREE_VIEW(gml->playlist.list),
		row_targets,
		G_N_ELEMENTS(row_targets),
		GDK_ACTION_MOVE);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);

	gml->playlist.scroll_widget = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(gml->playlist.scroll_widget),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	gtk_container_add(GTK_CONTAINER(gml->playlist.scroll_widget),
	                  gml->playlist.list);

	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.scroll_widget,
		             0, 7, 1, 2,
		             GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

	gml->playlist.column_pos = gtk_tree_view_column_new_with_attributes("#", 
	                                                         gml->renderer_text,
	                                                         "text", 0, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_pos);
	gtk_tree_view_column_set_resizable(gml->playlist.column_pos, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_pos,
	                                        gml->renderer_text,
	                                        pl_cell_data_function, gml, NULL);

	gml->playlist.column_id = gtk_tree_view_column_new_with_attributes("ID",
	                                                        gml->renderer_text,
	                                                        "text", 1, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_id);
	gtk_tree_view_column_set_resizable(gml->playlist.column_id, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_id,
	                                        gml->renderer_text,
	                                        pl_cell_data_function, gml, NULL);

	gml->playlist.column_artist = 
		gtk_tree_view_column_new_with_attributes("Artist", gml->renderer_text_ellipsize,
		                                         "text", 2, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_artist);
	gtk_tree_view_column_set_resizable(gml->playlist.column_artist, TRUE);
	gtk_tree_view_column_set_expand(gml->playlist.column_artist, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_artist,
	                                        gml->renderer_text_ellipsize,
	                                        pl_cell_data_function, gml, NULL);

	gml->playlist.column_track = 
		gtk_tree_view_column_new_with_attributes("Track", gml->renderer_text_ellipsize,
		                                         "text", 3, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_track);
	gtk_tree_view_column_set_resizable(gml->playlist.column_track, TRUE);
	gtk_tree_view_column_set_expand(gml->playlist.column_track, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_track,
	                                        gml->renderer_text_ellipsize,
	                                        pl_cell_data_function, gml, NULL);

	gml->playlist.column_album = 
		gtk_tree_view_column_new_with_attributes("Album [#]", gml->renderer_text_ellipsize,
		                                         "text", 4, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_album);
	gtk_tree_view_column_set_resizable(gml->playlist.column_album, TRUE);
	gtk_tree_view_column_set_expand(gml->playlist.column_album, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_album,
	                                        gml->renderer_text_ellipsize,
	                                        pl_cell_data_function, gml, NULL);


	renderer_pixbuf = gtk_cell_renderer_pixbuf_new();
	gml->playlist.column_rating = 
		gtk_tree_view_column_new_with_attributes("Rating", renderer_pixbuf,
		                                         "pixbuf", 5, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_rating);
	gtk_tree_view_column_set_resizable(gml->playlist.column_rating, FALSE);


	/* rating icons */
	gml->playlist.icon_rating[0] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank0_xpm);
	gml->playlist.icon_rating[1] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank1_xpm);
	gml->playlist.icon_rating[2] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank2_xpm);
	gml->playlist.icon_rating[3] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank3_xpm);
	gml->playlist.icon_rating[4] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank4_xpm);

	/* tooltips */
	tooltips = gtk_tooltips_new();
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_new_pl, 
	                     "New playlist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_save_pl, 
	                     "Save playlist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_shuffle_pl, 
	                     "Shuffle the playlist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_sort_pl, 
	                     "Sort the playlist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_add_rnd, 
	                     "Add 10 random songs to the playlist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_reload_pl, 
	                     "Refresh the playlist\n(Useful if another XMMS2 " \
	                     "client has altered the playlist)", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_track, 
	                     "Show currently played track", NULL);

	gml_pl_set_column_pos_visible   (gml, TRUE);
	gml_pl_set_column_id_visible    (gml, TRUE);
	gml_pl_set_column_artist_visible(gml, TRUE);
	gml_pl_set_column_track_visible (gml, TRUE);
	gml_pl_set_column_album_visible (gml, TRUE);
	gml_pl_set_column_rating_visible(gml, TRUE);

	g_signal_connect(G_OBJECT(gml->playlist.list),              "drag-data-get",
	                 G_CALLBACK(cb_pl_drag_data_get),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.list),              "drag_data_received",
	                 G_CALLBACK(cb_pl_drag_data_received),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.button_reload_pl),  "clicked",
	                 G_CALLBACK(cb_pl_reload_button_pressed),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.button_shuffle_pl), "clicked",
	                 G_CALLBACK(cb_pl_shuffle_button_pressed),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.button_sort_pl),    "button-press-event",
					 G_CALLBACK(cb_pl_sort_button_raw_pressed),
					 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.button_add_rnd),    "clicked",
	                 G_CALLBACK(cb_pl_add_rnd_button_pressed),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.button_new_pl),     "clicked",
	                 G_CALLBACK(cb_pl_clear_button_pressed),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.button_save_pl),    "clicked",
	                 G_CALLBACK(cb_pl_save_pl_button_pressed),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.button_track),      "clicked",
	                 G_CALLBACK(cb_pl_button_track_pressed),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.list),              "row-activated",
	                 G_CALLBACK(cb_pl_on_tree_view_row_activated),
	                 (gpointer)gml);
	gml->playlist.sel = 
		gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	g_signal_connect(G_OBJECT(gml->playlist.list),              "button-press-event",
	                 G_CALLBACK(cb_pl_selection_button_press),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.list),              "popup-menu",
	                 G_CALLBACK(cb_pl_selection_popup_menu),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.list),              "key-press-event",
	                 G_CALLBACK(cb_pl_selection_key_press),
	                 (gpointer)gml);
	gml_pl_setup_xmms_callbacks(gml);

	update_current_pl_pos(gml);
	gml->playlist.move_finished = TRUE;
	gml->playlist.refresh = FALSE;
	gml->playlist.refresh_counter = 0;
	gml->playlist.new_playlist_confirm = TRUE;
}
