/*
 *  $Id: preview.h 23006 2020-12-09 15:36:50Z yeti-dn $
 *  Copyright (C) 2015-2017 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef __GWY_PROCESS_PREVIEW_H__
#define __GWY_PROCESS_PREVIEW_H__

#include <string.h>
#include <libgwydgets/gwycolorbutton.h>
#include <libgwydgets/gwydataview.h>
#include <libgwydgets/gwylayer-basic.h>
#include <libgwydgets/gwylayer-mask.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwydgetutils.h>
#include <app/menu.h>
#include <app/gwymoduleutils.h>
#include <app/data-browser.h>

enum {
    /* Standard preview size. */
    PREVIEW_SIZE = 480,
    /* For slow synth modules or if there are lots of other things to fit. */
    PREVIEW_SMALL_SIZE = 360,
    /* When we need to fit two preview-sized areas. */
    PREVIEW_HALF_SIZE = 240,
};

enum {
    RESPONSE_RESET     = 1,
    RESPONSE_PREVIEW   = 2,
    RESPONSE_CLEAR     = 3,
    RESPONSE_INIT      = 4,
    RESPONSE_ESTIMATE  = 5,
    RESPONSE_REFINE    = 6,
    RESPONSE_CALCULATE = 7,
    RESPONSE_LOAD      = 8,
    RESPONSE_SAVE      = 9,
    RESPONSE_COPY      = 10,
};

typedef struct {
    GwyGraphModel *gmodel;
    GwyDataChooserFilterFunc filter;
    gpointer filter_data;
} TargetGraphFilterData;

G_GNUC_UNUSED
static GwyPixmapLayer*
create_basic_layer(GwyDataView *dataview, gint id)
{
    GwyPixmapLayer *layer = gwy_layer_basic_new();
    GQuark quark = gwy_app_get_data_key_for_id(id);
    const gchar *key = g_quark_to_string(quark);
    gchar buf[24];

    gwy_pixmap_layer_set_data_key(layer, key);
    key = g_quark_to_string(gwy_app_get_data_palette_key_for_id(id));
    gwy_layer_basic_set_gradient_key(GWY_LAYER_BASIC(layer), key);
    key = g_quark_to_string(gwy_app_get_data_range_type_key_for_id(id));
    gwy_layer_basic_set_range_type_key(GWY_LAYER_BASIC(layer), key);
    g_snprintf(buf, sizeof(buf), "/%d/base", id);
    gwy_layer_basic_set_min_max_key(GWY_LAYER_BASIC(layer), buf);
    gwy_data_view_set_base_layer(dataview, layer);

    return layer;
}

G_GNUC_UNUSED
static GwyPixmapLayer*
create_mask_layer(GwyDataView *dataview, gint id)
{
    GwyPixmapLayer *layer = gwy_layer_mask_new();
    GQuark quark = gwy_app_get_mask_key_for_id(id);
    const gchar *key = g_quark_to_string(quark);

    gwy_pixmap_layer_set_data_key(layer, key);
    gwy_layer_mask_set_color_key(GWY_LAYER_MASK(layer), key);
    gwy_data_view_set_alpha_layer(dataview, layer);

    return layer;
}

G_GNUC_UNUSED
static GwySelection*
create_vector_layer(GwyDataView *dataview, gint id, const gchar *name,
                    gboolean editable)
{
    GwyVectorLayer *layer;
    GType type;
    gchar *s, *t;

    s = g_strconcat("GwyLayer", name, NULL);
    type = g_type_from_name(s);
    g_free(s);
    g_return_val_if_fail(type, NULL);
    g_return_val_if_fail(g_type_is_a(type, GWY_TYPE_VECTOR_LAYER), NULL);

    layer = GWY_VECTOR_LAYER(g_object_new(type, NULL));
    s = g_strdup_printf("/%d/select/%s", id, name);
    t = strrchr(s, '/');
    g_assert(t);
    t++;
    *t = g_ascii_tolower(*t);
    gwy_vector_layer_set_selection_key(layer, s);
    g_free(s);
    gwy_vector_layer_set_editable(layer, editable);
    gwy_data_view_set_top_layer(dataview, layer);

    return gwy_vector_layer_ensure_selection(layer);
}

G_GNUC_UNUSED
static GtkWidget*
create_preview(GwyContainer *data,
               gint id, gint preview_size, gboolean have_mask)
{
    GQuark quark = gwy_app_get_data_key_for_id(id);
    const gchar *key = g_quark_to_string(quark);
    GwyDataView *dataview;
    GtkWidget *widget;

    widget = gwy_data_view_new(data);
    dataview = GWY_DATA_VIEW(widget);
    gwy_data_view_set_data_prefix(dataview, key);
    create_basic_layer(dataview, id);
    if (have_mask)
        create_mask_layer(dataview, id);
    gwy_set_data_preview_size(dataview, preview_size);

    return widget;
}

G_GNUC_UNUSED
static void
ensure_mask_color(GwyContainer *data, gint id)
{
    const gchar *key = g_quark_to_string(gwy_app_get_mask_key_for_id(id));
    GwyRGBA rgba;

    if (!gwy_rgba_get_from_container(&rgba, data, key)) {
        gwy_rgba_get_from_container(&rgba, gwy_app_settings_get(), "/mask");
        gwy_rgba_store_to_container(&rgba, data, key);
    }
}

G_GNUC_UNUSED
static void
load_mask_color_to_button(GtkWidget *color_button,
                          GwyContainer *data,
                          gint id)
{
    const gchar *key = g_quark_to_string(gwy_app_get_mask_key_for_id(id));
    GwyRGBA rgba;

    ensure_mask_color(data, id);
    gwy_rgba_get_from_container(&rgba, data, key);
    gwy_color_button_set_color(GWY_COLOR_BUTTON(color_button), &rgba);
}

G_GNUC_UNUSED
static void
mask_color_changed(GtkWidget *color_button)
{
    GObject *object = G_OBJECT(color_button);
    GtkWindow *dialog;
    GwyContainer *data;
    GQuark quark;
    gint id;

    data = GWY_CONTAINER(g_object_get_data(object, "data"));
    dialog = GTK_WINDOW(g_object_get_data(object, "dialog"));
    id = GPOINTER_TO_INT(g_object_get_data(object, "id"));
    quark = gwy_app_get_mask_key_for_id(id);
    gwy_mask_color_selector_run(NULL, dialog,
                                GWY_COLOR_BUTTON(color_button), data,
                                g_quark_to_string(quark));
    load_mask_color_to_button(color_button, data, id);
}

G_GNUC_UNUSED
static GtkWidget*
create_mask_color_button(GwyContainer *data, GtkWidget *dialog, gint id)
{
    GtkWidget *color_button;
    GObject *object;

    color_button = gwy_color_button_new();
    object = G_OBJECT(color_button);
    g_object_set_data(object, "data", data);
    g_object_set_data(object, "dialog", dialog);
    g_object_set_data(object, "id", GINT_TO_POINTER(id));

    gwy_color_button_set_use_alpha(GWY_COLOR_BUTTON(color_button), TRUE);
    load_mask_color_to_button(color_button, data, id);
    g_signal_connect(color_button, "clicked",
                     G_CALLBACK(mask_color_changed), NULL);

    return color_button;
}

G_GNUC_UNUSED
static GwyDataField*
create_mask_field(GwyDataField *dfield)
{
    dfield = gwy_data_field_new_alike(dfield, TRUE);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(dfield), NULL);
    return dfield;
}

static void
mask_merge_enabled_changed(GtkToggleButton *toggle, GtkWidget *widget)
{
    gtk_widget_set_sensitive(widget, gtk_toggle_button_get_active(toggle));
}

/* Pass NULL @pcheck if you do not want a checkbox.
 * XXX: Cannot work with both hscale and adjbar.  Must use with adjbar. */
G_GNUC_UNUSED
static void
create_mask_merge_buttons(GtkWidget *table, gint row, const gchar *name,
                          gboolean enabled, GCallback enabled_callback,
                          GwyMergeType merge, GCallback merge_type_callback,
                          gpointer user_data,
                          GtkWidget **pcheck, GSList **ptype)
{
    GtkTooltips *tips;
    GSList *group, *l;
    GtkWidget *hbox, *button, *label;
    GQuark quark = g_quark_from_string("gwy-radiobuttons-key");
    gint value;

    if (!name)
        name = _("Combine with existing mask:");

    tips = gwy_app_get_tooltips();

    button = gtk_radio_button_new(NULL);
    g_object_set_qdata(G_OBJECT(button), quark,
                       GINT_TO_POINTER(GWY_MERGE_INTERSECTION));
    gtk_container_add(GTK_CONTAINER(button),
                      gtk_image_new_from_stock(GWY_STOCK_MASK_INTERSECT,
                                               GTK_ICON_SIZE_BUTTON));
    gtk_tooltips_set_tip(tips, button, _("Intersection"), NULL);

    button = gtk_radio_button_new_from_widget(GTK_RADIO_BUTTON(button));
    g_object_set_qdata(G_OBJECT(button), quark,
                       GINT_TO_POINTER(GWY_MERGE_UNION));
    gtk_container_add(GTK_CONTAINER(button),
                      gtk_image_new_from_stock(GWY_STOCK_MASK_ADD,
                                               GTK_ICON_SIZE_BUTTON));
    gtk_tooltips_set_tip(tips, button, _("Union"), NULL);

    group = *ptype = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach(GTK_TABLE(table), hbox,
                     1, 2, row, row+1, GTK_FILL, 0, 0, 0);

    for (l = group; l; l = g_slist_next(l)) {
        button = (GtkWidget*)l->data;
        gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button), FALSE);
        gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);

        value = GPOINTER_TO_INT(g_object_get_qdata(G_OBJECT(button), quark));
        if (value == merge)
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
    }

    if (merge_type_callback) {
        for (l = group; l; l = g_slist_next(l)) {
            g_signal_connect_swapped(l->data, "clicked",
                                     merge_type_callback, user_data);
        }
    }

    if (pcheck) {
        label = *pcheck = gtk_check_button_new_with_mnemonic(name);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(label), enabled);
        gtk_widget_set_sensitive(hbox, enabled);
        g_signal_connect(label, "toggled",
                         G_CALLBACK(mask_merge_enabled_changed), hbox);
        if (enabled_callback) {
            g_signal_connect_swapped(*pcheck, "toggled",
                                     enabled_callback, user_data);
        }
    }
    else {
        label = gtk_label_new(name);
        gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    }
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 1, row, row+1, GTK_FILL, 0, 0, 0);
}

/* NB: The channel needs full-range linear mapping for this to work! */
G_GNUC_UNUSED
static void
set_gradient_for_residuum(GwyGradient *gradient,
                          gdouble fullmin, gdouble fullmax,
                          gdouble *dispmin, gdouble *dispmax)
{
    static const GwyRGBA rgba_negative = { 0.0, 0.0, 1.0, 1.0 };
    static const GwyRGBA rgba_positive = { 1.0, 0.0, 0.0, 1.0 };
    static const GwyRGBA rgba_neutral = { 1.0, 1.0, 1.0, 1.0 };

    gdouble zero, range;
    GwyGradientPoint zero_pt = { 0.5, rgba_neutral };
    GwyRGBA rgba;
    gint pos;

    gwy_gradient_reset(gradient);
    fullmin = fmin(fullmin, *dispmin);
    fullmax = fmax(fullmax, *dispmax);

    /* Stretch the scale to the range when all the data are too high or too
     * low. */
    if (*dispmin >= 0.0) {
        gwy_gradient_set_point_color(gradient, 0, &rgba_neutral);
        gwy_gradient_set_point_color(gradient, 1, &rgba_positive);
        return;
    }
    if (*dispmax <= 0.0) {
        gwy_gradient_set_point_color(gradient, 0, &rgba_negative);
        gwy_gradient_set_point_color(gradient, 1, &rgba_neutral);
        return;
    }
    if (!(*dispmax > *dispmin)) {
        gwy_gradient_set_point_color(gradient, 0, &rgba_negative);
        gwy_gradient_set_point_color(gradient, 1, &rgba_positive);
        pos = gwy_gradient_insert_point_sorted(gradient, &zero_pt);
        g_assert(pos == 1);
        return;
    }

    /* Otherwise make zero neutral and map the two colours to both side,
     * with the same scale. */
    range = fmax(-*dispmin, *dispmax);
    *dispmin = -range;
    *dispmax = range;

    if (-fullmin >= range && fullmax >= range) {
        /* Symmetrically extended display range lies within the full data
         * range.  Use this as the display range with fully extended colour
         * gradients. */
        gwy_gradient_set_point_color(gradient, 0, &rgba_negative);
        gwy_gradient_set_point_color(gradient, 1, &rgba_positive);
    }
    else {
        if (fullmax < range) {
            /* Map [-range, fullmax] to colours [negative, cut_positive]. */
            zero = range/(fullmax + range);
            *dispmax = fullmax;
            gwy_gradient_set_point_color(gradient, 0, &rgba_negative);
            gwy_rgba_interpolate(&rgba_neutral, &rgba_positive,
                                 (1.0 - zero)/zero, &rgba);
            gwy_gradient_set_point_color(gradient, 1, &rgba);
        }
        else {
            /* Map [fullmin, range] to colours [cut_negative, positive]. */
            zero = -fullmin/(range - fullmin);
            *dispmin = fullmin;
            gwy_rgba_interpolate(&rgba_neutral, &rgba_negative,
                                 zero/(1.0 - zero), &rgba);
            gwy_gradient_set_point_color(gradient, 0, &rgba);
            gwy_gradient_set_point_color(gradient, 1, &rgba_positive);
        }
        zero_pt.x = zero;
    }

    pos = gwy_gradient_insert_point_sorted(gradient, &zero_pt);
    g_assert(pos == 1);
}

G_GNUC_UNUSED
static gboolean
filter_target_graph_default(GwyContainer *data, gint id, gpointer user_data)
{
    GwyGraphModel *gmodel = (GwyGraphModel*)user_data;
    GwyGraphModel *targetgmodel;
    GQuark quark;

    if (!gmodel)
        return FALSE;

    g_return_val_if_fail(GWY_IS_GRAPH_MODEL(gmodel), FALSE);
    quark = gwy_app_get_graph_key_for_id(id);
    return (gwy_container_gis_object(data, quark, (GObject**)&targetgmodel)
            && gwy_graph_model_units_are_compatible(gmodel, targetgmodel));
}

G_GNUC_UNUSED
static gboolean
filter_target_graph_extra(GwyContainer *data, gint id, gpointer user_data)
{
    TargetGraphFilterData *tgf = (TargetGraphFilterData*)user_data;

    /* No graph model provided; this means the filter is based on something
     * else than graph model compatibility (most likely the module does not
     * display the graph). */
    if (!tgf->gmodel)
        return tgf->filter(data, id, tgf->filter_data);

    /* Otherwise the filter has veto power.  If it says OK we still check the
     * graph compatibility. */
    if (!tgf->filter(data, id, tgf->filter_data))
        return FALSE;

    return filter_target_graph_default(data, id, tgf->gmodel);
}

G_GNUC_UNUSED
static GtkWidget*
create_target_graph_extra_filter(GwyAppDataId *id, GtkWidget *table, gint row,
                                 GwyGraphModel *gmodel,
                                 GwyDataChooserFilterFunc filter,
                                 gpointer filter_data)
{
    GtkWidget *widget;
    GwyDataChooser *chooser;

    widget = gwy_data_chooser_new_graphs();
    chooser = GWY_DATA_CHOOSER(widget);
    gwy_data_chooser_set_none(chooser, _("New graph"));
    gwy_data_chooser_set_active(chooser, NULL, -1);
    if (filter) {
        TargetGraphFilterData *tgf = g_new(TargetGraphFilterData, 1);
        tgf->gmodel = gmodel;
        tgf->filter = filter;
        tgf->filter_data = filter_data;
        gwy_data_chooser_set_filter(chooser, filter_target_graph_extra,
                                    tgf, (GtkDestroyNotify)g_free);
    }
    else {
        gwy_data_chooser_set_filter(chooser, filter_target_graph_default,
                                    gmodel, NULL);
    }
    gwy_data_chooser_set_active_id(chooser, id);
    gwy_data_chooser_get_active_id(chooser, id);
    gwy_table_attach_adjbar(table, row++, _("Target _graph:"), NULL,
                            GTK_OBJECT(widget), GWY_HSCALE_WIDGET_NO_EXPAND);

    return widget;
}

#define create_target_graph(id,table,row,gmodel) \
    create_target_graph_extra_filter(id,table,row,gmodel,NULL,NULL)

G_GNUC_UNUSED
static void
set_graph_model_keeping_label_pos(GwyGraph *graph, GwyGraphModel *newgmodel)
{
    GwyGraphModel *oldgmodel;
    GwyGraphLabelPosition labelpos;
    gdouble xpos, ypos;

    oldgmodel = gwy_graph_get_model(graph);
    g_object_get(oldgmodel,
                 "label-position", &labelpos,
                 "label-relative-x", &xpos,
                 "label-relative-y", &ypos,
                 NULL);
    /* NB: We cannot use gwy_graph_set_model() if there is a target graph
     * filter because we directly pass the graph model to it as the template.
     * Maybe there should have been some indirection, but this is just how it
     * works. */
    gwy_serializable_clone(G_OBJECT(newgmodel), G_OBJECT(oldgmodel));
    g_object_set(newgmodel,
                 "label-position", labelpos,
                 "label-relative-x", xpos,
                 "label-relative-y", ypos,
                 NULL);
    g_object_unref(newgmodel);
}

G_GNUC_UNUSED
static GtkWidget*
set_up_wait_data_field_preview(GwyDataField *dfield,
                               GwyContainer *data, gint id)
{
    GwyContainer *mydata;
    GtkWidget *preview;

    mydata = gwy_container_new();
    gwy_container_set_object(mydata, gwy_app_get_data_key_for_id(0), dfield);
    preview = create_preview(mydata, 0, PREVIEW_SMALL_SIZE, FALSE);
    if (data && id >= 0) {
        gwy_app_sync_data_items(data, mydata, id, 0, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                0);
    }
    gwy_app_wait_set_preview_widget(preview);

    return preview;
}

G_GNUC_UNUSED
static void
set_widget_as_error_message(GtkWidget *widget)
{
    GdkColor gdkcolor_bad = { 0, 51118, 0, 0 };

    gtk_widget_modify_fg(widget, GTK_STATE_NORMAL, &gdkcolor_bad);
}

G_GNUC_UNUSED
static void
set_widget_as_warning_message(GtkWidget *widget)
{
    GdkColor gdkcolor_warning = { 0, 45056, 20480, 0 };

    gtk_widget_modify_fg(widget, GTK_STATE_NORMAL, &gdkcolor_warning);
}

G_GNUC_UNUSED
static void
set_widget_as_ok_message(GtkWidget *widget)
{
    gtk_widget_modify_fg(widget, GTK_STATE_NORMAL, NULL);
}

#endif

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
