/*
Gwenview - A simple image viewer for KDE
Copyright (C) 2000-2002 Aurlien Gteau

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// KDE includes
#include <kaccel.h>
#include <kaction.h>
#include <kapp.h>
#include <kcmdlineargs.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <kglobal.h>
#include <khelpmenu.h>
#include <kkeydialog.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kprogress.h>
#include <kstatusbar.h>
#include <kstdaccel.h>
#include <kstdaction.h>

// Our includes
#include "configdialog.h"
#include "dirview.h"
#include "fileview.h"
#include "fileoperation.h"
#include "gvpixmap.h"
#include "pixmapview.h"
#include "scrollpixmapview.h"
#include "statusbarprogress.h"

#include "mainwindow.moc"


enum { SB_FOLDER, SB_FILE};

const char* CONFIG_DOCK_GROUP="dock";
const char* CONFIG_MAINWINDOW_GROUP="main window";
const char* CONFIG_FILEWIDGET_GROUP="file widget";
const char* CONFIG_PIXMAPWIDGET_GROUP="pixmap widget";
const char* CONFIG_FILEOPERATION_GROUP="file operations";


MainWindow::MainWindow()
: KDockMainWindow(), mProgress(0L), mFullScreen(false)
{
	FileOperation::readConfig(KGlobal::config(),CONFIG_FILEOPERATION_GROUP);

// Backend
	mGVPixmap=new GVPixmap(this);

// GUI
	createWidgets();
	createActions();
	createAccels();
	createMenu();
	createFileViewPopupMenu();
	createScrollPixmapViewPopupMenu();
	createToolBar();

// Folder connections
	connect(mDirView,SIGNAL(folderChanged(QString)),
		this,SLOT(setPath(QString)) );

// Pixmap connections
	connect(mPixmapView,SIGNAL(selectPrevious()),
		mFileView,SLOT(slotSelectPrevious()) );
	connect(mPixmapView,SIGNAL(selectNext()),
		mFileView,SLOT(slotSelectNext()) );
	connect(mPixmapView,SIGNAL(escapePressed()),
		this,SLOT(escapePressed()) );

// Scroll pixmap connections
	connect(mPixmapView->scrollPixmapView(),SIGNAL(zoomChanged(double)),
		this,SLOT(updateFileStatusBar()) );


// Thumbnail view connections
	connect(mFileView,SIGNAL(updateStarted(int)),
		this,SLOT(thumbnailUpdateStarted(int)) );
	connect(mFileView,SIGNAL(updateEnded()),
		this,SLOT(thumbnailUpdateEnded()) );
	connect(mFileView,SIGNAL(updatedOneThumbnail()),
		this,SLOT(thumbnailUpdateProcessedOne()) );

// File view connections
	connect(mFileView,SIGNAL(filenameChanged(QString)),
		this,SLOT(setFilename(QString)) );
	connect(mFileView,SIGNAL(completed()),
		this,SLOT(updateStatusBar()) );
	connect(mFileView,SIGNAL(canceled()),
		this,SLOT(updateStatusBar()) );
		
// GVPixmap connections
	connect(mGVPixmap,SIGNAL(loading()),
		this,SLOT(pixmapLoading()) );
	connect(mGVPixmap,SIGNAL(loaded()),
		this,SLOT(pixmapLoaded()) );


// Command line
	KCmdLineArgs *args = KCmdLineArgs::parsedArgs();

	QString fullPath;
	if (args->count()==0) {
		fullPath=QDir::currentDirPath();
	} else {
		fullPath=QFile::decodeName(args->arg(0));
	}

// Go to requested file
	setPath(fullPath);
}


MainWindow::~MainWindow() {
	KConfig* config=KGlobal::config();
	FileOperation::writeConfig(config,CONFIG_FILEOPERATION_GROUP);
	mPixmapView->writeConfig(config,CONFIG_PIXMAPWIDGET_GROUP);
	mFileView->writeConfig(config,CONFIG_FILEWIDGET_GROUP);
	writeDockConfig(config,CONFIG_DOCK_GROUP);
	mAccel->writeSettings();
}


//-Public slots----------------------------------------------------------
void MainWindow::setPath(QString fullPath) {
	QFileInfo fileInfo(fullPath);
    bool filenameIsValid;

// Init mFolder and mFilename
	if (fileInfo.isFile()) {
		mFolder=fileInfo.dirPath(true);
		mFilename=fileInfo.fileName();
		filenameIsValid=true;
		mGVPixmap->load(fullPath);
	} else {
		filenameIsValid=false;
		mFolder=fileInfo.absFilePath();
		mFilename="";
	}

// Update actions
	mToggleFullScreen->setEnabled(filenameIsValid);
	mRenameFile->setEnabled(filenameIsValid);
	mCopyFile->setEnabled(filenameIsValid);
	mMoveFile->setEnabled(filenameIsValid);
	mDeleteFile->setEnabled(filenameIsValid);

// Update widgets
	mFileView->setFolderAndFile(mFolder,mFilename);
	mDirView->setFolder(mFolder);
	
	if (filenameIsValid) {
		mGVPixmap->load(fullPath);
	} else {
		mGVPixmap->reset();
	}
}


void MainWindow::setFilename(QString filename) {
	if (mFilename==filename) return;

	mFilename=filename;

	bool filenameIsValid= mFilename!="";

	mToggleFullScreen->setEnabled(filenameIsValid);
	mRenameFile->setEnabled(filenameIsValid);
	mCopyFile->setEnabled(filenameIsValid);
	mMoveFile->setEnabled(filenameIsValid);
	mDeleteFile->setEnabled(filenameIsValid);
	
	if (filenameIsValid) {
		mFileView->selectFilename(mFilename);
		mGVPixmap->load(mFolder + '/' + mFilename);
	} else {
		mGVPixmap->reset();
	}
	updateFileStatusBar();
}


//-Private slots---------------------------------------------------------
void MainWindow::pixmapLoading() {
	kapp->setOverrideCursor(QCursor(WaitCursor));
}


void MainWindow::pixmapLoaded() {
	kapp->restoreOverrideCursor();
	if ( (!mFilename.isEmpty()) && mGVPixmap->isNull()) {
		KMessageBox::sorry(this,i18n("Could not load <b>%1</b>").arg(mFilename));
	}
}


void MainWindow::toggleFullScreen() {
	KConfig* config=KGlobal::config();

	if (!mFullScreen) {
		mFullScreen=true;
		menuBar()->hide();
		toolBar()->hide();
		statusBar()->hide();
		writeDockConfig(config,CONFIG_DOCK_GROUP);
		makeDockInvisible(mFileDock);
		makeDockInvisible(mFolderDock);
		showFullScreen();

		mPixmapView->setFullScreen(true);
	} else {
		mFullScreen=false;
		readDockConfig(config,CONFIG_DOCK_GROUP);
		statusBar()->show();
		toolBar()->show();
		menuBar()->show();
		showNormal();

		mPixmapView->setFullScreen(false);
	}
}


void MainWindow::showConfigDialog() {
	ConfigDialog dialog(this,this);
	dialog.exec();
}


void MainWindow::showKeyDialog() {
	KKeyDialog::configureKeys(mAccel);
}


void MainWindow::escapePressed() {
	if (mFullScreen) {
		mToggleFullScreen->activate();
	}
}


void MainWindow::openFile() {
	QString path=KFileDialog::getOpenFileName();
	if (path.isNull()) return;
	setPath(path);
}

void MainWindow::openWithEditor() {
	if (mFilename.isEmpty()) return;

	FileOperation::openWithEditor(mFolder + '/' + mFilename);
}


void MainWindow::thumbnailUpdateStarted(int count) {
	mProgress=new StatusBarProgress(statusBar(),i18n("Generating thumbnails..."),count);
	mProgress->progress()->setFormat("%v/%m");
	mProgress->show();
	mStop->setEnabled(true);
}


void MainWindow::thumbnailUpdateEnded() {
	mStop->setEnabled(false);
	if (mProgress) {
		mProgress->hide();
		delete mProgress;
		mProgress=0L;
	}
}


void MainWindow::thumbnailUpdateProcessedOne() {
	mProgress->progress()->advance(1);
}



//-GUI-------------------------------------------------------------------
void MainWindow::updateStatusBar() {
	QString txt;
	uint count=mFileView->fileCount();
	if (count>1) {
		txt=i18n("%1 - %2 images");
	} else {
		txt=i18n("%1 - %2 image");
	}
	txt=txt.arg(mFolder).arg(count);

	statusBar()->changeItem( txt, SB_FOLDER );
	updateFileStatusBar();
}


void MainWindow::updateFileStatusBar() {
	QString txt;
	if (!mFilename.isEmpty()) {
		txt=QString("%1 %2x%3 @ %4%")
			.arg(mFilename).arg(mGVPixmap->width()).arg(mGVPixmap->height())
			.arg(int(mPixmapView->scrollPixmapView()->zoom()*100) );
	} else {
		txt="";
	}
	statusBar()->changeItem( txt, SB_FILE );
}


void MainWindow::createWidgets() {
	KConfig* config=KGlobal::config();

// Status bar
	statusBar()->insertItem("",SB_FOLDER);
	statusBar()->setItemAlignment(SB_FOLDER, AlignLeft|AlignVCenter);
	statusBar()->insertItem("",SB_FILE,1);
	statusBar()->setItemAlignment(SB_FILE, AlignLeft|AlignVCenter);

// Pixmap widgets
	mPixmapDock = createDockWidget("Image",SmallIcon("gwenview"),NULL,i18n("Image"));

	mPixmapView=new PixmapView(mPixmapDock,mGVPixmap);
	mPixmapDock->setWidget(mPixmapView);
	setView(mPixmapDock);
	setMainDockWidget(mPixmapDock);

// Folder widget
	mFolderDock = createDockWidget("Folders",SmallIcon("folder_open"),NULL,i18n("Folders"));
	mDirView=new DirView(mFolderDock);
	mFolderDock->setWidget(mDirView);

// File widget
	mFileDock = createDockWidget("Files",SmallIcon("image"),NULL,i18n("Files"));
	mFileView=new FileView(this);
	mFileDock->setWidget(mFileView);

// Default dock config
	setGeometry(20,20,600,400);
	mFolderDock->manualDock( mPixmapDock,KDockWidget::DockLeft,30);
	mFileDock->manualDock( mPixmapDock,KDockWidget::DockTop,30);

// Load config
	readDockConfig(config,CONFIG_DOCK_GROUP);
	mFileView->readConfig(config,CONFIG_FILEWIDGET_GROUP);
	mPixmapView->readConfig(config,CONFIG_PIXMAPWIDGET_GROUP);
}


void MainWindow::createActions() {
	mOpenFile=KStdAction::open(this, SLOT(openFile()),actionCollection() );

	mRenameFile=new KAction(i18n("&Rename..."),Key_F2,mFileView,SLOT(renameFile()),actionCollection(),"file_new");

	mCopyFile=new KAction(i18n("&Copy to..."),Key_F5,mFileView,SLOT(copyFile()),actionCollection(),"file_copy");

	mMoveFile=new KAction(i18n("&Move to..."),Key_F6,mFileView,SLOT(moveFile()),actionCollection(),"file_move");

	mDeleteFile=new KAction(i18n("&Delete"),"editdelete",Key_Delete,mFileView,SLOT(deleteFile()),actionCollection(),"file_delete");

	mOpenWithEditor=new KAction(i18n("&Open with editor"),"paintbrush",0,this,SLOT(openWithEditor()),actionCollection(),"file_edit");

	mToggleFullScreen=new KAction(i18n("Full screen"),"window_fullscreen",CTRL + Key_F,this,SLOT(toggleFullScreen()),actionCollection(),"view_fullscreen");

	mShowConfigDialog=new KAction(i18n("Configure Gwenview"),"configure",0,this,SLOT(showConfigDialog()),actionCollection());

	mShowKeyDialog=KStdAction::keyBindings(this,SLOT(showKeyDialog()),actionCollection());

	mStop=new KAction(i18n("Stop"),"stop",Key_Escape,mFileView,SLOT(cancel()),actionCollection());
	mStop->setEnabled(false);
}


void MainWindow::createAccels() {
// Associate actions with accelerator
	mAccel=new KAccel(this);
	int count=actionCollection()->count();

	for (int pos=0;pos<count;++pos) {
		KAction *action = actionCollection()->action(pos);
		if ( action->accel() ) {
			action->plugAccel(mAccel);
		}
	}
	mFileView->plugActionsToAccel(mAccel);
	mPixmapView->scrollPixmapView()->plugActionsToAccel(mAccel);

// Read user accelerator
	mAccel->readSettings();
}


void MainWindow::createMenu() {
	QPopupMenu* fileMenu = new QPopupMenu;

	mOpenFile->plug(fileMenu);
	fileMenu->insertSeparator();
	mOpenWithEditor->plug(fileMenu);
	mRenameFile->plug(fileMenu);
	mCopyFile->plug(fileMenu);
	mMoveFile->plug(fileMenu);
	mDeleteFile->plug(fileMenu);
	fileMenu->insertSeparator();
	KStdAction::quit( kapp, SLOT (closeAllWindows()) )->plug(fileMenu);
	menuBar()->insertItem(i18n("&File"), fileMenu);

	QPopupMenu* viewMenu = new QPopupMenu;
	mStop->plug(viewMenu);
	
	viewMenu->insertSeparator();
	mFileView->noThumbnails()->plug(viewMenu);
	mFileView->smallThumbnails()->plug(viewMenu);
	mFileView->medThumbnails()->plug(viewMenu);
	mFileView->largeThumbnails()->plug(viewMenu);
	
	viewMenu->insertSeparator();
	mToggleFullScreen->plug(viewMenu);
	mPixmapView->scrollPixmapView()->zoomIn()->plug(viewMenu);
	mPixmapView->scrollPixmapView()->zoomOut()->plug(viewMenu);
	mPixmapView->scrollPixmapView()->resetZoom()->plug(viewMenu);
	mPixmapView->scrollPixmapView()->autoZoom()->plug(viewMenu);
	menuBar()->insertItem(i18n("&View"), viewMenu);

	QPopupMenu* goMenu = new QPopupMenu;
	mFileView->selectFirst()->plug(goMenu);
	mFileView->selectPrevious()->plug(goMenu);
	mFileView->selectNext()->plug(goMenu);
	mFileView->selectLast()->plug(goMenu);
	menuBar()->insertItem(i18n("&Go"), goMenu);

	QPopupMenu* settingsMenu = new QPopupMenu;
	mShowConfigDialog->plug(settingsMenu);
	mShowKeyDialog->plug(settingsMenu);
	menuBar()->insertItem(i18n("&Settings"),settingsMenu);

	menuBar()->insertItem(i18n("&Windows"), dockHideShowMenu());

	menuBar()->insertItem(i18n("&Help"), helpMenu());

	menuBar()->show();
}


void MainWindow::createFileViewPopupMenu() {
	QPopupMenu* menu=new QPopupMenu(this);

	mOpenWithEditor->plug(menu);
	mRenameFile->plug(menu);
	mCopyFile->plug(menu);
	mMoveFile->plug(menu);
	mDeleteFile->plug(menu);

	mFileView->installRBPopup(menu);
}


void MainWindow::createScrollPixmapViewPopupMenu() {
	QPopupMenu* menu=new QPopupMenu(this);

	mPixmapView->scrollPixmapView()->zoomIn()->plug(menu);
	mPixmapView->scrollPixmapView()->zoomOut()->plug(menu);
	mPixmapView->scrollPixmapView()->resetZoom()->plug(menu);
	mPixmapView->scrollPixmapView()->autoZoom()->plug(menu);

	menu->insertSeparator();
	mToggleFullScreen->plug(menu);

	menu->insertSeparator();
	mFileView->selectFirst()->plug(menu);
	mFileView->selectPrevious()->plug(menu);
	mFileView->selectNext()->plug(menu);
	mFileView->selectLast()->plug(menu);

	menu->insertSeparator();
	mOpenWithEditor->plug(menu);
	mRenameFile->plug(menu);
	mCopyFile->plug(menu);
	mMoveFile->plug(menu);
	mDeleteFile->plug(menu);

	mPixmapView->scrollPixmapView()->installRBPopup(menu);
}



void MainWindow::createToolBar() {
	mFileView->selectFirst()->plug(toolBar());
	mFileView->selectPrevious()->plug(toolBar());
	mFileView->selectNext()->plug(toolBar());
	mFileView->selectLast()->plug(toolBar());
	toolBar()->insertLineSeparator();

	mStop->plug(toolBar());
	toolBar()->insertLineSeparator();
	mFileView->noThumbnails()->plug(toolBar());
	mFileView->smallThumbnails()->plug(toolBar());
	mFileView->medThumbnails()->plug(toolBar());
	mFileView->largeThumbnails()->plug(toolBar());
	
	toolBar()->insertLineSeparator();
	mToggleFullScreen->plug(toolBar());
	mPixmapView->scrollPixmapView()->zoomIn()->plug(toolBar());
	mPixmapView->scrollPixmapView()->zoomOut()->plug(toolBar());
	mPixmapView->scrollPixmapView()->resetZoom()->plug(toolBar());
	mPixmapView->scrollPixmapView()->autoZoom()->plug(toolBar());
	toolBar()->show();
}

