/*
Gwenview - A simple image viewer for KDE
Copyright (C) 2000-2002 Aurlien Gteau

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// QT includes
#include <qlayout.h>
#include <qpixmap.h>
#include <qpixmapcache.h>
#include <qpushbutton.h>

// KDE includes
#include <kapp.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
#include <kurldrag.h>

// Our includes
#include "thumbnailloadjob.h"
#include "filethumbnailview.moc"

static const char* CONFIG_THUMBNAIL_SIZE="thumbnail size";
static const char* CONFIG_MARGIN_SIZE="margin size";
static const char* CONFIG_WORD_WRAP_FILENAME="word wrap filename";




/**
 * We override the QIconViewItem to control the look of selected items
 */
class ThumbnailViewItem : public QIconViewItem {
public:
	ThumbnailViewItem(QIconView* parent,const QString& text,const QPixmap& icon)
	: QIconViewItem(parent,text,icon) {}
protected:
	void paintItem(QPainter* painter, const QColorGroup& colorGroup) {
		if (!isSelected()) {
			QIconViewItem::paintItem(painter,colorGroup);
			return;
		}
		
	// Paint icon
		painter->drawPixmap(
			x()+(width()-pixmap()->width())/2,
			y(),
			*pixmap());

	// Paint focus border
		QRect pRect=pixmapRect(false);
		QRect tRect=textRect(false);
	
		painter->save();
		painter->setPen( QPen( colorGroup.highlight() ) );
		painter->drawRect(pRect | tRect);

	// Paint text
		painter->fillRect( pRect.x(),tRect.y(),pRect.width(),tRect.height(), colorGroup.highlight() );
		painter->setPen( QPen( colorGroup.highlightedText() ) );

		int align = AlignHCenter;
		QString itemText;
		if (iconView()->wordWrapIconText() ) {
			align |= WordBreak;
			itemText=text();
		} else {
			itemText=cutText();
		}
		painter->drawText(
			textRect(false),
			align,
			itemText );
		painter->restore();

	}
	
	
	QString cutText() {
		QString itemText=text();
		int width=iconView()->maxItemWidth()-4;
		QFontMetrics metrics=iconView()->fontMetrics();
		
		if (metrics.width(itemText)<width) {
			return itemText;
		}

		QString tmpText="...";
		for(int pos=0;metrics.width(tmpText + itemText[pos]) < width;++pos) {
			tmpText+=itemText[pos];
		}
		tmpText.remove( 0, 3);
		tmpText += "...";
		return tmpText;
	}
	
	
	void paintFocus(QPainter*, const QColorGroup&) {
	}
};





FileThumbnailView::FileThumbnailView(QWidget* parent)
: QIconView(parent), KFileView(), mThumbnailLoadJob(0L)
{
	setAutoArrange(true);
	setItemsMovable(false);
	setResizeMode(Adjust);
	setShowToolTips(true);
	setSpacing(0);
	//QIconView::setSelectionMode(Extended); // FIXME : Find a way to change which item is current on multi-select before enabling this
}


FileThumbnailView::~FileThumbnailView() {
    stopThumbnailUpdate();
}


void FileThumbnailView::insertItem(KFileViewItem* item) {
	if (!item) return;
	int pixelSize=mThumbnailSize.pixelSize();

// Create an empty thumbnail
	QPixmap thumbnail(pixelSize,pixelSize);
	QPainter painter(&thumbnail);
	painter.eraseRect(0,0,pixelSize,pixelSize);
	painter.drawRect(0,0,pixelSize,pixelSize);

// Create item
	QIconViewItem* iconItem=new ThumbnailViewItem(this,item->text(),thumbnail);
	item->setViewItem(this,iconItem);
}


void FileThumbnailView::ensureItemVisible(const KFileViewItem* fileItem) {
	if (!fileItem) return;

	QIconViewItem* iconItem=(QIconViewItem*)( fileItem->viewItem(this) );
	QIconView::ensureItemVisible(iconItem);
}


void FileThumbnailView::setSelected(const KFileViewItem* fileItem,bool enable) {
	if (!fileItem) return;

	QIconViewItem* iconItem=(QIconViewItem*)( fileItem->viewItem(this) );
	QIconView::setSelected(iconItem,enable);
}


bool FileThumbnailView::isSelected(const KFileViewItem* fileItem) const {
	if (!fileItem) return false;

	QIconViewItem* current=currentItem();
	if (!current) {
		return false;
	}
	return current->text()==fileItem->text();
}


void FileThumbnailView::removeItem(const KFileViewItem* fileItem) {
	if (!fileItem) return;

// Remove it from the image preview job
	if (!mThumbnailLoadJob.isNull())
		mThumbnailLoadJob->itemRemoved(fileItem);

// Remove it from our view
	const QIconViewItem* iconItem=static_cast<const QIconViewItem*>( fileItem->viewItem(this) );
	delete iconItem;
	KFileView::removeItem(fileItem);
	arrangeItemsInGrid();
}


void FileThumbnailView::clearView() {
	for (KFileViewItem* item = KFileView::firstItem();item; item=item->next()) {
		item->setViewItem(this,0L);
	}
	QIconView::clear();
}


void FileThumbnailView::setThumbnailSize(ThumbnailSize value) {
	if (value==mThumbnailSize) return;
	mThumbnailSize=value;
	setGridX(mThumbnailSize.pixelSize() + mMarginSize);
}


void FileThumbnailView::setMarginSize(int value) {
	if (value==mMarginSize) return;
	mMarginSize=value;
	setGridX(mThumbnailSize.pixelSize() + mMarginSize);
}


void FileThumbnailView::setThumbnailPixmap(const KFileViewItem* fileViewItem,const QPixmap& thumbnail) {
	QIconViewItem* item=static_cast<QIconViewItem*>( const_cast<void*>(fileViewItem->viewItem(this)) );
	int pixelSize=mThumbnailSize.pixelSize();

// Draw the thumbnail to the center of the icon
	QPainter painter(item->pixmap());
	painter.eraseRect(0,0,pixelSize,pixelSize);
	painter.drawPixmap(
		(pixelSize-thumbnail.width())/2,
		(pixelSize-thumbnail.height())/2,
		thumbnail);
	item->repaint();

// Notify others that one thumbnail has been updated
	emit updatedOneThumbnail();
}


void FileThumbnailView::startThumbnailUpdate()
{
	stopThumbnailUpdate(); // just in case

	mThumbnailLoadJob = new ThumbnailLoadJob(this->items(),mThumbnailSize);
	connect(mThumbnailLoadJob, SIGNAL(thumbnailLoaded(const KFileViewItem*,const QPixmap&)),
		this, SLOT(setThumbnailPixmap(const KFileViewItem*,const QPixmap&)) );
	connect(mThumbnailLoadJob, SIGNAL(result(KIO::Job*)),
		this, SIGNAL(updateEnded()) );
	emit updateStarted(QIconView::count());
	mThumbnailLoadJob->start();
}


void FileThumbnailView::stopThumbnailUpdate()
{
	if (!mThumbnailLoadJob.isNull()) {
		emit updateEnded();
		mThumbnailLoadJob->kill();
	}
}


//-Browsable methods--------------------------------------------------------
/*void dumpItem(const QIconViewItem* item) {
	QString selected=item->isSelected()?"x":" ";
	kdWarning() << "  [" << selected << "] " << item->text() << "\n";
}

void dumpItemList(const QString& name,const QIconView* list) {
	kdWarning() << "Dumping content of " << name << "\n";

	const QIconViewItem* item;
	for (item=list->firstItem();item!=NULL;item=item->nextItem()) {
		dumpItem(item);
	}
}*/

void FileThumbnailView::selectFirst() {
	QIconViewItem* item=QIconView::firstItem();
	if (!item) return;

	QIconView::setCurrentItem(item);
	QIconView::setSelected(item,true);
	QIconView::ensureItemVisible(item);
}


void FileThumbnailView::selectLast() {
	QIconViewItem* item=lastItem();
	if (!item) return;

	QIconView::setCurrentItem(item);
	QIconView::setSelected(item,true);
	QIconView::ensureItemVisible(item);
}


void FileThumbnailView::selectPrevious() {
	QIconViewItem* item=currentItem();
	if (!item) return;

	item=item->prevItem();
	if (!item) return;

	QIconView::setCurrentItem(item);
	QIconView::setSelected(item,true);
	QIconView::ensureItemVisible(item);
}


void FileThumbnailView::selectNext() {
	QIconViewItem* item=currentItem();
	if (!item) return;

	item=item->nextItem();
	if (!item) return;

	QIconView::setCurrentItem(item);
	QIconView::setSelected(item,true);
	QIconView::ensureItemVisible(item);
}


QString FileThumbnailView::previousItemText() const {
	QIconViewItem* item=currentItem();
	if (!item) return "";

	item=item->prevItem();
	if (!item) return "";
	return item->text();
}


QString FileThumbnailView::currentItemText() const {
	QIconViewItem* item=currentItem();
	if (!item) return "";
	return item->text();
}


QString FileThumbnailView::nextItemText() const {
	QIconViewItem* item=currentItem();
	if (!item) return "";

	item=item->nextItem();
	if (!item) return "";
	return item->text();
}


bool FileThumbnailView::currentIsFirst() const {
	return currentItem()==QIconView::firstItem();
}


bool FileThumbnailView::currentIsLast() const {
	return currentItem()==lastItem();
}


//-Protected----------------------------------------------------------------
void FileThumbnailView::startDrag() {
	KFileViewItem* item=selectedItems()->getFirst();
	if (!item) {
		kdWarning() << "No item to drag\n";
		return;
	}
	KURL::List urls;
	urls.append(item->url());
	QUriDrag* drag = KURLDrag::newDrag( urls, this );
	drag->setPixmap( item->pixmap() );
	drag->dragCopy();
}


//-Configuration------------------------------------------------------------
void FileThumbnailView::readConfig(KConfig* config,const QString& group) {
	config->setGroup(group);

	mThumbnailSize=config->readEntry(CONFIG_THUMBNAIL_SIZE);
	mMarginSize=config->readNumEntry(CONFIG_MARGIN_SIZE,5);

	setGridX(mThumbnailSize.pixelSize() + mMarginSize);
	setWordWrapIconText(config->readBoolEntry(CONFIG_WORD_WRAP_FILENAME,false));
	arrangeItemsInGrid();
}


void FileThumbnailView::writeConfig(KConfig* config,const QString& group) const {
	config->setGroup(group);
	config->writeEntry(CONFIG_THUMBNAIL_SIZE,QString(mThumbnailSize));
	config->writeEntry(CONFIG_MARGIN_SIZE,mMarginSize);
	config->writeEntry(CONFIG_WORD_WRAP_FILENAME,wordWrapIconText());
}
