//$Id: menu-win.cc,v 1.19 2001/11/25 19:50:19 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "menu-win.h"

#include "config.h"
#include <libgnome/libgnome.h>

#ifdef GUIKACHU_USE_WM_ICONS
#include <libgnomeui/gnome-window-icon.h>
#endif

#include <sigc++/retbind.h>
#include "argumentsink.h"

using namespace Guikachu;
using std::vector;
using std::string;

GUI::MenuWindow::MenuWindow (Guikachu::Resources::Menu *res_):
    res (res_),
    update_block (false)
{
    gui = glade_xml_new (GLADEDIR "/guikachu.glade", "res_menu");

    using Gtk::wrap;
    using namespace SigC;
    
    window = wrap (GTK_WINDOW (glade_xml_get_widget (gui, "res_menu")));
    window->delete_event.connect (slot (this, &MenuWindow::delete_event_impl));

#ifdef GUIKACHU_USE_WM_ICONS
    gnome_window_icon_set_from_file (window->gtkobj (), GNOME_ICONDIR "/guikachu.png");
#endif    

    id_entry = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "menu_id")));

    // Menu items list
    clist = wrap (GTK_CLIST (glade_xml_get_widget (gui, "menu_tree")));
    btnUp = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_up")));
    btnDown = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_down")));
    btnRemove = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_remove")));

    // New item buttons
    btnSubmenu = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_submenu")));
    btnItem = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_item")));
    btnSeparator = wrap (GTK_BUTTON (glade_xml_get_widget (gui, "menu_btn_separator")));

    // Menu item properties
    item_label = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "menuitem_label")));
    lbl_item_label = wrap (GTK_LABEL (glade_xml_get_widget (gui, "menuitem_label_label")));

    item_id = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "menuitem_id")));
    lbl_item_id = wrap (GTK_LABEL (glade_xml_get_widget (gui, "menuitem_id_label")));

    item_shortcut = wrap (GTK_ENTRY (glade_xml_get_widget (gui, "menuitem_shortcut")));
    lbl_item_shortcut = wrap (GTK_LABEL (glade_xml_get_widget (gui, "menuitem_shortcut_label")));

    id_entry->activate.connect (slot (this, &MenuWindow::id_entry_cb));
    id_entry->focus_out_event.connect (
	hide<int, GdkEventFocus*> (retbind (slot (this, &MenuWindow::id_entry_cb),0)));

    item_id->activate.connect (slot (this, &MenuWindow::item_id_entry_cb));
    item_id->focus_out_event.connect (
	hide<int, GdkEventFocus*> (retbind (slot (this, &MenuWindow::item_id_entry_cb),0)));
    item_label->changed.connect (slot (this, &MenuWindow::item_label_entry_cb));
    item_shortcut->changed.connect (slot (this, &MenuWindow::item_shortcut_entry_cb));

    btnUp->clicked.connect (slot (this, &MenuWindow::btn_up_cb));
    btnDown->clicked.connect (slot (this, &MenuWindow::btn_down_cb));
    btnRemove->clicked.connect (slot (this, &MenuWindow::btn_remove_cb));

    btnSubmenu->clicked.connect (slot (this, &MenuWindow::btn_submenu_cb));
    btnItem->clicked.connect (bind (slot (this, &MenuWindow::btn_item_cb), false));
    btnSeparator->clicked.connect (bind (slot (this, &MenuWindow::btn_item_cb), true));
    
    clist->select_row.connect (slot (this, &MenuWindow::clist_sel_row));
    
    res->changed.connect (slot (this, &MenuWindow::update));
    update ();
}

GUI::MenuWindow::~MenuWindow ()
{
//    window->destroy ();
    gtk_widget_destroy (GTK_WIDGET (window->gtkobj ()));
    gtk_object_unref (GTK_OBJECT (gui));
}

int GUI::MenuWindow::delete_event_impl (GdkEventAny *e)
{
    window->hide ();
    return true;
}

void GUI::MenuWindow::show ()
{
    window->show_all ();
    window->get_window ().raise ();
}

void GUI::MenuWindow::update ()
{
    update_block = true;
    
    gchar *title_buf = g_strdup_printf (_("Menu: %s"), res->id ().c_str ());
    window->set_title (title_buf);
    g_free (title_buf);

    id_entry->set_text (res->id);

    submenus = res->get_submenus ();

    int selected_row = 0;
    if (clist->selection ().size ())
	selected_row = clist->selection ().begin ()->get_row_num ();

    clist->clear ();
    clist_rows.clear ();
    reset_controls ();
    
    unsigned int submenu_num = 0, menuitem_num = 0;
    
    for (vector<Resources::Submenu>::iterator i = submenus.begin ();
	 i != submenus.end (); i++, submenu_num++)
    {
	RowData row_data;
	row_data.is_submenu = true;
	row_data.index = submenu_num;
	clist_rows.push_back (row_data);
	
	vector<string> row;
	row.push_back (i->label);
	row.push_back ("");
	row.push_back ("");
	
	clist->rows ().push_back (row);
	
	menuitem_num = 0;
	for (vector<Resources::MenuItem>::iterator j = i->items.begin ();
	     j != i->items.end (); j++, menuitem_num++)
	{
	    vector<string> row;
	    
	    if (!j->separator)
	    {
		gchar *shortcut_str = g_strdup_printf ("%c", j->shortcut);
		
		row.push_back ("    " + j->label);
		row.push_back (j->id);
		row.push_back (shortcut_str);

		g_free (shortcut_str);
		
	    } else {
		
		row.push_back ("    ------");
		row.push_back ("");
		row.push_back ("");
	    }

	    RowData row_data;
	    row_data.is_submenu = false;
	    row_data.index = menuitem_num;
	    row_data.parent_index = submenu_num;
	    clist_rows.push_back (row_data);

	    clist->rows ().push_back (row);
	}
    }

    clist->select_row (selected_row);

    update_block = false;
}

void GUI::MenuWindow::reset_controls ()
{
    btnDown->set_sensitive (false);
    btnRemove->set_sensitive (false);
    btnUp->set_sensitive (false);

    btnItem->set_sensitive (false);
    btnSeparator->set_sensitive (false);
    
    item_label->set_text ("");
    item_label->set_sensitive (false);
    lbl_item_label->set_sensitive (false);

    item_id->set_text ("");
    item_id->set_sensitive (false);
    lbl_item_id->set_sensitive (false);

    item_shortcut->set_text ("");
    item_shortcut->set_sensitive (false);
    lbl_item_shortcut->set_sensitive (false);
}

void GUI::MenuWindow::clist_sel_row (gint row, gint column,
				     GdkEvent *event)
{
    update_block = true;
    
    RowData& row_data = clist_rows[row];

    if (clist->selection ().size () != 1)
	return;

    reset_controls ();
    
    btnRemove->set_sensitive (true);
    btnItem->set_sensitive (true);
    btnSeparator->set_sensitive (true);
    
    if (row_data.is_submenu)
    {
	// This is a submenu
	
	Resources::Submenu& submenu = submenus[row_data.index];

	btnUp->set_sensitive (row_data.index != 0);
	btnDown->set_sensitive (row_data.index < (submenus.size () - 1));

	item_label->set_text (submenu.label);
	item_label->set_sensitive (true);
	lbl_item_label->set_sensitive (true);

    } else {
	// This is a menu item

	Resources::Submenu& submenu = submenus[row_data.parent_index];
	Resources::MenuItem& menuitem = submenu.items[row_data.index];

	btnUp->set_sensitive (row_data.index != 0);
	btnDown->set_sensitive (row_data.index < (submenu.items.size () - 1));
	
	if (!menuitem.separator)
	{
	    item_label->set_text (menuitem.label);
	    item_label->set_sensitive (true);
	    lbl_item_label->set_sensitive (true);
	    
	    item_id->set_text (menuitem.id);
	    item_id->set_sensitive (true);
	    lbl_item_id->set_sensitive (true);

	    gchar *shortcut_str = g_strdup_printf ("%c", menuitem.shortcut);
	    item_shortcut->set_text (shortcut_str);
	    item_shortcut->set_sensitive (true);
	    lbl_item_shortcut->set_sensitive (true);
	    g_free (shortcut_str);
	}
    }

    update_block = false;
}

void GUI::MenuWindow::id_entry_cb ()
{
    if (update_block)
	return;

    update_block = true;
    res->id = id_entry->get_text ();
    update_block = false;

    id_entry->set_text (res->id);
}

void GUI::MenuWindow::item_id_entry_cb ()
{
    if (update_block)
	return;
    
    if (clist->selection ().size () != 1)
	return;

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    if (clist_rows[row].is_submenu)
	return;

    vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
    items[index].id = item_id->get_text ();

    res->set_submenus (submenus);
}

void GUI::MenuWindow::item_label_entry_cb ()
{
    if (update_block)
	return;
    
    if (clist->selection ().size () != 1)
	return;

    int saved_pos = item_label->get_position ();
    
    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    if (clist_rows[row].is_submenu)
    {
	submenus[index].label = item_label->get_text ();

	res->set_submenus (submenus);

    } else {

	vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
	items[index].label = item_label->get_text ();
	
	res->set_submenus (submenus);
    }

    item_label->set_position (saved_pos);
    item_label->grab_focus ();
}

void GUI::MenuWindow::item_shortcut_entry_cb ()
{
    if (update_block)
	return;
    
    if (clist->selection ().size () != 1)
	return;

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    if (clist_rows[row].is_submenu)
	return;

    vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
    string shortcut_str = item_shortcut->get_text ();

    if (shortcut_str == "")
	items[index].shortcut = 0;
    else
	items[index].shortcut = shortcut_str[0];

    res->set_submenus (submenus);
    
    item_shortcut->grab_focus ();
}

void GUI::MenuWindow::btn_up_cb ()
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;
    g_return_if_fail (index != 0);
    
    if (clist_rows[row].is_submenu)
    {
	vector<Resources::Submenu>::iterator curr = submenus.begin() + index;
	vector<Resources::Submenu>::iterator prev = curr - 1;
	int offset = prev->items.size() + 1;

	iter_swap (curr, prev);
	res->set_submenus (submenus);

	clist->select_row (row - offset);
	
    } else {
	
	vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
	vector<Resources::MenuItem>::iterator curr = items.begin() + index;
	vector<Resources::MenuItem>::iterator prev = curr - 1;

	iter_swap (curr, prev);
	res->set_submenus (submenus);

	clist->select_row (row - 1);
    }
}

void GUI::MenuWindow::btn_down_cb ()
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    
    if (clist_rows[row].is_submenu)
    {
	g_return_if_fail (index < (submenus.size() - 1));
    
	vector<Resources::Submenu>::iterator curr = submenus.begin() + index;
	vector<Resources::Submenu>::iterator next = curr + 1;
	int offset = next->items.size() + 1;

	iter_swap (curr, next);
	res->set_submenus (submenus);
    
	clist->select_row (row + offset);

    } else {
	vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
	vector<Resources::MenuItem>::iterator curr = items.begin() + index;
	vector<Resources::MenuItem>::iterator next = curr + 1;

	g_return_if_fail (index < (items.size () - 1));
	
	iter_swap (curr, next);
	res->set_submenus (submenus);

	clist->select_row (row + 1);
    }
}

void GUI::MenuWindow::btn_remove_cb ()
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;

    if (clist_rows[row].is_submenu)
    {
	submenus.erase (submenus.begin () + index);
	res->set_submenus (submenus);
	
    } else {

	vector<Resources::MenuItem>& items = submenus[clist_rows[row].parent_index].items;
	items.erase (items.begin () + index);

	res->set_submenus (submenus);	
    }
}

void GUI::MenuWindow::btn_submenu_cb ()
{
    if (clist->selection ().size () == 0)
    {
	Resources::Submenu submenu (_("New submenu"));
	submenus.push_back (submenu);

	res->set_submenus (submenus);

	item_label->grab_focus ();

	return;
    }

    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;
    vector<Resources::Submenu>::iterator new_index;
    Resources::Submenu submenu (_("New submenu"));
    int new_row;
    
    if (clist_rows[row].is_submenu)
    {
	new_index = submenus.begin () + index + 1;
	new_row = row + submenus[clist_rows[row].index].items.size () + 1;
    } else {
	new_index = submenus.begin () + clist_rows[row].parent_index + 1;

	new_row = row - index +
	    submenus[clist_rows[row].parent_index].items.size ();
    }

    submenus.insert (new_index, submenu);
    
    clist->select_row (new_row);
    res->set_submenus (submenus);
    clist->select_row (new_row);

    item_label->grab_focus ();
}

void GUI::MenuWindow::btn_item_cb (bool separator)
{
    g_return_if_fail (clist->selection ().size () == 1);

    unsigned int row = clist->selection ().begin ()->get_row_num ();
    unsigned int index = clist_rows[row].index;
    unsigned int parent_index = clist_rows[row].parent_index;
    vector<Resources::MenuItem>::iterator pos;
    unsigned int new_row;
    
    Resources::MenuItem menuitem;
    menuitem.separator = separator;
    
    if (!separator)
    {
	menuitem.id = ResourceManager::create_id ("MenuItem_");
	menuitem.label = menuitem.id;
	menuitem.shortcut = 0;
    }
    
    if (clist_rows[row].is_submenu)
    {
	pos = submenus[index].items.end ();	
	submenus[index].items.insert (pos, menuitem);
	new_row = row + submenus[index].items.size ();
	
    } else {
	
	pos = submenus[parent_index].items.begin () + index;
	submenus[parent_index].items.insert (pos, menuitem);
	new_row = row;
    }

    clist->select_row (new_row);
    res->set_submenus (submenus);

    item_id->grab_focus ();
}
