/*
 * Guifications - The end all, be all, toaster popup plugin
 * Copyright (C) 2003-2005 Gary Kramlich
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <glib.h>
#include <gdk/gdk.h>

#include <debug.h>
#include <xmlnode.h>


#include "gf_event_info.h"
#include "gf_gtk_utils.h"
#include "gf_item.h"
#include "gf_notification.h"
#include "gf_theme.h"

struct _GfItemImage {
	GfItem *item;
	gchar *filename;
};

#include "gf_item_image.h"

void
gf_item_image_destroy(GfItemImage *item_image) {
	g_return_if_fail(item_image);

	item_image->item = NULL;

	if(item_image->filename) {
		g_free(item_image->filename);
		item_image->filename = NULL;
	}

	g_free(item_image);
	item_image = NULL;
}

GfItemImage *
gf_item_image_new(GfItem *item) {
	GfItemImage *item_image;

	g_return_val_if_fail(item, NULL);

	item_image = g_new0(GfItemImage, 1);
	item_image->item = item;

	return item_image;
}

GfItemImage *
gf_item_image_new_from_xmlnode(GfItem *item, xmlnode *node) {
	GfItemImage *item_image;

	g_return_val_if_fail(item, NULL);
	g_return_val_if_fail(node, NULL);

	item_image = gf_item_image_new(item);

	item_image->filename = g_strdup(xmlnode_get_attrib(node, "filename"));
	if(!item_image) {
		purple_debug_info("Guifications",
						"** Error loading image item: 'Unknown filename'\n");
		gf_item_image_destroy(item_image);
		return NULL;
	}

	return item_image;
}

GfItemImage *
gf_item_image_copy(GfItemImage *image) {
	GfItemImage *new_image;

	g_return_val_if_fail(image, NULL);

	new_image = gf_item_image_new(image->item);

	if(image->filename)
		new_image->filename = g_strdup(image->filename);

	return new_image;
}

xmlnode *
gf_item_image_to_xmlnode(GfItemImage *image) {
	xmlnode *parent;

	parent = xmlnode_new("image");
	xmlnode_set_attrib(parent, "filename", image->filename);

	return parent;
}

void
gf_item_image_set_item(GfItemImage *item_image, GfItem *item) {
	g_return_if_fail(item_image);
	g_return_if_fail(item);

	item_image->item = item;
}

GfItem *
gf_item_image_get_item(GfItemImage *item_image) {
	g_return_val_if_fail(item_image, NULL);

	return item_image->item;
}

void
gf_item_image_set_image(GfItemImage *item_image, const gchar *image) {
	g_return_if_fail(item_image);
	g_return_if_fail(image);

	item_image->filename = g_strdup(image);
}

const gchar *
gf_item_image_get_image(GfItemImage *item_image) {
	g_return_val_if_fail(item_image, NULL);

	return item_image->filename;
}

void
gf_item_image_render(GfItemImage *item_image, GdkPixbuf *pixbuf,
					 GfEventInfo *info)
{
	GfNotification *notification;
	GfTheme *theme;
	GdkPixbuf *image;
	gint x, y;
	gint width, height;
	gchar *filename;

	g_return_if_fail(item_image);
	g_return_if_fail(pixbuf);
	g_return_if_fail(info);

	notification = gf_item_get_notification(item_image->item);
	theme = gf_notification_get_theme(notification);
	filename = g_build_filename(gf_theme_get_path(theme),
								item_image->filename, NULL);
	image = gdk_pixbuf_new_from_file(filename, NULL);
	g_free(filename);

	if(!image)
		return;

	width = gdk_pixbuf_get_width(image);
	height = gdk_pixbuf_get_height(image);

	gf_item_get_render_position(&x, &y,
								width, height,
								gdk_pixbuf_get_width(pixbuf),
								gdk_pixbuf_get_height(pixbuf),
								item_image->item);

	gf_gtk_pixbuf_clip_composite(image, x, y, pixbuf);

	g_object_unref(G_OBJECT(image));
}
