/* GTS - Library for the manipulation of triangulated surfaces
 * Copyright (C) 1999 Stphane Popinet
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include "gts.h"


/* set - compute set operations between surfaces */
int main (int argc, char * argv[])
{
  GtsSurface * s1, * s2, * s3;
  GtsSurfaceInter * si;
  GNode * tree1, * tree2;
  FILE * fp;
  guint line;

  if (argc != 4) {
    fprintf (stderr, 
	     "%s: compute set operations between surfaces\n"
	     "usage: %s OPERATION FILE1 FILE2\n"
	     "where OPERATION is either union, inter, diff\n",
	     argv[0], argv[0]);
    return 1;
  }

  /* open first file */
  if ((fp = fopen (argv[2], "rt")) == NULL) {
    fprintf (stderr, "%s: can not open file `%s'\n", argv[0], argv[2]);
    return 1;
  }
  /* reads in first surface file */
  s1 = GTS_SURFACE (gts_object_new (GTS_OBJECT_CLASS (gts_surface_class ())));
  if ((line = gts_surface_read (s1, fp))) {
    fprintf (stderr, "%s: `%s' is not a valid GTS surface file\n", 
	     argv[0], argv[2]);
    fprintf (stderr, "Format error at line %d\n", line);
    return 1;
  }
  fclose (fp);

  /* open second file */
  if ((fp = fopen (argv[3], "rt")) == NULL) {
    fprintf (stderr, "%s: can not open file `%s'\n", argv[0], argv[3]);
    return 1;
  }
  /* reads in second surface file */
  s2 = GTS_SURFACE (gts_object_new (GTS_OBJECT_CLASS (gts_surface_class ())));
  if ((line = gts_surface_read (s2, fp))) {
    fprintf (stderr, "%s: `%s' is not a valid GTS surface file\n", 
	     argv[0], argv[3]);
    fprintf (stderr, "Format error at line %d\n", line);
    return 1;
  }
  fclose (fp);

  /* display summary information about both surfaces */
  gts_surface_print_stats (s1, stderr);
  gts_surface_print_stats (s2, stderr);

  /* check that the surfaces are orientable manifolds */
  if (!gts_surface_is_orientable (s1)) {
    fprintf (stderr, "surface `%s' is not an orientable manifold\n",
	     argv[2]);
    return 1;
  }
  if (!gts_surface_is_orientable (s2)) {
    fprintf (stderr, "surface `%s' is not an orientable manifold\n",
	     argv[3]);
    return 1;
  }

  /* build bounding box tree for first surface */
  tree1 = gts_bb_tree_surface (s1);

  /* build bounding box tree for second surface */
  tree2 = gts_bb_tree_surface (s2);

  si = gts_surface_inter_new (gts_surface_inter_class (),
			      s1, s2, tree1, tree2);
  g_assert (gts_surface_inter_check (si));
  s3 = gts_surface_new (gts_surface_class (),
			gts_face_class (),
			gts_edge_class (),
			gts_vertex_class ());  
  if (!strcmp (argv[1], "union")) {
    gts_surface_inter_boolean (si, s3, GTS_1_OUT_2);
    gts_surface_inter_boolean (si, s3, GTS_2_OUT_1);
  }
  else if (!strcmp (argv[1], "inter")) {
    gts_surface_inter_boolean (si, s3, GTS_1_IN_2);
    gts_surface_inter_boolean (si, s3, GTS_2_IN_1);
  }
  else if (!strcmp (argv[1], "diff")) {
    gts_surface_inter_boolean (si, s3, GTS_1_OUT_2);
    gts_surface_inter_boolean (si, s3, GTS_2_IN_1);
    gts_surface_foreach_face (si->s2, (GtsFunc) gts_triangle_revert, NULL);
    gts_surface_foreach_face (s2, (GtsFunc) gts_triangle_revert, NULL);
  }
  else {
    fprintf (stderr, "%s: operation `%s' unknown\n", argv[0], argv[1]);
    return 1;
  }
  
  /* display summary information about the resulting surface */
  gts_surface_print_stats (s3, stderr);
  /* write resulting surface to standard output */
  gts_surface_write (s3, stdout);

  /* destroy surfaces */
  gts_object_destroy (GTS_OBJECT (s1));
  gts_object_destroy (GTS_OBJECT (s2));
  gts_object_destroy (GTS_OBJECT (s3));
  gts_object_destroy (GTS_OBJECT (si));

  /* destroy bounding box trees (including bounding boxes) */
  gts_bb_tree_destroy (tree1, TRUE);
  gts_bb_tree_destroy (tree2, TRUE);  

  return 0;
}
