/* The Type of Track representing the Fs edit stuff */

#include <sys/wait.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include "int.h"

#include "tracks.h"
#include "piping.h"
#include "preferences.h"
#include "varmanwidgets.h"
#include "isotrack.h"
#include "colors.h"
#include "fsedit.h"
#include "updatehandlers.h"
#include "multisession.h"
#include "main.h"
#include "calc.h"
#include "stdfiletrack.h"
#include "cddrives.h"
#include "dependencies.h"

#define isotrack_getprecachingflag \
     (!strcmp(varman_getvar(global_defs,"isotrack_precache"),"true")) \
      *atwrite

void isotrack_updatehandler(tracks_trackinfo *ti)
{
   updatehandlers_call(ti->updatehandlers);
}
;

/* configure variables for running mkisofs with  current settings */
void isotrack_configuremkisofsoptions()
{
   /* if bootable support was selected */
   if (!strcasecmp(varman_getvar(global_defs,"isotrack_makebootable"),
		   "true"))
     {
	char *eltorito;

	/* create boot options and store them as a dynamic var
	 * (dynamic vars ==> not stored to the config file */
	eltorito=varman_getvar_copy(global_defs,"isotrack_eltorito");
	varman_replacevars(global_defs,eltorito);
	varman_replacevars(dynamic_defs,eltorito);
	varman_setvar(dynamic_defs,"$eltorito",eltorito);
	free(eltorito);
     }
   else
     /* do not specify eltorito options */
     varman_setvar(dynamic_defs,"$eltorito","");

   /* when creating a multisession cd and a valid recorder could be found */
   if ((multisession_lastsession!=NULL)&&(cddrives_getrecorder()!=NULL))
     {
	char *msession;

	/* create multisession options and store them as a dynamic var
	 * (dynamic vars ==> not stored to the config file */
	msession=varman_getvar_copy(global_defs,"isotrack_msoption");
	varman_replacestring(msession,"$filesystem",
			     cddrives_getrecorder()->device);
	varman_replacevars(global_defs,msession);
	varman_replacevars(dynamic_defs,msession);
	varman_setvar(dynamic_defs,"$msession",msession);
	free(msession);
     }
   else
     /* do not specify multisession options */
     varman_setvar(dynamic_defs,"$msession","");

};

int isotrack_openpipe(void*trackinfo)
{
   int outp=-1;
   char *call;
   isotrack_info *i;

   i=(isotrack_info*)((tracks_trackinfo*)trackinfo)->data;

   isotrack_configuremkisofsoptions();

   /* this option is always "" when track is actually opened for reading,
    * it's used by tracks->tracksize to detect the accurate tracksize of
    * an iso track */
   varman_setvar(dynamic_defs,"$getts","");

   call=varman_getvar_copy(global_defs,
			   "isotrack_client");
   varman_replacevars(global_defs,call);
   varman_replacevars(dynamic_defs,call);
   i->mkisofspid=piping_create(call,NULL,&outp,NULL);
   i->mkisofspipe=outp;
   free(call);
   return outp;
}
;

void isotrack_closepipe(void*trackinfo)
{
   isotrack_info *i;

   i=(isotrack_info*)((tracks_trackinfo*)trackinfo)->data;
   if (i->mkisofspipe!=-1)
     close(i->mkisofspipe);
   waitpid(i->mkisofspid,&i->mkisofsstatus,0); // wait for client to finish
}
;

int isotrack_tracksize(void*trackinfo)
{
   char *call;
   int  result;

   /* if mkisofs is to be used to determine tracksize */
   if (!strcasecmp(varman_getvar(global_defs,"isotrack_detwithcreate"),
		   "true"))
     {
	char *tsopt;

	isotrack_configuremkisofsoptions();

	/* create tsdetect options and store them as a dynamic var
	 * (dynamic vars ==> not stored to the config file */
        tsopt=varman_getvar_copy(global_defs,"isotrack_detwithoption");
	varman_replacevars(global_defs,tsopt);
	varman_replacevars(dynamic_defs,tsopt);
	varman_setvar(dynamic_defs,"$getts",tsopt);
	free(tsopt);

	call=varman_getvar_copy(global_defs,
				"isotrack_client");
     }
   else
     call=varman_getvar_copy(global_defs,"isotrack_tsclient");

   varman_replacevars(global_defs,call);
   varman_replacevars(dynamic_defs,call);

   result=piping_create_scantextoutput(call,
				       varman_getvar(global_defs,"isotrack_tscalc"),
				       PIPING_WATCHSTDERR);
   /* Quick hack! Call again watching stdout this time.
    * For other mkisofs versions */
   if (!result)
     result=piping_create_scantextoutput(call,
					 varman_getvar(global_defs,"isotrack_tscalc"),				  
					 PIPING_WATCHSTDOUT);
   return result;
};

int isotrack_dataavail(void*trackinfo)
{
   isotrack_info *i;
   int dataavail;

   i=(isotrack_info*)((tracks_trackinfo*)trackinfo)->data;
   dataavail=1;
   if (waitpid(i->mkisofspid,&i->mkisofsstatus,WNOHANG)!=0)
     dataavail=0;

   return dataavail;
}
;

int  isotrack_valid(void*trackinfo)  /* isotrack is always valid */
{
   char *call=varman_getvar_copy(global_defs,"isotrack_client");
   int result;
   isotrack_configuremkisofsoptions();
   varman_replacevars(global_defs,call);
   varman_replacevars(dynamic_defs,call);
   result=piping_isvalid_command(call);
   if (!result)
     dependencies_showmissingexec(call);
   free(call);
   return result;
}
;

void isotrack_toggleprecaching(tracks_trackinfo *it)
{
   it->precacherequired=isotrack_getprecachingflag;
};

void isotrack_pc_policy_changed(VARMAN_VAR_ENTRY *var,gpointer data)
{
   isotrack_toggleprecaching((tracks_trackinfo*)data);
};

void isotrack_destroy(void*trackinfo)
{
   isotrack_info *i;

   i=(isotrack_info*)((tracks_trackinfo*)trackinfo)->data;

   updatehandlers_unregister(&fsedit_updatehandlers,
			     (updatehandlers_handler)isotrack_updatehandler,
			     (gpointer)trackinfo);

   varman_remove_handler(global_defs,"isotrack_precache",
			 isotrack_pc_policy_changed,
			 trackinfo);

   if ((isotrack_dataavail(trackinfo))&&(i->mkisofspipe!=-1))
     close (i->mkisofspipe);

   /* delete directory structure */
   virtualdir_deletestructure(i->root);

   free(i);
}
;

tracks_trackinfo *isotrack_create()
{
   tracks_trackinfo *it;
   isotrack_info *info;

   info=(isotrack_info*)malloc(sizeof(isotrack_info));

   /* create the isofs root directory */
   info->root=virtualdir_create(varman_getvar(global_defs,
					      "fsedit_volid"),
				0,0,NULL);

   it=tracks_create(_("Gnometoaster-Filesystem (ISO)"),
		    
		    varman_getvar(global_defs,"fsedit_volid"),
		    _("Gnome Toaster"),
		    
		    "data",isotrack_getprecachingflag,
		    isotrack_openpipe,
		    isotrack_closepipe,
		    isotrack_tracksize,
		    isotrack_dataavail,
		    isotrack_valid,
		    isotrack_destroy,
		    info);

   /* FIXME: to allow for more than one *edited* iso tracks in
    * one tracklist, we have to replace this by some dynamic edit bindings
    * one day */
   updatehandlers_register(&fsedit_updatehandlers,
			   (updatehandlers_handler)isotrack_updatehandler,
			   (gpointer)it);

   varman_install_handler(global_defs,"isotrack_precache",
			  isotrack_pc_policy_changed,
			  it);

   return it;
}
;

/* this is called by main.c to do some init stuff,e.g. creating the preferences
 * entry */
void isotrack_init()
{
   GtkWidget *isotrack_prefs;
   GtkWidget *label;

   varmanwidgets_widget *itc;
   varmanwidgets_widget *itc_ms;
   varmanwidgets_widget *eltorito;
   varmanwidgets_widget *detwithcreate;
   varmanwidgets_widget *detwithoption;
   varmanwidgets_widget *tracksizedet;
   varmanwidgets_widget *tracksizecalc;
        /* for mounting/umounting iso images we use an exec call to the
	 * os's mount command instead of the corresponding libc function
	 * as I understand that mounting/umounting a filesystem is
	 * a very operating-specific task,so it should be as configurable
	 * as possible.
	 * Also, the linux libc function requires the
	 * caller to be root no matter
	 * what,while I can exec the mount command as any user if the
	 * specified mount task was marked as user in /etc/fstab */
   varmanwidgets_widget *mountiso;  /* command to call to mount an iso image */
   varmanwidgets_widget *umountiso; /* the same to umount an image */
   varmanwidgets_widget *precache;
   varmanwidgets_widget *tempdir;

   isotrack_prefs=gtk_vbox_new(0,0);
   gtk_widget_show(isotrack_prefs);
   label=gtk_label_new("ISO9660");
   gtk_widget_show(label);
   preferences_append_page(isotrack_prefs,label);

   itc=varmanwidgets_entry_new(_("Create ISO Track with"),
			       "isotrack_client",
			       global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),itc->visual,0,0,0);

   itc_ms=varmanwidgets_entry_new(_("Multisession Options"),
				  "isotrack_msoption",
				  global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),itc_ms->visual,0,0,0);

   eltorito=varmanwidgets_entry_new(_("Bootable Options"),
				    "isotrack_eltorito",
				    global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),eltorito->visual,0,0,0);

   detwithcreate=varmanwidgets_checkbox_new(_("Use create call to determine tracksize"),
					    "isotrack_detwithcreate",
					    global_defs,
					    APPLYMODE_BUTTON,
					    160,
					    "true",
					    "false"
					    );
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),detwithcreate->visual,0,0,0);

   detwithoption=varmanwidgets_entry_new(_("Determine tracksize option"),
					 "isotrack_detwithoption",
					 global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),detwithoption->visual,0,0,0);

   tracksizedet=varmanwidgets_entry_new(_("Determine tracksize with"),
					"isotrack_tsclient",
					global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),tracksizedet->visual,0,0,0);

   tracksizecalc=varmanwidgets_entry_new(_("Calculate tracksize as"),
					 "isotrack_tscalc",
					 global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),tracksizecalc->visual,0,0,0);

   mountiso=varmanwidgets_entry_new(_("Mount an iso image with"),
				    "isotrack_mount",
				    global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),mountiso->visual,0,0,0);

   umountiso=varmanwidgets_entry_new(_("Unmount an iso image with"),
				     "isotrack_umount",
				     global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),umountiso->visual,0,0,0);

   precache=varmanwidgets_checkbox_new(_("Precache before writing"),
				       "isotrack_precache",
				       global_defs,
				       APPLYMODE_BUTTON,
				       160,
				       "true",
				       "false"
				       );
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),precache->visual,0,0,0);

   tempdir=varmanwidgets_entry_new(_("ISO Temporary files"),
				    "isotrack_tempdir",
				    global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(isotrack_prefs),tempdir->visual,0,0,0);   

}
;

