/* this file implements Gnometoaster's native mp3 functionality.
 * Currently, this functionality is limited to calculating the
 * tracksize of mp3 files with special emphasis put on it's ability
 * to deliver the tracksize in bytes of raw audio data in
 * 16bit/44.1khz/stereo and the capability to estimate the playing
 * time of broken mp3 files and variable bitrate mp3s.
 * To accomplish those goals, this function scans through the whole
 * mp3 file which is quite naturally taking longer than other methods
 * of mp3 playing time calculation */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

#include "preferences.h"
#include "helpings.h"

int mpg1_lookup_bitrate_l1[]=
{  32, 64, 96,128,160,192,224,256,288,320,352,384,416,448,0,0 };
int mpg1_lookup_bitrate_l2[]=
{  32, 48, 56, 64, 80, 96,112,128,160,192,224,256,320,384,0,0 };
int mpg1_lookup_bitrate_l3[]=
{  32, 40, 48, 56, 64, 80, 96,112,128,160,192,224,256,320,0,0 };

int *mpg1_lookup_bitrate[]=
{
   mpg1_lookup_bitrate_l1,
   mpg1_lookup_bitrate_l2,
   mpg1_lookup_bitrate_l3     
};

int mpg2_lookup_bitrate_l1[]=
{32,48,56,64,80,96,112,128,144,160,176,192,224,256,0,0};
int mpg2_lookup_bitrate_l2[]=
{8,16,24,32,40,48,56,64,80,96,112,128,144,160,0,0};
int mpg2_lookup_bitrate_l3[]=
{8,16,24,32,40,48,56,64,80,96,112,128,144,160,0,0};  
  
int *mpg2_lookup_bitrate[]=
{
   mpg2_lookup_bitrate_l1,
   mpg2_lookup_bitrate_l2,
   mpg2_lookup_bitrate_l3,
};
   	
int mpg1_lookup_samplingfreq[]=
{
   44100,48000,32000,0
};

int mpg2_lookup_samplingfreq[]=
{
   22050,24000,16000,0
};

char *mode[]=
{
   "stereo","joint stereo","dual channel","single channel"
};     

typedef struct 
{
   unsigned short int valid; /* contains sync bits */
   int version; /* 1 or 2 */
   int layer;
   int has_error_protection;
   int bitrate;
   int freq;
   int has_padding;
   int extension;
   int mode;
   int mode_ext;
   int has_copyright;
   int is_original;
   int emphasis;
   
   int main_data;
} mpeg_header;

/* we are mem mapping the mp3 for simplified handling */
unsigned char *mp3;

/* the following structs and functions define convenient ways 
 * of accessing bit fields within our memory-mapped mp3 file */

/* FIXME: move this into helpings.c */
unsigned int bitswap(unsigned int b)
{
   unsigned int b2=0;
   int x,y;
   unsigned char *c=(unsigned char*)&b;
   unsigned char *c2=(unsigned char*)&b2;
   
   for (y=0;y<4;y++)
     {	
	for (x=0;x<8;x++)
	  {
	     if (c[y]&(1<<x))
	       {
		  c2[y]+=(1<<(7-x));
	       };
	  };
     };
   return b2;
};
   
/* get a bit at a certain position within buf where buf may be one of
 * byte,unsigned short int or unsigned int */
#define bitget(buf,at,count) ((bitswap(buf)>>at)&((1<<count)-1))

/* those are taken from the first valid block,
 * all other blocks are compared against those values to 
 * ensure their validity */
int mp3func_samplingrate=-1;
int mp3func_layer=-1;
int mp3func_version=-1;
int mp3func_s_mode=-1;
int mp3func_has_copyright=-1;
int mp3func_is_original=-1;

/* sync header */
#define SYNC 0xfff

/* FIXME: get_aligned_int has to be used on almost all but intel architectures,
 * so we should introduce some generic detect alghorithm in our configure 
 * script and allow intel CPUs to directly access the requested data 
 * 
 * Until then, direct access will be disabled to make Gnometoaster work 
 * on a variety of different platforms, not just intel and Sparc for which
 * the compiler switch mentioned below is taking care 
 * 
 * Update: There's now an implementation of a generic 32bit align detection
 * alghorithm in preferences.c. However, instead of using it 
 * we can always use the "safe" method as both take about the same CPU
 * time anyway 
 * 
 * Note: get_aligned_int has been replaced by helpings_readuint, a macro
 * stored in helpings.h */

/* #ifdef __svr4__ */
#define pbitget(at,count) bitget(helpings_readuint(buffer,pos),at,count)

/*#else
#define pbitget(at,count) bitget(*((unsigned int*)(&buffer[pos])),at,count)
#endif*/

/* scans a 2kb buffer for a valid mpeg header and returns
 * the mpeg_header structure describing the frame associated with it */
mpeg_header *parse_header(unsigned int pos, 
			  unsigned int size,
			  unsigned char *buffer)
{
   mpeg_header *h;
   int valid=0;
   
   h=(mpeg_header*)malloc(sizeof(mpeg_header));
   if (h!=NULL)
     {	
	/* URGH! Get me a doctor, please, I got Mad Programmer's disease.
	 * Those unsigned short ints are crucial to prevent endianness problems */
	h->valid=((unsigned short int)buffer[pos])+((((unsigned short int)(buffer[pos+1]&0xf0)))<<4);
	if (h->valid==SYNC)
	  {	
	     h->version=pbitget(12,1)?1:2;
	     h->layer=pbitget(13,2)+1;
	     h->has_error_protection=!pbitget(15,1);
	     if ((h->layer<4)&&(h->layer>0))
	       {
		  int bridx=(bitswap(pbitget(16,4))>>4)-1;
		  h->bitrate=(h->version==1)?(mpg1_lookup_bitrate[h->layer-1])
		    [bridx]
		    :(mpg2_lookup_bitrate[h->layer-1])
		      [bridx];
	       };
	     h->freq=(h->version==1)?mpg1_lookup_samplingfreq[pbitget(20,2)]
	       :mpg2_lookup_samplingfreq[pbitget(20,2)];
	     h->has_padding=pbitget(22,1);
	     h->extension=pbitget(23,1);
	     h->mode=pbitget(24,2);
	     h->mode_ext=pbitget(26,2);
	     h->has_copyright=pbitget(28,1);
	     h->is_original=pbitget(29,1);
	     h->emphasis=pbitget(30,2);
	     
	  };
   
	valid=((h->valid==SYNC)
	       &&(h->layer!=0)&&(h->freq!=0)&&(h->bitrate!=0));
	if (valid)
	  {	     
	     if (mp3func_samplingrate==-1)
	       mp3func_samplingrate=h->freq;
	     if (mp3func_layer==-1)
	       mp3func_layer=h->layer;
	     if (mp3func_version==-1)
	       mp3func_version=h->version;
	     if (mp3func_s_mode==-1)
	       mp3func_s_mode=h->mode;
	     if (mp3func_has_copyright==-1)
	       mp3func_has_copyright=h->has_copyright;
	     if (mp3func_is_original==-1)
	       mp3func_is_original=h->is_original;
	     if ((mp3func_samplingrate!=h->freq)
		 ||(mp3func_layer!=h->layer)
		 ||(mp3func_version!=h->version)
		 ||(mp3func_s_mode!=h->mode)
		 ||(mp3func_has_copyright!=h->has_copyright)
		 ||(mp3func_is_original!=h->is_original)
		 )
	       valid=0;
	  };
	/* sort out invalid headers */
	if (!valid)
	  {	
	     free(h);
	     h=NULL;
	  };
     };
   return h;
};

unsigned int mp3func_playingtime_rawbytes(char *filename)
{
   int fd;
   int headers=0;
   mpeg_header *ph;
   unsigned int filesize;
   unsigned int counter;
   double timeconst;
   

   mp3func_samplingrate=-1;
   mp3func_layer=-1;
   mp3func_version=-1;
   mp3func_s_mode=-1;
   mp3func_has_copyright=-1;
   mp3func_is_original=-1;
   
   fd=open(filename,O_RDONLY);
   if (fd!=-1)
     {	
	filesize=lseek(fd,0,SEEK_END);
	mp3=(unsigned char*)mmap(0,filesize,PROT_READ,MAP_PRIVATE,fd,0);
	if ((int)mp3!=-1)
	  {	     	
	     counter=0;
	     do 
	       {
		  ph=parse_header(counter,filesize,mp3);
		  if (ph!=NULL)
		    {
		       int addconst=144000;
		       
		       /* continue scanning behind last header */
		       
		       if (ph->version==1)
			 switch (ph->layer)
			   {
			    case 1:
			      addconst=48000;
			      break;
			    case 2:
			    case 3:
			      addconst=144000;
			      break;
			   }
		       else
			 switch (ph->layer)
			   {
			    case 1:
			      addconst=24000;
			      break;
			    case 2:
			    case 3:
			      addconst=72000;
			      break;
			   };
		       counter+=(addconst*ph->bitrate)/ph->freq;
		       free(ph);
		       headers++;
		    }
		  else
		    counter++;
	       }
	     while (counter+8<filesize);
	     close(fd);
	     munmap(mp3,filesize);
	     timeconst=(1152/(double)mp3func_samplingrate);
	     timeconst/=mp3func_version;
#ifdef DEBUG
	     printf ("found %i headers. at %iHz this is %2i:%02i of playing time.\n",
		     headers,
		     mp3func_samplingrate,
		     (int)(((double)headers*timeconst)/60),
		     ((int)((double)headers*timeconst))%60);
#endif
	
	     /* return the length of the raw stream containing this mp3 file in bytes 
	      * or 0 if the processed file wasn't a valid mp3 stream */
	     if (mp3func_samplingrate!=-1)
	       return (int)((timeconst*headers)*44100*4);
	     else
	       return 0;
	  }
	else
	  return 0;
     }
   else     
     return 0;
};
   

   
   
