/* The Type of Track representing an audiotrack located on a cdrom */

#include <sys/wait.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "int.h"

#include "tracks.h"
#include "piping.h"
#include "preferences.h"
#include "varmanwidgets.h"
#include "varman.h"
#include "audiotrack.h"
#include "dependencies.h"
#include "cdromlow.h"
#include "cddrives.h"
#include "cddb.h"
#include "updatehandlers.h"

/* uncomment if you want to see output of cdda2wav in recording terminal */
/* #define DEBUG */

#ifdef DEBUG
# include "rterm.h"
# include "colors.h"
rterm_pipeinfo *debugtag;
#endif

#define audiotrack_getprecachingflag \
   (!strcmp(varman_getvar(global_defs,"audiotrack_precache"),"true")) \
   *atadd

char *audiotrack_getdaecall(void*trackinfo)
{
   char *call;
   audiotrack_info *i=(audiotrack_info*)((tracks_trackinfo*)trackinfo)->data;
   char *scsidevice=cddrives_getrombydevicename(i->device)->scsiid;
   char *normaldevice=i->device;
   call=varman_getvar_copy(global_defs,"audiotrack_read");
   varman_replacevars(global_defs,call);
   varman_replacestring(call,"$device",normaldevice);
   if (scsidevice!=NULL)
     varman_replacestring(call,
			  "$scsiid",
			  scsidevice);
   if ((cddrives_getrombydevicename(i->device)->dae_usesg)&&(scsidevice!=NULL))
     varman_replacestring(call,
			  "$source",
			  scsidevice);
   else
     varman_replacestring(call,
			  "$source",
			  normaldevice);
   varman_replacestringbyfloat(call,"$tracknum",i->tracknum);
   return call;
};

int audiotrack_openpipe(void*trackinfo)
{
   int outp=-1;
#ifdef DEBUG
   int errp=-1;
#endif
   int *perrp;
   /* assemble call to DAE client */
   char *call=audiotrack_getdaecall(trackinfo);
   audiotrack_info *i=(audiotrack_info*)((tracks_trackinfo*)trackinfo)->data;
   /* only get a std err pipe if in debugging mode,
    * otherwise redirect output to /dev/zero (piping.c) */
#ifdef DEBUG
   perrp=&errp;
#else
   perrp=NULL;
#endif
   i->cdda2wavpid=piping_create(call,NULL,&outp,perrp);
   i->cdda2wavpipe=outp;
#ifdef DEBUG
   debugtag=rterm_connectpipe(&Red,errp);
#endif
   if (outp!=-1)
     cddrives_updatedisable();
   free(call);
   return outp;
}
;

void audiotrack_closepipe(void*trackinfo)
{
   audiotrack_info *i;

   i=(audiotrack_info*)((tracks_trackinfo*)trackinfo)->data;
   if (i->cdda2wavpipe!=-1)
     {
	cddrives_updateenable();
	close(i->cdda2wavpipe);
	waitpid(i->cdda2wavpid,&i->cdda2wavstatus,0);
	/* wait for client to finish */
#ifdef DEBUG
	rterm_disconnectpipe(debugtag);
#endif
     };
}
;

int audiotrack_tracksize(void*trackinfo)
{
   audiotrack_info *i;

   i=(audiotrack_info*)((tracks_trackinfo*)trackinfo)->data;

   return i->tracksize;
}
;

int audiotrack_dataavail(void*trackinfo)
{
   audiotrack_info *i;
   int dataavail;

   i=(audiotrack_info*)((tracks_trackinfo*)trackinfo)->data;
   dataavail=1;
   if (waitpid(i->cdda2wavpid,&i->cdda2wavstatus,WNOHANG)!=0)
     dataavail=0;

   return dataavail;
}
;

int audiotrack_valid(void*trackinfo)
{
   int result;
   /* first, determine what to do to open an audio track */
   char *call=audiotrack_getdaecall(trackinfo);
   /* is this call legal ? If not, show exec not found
    * plus dependency informations */
   result=piping_isvalid_command(call);
   if (!result)
     dependencies_showmissingexec(call);
   free(call);
   /* return validity flag */
   return result;
}
;

void audiotrack_destroy(void*trackinfo)
{
   audiotrack_info *i;

   i=(audiotrack_info*)((tracks_trackinfo*)trackinfo)->data;

   if ((audiotrack_dataavail(trackinfo))&&(i->cdda2wavpipe!=-1))
     close (i->cdda2wavpipe);
   if (i->cddbrequestid)
     cddb_killjob(i->cddbrequestid);
   free(i);
}
;

void audiotrack_pc_policy_changed(VARMAN_VAR_ENTRY *var,gpointer data)
{
   ((tracks_trackinfo*)data)->precacherequired=audiotrack_getprecachingflag;
};

char *audiotrack_buildtrackname(int tracknum,
				const char *title,
				const char *artist,
				const char *cdtitle)
{
   const char *layout="xx s (s,s)"; // this is just for stringlength calculation
   char *name=NULL;
   if (!title)
     title="";
   if (!artist)
     artist="";
   if (!cdtitle)
     cdtitle="";
   name=(char*)malloc(strlen(layout)+strlen(title)+strlen(artist)+strlen(cdtitle)+3);
   sprintf(name,"%02i %s",tracknum,title);
   if (strlen(artist)||strlen(cdtitle))
     {
	strcat(name," (");
	strcat(name,artist);
	if (strlen(artist)&&strlen(cdtitle))
	  strcat(name,",");
	strcat(name,cdtitle);
	strcat(name,")");
     };
   return name;
};

void audiotrack_cddbcallback(cddb_infostruct *info,
			     gpointer data)
{
   tracks_trackinfo *track=(tracks_trackinfo*)data;
   char *newname=NULL;
   if (info->track_title)
     strcpy(track->title,info->track_title);
   if (info->artist)
     strcpy(track->performer,info->artist);

   newname=audiotrack_buildtrackname(info->tracknum,
				     track->title,
				     track->performer,
				     info->cd_title);
   strcpy(track->name,newname);
   free(newname);
   updatehandlers_call(track->updatehandlers);
   ((audiotrack_info*)track->data)->cddbrequestid=0;
};

tracks_trackinfo *audiotrack_create(char *device,
				    int tracknum,
				    const char *title,
				    const char *performer,
				    cddb_essentials *es)
{
   tracks_trackinfo *at;
   audiotrack_info *info;
   char name[255];
   int tochandle=cdromlow_gettochandle(device);

   info=(audiotrack_info*)malloc(sizeof(audiotrack_info));
   strcpy(info->device,device);
   info->tracknum=tracknum;
   if (tochandle)
     {
	info->tracksize=cdromlow_tracksize(tochandle,tracknum);
     }
   else
     info->tracksize=0;

   if (title||performer)
     {
	char *newname=audiotrack_buildtrackname(tracknum,
						title,
						performer,
						NULL);
	strcpy(name,newname);
	free(newname);
     }
   else
     {
	if (tochandle)
	  sprintf(name,_("CD %#x Audiotrack %02d"),
		  cdromlow_cddbnumber(tochandle),
		  info->tracknum);
	else
	  sprintf(name,_("CD in %s Audiotrack %02d"),
			 device,
			 info->tracknum);
     };

   at=tracks_create(name,
		    title,
		    performer,
		    "audio",audiotrack_getprecachingflag,
		    audiotrack_openpipe,
		    audiotrack_closepipe,
		    audiotrack_tracksize,
		    audiotrack_dataavail,
		    audiotrack_valid,
		    audiotrack_destroy,
		    info);

   /* perform a cddb database lookup */
   info->cddbrequestid=cddb_lookup(es,
				   info->tracknum,
				   audiotrack_cddbcallback,
				   at);
#ifdef DEBUG
   printf ("audiotrack_create: queued cddb lookup request\n");
#endif

   varman_install_handler(global_defs,"audiotrack_precache",
			  audiotrack_pc_policy_changed,
			  at);

   if (tochandle)
     cdromlow_closetochandle(tochandle);

   return at;
}
;

/* configure the cdda audio stream from cdrom tracktype */
void audiotrack_init()
{
   GtkWidget *audiotrack_prefs;
   GtkWidget *label;

   varmanwidgets_widget *atc;
   varmanwidgets_widget *read;
   varmanwidgets_widget *precache;

   varmanwidgets_widget *cdtext;

   audiotrack_prefs=gtk_vbox_new(0,0);
   gtk_widget_show(audiotrack_prefs);
   label=gtk_label_new(_("CD Digital Audio"));
   gtk_widget_show(label);
   preferences_append_page(audiotrack_prefs,label);

   atc=varmanwidgets_entry_new(_("Audio track Client:"),
			       "audiotrack_client",
			       global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(audiotrack_prefs),atc->visual,0,0,0);

   read=varmanwidgets_entry_new(_("Audio track reading:"),
				"audiotrack_read",
				global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(audiotrack_prefs),read->visual,0,0,0);

   cdtext=varmanwidgets_entry_new(_("CD Text retrieval:"),
				  "audiotrack_readcdtext",
				  global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(audiotrack_prefs),cdtext->visual,0,0,0);
   cdtext=varmanwidgets_entry_new(_("CD Text parse Title:"),
				  "audiotrack_parsecdtext_title",
				  global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(audiotrack_prefs),cdtext->visual,0,0,0);
   cdtext=varmanwidgets_entry_new(_("CD Text parse Artist:"),
				  "audiotrack_parsecdtext_performer",
				  global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(audiotrack_prefs),cdtext->visual,0,0,0);
   cdtext=varmanwidgets_entry_new(_("CD Text parse Disc Title:"),
				  "audiotrack_parsecdtext_cdtitle",
				  global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(audiotrack_prefs),cdtext->visual,0,0,0);
   cdtext=varmanwidgets_entry_new(_("CD Text parse Disc Artist:"),
				  "audiotrack_parsecdtext_cdperformer",
				  global_defs,APPLYMODE_BUTTON,160,320);
   gtk_box_pack_start(GTK_BOX(audiotrack_prefs),cdtext->visual,0,0,0);

   precache=varmanwidgets_checkbox_new(_("Precache Audio Tracks:"),
				       "audiotrack_precache",
				       global_defs,APPLYMODE_BUTTON,160,
				       "true","false");
   gtk_box_pack_start(GTK_BOX(audiotrack_prefs),precache->visual,0,0,0);
}
;

