/*  GTKtalog.
 *  Copyright (C) 1999-2000  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define _USE_BSD
#include <config.h>
#if defined(OS_FREEBSD) || defined(OS_NETBSD) || defined(OS_OPENBSD)
#    include <sys/param.h>
#    include <sys/mount.h>
#endif
#include <gnome.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#if defined(USE_PTHREADS)
#    include <pthread.h>
#endif
#include <signal.h>

#include "addisk.h"
#include "config_common.h"
#include "fastaddisk.h"
#include "folder.h"
#include "interface.h"
#include "compare.h"
#include "io.h"
#include "progressbar.h"
#include "vfs.h"
#include "thread_utils.h"

#define READ   0
#define WRITE 1

static char *local_foldername = NULL;
static char *local_diskname = NULL;
static gboolean flag_to_umount;
static gboolean flag_to_mount;
static gint added_disk_is_ok;

static GNode *local_gn;

static gboolean is_message_to_display;
static GString *message_to_display;

static gboolean vfs_scan_nok;

static GtkWidget *stop_window = NULL;

static gint current_progressbar;

#ifdef OS_PPC_DARWIN
static gchar *mosx_cd_mount_point = NULL;
#endif

gint
spawn_proc (gchar * path, gint narg, gchar ** arg)
{
  gchar **arg2;
  int i, pid;
  GString *gstr = g_string_new (arg[0]);


  if (test_plugin_existence (gstr))
    {
      g_string_free (gstr, TRUE);
      pid = fork ();
      if (pid != 0)
	return pid;

      /* Child */

      arg2 = g_new0 (gchar *, (narg + 1));
      for (i = 0; i < narg; i++)
	arg2[i] = arg[i];
      arg2[narg] = NULL;
#if !defined(__DEBUG__)
      close (0);
      close (1);
      close (2);
#endif

      return execvp (arg2[0], arg2);
    }
  else
    {
      g_string_free (gstr, TRUE);
      return (-1);
    }
}

guint
du_s (gchar * path)
{
  int p[2];
  static gchar buffer[1024];
  gchar *argsdu[4];
  gchar *b2;
  ssize_t len;
  int pid;

  if (pipe (p) == 0)
    {
      if ((pid = fork ()) == 0)
	{
	  close (1);
	  dup (p[1]);
	  /* FIXME: Stderr should be closed too or redirected
	   * in order not to have error messages on the console
	   */
	  argsdu[0] = g_strdup ("du");
#if defined(OS_LINUX)
	  argsdu[1] = g_strdup ("-sb");
#elif defined(OS_FREEBSD) || defined(OS_NETBSD) || defined(OS_OPENBSD) || defined(OS_PPC_DARWIN)
	  argsdu[1] = g_strdup ("-s");
#endif
	  argsdu[2] = path;
	  argsdu[3] = NULL;
	  execvp (argsdu[0], argsdu);
	}
      else
	{
	  close (0);
	  dup (p[0]);
	  close (p[1]);
	  len = read (0, buffer, 1023);
	  buffer[len] = 0;
	  b2 = buffer;
	  while ((b2[0] >= '0') && (b2[0] <= '9'))
	    b2++;
	  b2[0] = 0;
	  wait4 (pid, NULL, 0, 0);
	  close (p[0]);
	  /* FIXME: STDIN should be reopened to the real STDIN. */
	}
    }
  return (atol (buffer));
}

void
init_message_to_display ()
{
  is_message_to_display = FALSE;
}

void
display_message_if_necessary (void)
{
  if (is_message_to_display)
    {
      is_message_to_display = FALSE;
      gnome_dialog_run_and_close (GNOME_DIALOG
				  (gnome_warning_dialog_parented
				   (message_to_display->str,
				    GTK_WINDOW (main_window))));
      g_string_free (message_to_display, TRUE);
    }

}

void
set_message_to_display (GString * message)
{
  message_to_display = message;
  is_message_to_display = TRUE;
}

gboolean
eject_disk (CONFIG * my_config)
{
  int status, pid;
  char *args[3];

  if (my_config->use_automount)
    {
#ifdef OS_LINUX
      if (!my_config->use_supermount)
#endif
	return (TRUE);
    }


  seteuid (getuid ());
  if (my_config->use_sudo)
    {
  args[0] = my_config->sudo;
  args[1] = my_config->eject_prog->str;
  args[2] = my_config->mount_device->str;
  pid = spawn_proc (NULL, 3, args);
    }
  else
    {
  args[0] = my_config->eject_prog->str;
  args[1] = my_config->mount_device->str;
  pid = spawn_proc (NULL, 2, args);
  }
  if (pid == -1)
    {
      set_message_to_display (g_string_new
			      (_
			       ("Could not execute eject program, check your config")));
      return (FALSE);
    }

  wait4 (pid, &status, 0, NULL);
  if (WIFEXITED (status) == 0)
    {
      set_message_to_display (g_string_new
			      (_("Eject did not exit normally")));
      return (FALSE);
    }
  if (WEXITSTATUS (status) != 0)
    {
      set_message_to_display (g_string_new (_("Eject failed! Disk in use?")));
      return (FALSE);
    }
  return (TRUE);
}

gboolean
mount_disk (CONFIG * my_config)
{
#if defined(OS_LINUX) || defined(OS_NETBSD) || defined(OS_OPENBSD)
  int pid, status;
  char *args[3];
  int nargs;
  int tries = my_config->mount_retry;

  seteuid (getuid ());
  if (my_config->use_sudo)
    {
  args[0] = my_config->sudo;
  args[1] = my_config->mount->str;
  args[2] = my_config->mount_point->str;
  nargs = 3;
    }
  else
    {
  args[0] = my_config->mount->str;
  args[1] = my_config->mount_point->str;
      nargs = 2;
    }

  while (tries >= 0)
    {
      pid = spawn_proc (NULL, nargs, args);
      if (pid == -1)
	{
	  set_message_to_display (g_string_new
				  (_
				   ("Could not execute mount program, check your config")));
	  return (FALSE);
	}

      wait4 (pid, &status, 0, NULL);
      if (WIFEXITED (status) == 0)
	{
	  set_message_to_display (g_string_new
				  (_("Mount did not exit normally")));
	  return (FALSE);
	}
      if (WEXITSTATUS (status) == 0)
	{
	  return (TRUE);
	}
      tries--;
      if (tries >= 0)
	sleep (MOUNT_RETRY_TIMEOUT);
    }

  set_message_to_display (g_string_new (_("Mount failed! Disk in use?")));
  return (FALSE);
#elif defined(OS_FREEBSD)
  int status;
  struct iso_args args;

  args.export.ex_flags = MNT_EXRDONLY;
  args.fspec = "/dev/cd0a";	// Will be changed
  args.export.ex_root = -2;
  status =
    mount (MOUNT_CD9660, my_config->mount_point->str, MNT_RDONLY, &args);
  /* error saved in errno variable, do something with it */
  if (status == 0)
    return (TRUE);
  else
    return (FALSE);
#elif defined(OS_PPC_DARWIN)
  /*
   * with MacOS X, CD are mounted automatically so, 
   * if mosx_get_cd_mount_point() retuns NULL, no CD 
   * is available or an error has occured
   */

  return (mosx_get_cd_mount_point (FALSE) == NULL ? FALSE : TRUE);

#endif
}

gboolean
umount_disk (CONFIG * my_config)
{
#if defined(OS_LINUX) || defined(OS_NETBSD) || defined(OS_OPENBSD)
  int pid, status;
  int nargs;
  char *args[3];

  seteuid (getuid ());
  if (my_config->use_sudo)
    {
  args[0] = my_config->sudo;
  args[1] = my_config->umount->str;
  args[2] = my_config->mount_point->str;
  nargs = 3;
    }
  else
    {
  args[0] = my_config->umount->str;
  args[1] = my_config->mount_point->str;
      nargs = 2;
    }
  pid = spawn_proc (NULL, nargs, args);
  if (pid == -1)
    {
      set_message_to_display (g_string_new
			      (_
			       ("Could not execute umount program, check your config")));
      return (FALSE);
    }

  wait4 (pid, &status, 0, NULL);
  if (WIFEXITED (status) == 0)
    {
      set_message_to_display (g_string_new
			      (_("Umount did not exit normally")));
      return (FALSE);
    }
  if (WEXITSTATUS (status) != 0)
    {
      set_message_to_display (g_string_new
			      (_("Umount failed! Disk in use?")));
      return (FALSE);
    }
  return (TRUE);
#elif defined(OS_FREEBSD)
  int status;

  status = unmount (my_config->mount_point->str, NULL);
  if (status == 0)
    return (TRUE);
  else
    return (FALSE);
#elif defined(OS_PPC_DARWIN)
  set_message_to_display (g_string_new (_("Umount failed! Disk in use?")));
  return (FALSE);
#endif
}

void
umount_and_eject_if_necessary (void)
{
  if (flag_to_umount == TRUE)
    {
      /* umount the drive and eject it */
	  if ((my_config->use_supermount == TRUE)
	      || (my_config->use_automount == TRUE))
	{
	  if (my_config->eject_disk == TRUE)
	    {
	    eject_disk (my_config);
	    }
	}
	  else if (umount_disk (my_config) == TRUE)
	{
	  if (my_config->eject_disk == TRUE)
	    {
	    eject_disk (my_config);
	}
    }
    }

}

void
do_not_forget_to_umount (gboolean gb)
{
  flag_to_umount = gb;
}

void
do_not_forget_to_mount (gboolean gb)
{
  flag_to_mount = gb;
}

void
reset_added_disk (void)
{
  added_disk_is_ok = ADDED_DISK_UNDEF;
}

void
set_added_disk (gint gb)
{
  if ((added_disk_is_ok != ADDED_DISK_NOK)
      && ((gb == ADDED_DISK_DELETE) || (gb == ADDED_DISK_STOP)
	  || (gb == ADDED_DISK_UNDEF)
	  || (added_disk_is_ok == ADDED_DISK_UNDEF)))
    {
      added_disk_is_ok = gb;
    }
}

gboolean
test_abort_scan (void)
{
  if ((added_disk_is_ok == ADDED_DISK_DELETE)
      || (added_disk_is_ok == ADDED_DISK_STOP))
    return (TRUE);
  return (FALSE);

}

void
init_vfs_scan_status (void)
{
  vfs_scan_nok = FALSE;
}

gint
my_timeout (gpointer data)
{
  FOLDER *racine = data;
  gint i;
/*  FILE_DATA *fd;*/
  display_message_if_necessary ();
  if (progress_needReset ())
    {
      progress_reset ();
    }
  if ((get_thread_status () == THREAD_SCAN_RUNNING_ENDED)
      && (added_disk_is_ok != ADDED_DISK_UNDEF))
    {
      if (added_disk_is_ok == ADDED_DISK_STOP)
	added_disk_is_ok = ADDED_DISK_OK;
      if (added_disk_is_ok == ADDED_DISK_OK)
	{
	  gtk_clist_freeze (GTK_CLIST (racine->ctree));
	  change_name (local_gn, local_diskname);
	  change_information (local_gn, local_foldername);
	  g_node_traverse (local_gn,
			   G_PRE_ORDER, G_TRAVERSE_ALL, -1,
			   add_gnode_to_ctree, racine);
	  gtk_clist_thaw (GTK_CLIST (racine->ctree));
	  update_tree (racine);
	}
      else if (added_disk_is_ok == ADDED_DISK_DELETE)
	{
	  if (g_node_n_nodes (local_gn, G_TRAVERSE_ALL) <= 1)
	    {
	      FILE_DATA *fd = local_gn->data;
	      for (i = 0; i < racine->datas->len; i++)
		{
		  fd = g_ptr_array_index (racine->datas, i);
		  fd->id = i;
		}
	      g_ptr_array_index (racine->datas, fd->id) = NULL;
	      g_free (fd);
	      while (g_ptr_array_remove_fast (racine->datas, NULL));
	      for (i = 0; i < racine->datas->len; i++)
		{
		  fd = g_ptr_array_index (racine->datas, i);
		  fd->id = i;
		}
	      g_node_unlink (local_gn);
	      g_node_destroy (local_gn);

	    }
	  else
	    {
	      suppress_dir (local_gn, FALSE);
	    }
	}
      if (stop_window)
	{
	  progress_setCurrent (progress_getTotal_m (GNOMEAPPBAR_SCAN),progress_getTotal_b (GNOMEAPPBAR_SCAN),
			       GNOMEAPPBAR_SCAN);
	  progress_timeout (GINT_TO_POINTER (GNOMEAPPBAR_SCAN));
	  current_progressbar = GNOMEAPPBAR_MAIN;
	  progress_setGnomeAppBar (NULL, GNOMEAPPBAR_SCAN, FALSE);
	  gtk_widget_destroy (stop_window);
	  stop_window = NULL;
	}
      if (local_diskname)
	{
	  g_free (local_diskname);
	  local_diskname = NULL;
	}
      if (local_foldername)
	{
	  g_free (local_foldername);
	  local_foldername = NULL;
	}
/************** If umount and eject have to be in the main thread, remove the comments ********/
//umount_and_eject_if_necessary();
/*********************************************************************************************/
      progress_setCurrent (progress_getTotal_b (current_progressbar),progress_getTotal_b (current_progressbar),
			   current_progressbar);
      progress_timeout (GINT_TO_POINTER (current_progressbar));

      reinit_thread_status ();
      progress_addCurrent ( 1,
			   GNOMEAPPBAR_MAIN);
      progress_timeout (GINT_TO_POINTER (GNOMEAPPBAR_MAIN));
      if (vfs_scan_nok)
	{
	  WARNING_DIALOG (_
			  ("At least one VFS could not be read successfully"),
			  main_window);
	}
      gtk_ctree_sort_node (GTK_CTREE (racine->ctree), NULL);

    }
  else
    {
      progress_timeout (GINT_TO_POINTER (current_progressbar));
      gtk_timeout_add (PROGRESS_BAR_TIMEOUT, my_timeout, racine);
    }
  return (0);
}

void
fad_wait_window_cb (GtkWidget * w, gpointer data)
{
  progress_addCurrent ( 1,
		       GNOMEAPPBAR_SCAN);
  progress_timeout (GINT_TO_POINTER (GNOMEAPPBAR_SCAN));
  current_progressbar = GNOMEAPPBAR_MAIN;
  progress_setGnomeAppBar (NULL, GNOMEAPPBAR_SCAN, FALSE);
  set_added_disk (GPOINTER_TO_INT (data));
  gtk_widget_destroy (stop_window);
  stop_window = NULL;
}

static void
destroy_fad_window (GtkWidget * w, GdkEventKey * event, gpointer data)
{
  progress_addCurrent ( 1,
		       GNOMEAPPBAR_SCAN);
  progress_timeout (GINT_TO_POINTER (GNOMEAPPBAR_SCAN));
  current_progressbar = GNOMEAPPBAR_MAIN;
  progress_setGnomeAppBar (NULL, GNOMEAPPBAR_SCAN, FALSE);

  gtk_widget_destroy (w);
  fad_wait_window ();
}

void
fad_wait_window (void)
{
  GtkWidget *label;
  GtkWidget *progressbar;

  /* Window */
  stop_window = gnome_dialog_new (_("Scanning"), _("Stop"), _("Abort"), NULL);

  /* FIXME: Set Parent to help the window manager */
  //gnome_dialog_set_parent(GNOME_DIALOG(dlg), GTK_WINDOW(app));

  gtk_window_set_position (GTK_WINDOW (stop_window), GTK_WIN_POS_MOUSE);

  /* Label */
  label = gtk_label_new (_("Scanning..."));
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (stop_window)->vbox), label, TRUE,
		      TRUE, 0);
  gtk_widget_show (label);

  /* Progress bar */
  progressbar = gtk_progress_bar_new ();
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (stop_window)->vbox), progressbar,
		      TRUE, TRUE, 0);
  gtk_widget_show (progressbar);


  progress_setGnomeAppBar (progressbar, GNOMEAPPBAR_SCAN, FALSE);
  progress_set_activity_mode (GNOMEAPPBAR_SCAN, FALSE);
  progress_setCurrent (progress_getCurrent_m (GNOMEAPPBAR_MAIN),progress_getCurrent_b (GNOMEAPPBAR_MAIN),
		       GNOMEAPPBAR_SCAN);
  progress_setTotal (progress_getTotal_m (GNOMEAPPBAR_MAIN), progress_getTotal_b (GNOMEAPPBAR_MAIN), GNOMEAPPBAR_SCAN);
  progress_setCurrent (progress_getTotal_m (GNOMEAPPBAR_MAIN),progress_getTotal_b (GNOMEAPPBAR_MAIN),
		       GNOMEAPPBAR_MAIN);
  current_progressbar = GNOMEAPPBAR_SCAN;
#if !defined(USE_PTHREADS)
  my_timeout (my_config->racine);
//      progress_timeout (GINT_TO_POINTER (current_progressbar));
#endif

  /* Callbacks */
  gnome_dialog_button_connect (GNOME_DIALOG (stop_window), 0,
			       fad_wait_window_cb,
			       GINT_TO_POINTER (ADDED_DISK_STOP));
  gnome_dialog_button_connect (GNOME_DIALOG (stop_window), 1,
			       fad_wait_window_cb,
			       GINT_TO_POINTER (ADDED_DISK_DELETE));

  gtk_window_set_modal (GTK_WINDOW (stop_window), TRUE);
  gtk_window_set_transient_for (GTK_WINDOW (stop_window),
				GTK_WINDOW (main_window));
  gtk_signal_connect (GTK_OBJECT (stop_window),
		      "delete_event", destroy_fad_window, NULL);
  gtk_signal_connect (GTK_OBJECT (stop_window),
		      "close", destroy_fad_window, NULL);
  gtk_widget_realize (stop_window);
  gtk_widget_show (stop_window);

}

void
set_local_disk_name (gchar * s)
{
  if (added_disk_is_ok == ADDED_DISK_UNDEF)
    local_diskname = g_strdup (s);
  return;
}

void
set_local_folder_name (gchar * s)
{
  local_foldername = g_strdup (s);
  return;
}

gchar *
launch_information_plugin (gchar * file, MIME_EXTENSIONS * m)
{
  char buff[250];
  int fd[2];
  int b;
  gint pid;
  gint status;
  GString *information;
  gchar *ret;
  gchar **args;
  gchar *cmdline;
  gint nb_args;
  gchar **args_ptr;

  if (compare ("^[ \t]*$", m->prog->str, COMPARE_REGEX, FALSE))
    {
      if (pipe (fd) == -1)
	{
	  g_error ("Couldn't create a pipe");
	  return (NULL);
	}
      switch (pid = fork ())
	{
	case 0:		/*fils1 */
	  close (fd[READ]);
	  close (1);
	  dup (fd[WRITE]);
	  close (fd[WRITE]);
	  if (compare ("^[ \t]*$", m->arg->str, COMPARE_REGEX, FALSE))
	    {

	      cmdline = g_strconcat (m->prog->str, " ", m->arg->str, NULL);
	      args = g_strsplit (cmdline, " ", 0);
	      nb_args = 0;
	      args_ptr = args;
	      while (args_ptr[0] != NULL)
		{
		  nb_args++;
		  args_ptr++;
		}
	      args =
		(gchar **) g_realloc (args, sizeof (gchar *) * (nb_args + 2));
	      args[nb_args] = g_strdup (file);
	      args[nb_args + 1] = NULL;
	      execvp (args[0], args);
	    }
	  else
	    {
	      execlp (m->prog->str, m->prog->str, file, NULL);
	    }
	case -1:		/*erreur */
	  g_error (_("Could not fork\n"));
	  exit (3);
	default:
	  close (fd[WRITE]);
	  close (0);
	  dup (fd[READ]);
	  close (fd[READ]);
	  information = g_string_new ("");
	  b = read (0, buff, 249 * sizeof (char));
	  while (b != 0)
	    {
	      buff[b] = '\0';
	      information = g_string_append (information, buff);
	      b = read (0, buff, 249 * sizeof (char));
	    }
	  wait4 (pid, &status, 0, NULL);
	  ret = g_strdup (information->str);
	  g_string_free (information, TRUE);
	  return (ret);

	}
    }
  else
    {
      return (NULL);
    }

}

gchar *
read_information (gchar * file, gchar * mime)
{
  MIME_EXTENSIONS *m;
  gchar *information_string = NULL;
#ifdef OS_PPC_DARWIN
  int file_len;
#endif

  if (mime)
    {
      if ((m = find_extension (mime)))
	{
	  information_string = launch_information_plugin (file, m);
	}
    }
  if (!information_string)
    {
      if (my_config->use_default_information_plugin)
	{
	  m = my_config->default_information_plugin;
	  if (m->state == TRUE)
	    {
	      information_string = launch_information_plugin (file, m);
	    }
	}
    }

  if (information_string)
    {
#ifdef OS_PPC_DARWIN
      /* in MacOS X, the file utility doesn't accept the -b parameter */
      /* so we remove the file name and ": " from the beginning of information_string */
      file_len = strlen (file) /*file_len */ ;
      if (strncmp (information_string, file, file_len) == 0)
	{
	  if (strncmp (&information_string[file_len], ": ", 2) == 0)
	    {
	      strcpy (information_string, &information_string[file_len + 2]);
	    }
	  else
	    {
	      strcpy (information_string, &information_string[file_len]);
	    }
	}
#endif
      /* remove the last char which is a \n */
      while ((strlen (information_string) > 0)
	     && (information_string[strlen (information_string) - 1] == '\n'))
	information_string[strlen (information_string) - 1] = 0;
    }
  return (information_string);
}

gboolean
is_dir_openable (char *dir_name)
{
  DIR *dir;

  if ((dir = opendir (dir_name)) == NULL)
    return (FALSE);
  closedir (dir);
  return (TRUE);
}

gboolean
create_folder_tree (FOLDER * racine, GNode * parent, const gchar * path,
		    gboolean getInformation, gint testvfs)
/* Stocks the directory (ans subdirs) into a FOLDER structure
 * folder is the FOLDER directory where the directory should be stored
 *   WARNING: folder should not be NULL!!!
 * path is the full path of the directory
 */
{
  DIR *dir;
  struct dirent *fichier;
  struct stat fileinfo;
  gchar *file;
  GList *gl;
  GList *new_dirs = NULL;
  GNode *gn;
  GString *my_path;
  gchar *infostring;
  gchar *mime;
  gpointer scan_it;

  if (added_disk_is_ok == ADDED_DISK_DELETE)
    {
      return (FALSE);
    }

  if (added_disk_is_ok == ADDED_DISK_STOP)
    {
      return (FALSE);
    }
  my_path = g_string_new (path);

  my_path = g_string_append_c (my_path, '/');

  /*  kill (my_config->current_pid, SIGUSR2); */

  if ((dir = opendir (my_path->str)) == NULL)
    {
      char *tmp;
      tmp = g_strdup_printf (_("Can't read the directory: %s"), my_path->str);
      while (is_message_to_display)
	sleep (1);
      set_message_to_display (g_string_new (tmp));
      g_free (tmp);
      return (FALSE);
    }
  while (((fichier = readdir (dir)) != NULL)
	 && (added_disk_is_ok != ADDED_DISK_DELETE)
	 && (added_disk_is_ok != ADDED_DISK_STOP))
    {
      file = g_strconcat (my_path->str, fichier->d_name, NULL);
      if ((strcmp (fichier->d_name, ".") )
	  && (strcmp (fichier->d_name, "..") ))
	{

	  if (lstat (file, &fileinfo) == -1)
	    {
	      gn = folder_add (racine, parent, fichier->d_name, NULL,
			       IS_UNREADABLE, NULL, 0, 0, 0, 0);
	      /* Progress Bar */
	    }
	  /* This is a directory */
	  else if (S_ISLNK (fileinfo.st_mode))
	    {
	      char tmp[1024];
	      char tmp2[2048];

	      progress_addCurrent (fileinfo.st_size, current_progressbar);

	      memset (tmp, 0, 1024);

	      if (readlink (file, tmp, 1023) == -1)
		{
		  char *tmp;
		  tmp =
		    g_strdup_printf (_
				     ("Problem with symbolic link %s, we forget it!"),
				     file);
		  while (is_message_to_display)
		    sleep (1);
		  set_message_to_display (g_string_new (tmp));
		  g_free (tmp);
		}
	      else
		{
		  snprintf (tmp2, 2047, "%s -> %s", fichier->d_name, tmp);
		  folder_add_link (racine, parent, tmp, tmp2, NULL,
				   fileinfo.st_size, fileinfo.st_ctime, 0, 0);
		}
	    }
	  else if (S_ISDIR (fileinfo.st_mode))
	    {
	      progress_addCurrent (fileinfo.st_size, current_progressbar);

	      if (is_dir_openable (file))
		{
		  gn =
		    folder_add (racine, parent, fichier->d_name, NULL,
				IS_DIR, NULL, fileinfo.st_size,
				fileinfo.st_ctime, 0, 0);
		  new_dirs = g_list_append (new_dirs, gn);
		}
	    }

	  else
	    {
	      /* This is a file */
	      progress_addCurrent (fileinfo.st_size, current_progressbar);

	      mime = NULL;
	      if (my_config->scan_mime_types)
		{
		  mime = g_strdup (gnome_mime_type_from_magic (file));
		}
	      if (!mime)
		mime = g_strdup (gnome_mime_type_or_default (file, NULL));
	      if (mime)
		{
		  scan_it =
		    g_tree_lookup (my_config->mime_to_skip_during_a_scan,
				   mime);
		}
	      else
		{
		  scan_it = NULL;
		}
	      if (GPOINTER_TO_INT (scan_it) != 1)
		{
		  if (getInformation)
		    {
		      /* use a plugin to get informations about the file */
		      infostring = read_information (file, mime);
		    }
		  else
		    {
		      infostring = NULL;
		    }
		  gn =
		    folder_add (racine, parent, fichier->d_name, infostring,
				IS_FILE, mime, fileinfo.st_size,
				fileinfo.st_ctime, 0, 0);
		  if (mime)
		    g_free (mime);
		  if (infostring)
		    g_free (infostring);
		  if (testvfs != TEST_VFS_NO)
		    {
		      if (vfs_read (gn, fichier->d_name, testvfs, racine))
			vfs_scan_nok = TRUE;
		    }
		}
	      else
		{
		  if (mime)
		    g_free (mime);
		}
	    }
	} else {
		  if (lstat (file, &fileinfo) >=0)
	      progress_addCurrent (fileinfo.st_size, current_progressbar);

	}
      g_free (file);
#if !defined(USE_PTHREADS)
      gui_update ();
#endif
    }
  closedir (dir);
  g_string_free (my_path, TRUE);

  while ((new_dirs) && (added_disk_is_ok != ADDED_DISK_DELETE)
	 && (added_disk_is_ok != ADDED_DISK_STOP))
    {
      gn = (GNode *) new_dirs->data;
      file = g_strconcat (path, "/", folder_get_name (gn), NULL);
      if (create_folder_tree (racine, gn, file, getInformation, testvfs))
	vfs_scan_nok = TRUE;
      g_free (file);
      gl = new_dirs;
      new_dirs = g_list_remove_link (new_dirs, new_dirs);
      g_list_free_1 (gl);
    }
  return (vfs_scan_nok);
}

gboolean
check_mounted (CONFIG * my_config)
{

#ifdef OS_LINUX
  FILE *fp;
  char line[1024];
  char s1[1024];
  char s2[1024];
  char s3[1024];
  int rc;

  fp = fopen ("/proc/mounts", "r");
  if (fp == NULL)
    ERROR_DIALOG (_("Unable to open /proc/mounts\n"), main_window);

  while (fgets (line, sizeof (line), fp) != 0)
    {
      rc = sscanf (line, "%1023s %1023s %1023s", s1, s2, s3);
      if ((rc >= 3) && (strcmp (s2, my_config->mount_point->str) == 0))
	{
	  fclose (fp);
	  return TRUE;
	}
    }
  fclose (fp);
  return FALSE;
#elif (defined(OS_PPC_DARWIN))

  /*
   * with MacOS X, CD are mounted automatically so, 
   * if mosx_get_cd_mount_point() retuns NULL, no CD 
   * is available or an error has occured
   */

  return (mosx_get_cd_mount_point (FALSE) == NULL ? FALSE : TRUE);

#else
/* FIXME: in many cases, this is a bug as we do not know if the CD is mounted
 * or not.
 * If you are porting gtktalog to another platform, adapt this function to
 * your platform and register that platform in configure.in
 * Do not forget to send me (Read the AUTHORS file to know who) the patch ;-)
 */
  return FALSE;
#endif
}

int
mount_disk_if_necessary (void)
{
  if (flag_to_mount == TRUE)
    {
      flag_to_umount = FALSE;
      if ((my_config->use_supermount == FALSE)
	  && (my_config->use_automount == FALSE))
	{
	  if (check_mounted (my_config))
	    {
	      if (my_config->warn_mount)
		{
		  /* FIXME: Error Message asking if we should continue */
		  set_message_to_display (g_string_new
					  (_
					   ("The CD player is already mounted.")));
		}
	    }
	  else
	    {
	      if (mount_disk (my_config) == FALSE)
		{
		  /* Display Error */
		  set_message_to_display (g_string_new
					  (_
					   ("Could not mount the device.\n"
					    "Check there is a CD in the CD player\n"
					    "If yes, check the mount point (see the preferences)\n"
					    "Then try again.\n"
					    "If the problem persists, try to manually mount your CDROM.")));
		  added_disk_is_ok = ADDED_DISK_NOK;
		  set_thread_status_to_ended ();
		  vfs_scan_nok = FALSE;
#if defined(__DEBUG__)
		  GML_log (2, "Exiting start_thread_scan(): error\n");
#endif
		  return (0);
		}
	      else
		flag_to_umount = TRUE;
	    }
	}
      else
	flag_to_umount = TRUE;
    }
  return (1);
}

void *
start_thread_scan (void *arg)
{
  FOLDER *racine = arg;
  guint l;
#if defined(USE_PTHREADS)
  current_progressbar = GNOMEAPPBAR_MAIN;
  progress_setTotal (0,0, GNOMEAPPBAR_SCAN);
#else
  current_progressbar = GNOMEAPPBAR_SCAN;
#endif
  progress_setCurrent (0,0, current_progressbar);
  progress_setTotal (0,0, current_progressbar);
#if defined(__DEBUG__)
  GML_log (2, "start_thread_scan()\n");
#endif
  init_message_to_display ();
#if defined(USE_PTHREADS)
  pthread_detach (pthread_self ());
#endif


  /* Mount the CD if we are in fastadddisk */
  if (!mount_disk_if_necessary ())
    {
#if defined(USE_PTHREADS)
      pthread_exit (NULL);
#endif
      return (NULL);
    }

  if (my_config->use_du_s == TRUE)
    l = du_s (local_foldername);
  else
    l = getStats (my_config->mount_device->str);
  progress_setTotal (0,l, current_progressbar);

  /* Launch the update of the progress bar */
/*  progress_createTimeout ();*/

#ifdef OS_PPC_DARWIN
  if (strcmp (local_foldername, mosx_get_cd_mount_point (FALSE)))
#else
  if (strcmp (local_foldername, my_config->mount_point->str))
#endif
    {
      local_gn =
	folder_add (racine, racine->tree,
		    local_diskname, local_foldername, IS_DISK, NULL, 0, 0, 0,
		    0);
    }
  else
    {
      local_gn =
	folder_add (racine, racine->tree,
		    local_diskname, NULL, IS_CD, NULL, 0, 0, 0, 0);
    }
#if !defined(USE_PTHREADS)
  gui_update ();
#endif
  vfs_scan_nok = create_folder_tree (racine, local_gn, local_foldername,
				     my_config->getInformation,
				     my_config->testvfs);

  umount_and_eject_if_necessary ();
  set_thread_status_to_ended ();
  racine->is_modified = TRUE;
#if defined(__DEBUG__)
  GML_log (2, "Exiting start_thread_scan(): OK\n");
#endif
#if defined(USE_PTHREADS)
  pthread_exit (NULL);
#endif
  return (NULL);
}

gchar *
escapespaces (gchar * txt)
{
  gchar *ptr, *ptmp;
  gint n;

  n = 1;
  ptr = txt;
  while (ptr[0])
    {
      if (ptr[0] == ' ')
	n++;
      n++;
      ptr++;
    }
  ptr = (gchar *) g_malloc (sizeof (gchar) * n);
  ptmp = ptr;
  while (txt[0])
    {
      if (txt[0] == ' ')
	{
	  ptmp[0] = '\\';
	  ptmp++;
	}
      ptmp[0] = txt[0];
      ptmp++;
      txt++;
    }
  ptmp[0] = 0;
  return (ptr);
}

#ifdef OS_PPC_DARWIN
gchar *
mosx_get_cd_mount_point (int force)
{
  FILE *fpipe;
  char buffer[1024], *p = NULL, device[32], dum[64];
  int unit = -1;
  int ejectable, writable, typecd, found, newdevice;

  if (force)
    {
      if (mosx_cd_mount_point != NULL)
	{
	  g_free (mosx_cd_mount_point);
	  mosx_cd_mount_point = NULL;
	}
    }

  if (mosx_cd_mount_point != NULL)
    return mosx_cd_mount_point;

  fpipe = popen ("ioreg -c IOMedia -w 0", "r");
  if (fpipe == NULL)
    {
      fprintf (stderr, "popen ioreg failed\n");
      return NULL;
    }

  found = 0;
  newdevice = 0;
  ejectable = writable = typecd = -1;

  while (fgets (buffer, sizeof (buffer), fpipe) != NULL)
    {
      if (found)
	continue;		/* don't know if the pipe can be closed */
      p = strchr (buffer, '\n');
      if (p)
	*p = 0;
      if ((p = strstr (buffer, "\"BSD Unit\" = ")))
	{
	  p += 13;
	  if (atoi (p) != unit)
	    {
	      newdevice = 1;
	      if (unit == -1)
		{
		  unit = atoi (p);
		  continue;
		}
	      if (typecd == 1 || (ejectable == 1 && writable == 0))
		{
		  /* a CD-ROM drive is found */
		  found = 1;
		}
	      else
		{
		  unit = atoi (p);
		  typecd = ejectable = writable = -1;
		  *device = 0;
		}
	    }
	  else
	    newdevice = 0;
	}			/* endif BSD Unit */
      if (newdevice && (p = strstr (buffer, "\"Type\" = ")))
	{
	  if (!strncmp (p + 9, "\"CD-ROM\"", 8))
	    {
	      typecd = 1;
	    }
	}
      if (newdevice && (p = strstr (buffer, "\"Ejectable\" = ")))
	{
	  if (!strncmp (p + 14, "Yes", 3))
	    ejectable = 1;
	  else if (!strncmp (p + 14, "No", 2))
	    ejectable = 0;
	}
      if (newdevice && (p = strstr (buffer, "\"Writable\" = ")))
	{
	  if (!strncmp (p + 13, "Yes", 3))
	    writable = 1;
	  else if (!strncmp (p + 13, "No", 2))
	    writable = 0;
	}
      if (newdevice && (p = strstr (buffer, "\"BSD Name\" = \"")))
	{
	  strcpy (device, p + 14);
	  p = strchr (device, '\"');
	  if (p)
	    *p = 0;
	}
    }				/* end while */
  fclose (fpipe);

  if (!(typecd == 1 || (ejectable == 1 && writable == 0)))
    {
      return NULL;		/* no CD-ROM found */
    }

  strcpy (buffer, "df | grep ");
  strcat (buffer, device);
  fpipe = popen (buffer, "r");
  if (fpipe == NULL)
    {
      fprintf (stderr, "popen df failed\n");
      return NULL;
    }

  found = 0;
  strcpy (dum, "/dev/");
  strcat (dum, device);
  while (fgets (buffer, sizeof (buffer), fpipe) != NULL)
    {
      if (found)
	continue;		/* don't know if the pipe can be closed */
      p = strchr (buffer, '\n');
      if (p)
	*p = 0;
      if (strncmp (buffer, dum, strlen (dum)) == 0)
	{
	  p = strstr (buffer, "/Volumes/");
	  if (p)
	    found = 1;
	}
    }				/* end while */
  fclose (fpipe);

  if (found)
    {
      mosx_cd_mount_point = g_strdup (p);
      return mosx_cd_mount_point;
    }
  else
    {
      return NULL;
    }
}
#endif
