/* ----------------------------------------------------------------------------
 * themes.c
 * functions to handle themes in gtkpbbuttons.
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "themes.h"
#include "audio.h"

struct theme *
theme_init (char *name)
{
	int err;
	static struct theme td;
	struct themeoption themedef[] = {
		{GLOBAL,     "layout",  theme_layout, NULL},
		{GLOBAL,     "bgimage", theme_image,  &td.bgimage},
		{BRIGHTNESS, "active",  theme_bool,   &td.popup_active[POPUP_BRIGHTNESS]},
		{BRIGHTNESS, "timeout", theme_int,    &td.popup_time[POPUP_BRIGHTNESS]},
		{VOLUME,     "active",  theme_bool,   &td.popup_active[POPUP_VOLUME]},
		{VOLUME,     "timeout", theme_int,    &td.popup_time[POPUP_VOLUME]},
		{VOLUME,     "sample",  theme_sound,  &td.volsound},
		{NOAUDIO,    "timeout", theme_int,    &td.popup_time[POPUP_NOAUDIO]},
		{MOUSE,      "active",  theme_bool,   &td.popup_active[POPUP_MOUSE]},
		{MOUSE,      "timeout", theme_int,    &td.popup_time[POPUP_MOUSE]},
		{BATTERY,    "active",  theme_bool,   &td.popup_active[POPUP_BATTERY]},
		{BATTERY,    "timeout", theme_int,    &td.popup_time[POPUP_BATTERY]},
		{BATTERY,    "sample",  theme_sound,  &td.warnsound},
		{SLEEP,      "active",  theme_bool,   &td.popup_active[POPUP_SLEEP]},
		{SLEEP,      "timeout", theme_int,    &td.popup_time[POPUP_SLEEP]},
		{KBDILLU,    "active",  theme_bool,   &td.popup_active[POPUP_KBDILLU]},
		{KBDILLU,    "timeout", theme_int,    &td.popup_time[POPUP_KBDILLU]},
		{CDROM,      "active",  theme_bool,   &td.popup_active[POPUP_CDROM]},
		{CDROM,      "timeout", theme_int,    &td.popup_time[POPUP_CDROM]}
	};

	td.bgimage		= NULL;  /* window background image */
	td.layout_init		= NULL;  /* callbacks for layout renderer */
	td.layout_attach	= NULL;
	td.layout_cleanup	= NULL;
	td.layout_update	= NULL;
	td.user_data		= NULL;  /* layout dependent theme extension */
	td.havesound		= 0;
	td.popup_time[POPUP_BRIGHTNESS] = TIMEOUT_LEVER;
	td.popup_time[POPUP_VOLUME]     = TIMEOUT_LEVER;
	td.popup_time[POPUP_NOAUDIO]    = TIMEOUT_NOTICE;
	td.popup_time[POPUP_MOUSE]      = TIMEOUT_LEVER;
	td.popup_time[POPUP_BATTERY]    = TIMEOUT_NOTICE;
	td.popup_time[POPUP_SLEEP]      = TIMEOUT_NOTICE;
	td.popup_time[POPUP_KBDILLU]    = TIMEOUT_LEVER;
	td.popup_time[POPUP_CDROM]      = TIMEOUT_NOTICE;

	strncpy(td.name, name, sizeof(td.name));

	err = theme_load_data(&td, themedef, sizeof(themedef) / sizeof(struct themeoption));
	td.popup_active[POPUP_MUTE]    = td.popup_active[POPUP_VOLUME];
	td.popup_active[POPUP_NOAUDIO] = td.popup_active[POPUP_VOLUME];
	td.popup_time[POPUP_MUTE]      = td.popup_time[POPUP_VOLUME];
	if (err == 0)
		err = td.layout_init(&td);
	if (err != 0) {
		if (td.bgimage != NULL)
			g_object_unref(td.bgimage);
		return NULL;
	} else
		return &td;
}

void
theme_exit (struct theme *td)
{
	if (td) {
		td->layout_cleanup(td);
		if (td->volsound) {
			free(td->volsound->audiodata);
			free(td->volsound);
			td->volsound = NULL;
		}
		if (td->warnsound) {
			free(td->warnsound->audiodata);
			free(td->warnsound);
			td->warnsound = NULL;
		}
		td->havesound = 0;
	}
}

int
is_theme (char *name)
{
	struct stat stat_buf;
	char buffer[200];

	snprintf(buffer, sizeof(buffer), "%s/themes/%s/theme.desc", PACKAGE_DATA_DIR, name);
	if ((stat(buffer, &stat_buf)) == -1)
		return errno;
	return 0;
}

int
theme_load_data (struct theme *td, struct themeoption *tab, int tablen)
{
	char linebuffer[150], buffer[200];
	char *token, *arg, *delim;
	char *section[] = {"", "brightness", "volume", "mute",
	                  "mouse", "battery", "sleep", "noaudio",
	                  "kbdillu", "cdrom"};
	int n, err = 0;
	FILE *fd;

	snprintf(buffer, sizeof(buffer), "%s/themes/%s/theme.desc", PACKAGE_DATA_DIR, td->name);
	if ((fd = fopen(buffer, "r"))) {
		while (fgets(linebuffer, sizeof(linebuffer), fd)) {
			if (linebuffer[0] != '#') {
				theme_cleanup_buffer(linebuffer);
				if ((token = strtok(linebuffer, "=\n"))) {
					arg = strtok(0, "#;\n");
					for (n=0; n < tablen; n++) {
						delim = tab[n].popup ? "." : "";
						snprintf(buffer, sizeof(buffer),"%s%s%s",
						   section[tab[n].popup], delim, tab[n].option);
						if (!strcasecmp (buffer, token)) {
							err = tab[n].process (td, arg, tab[n].data);
							break;
						}
					}
					if (n == -1)
						strtok(0,"=\n");
				}
			}
			if (err != 0) break;
		}
		fclose(fd);
	} else {
		print_error (_("ERROR: Couldn't read theme description for %s, %s.\n"), td->name, strerror(errno));
		return E_NOFILE;
	}
	return err;
}

int
theme_bool (struct theme *td, char *arg, int *data)
{
	if (!strncasecmp("true", arg, 4))
		*data = 1;
	else
		*data = 0;
	return 0;
}

int
theme_int (struct theme *td, char *arg, int *data)
{
	*data = atoi (arg);
	return 0;
}

int
theme_string (struct theme *td, char *arg, gchar **data)
{
	*data = g_strdup (arg);
	return 0;
}

int
theme_layout (struct theme *td, char *arg, int *data)
{
	struct layoutengine layouts[] = {
		{"classic", layout_classic_init,
		            layout_classic_attach,
					layout_classic_exit,
					layout_classic_update},
		{"macosx",  layout_macosx_init,
		            layout_macosx_attach,
					layout_macosx_exit,
					layout_macosx_update}};
	int n;

	for (n=1; n >= 0; n--)
		if (!strncmp(layouts[n].name, arg, sizeof(layouts[n].name))) {
			td->layout_init    = layouts[n].layout_init;
			td->layout_attach  = layouts[n].layout_attach;
			td->layout_cleanup = layouts[n].layout_cleanup;
			td->layout_update  = layouts[n].layout_update;
			break;
		}
	if (n < 0) {
		print_error (_("ERROR: Unknown theme layout %s.\n"), arg);
		return -1;
	}
	return 0;
}

int
theme_bgtype (struct theme *td, char *arg, int *data)
{
	char *bgtypes[] = {"gtk", "image"};
	int n;

	for (n=1; n >= 0; n--)
		if (!strncasecmp(bgtypes[n], arg, sizeof(bgtypes[n]))) {
			*data = n;
			break;
		}
	if (n < 0) {
		print_error (_("ERROR: Unknown bgtype %s.\n"), arg);
		return -1;
	}
	return 0;
}

/* this function loads a GdkPixbuf from file and creates
 * a GtkImage from it. This is done to bypass Gtk's error
 * handling for images. This function returns -1 if the
 * image couldn't be loaded.
 * A reference will be obtained to the new image and the
 * floating reference of the GdkPixbuf will be released.
 * If arg was a null-pointer, *data would also be a null-
 * ponter. This is not an error condition. The image is
 * simply not defined.
 */
int
theme_image (struct theme *td, char *arg, GtkWidget **data)
{
	GtkWidget *image = NULL;
	GdkPixbuf *pixbuf;
	GError *error = NULL;
	char buffer[200];

	if (strlen(arg) != 0) {
		snprintf(buffer, sizeof(buffer), "%s/themes/%s/%s", PACKAGE_DATA_DIR, td->name, arg);
		pixbuf = gdk_pixbuf_new_from_file (buffer, &error);
		if (pixbuf) {
			image = gtk_image_new_from_pixbuf(pixbuf);
			g_object_ref(image);     /* mark image as used */
			g_object_unref(pixbuf);  /* free pixbuf */
		} else {
			print_error (_("ERROR: %s.\n"), error->message);
			g_error_free (error);
			return -1;
		}
	}
	*data = image;
	return 0;
}

int
theme_sound (struct theme *td, char *arg, struct sample **data)
{
#ifdef HAVE_SOUND
	char buffer[200];

	snprintf(buffer, sizeof(buffer), "%s/themes/%s/%s", PACKAGE_DATA_DIR, td->name, arg);
	if ((*data = load_sample(buffer)))
		td->havesound = 1;
#endif
	return 0;
}

/* This function removes all whitespaces from the buffer except
 * quoted ones.
 */
void
theme_cleanup_buffer(char *buffer)
{
	char *buf2 = buffer, quotchar = 0;

	while (*buffer != 0) {
		*buf2 = *buffer++;
		if (quotchar == 0) {
			if ((*buf2 == '"') || (*buf2 == '\'')) {
				quotchar = *buf2;
				continue;
			}
			if ((*buf2 != ' ') && (*buf2 != '\t'))
				buf2++;			
		} else {
			if (*buf2 == quotchar) {
				quotchar = 0;
				continue;
			}
			buf2++;
		}
	}
	*buf2 = 0;   /* terminate cleaned-up buffer again */
}

