/*
 * Back-end main header
 * This is the most important file to understand the back-end.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifndef __GDIFF_GDIFF_H__
#define __GDIFF_GDIFF_H__

#include <time.h> /* for time_t */
#include <glib.h>

#ifdef __cplusplus 
extern "C" {
#endif /* __cplusplus */

/*
 * General description of data structures
 *
 * DiffDir is allocated for one comparison(diff) of files or directories.
 * DiffDir manages a linked list of DiffFiles.
 * Each node(DiffFiles) is allocated for each compared (regular)files
 * under the directories.
 * DiffFiles has a general information of the files,
 * such as filename or the number of lines.
 * In addition, DiffFiles manages a linked list of DiffLines.
 * Each node(DiffLines) is allocated for different ranges(=lines),
 * which has the type(DiffType) and the line numbers of the difference.
 *
 * DiffDir and DiffFiles are managed by reference count.
 * The initial value is zero, but the creator usually owns it
 * and increments the count.
 * DiffDir is typically owned by GUI directory view.
 * DiffFiles is owned by DiffDir and also by GUI file views.
 */

/* Forward declarations of types. */
typedef struct _DiffDir DiffDir;
typedef struct _DiffFiles DiffFiles;
typedef struct _FileInfo FileInfo;
typedef struct _DiffLines DiffLines;
typedef struct _DiffBeginEnd DiffBeginEnd;

/*
 * Constant number
 */
/* IMPORTANT. These numbers are used in DiffLines structure.
 * Basically, they represent the type for each difference.
 * In addition, used as the return value from parse_diff_line() */
typedef enum {
	F1ONLY = (1<<0),/* Lines taken from just the first file. (known as DELETE) */
	F2ONLY = (1<<1),/* Lines taken from just the second file. (known as ADD) */
	CHANGE = (1<<2),

	/* Used only in diff3 case. */
	F3ONLY = (1<<3),

	/* These symbols are always bit-wised OR with FnONLY(n=1,2,3) */
	ONLY_ADD = (1<<4),
	/* ONLY_DEL = (1<<5), No need */
	ONLY_CHANGE = (1<<6),

	F12ADD = (1<<7),
	F23ADD = (1<<8),
	F31ADD = (1<<9),

	/* The followings are not used in DiffLines,
	   but they are used as a return value from parse_diff_line() */
	FINDFILE,
	FINDBINFILE,
	IGNORE,
	ERROR
} DiffType;


#define MAX_NUM_COMPARE_FILES		3
typedef enum {
	FIRST_FILE = 0,
	SECOND_FILE = 1,
	THIRD_FILE = 2
} WhichFile;

/* Status of the compared files.
   I'm afraid you feel confused about this and DiffType above.
   DiffType exists for each different ranges(lines).
   On the other hand, DiffStatus is related to the compared files(directories).
   These shifted bits are used in row_hide_stat_mask of DViewPref(see properties.h). */
typedef enum {
	BINARY_FILES		= (1<<0),
	ONLY_FILE1_EXISTS	= (1<<1),
	ONLY_FILE2_EXISTS	= (1<<2),
	/* ONLY_FILE3_EXISTS	= (1<<3), impossible */
	DIFFERENT_FILES		= (1<<4),
	IDENTICAL_FILES		= (1<<5),
	DIRECTORIES			= (1<<6)
} FilesStatus;

/* Used by dfiles_has_file_modified().
   Specifying that the files are modified or not. */
typedef enum {
	MOD_MODIFIED,
	MOD_NOT_MODIFIED,
	MOD_DONTCARE,
	MOD_ERROR
} FileModStatue;


/*
 * Data structure definitions
 */

/* Directory information data
 * Note:
 * Comparing regular files is dealt with as one special case of comparing directories.
 * - When comparing directories, the first node of dfiles_list
 *   has the directories info, and the following nodes has each files info.
 * - When comparing regular files, the first node has the files info. */
struct _DiffDir {
	int ref_count;			/* Reference count */
	gboolean is_diff3;		/* TRUE if comparing three dirs */
	GSList *dfiles_list;	/* Singly linked list of DiffFiles */
};


/* Compared files information data
 * Some special cases as follows,
 * - dlines_list==NULL implies the files are identical.
 * - One of fileinfo[]'s fname=="" implies that
 *   a file exists only in one directory. */
struct _DiffFiles {
	int ref_count;		/* Reference count */
	gboolean binary;	/* TRUE even if one of files is a binary file */
	gboolean is_diff3;	/* TRUE if comparing three files */

	struct _FileInfo {
		gboolean f_dir;	/* TRUE if directory */
		char *fname;	/* file name. "" implies non-existence */
		int nlines;		/* Number of lines */
		char *buf;		/* (mmap'd)buffer */
		int lenb;		/* Length(byte) of the buffer */
		time_t mtime;	/* time of last modification */
		time_t ctime;	/* time of last change */
	} fileinfo[MAX_NUM_COMPARE_FILES];
	GList *dlines_list;/* Doubly linked list of DiffLines */
};


/* Diff lines information data
 * difftype and the line numbers. */
struct _DiffLines {
	DiffType difftype;
	struct _DiffBeginEnd {
		/* Line numbers. */
		int begin;
		int end;
	} between[MAX_NUM_COMPARE_FILES];
};


/* Function type definitions */
typedef const GList* (*DiffFindFunc)(const DiffFiles *dfiles, const GList *cur_dlines_node);
	

/* Global variable declarations */
/* None */


/* Global function declarations */
/* diffdata.c */
extern DiffDir* diffdir_new(const char *fname1, const char *fname2, const char *fname3, const char *args);
extern void diffdir_ref(DiffDir *diffdir);
extern void diffdir_unref(DiffDir *diffdir);
extern DiffFiles* diffdir_add_dfiles(DiffDir *diffdir, const char *fname1, const char *fname2, const char *fname3);
extern void diffdir_remove_dfiles(DiffDir *diffdir, DiffFiles *dfiles);

extern void dfiles_ref(DiffFiles *dfiles);
extern void dfiles_unref(DiffFiles *dfiles);
extern const FileInfo* dfiles_get_fileinfo(DiffFiles *dfiles, WhichFile n, gboolean f_mmap);
extern void dfiles_add_dlines(DiffFiles* dfiles, DiffType dtype, const int *begin, const int *end);


/* rundiff.c */
extern void run_diff(DiffDir *diffdir, const char *filename1, const char *filename2, const char *args, DiffFiles *cur_files);

/* rundiff3.c */
extern void run_diff3(DiffDir *diffdir, const char *filename1, const char *filename2, const char *filename3, const char *args, DiffFiles *cur_files);

/* diffmisc.c */
extern FilesStatus dfiles_get_status(DiffFiles *dfiles);
extern FileModStatue dfiles_has_file_modified(DiffFiles *dfiles);
extern int dfiles_calc_total_nlines(const DiffFiles *dfiles, WhichFile n);
extern int dfiles_get_max_nlines(DiffFiles *dfiles);

/* DiffFindFunc functions */
extern const GList* dfiles_get_curl(const DiffFiles *dfiles, const GList *cur_dlines_node);
extern const GList* dfiles_get_firstl(const DiffFiles *dfiles, const GList *cur_dlines_node);
extern const GList* dfiles_get_lastl(const DiffFiles *dfiles, const GList *cur_dlines_node);
extern const GList* dfiles_find_nextl(const DiffFiles *dfiles, const GList *cur_dlines_node);
extern const GList* dfiles_find_prevl(const DiffFiles *dfiles, const GList *cur_dlines_node);

extern const GList* dfiles_find_rel_curl(const DiffFiles *dfiles, WhichFile n, int cur_line);
extern const GList* dfiles_find_rel_nextl(const DiffFiles *dfiles, WhichFile n, int cur_line);
extern const GList* dfiles_find_rel_prevl(const DiffFiles *dfiles, WhichFile n, int cur_line);

extern const GList* dfiles_find_includel(const DiffFiles *dfiles, WhichFile n, int cur_line);

#ifdef __cplusplus
} 
#endif /* __cplusplus */
 
#endif /* __GDIFF_GDIFF_H__ */
