/*
 *     gtkatlantic - the gtk+ monopd client, enjoy network monopoly games
 *
 *
 *  Copyright © 2002-2014 Sylvain Rochet
 *
 *  gtkatlantic is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* ** functions for connect/disconnect server, receive/send data to/from server */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#ifndef WIN32
#include <netdb.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <fcntl.h>

#define closesocket close
#define my_io_channel_socket_new g_io_channel_unix_new
#else
#include <winsock2.h>

#define my_io_channel_socket_new g_io_channel_win32_new_socket
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <time.h>

#include "config.h"
#include "xmlparse.h"
#include "global.h"
#include "client.h"
#include "game.h"

#if !defined(HAVE_INET_ATON) || defined(WIN32)
#ifndef INADDR_NONE
#define INADDR_NONE 0xffffffff
#endif

int inet_aton(const char *cp, struct in_addr *addr) {

	addr->s_addr = inet_addr(cp);
	return addr->s_addr != INADDR_NONE;
}
#endif


/* -- connect to server
     >
     > return false if:
     >   - max number of connection reached
     >   - can't resolve host
     >   - can't open port or socket
*/
gboolean client_connect(gint32 *idconnect, gchar* host, gint32 port)  {

	struct sockaddr_in addr;
	struct hostent *ent;
	struct in_addr in;
	gint32 addr_len, test;
	gpointer errormsg;
	gchar *text;
	gint32 idcon;
	socket_t sock;
	GIOChannel *channel;


	if( !connect_get_valid_id(&idcon) )  {

		errormsg = g_strdup("[ERROR] Max number of connections reached");
		text_insert_message(errormsg, strlen(errormsg));
		g_free(errormsg);
		return(FALSE);
	}

	/** resolve an address */
	if (!inet_aton(host, &in) ) {

		if (! (ent = gethostbyname(host)) )  {

			errormsg = g_strdup_printf("[ERROR] Can't resolve host \"%s\"", host);
			text_insert_message(errormsg, strlen(errormsg));
			g_free(errormsg);
			return(FALSE);

		}  else  {

			in = *(struct in_addr *)ent->h_addr;
			addr.sin_family = ent->h_addrtype;
		}
	}
	else  addr.sin_family = AF_INET;

	addr.sin_addr = in;
	addr.sin_port = htons(port);
	addr_len = sizeof(addr);

	/* -- now host is resolved, continue connection */
	sock = socket(addr.sin_family, SOCK_STREAM, IPPROTO_IP);
		// see /usr/include/netinet/in.h for IPPROTO_x
	channel = my_io_channel_socket_new(sock);
	g_io_channel_set_encoding(channel, NULL, NULL);

	test = connect(sock, (struct sockaddr*) &addr, addr_len);

	if(test < 0) {

		errormsg = g_strdup_printf("[ERROR] Host \"%s\" resolved but can't open port \"%d\" or socket", host, port);
		text_insert_message(errormsg, strlen(errormsg));
		g_free(errormsg);
		g_io_channel_unref(channel);
		return(FALSE);
	}

#ifndef WIN32
	/* set socket to non-blocking */
	int flags = fcntl(sock, F_GETFL);
	if(flags >= 0) {
		flags |= O_NONBLOCK;
		fcntl(sock, F_SETFL, flags);
	}
#else
	u_long on = 1;
	ioctlsocket(sock, FIONBIO, &on);
#endif

	/* -- connection established :), finishing it */
	connection[idcon] = g_malloc0(sizeof (_connect) );
	connection_is_open[idcon] = TRUE;

	connection[idcon]->socket = sock;
	connection[idcon]->host = g_strdup(host);
	connection[idcon]->port = port;
	connection[idcon]->channel = channel;
	connection[idcon]->connected = TRUE;
	*idconnect = idcon;

	connection[idcon]->event_source_id = g_io_add_watch(channel, G_IO_IN|G_IO_ERR|G_IO_HUP, (GIOFunc)client_event, GINT_TO_POINTER(idcon));

	text = g_strdup_printf("[CONNECT %d] connected on \"%s:%d\", use socket %d", idcon, host, port, sock);
	text_insert_message(text, strlen(text));
	g_free(text);

	return(TRUE);
}


/* -- close the connection */
gboolean client_disconnect(gint32 idconnect)  {

	gpointer errormsg;
	gchar *text;

	if(!connection_is_open[idconnect])   return(FALSE);
	if(!connection[idconnect]->connected)   return(FALSE);

	shutdown (connection[idconnect]->socket, 2);

	if(closesocket(connection[idconnect]->socket))  {

		errormsg = g_strdup_printf("[ERROR] Connection \"%d\" : can't close socket", idconnect);
		text_insert_message(errormsg, strlen(errormsg));
		g_free(errormsg);
		return(FALSE);
	}

	g_source_remove(connection[idconnect]->event_source_id);
	g_io_channel_unref(connection[idconnect]->channel);

	text = g_strdup_printf("[CONNECT %d] socket %d closed", idconnect, connection[idconnect]->socket);
	text_insert_message(text, strlen(text));
	g_free(text);

	if(connection[idconnect]->buffer_in)
		g_free(connection[idconnect]->buffer_in);

	g_free(connection[idconnect]->host);
	g_free(connection[idconnect]->server_version);

	g_free( connection[idconnect] );
	connection_is_open[idconnect] = FALSE;

	return(TRUE);
}


gboolean client_event(GIOChannel *channel, GIOCondition condition, gpointer data) {
	gint32 connect_id = GPOINTER_TO_INT(data);
	_connect *connect = connection[connect_id];

	if (condition == G_IO_IN) {
		GIOStatus status;
		gchar chunk[1024 +1];
		gsize bytes_read;
		GError *err = NULL;
		gchar *buffer, *beg, *cur;

		status = g_io_channel_read_chars(channel, chunk, 1024, &bytes_read, &err);
		switch (status) {
			case G_IO_STATUS_NORMAL:
				break;

			case G_IO_STATUS_AGAIN:
				return TRUE;

			case G_IO_STATUS_ERROR:
			case G_IO_STATUS_EOF:
				client_disconnect(connect_id);
				/* FIXME: we need a callback here, so the upper level protocol know the connection is closed */
				return TRUE;
		}

		chunk[bytes_read] = '\0';

		if(connect->buffer_in == NULL) {
			buffer = g_strdup(chunk);
		} else {
			buffer = g_strconcat(connect->buffer_in, chunk, NULL);
			g_free(connect->buffer_in);
			connect->buffer_in = NULL;
		}

		/* parse data */
		for(beg = cur = buffer; *cur; cur++)  {
			if( *cur == '\n') {
				*cur = '\0';
#if DEBUG
				fprintf(stdout, "\033[32mID(%.2d): GET(%.4d): [%s]\033[m\n", connect_id, (int)strlen(beg), beg);
#endif
				switch(connect->type)  {

					case CONNECT_TYPE_MONOPD_GETGAME:
						xmlparse_getgamelist_plugger(connect_id, beg);
						break;
					case CONNECT_TYPE_MONOPD_GAME:
						xmlparse_game_plugger(connect_id, beg);
						break;

					case CONNECT_TYPE_META_GETLIST:
					case CONNECT_TYPE_META_GETGAME:
						xmlparse_metaserver(connect_id, beg);
						break;
				}

				beg = cur+1;
			}
		}

		/* bufferise remaining data */
		if (beg != cur) {
			connect->buffer_in = g_strdup(beg);
		}

		g_free(buffer);
	}

	else /* if (condition == G_IO_ERR || condition == G_IO_HUP) */ {
		client_disconnect(connect_id);
		/* FIXME: we need a callback here, so the upper level protocol know the connection is closed */
		return TRUE;
	}

	return TRUE;
}


/* -- send data to server
      >> get from send_tmp buffer
      >> flush send_tmp buffer */
gboolean client_send(gint32 id, gchar *data, gint32 lenght_data)  {

	gint32 t;
	gpointer errormsg;

	if(!connection_is_open[id])  return(TRUE);
	if(!connection[id]->connected)  return(TRUE);

	t = send(connection[id]->socket, data, lenght_data, 0);

	if(t != lenght_data) {

		errormsg = g_strdup("[ERROR] Can't send data to server");
		text_insert_message(errormsg, strlen(errormsg));
		g_free(errormsg);
		return(FALSE);
	}

#if DEBUG
	fprintf(stdout, "\033[31mID(%.2d): SEND(%.4d): [%s]\033[m\n", id, lenght_data, data);
#endif

	return(TRUE);
}
