#include "gradient.h"
#include "icegradient_theme.h"
#include <math.h>


static inline void
extract_rgb_components (GdkVisual * visual, GdkColor * color)
{
  color->red =
    ((color->pixel >> visual->red_shift) << (16 - visual->red_prec)) & 0xFFFF;
  color->green =
    ((color->pixel >> visual->green_shift) << (16 - visual->green_prec)) &
    0xFFFF;
  color->blue =
    ((color->pixel >> visual->blue_shift) << (16 - visual->blue_prec)) &
    0xFFFF;
}


gboolean
indexed_visual (GtkStyle * style)
{
  GdkVisual *visual;

  visual = gdk_colormap_get_visual (style->colormap);
  return (visual->type != GDK_VISUAL_DIRECT_COLOR)
    && (visual->type != GDK_VISUAL_TRUE_COLOR);
}


Detail *
get_detail (GtkStyle * style, gchar * detail)
{
  ThemeData *data;
  Detail *result;
  static Detail default_detail =
    { -1, "theme_hardcoded_default", CLEAN, NONE, NORTH_EAST, AUTOMATIC, 7,
    5
  };
  if (!style || !style->engine_data) {
    return &default_detail;
  }

  data = style->engine_data;
  if (!data->detail_list) {
    return &default_detail;
  }

  if (!detail) {
    result = g_hash_table_lookup (data->detail_list, "default");
    if (!result) {
      result = &default_detail;
    }
    return result;
  }

  result = g_hash_table_lookup (data->detail_list, detail);
  if (!result) {
#ifdef DEBUG
    printf ("get_detail: %s not found\n", detail);
#endif
    result = g_hash_table_lookup (data->detail_list, "default");
  }
  if (!result) {
    result = &default_detail;
  }
  return result;
}


static inline GtkOrientation
extract_orientation (GtkStyle * style,
		     gchar * detail, gint width, gint height)
{
  Detail *d;

  d = get_detail (style, detail);
  switch (d->aspect) {
  case HORIZONTAL:
    return GTK_ORIENTATION_HORIZONTAL;
    break;
  case VERTICAL:
    return GTK_ORIENTATION_VERTICAL;
    break;
  case AUTOMATIC:
    if (height > width) {
      return GTK_ORIENTATION_VERTICAL;
    } else {
      return GTK_ORIENTATION_VERTICAL;
    }
    break;
  }
}


static void
draw_gradient (GtkStyle * style,
	       GdkWindow * window,
	       GtkStateType state_type,
	       GdkRectangle * area,
	       gint x,
	       gint y,
	       gint width,
	       gint height,
	       GdkColor light, GdkColor dark, GtkOrientation orientation)
{
  GdkGC *gc;
  GdkColor color;
  gint i;

  gc = gdk_gc_new (window);
  gdk_gc_set_function (gc, GDK_COPY);
  gdk_gc_set_line_attributes (gc, 1, GDK_LINE_SOLID, GDK_CAP_NOT_LAST,
			      GDK_JOIN_MITER);
  gdk_gc_set_clip_rectangle (gc, area);

  switch (orientation) {
  case GTK_ORIENTATION_HORIZONTAL:
    for (i = 0; i < height; i++) {
      color.red =
	light.red +
	(float) ((dark.red - light.red)) * ((float) i / (float) height);
      color.green =
	light.green +
	(float) ((dark.green - light.green)) * ((float) i / (float) height);
      color.blue =
	light.blue +
	(float) ((dark.blue - light.blue)) * ((float) i / (float) height);

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x, y + i, x + width, y + i);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }
    break;
  case GTK_ORIENTATION_VERTICAL:
    for (i = 0; i < width; i++) {
      color.red =
	light.red +
	(float) ((dark.red - light.red)) * ((float) i / (float) width);
      color.green =
	light.green +
	(float) ((dark.green - light.green)) * ((float) i / (float) width);
      color.blue =
	light.blue +
	(float) ((dark.blue - light.blue)) * ((float) i / (float) width);

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x + i, y, x + i, y + height);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }
    break;
  }

  gdk_gc_destroy (gc);
}


/*
 * Explanation of the formula:
 * y|
 *  |\        /        If you use linear gradients in buttons the text is difficult to read because
 *  | \      /         the color range of the background is too wide. The idea is to make the range
 *  |  \    /          shorter in the center where the text will be. An easy way to do it would be
 *  !   -__-           to set the relative distances between steps corresponding to a something like
 * -+----------|-x                            y = (x - w/2) for x in [0,w]
 *  |          w       where w is the number of steps in the gradient.
 *                     At step w you'll have the acumulated sum of all steps and thus
 *                                   int(y, 0, w) = w/12
 * At each step you would get
 *                               int(y) = x/3 - wx/2 + wx/4
 * If you want a coefficient in [0,1]
 *                      int(y) / int(y, 0, w) = 4x/w - 6x/w + 3x/w =
 *                                            = ax + bx + cx
*/
static void
draw_quadratic_gradient (GtkStyle * style,
			 GdkWindow * window,
			 GtkStateType state_type,
			 GdkRectangle * area,
			 gint x,
			 gint y,
			 gint width,
			 gint height,
			 GdkColor light,
			 GdkColor dark, GtkOrientation orientation)
{
  GdkGC *gc;
  GdkColor color;
  gint i;
  float a, b, c, power, delta;

  gc = gdk_gc_new (window);
  gdk_gc_set_function (gc, GDK_COPY);
  gdk_gc_set_line_attributes (gc, 1, GDK_LINE_SOLID, GDK_CAP_NOT_LAST,
			      GDK_JOIN_MITER);
  gdk_gc_set_clip_rectangle (gc, area);

  switch (orientation) {
  case GTK_ORIENTATION_HORIZONTAL:
    a = 4.0 / (height * height * height);
    b = -6.0 / (height * height);
    c = 3.0 / height;
    for (i = 0; i < height; i++) {
      power = i * i * i;
      delta = a * power;
      power = i * i;
      delta += b * power;
      power = i;
      delta += c * power;

      color.red = light.red + (float) ((dark.red - light.red)) * delta;
      color.green =
	light.green + (float) ((dark.green - light.green)) * delta;
      color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x, y + i, x + width, y + i);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }
    break;
  case GTK_ORIENTATION_VERTICAL:
    a = 4.0 / (width * width * width);
    b = -6.0 / (width * width);
    c = 3.0 / width;
    for (i = 0; i < width; i++) {
      power = i * i * i;
      delta = a * power;
      power = i * i;
      delta += b * power;
      power = i;
      delta += c * power;

      color.red = light.red + (float) ((dark.red - light.red)) * delta;
      color.green =
	light.green + (float) ((dark.green - light.green)) * delta;
      color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x + i, y, x + i, y + height);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }
    break;
  }

  gdk_gc_destroy (gc);
}


static void
draw_quadratic_edges (GtkStyle * style,
		      GdkWindow * window,
		      GtkStateType state_type,
		      GdkRectangle * area,
		      gchar * detail,
		      gint x,
		      gint y,
		      gint width,
		      gint height,
		      GdkColor light,
		      GdkColor dark, GtkOrientation orientation)
{
  GdkGC *gc;
  GdkColor color;
  gint i;
  float a, b, c, power, delta;
  Detail *d;

  d = get_detail (style, detail);
  if ((orientation == GTK_ORIENTATION_HORIZONTAL)
      && (height <= d->edge_size * 2)) {
    draw_quadratic_gradient (style, window, state_type, area, x, y, width,
			     height, light, dark, orientation);
    return;
  }
  if ((orientation == GTK_ORIENTATION_VERTICAL)
      && (width <= d->edge_size * 2)) {
    draw_quadratic_gradient (style, window, state_type, area, x, y, width,
			     height, light, dark, orientation);
    return;
  }

  gc = gdk_gc_new (window);
  gdk_gc_set_function (gc, GDK_COPY);
  gdk_gc_set_line_attributes (gc, 1, GDK_LINE_SOLID, GDK_CAP_NOT_LAST,
			      GDK_JOIN_MITER);
  gdk_gc_set_clip_rectangle (gc, area);

  switch (orientation) {
  case GTK_ORIENTATION_HORIZONTAL:
    a = 4.0 / (d->edge_size * 2 * d->edge_size * 2 * d->edge_size * 2);
    b = -6.0 / (d->edge_size * 2 * d->edge_size * 2);
    c = 3.0 / (d->edge_size * 2);
    for (i = 0; i < d->edge_size; i++) {
      power = i * i * i;
      delta = a * power;
      power = i * i;
      delta += b * power;
      power = i;
      delta += c * power;

      color.red = light.red + (float) ((dark.red - light.red)) * delta;
      color.green =
	light.green + (float) ((dark.green - light.green)) * delta;
      color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x, y + i, x + width, y + i);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }

    power = i * i * i;
    delta = a * power;
    power = i * i;
    delta += b * power;
    power = i;
    delta += c * power;

    color.red = light.red + (float) ((dark.red - light.red)) * delta;
    color.green = light.green + (float) ((dark.green - light.green)) * delta;
    color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

    gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
    gdk_gc_set_foreground (gc, &color);
    gdk_draw_rectangle (window, gc, 1, x, y + i, width,
			height - d->edge_size * 2);
    gdk_colormap_free_colors (style->colormap, &color, 1);

    for (; i < d->edge_size * 2; i++) {
      power = i * i * i;
      delta = a * power;
      power = i * i;
      delta += b * power;
      power = i;
      delta += c * power;

      color.red = light.red + (float) ((dark.red - light.red)) * delta;
      color.green =
	light.green + (float) ((dark.green - light.green)) * delta;
      color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x, y + i + height - d->edge_size * 2,
		     x + width, y + i + height - d->edge_size * 2);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }
    break;
  case GTK_ORIENTATION_VERTICAL:
    a = 4.0 / (d->edge_size * 2 * d->edge_size * 2 * d->edge_size * 2);
    b = -6.0 / (d->edge_size * 2 * d->edge_size * 2);
    c = 3.0 / (d->edge_size * 2);
    for (i = 0; i < d->edge_size; i++) {
      power = i * i * i;
      delta = a * power;
      power = i * i;
      delta += b * power;
      power = i;
      delta += c * power;

      color.red = light.red + (float) ((dark.red - light.red)) * delta;
      color.green =
	light.green + (float) ((dark.green - light.green)) * delta;
      color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x + i, y, x + i, y + height);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }

    power = i * i * i;
    delta = a * power;
    power = i * i;
    delta += b * power;
    power = i;
    delta += c * power;

    color.red = light.red + (float) ((dark.red - light.red)) * delta;
    color.green = light.green + (float) ((dark.green - light.green)) * delta;
    color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

    gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
    gdk_gc_set_foreground (gc, &color);
    gdk_draw_rectangle (window, gc, 1, x + i, y, width - d->edge_size * 2,
			height);
    gdk_colormap_free_colors (style->colormap, &color, 1);

    for (; i < d->edge_size * 2; i++) {
      power = i * i * i;
      delta = a * power;
      power = i * i;
      delta += b * power;
      power = i;
      delta += c * power;

      color.red = light.red + (float) ((dark.red - light.red)) * delta;
      color.green =
	light.green + (float) ((dark.green - light.green)) * delta;
      color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x + i + width - d->edge_size * 2, y,
		     x + i + height - d->edge_size * 2, y + height);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }
    break;
  }

  gdk_gc_destroy (gc);
}


static void
draw_quadratic_corners (GtkStyle * style,
			GdkWindow * window,
			GtkStateType state_type,
			GdkRectangle * area,
			gchar * detail,
			gint x,
			gint y,
			gint width,
			gint height, GdkColor light, GdkColor dark)
{
  GdkGC *gc;
  GdkColor color;
  gint i;
  float a, b, c, power, delta;
  Detail *d;

  d = get_detail (style, detail);
  if ((height <= d->corner_size * 2) || (width <= d->corner_size * 2)) {
    draw_quadratic_gradient (style, window, state_type, area, x, y, width,
			     height, light, dark, GTK_ORIENTATION_HORIZONTAL);
    return;
  }

  gc = gdk_gc_new (window);
  gdk_gc_set_function (gc, GDK_COPY);
  gdk_gc_set_line_attributes (gc, 1, GDK_LINE_SOLID, GDK_CAP_NOT_LAST,
			      GDK_JOIN_MITER);
  gdk_gc_set_clip_rectangle (gc, area);

  a = 4.0 / (d->corner_size * 2 * d->corner_size * 2 * d->corner_size * 2);
  b = -6.0 / (d->corner_size * 2 * d->corner_size * 2);
  c = 3.0 / (d->corner_size * 2);
  for (i = 0; i < d->corner_size; i++) {
    power = i * i * i;
    delta = a * power;
    power = i * i;
    delta += b * power;
    power = i;
    delta += c * power;

    color.red = light.red + (float) ((dark.red - light.red)) * delta;
    color.green = light.green + (float) ((dark.green - light.green)) * delta;
    color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

    gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
    gdk_gc_set_foreground (gc, &color);
    gdk_draw_line (window, gc, x + i, y + i, x + width - i - 1, y + i);
    gdk_draw_line (window, gc, x + i, y + i, x + i, y + height - i - 1);
    gdk_colormap_free_colors (style->colormap, &color, 1);
  }

  power = i * i * i;
  delta = a * power;
  power = i * i;
  delta += b * power;
  power = i;
  delta += c * power;

  color.red = light.red + (float) ((dark.red - light.red)) * delta;
  color.green = light.green + (float) ((dark.green - light.green)) * delta;
  color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

  gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
  gdk_gc_set_foreground (gc, &color);
  gdk_draw_rectangle (window, gc, 1, x + i, y + i, width - d->corner_size * 2,
		      height - d->corner_size * 2);
  gdk_colormap_free_colors (style->colormap, &color, 1);

  for (; i < d->corner_size * 2; i++) {
    power = i * i * i;
    delta = a * power;
    power = i * i;
    delta += b * power;
    power = i;
    delta += c * power;

    color.red = light.red + (float) ((dark.red - light.red)) * delta;
    color.green = light.green + (float) ((dark.green - light.green)) * delta;
    color.blue = light.blue + (float) ((dark.blue - light.blue)) * delta;

    gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
    gdk_gc_set_foreground (gc, &color);
    gdk_draw_line (window, gc,
		   x + i + width - d->corner_size * 2,
		   y + i + height - 1 - d->corner_size * 2,
		   x + d->corner_size * 2 - i,
		   y + i + height - 1 - d->corner_size * 2);
    gdk_draw_line (window, gc, x + i + width - 1 - d->corner_size * 2,
		   y + i + height - d->corner_size * 2,
		   x + i + width - 1 - d->corner_size * 2,
		   y + d->corner_size * 2 - i);
    gdk_colormap_free_colors (style->colormap, &color, 1);
  }

  gdk_gc_destroy (gc);
}


static void
draw_diagonal_gradient (GtkStyle * style,
			GdkWindow * window,
			GtkStateType state_type,
			GdkRectangle * area,
			gint x,
			gint y,
			gint width,
			gint height,
			GdkColor light,
			GdkColor dark, diagonal_aspect_style orientation)
{
  GdkGC *gc;
  GdkGCValues gc_values;
  GdkColor color;
  gint i;
  gint scale;
  GdkRectangle our_area;

  our_area.x = x;
  our_area.y = y;
  our_area.width = width;
  our_area.height = height;

  gc = gdk_gc_new (window);
  gdk_gc_set_function (gc, GDK_COPY);
  gdk_gc_set_line_attributes (gc, 1, GDK_LINE_SOLID, gc_values.cap_style,
			      gc_values.join_style);
  gdk_gc_set_clip_rectangle (gc, &our_area);

  switch (orientation) {
  case NORTH_EAST:
  case SOUTH_EAST:
    break;
  case NORTH_WEST:
  case SOUTH_WEST:
    color = light;
    light = dark;
    dark = color;
  }

  switch (orientation) {
  case NORTH_EAST:
  case SOUTH_WEST:
    scale = width + height - 1;
    for (i = 0; i < scale; i++) {
      color.red =
	light.red +
	(float) ((dark.red - light.red)) * ((float) i / (float) scale);
      color.green =
	light.green +
	(float) ((dark.green - light.green)) * ((float) i / (float) scale);
      color.blue =
	light.blue +
	(float) ((dark.blue - light.blue)) * ((float) i / (float) scale);

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x + i, y, x, y + i);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }
    break;
  case SOUTH_EAST:
  case NORTH_WEST:
    scale = width + height - 1;
    for (i = 0; i < scale; i++) {
      color.red =
	dark.red -
	(float) ((dark.red - light.red)) * ((float) i / (float) scale);
      color.green =
	dark.green -
	(float) ((dark.green - light.green)) * ((float) i / (float) scale);
      color.blue =
	dark.blue -
	(float) ((dark.blue - light.blue)) * ((float) i / (float) scale);

      gdk_colormap_alloc_color (style->colormap, &color, FALSE, TRUE);
      gdk_gc_set_foreground (gc, &color);
      gdk_draw_line (window, gc, x + width - 1 - i, y, x + width - 1, y + i);
      gdk_colormap_free_colors (style->colormap, &color, 1);
    }
    break;
  }

  gdk_gc_destroy (gc);
}


void
draw_generic_gradient (GtkStyle * style,
		       GdkWindow * window,
		       GtkStateType state_type,
		       GtkShadowType shadow_type,
		       GdkRectangle * area,
		       gchar * detail,
		       gint x,
		       gint y,
		       gint width,
		       gint height,
		       background_style background,
		       diagonal_aspect_style diagonal_orientation)
{
  GdkVisual *visual;
  GdkGC *gc;
  GdkGCValues gc_values;
  GdkColor light, dark, color;
  GtkOrientation orientation;

  gdk_gc_get_values (style->light_gc[state_type], &gc_values);
  light.pixel = gc_values.foreground.pixel;
  gdk_gc_get_values (style->dark_gc[state_type], &gc_values);
  dark.pixel = gc_values.foreground.pixel;

  visual = gdk_colormap_get_visual (style->colormap);
  extract_rgb_components (visual, &light);
  extract_rgb_components (visual, &dark);

  switch (shadow_type) {
  case GTK_SHADOW_IN:
  case GTK_SHADOW_ETCHED_IN:
    color = light;
    light = dark;
    dark = color;
    break;
  default:
    break;
  }

  switch (background) {
  case GRADIENT:
    orientation = extract_orientation (style, detail, width, height);
    draw_gradient (style, window, state_type, area, x, y, width, height,
		   light, dark, orientation);
    break;
  case QUADRATIC_GRADIENT:
    orientation = extract_orientation (style, detail, width, height);
    draw_quadratic_gradient (style, window, state_type, area, x, y, width,
			     height, light, dark, orientation);
    break;
  case EDGE_GRADIENT:
    orientation = extract_orientation (style, detail, width, height);
    draw_quadratic_edges (style, window, state_type, area, detail, x, y,
			  width, height, light, dark, orientation);
    break;
  case CORNER_GRADIENT:
    draw_quadratic_corners (style, window, state_type, area, detail, x, y,
			    width, height, light, dark);
    break;
  case DIAGONAL_GRADIENT:
    draw_diagonal_gradient (style, window, state_type, area, x, y, width,
			    height, light, dark, diagonal_orientation);
    break;
  default:
    break;
  }
}
