/* gtkplotarray - array plots widget for gtk+
 * Copyright 1999-2001  Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <gtk/gtk.h>
#include "gtkplot.h"
#include "gtkplot3d.h"
#include "gtkplotdata.h"
#include "gtkplotarray.h"
#include "gtkpsfont.h"


static void gtk_plot_array_class_init 	(GtkPlotArrayClass *klass);
static void gtk_plot_array_init 	(GtkPlotArray *array);
static void gtk_plot_array_destroy 	(GtkObject *object);
static void gtk_plot_array_set_arg      (GtkObject *object,
                                         GtkArg    *arg,
                                         guint      arg_id);
static void gtk_plot_array_get_arg      (GtkObject *object,
                                         GtkArg    *arg,
                                         guint      arg_id);

static void gtk_plot_array_list_class_init (GtkPlotArrayListClass *klass);
static void gtk_plot_array_list_init 	(GtkPlotArrayList *array_list);
static void gtk_plot_array_list_destroy 	(GtkObject *object);

static GtkObjectClass *array_parent_class = NULL;
static GtkObjectClass *array_list_parent_class = NULL;

enum {
  ARG_0,
  ARG_NAME,
  ARG_LABEL,
  ARG_DESCRIPTION,
  ARG_TYPE,
  ARG_SIZE,
  ARG_DATA,
  ARG_SCALE,
  ARG_REQUIRED,
  ARG_INDEPENDENT,
  ARG_OWN,
};

GtkType
gtk_plot_array_get_type (void)
{
  static GtkType array_type = 0;

  if (!array_type)
    {
      GtkTypeInfo data_info =
      {
	"GtkPlotArray",
	sizeof (GtkPlotArray),
	sizeof (GtkPlotArrayClass),
	(GtkClassInitFunc) gtk_plot_array_class_init,
	(GtkObjectInitFunc) gtk_plot_array_init,
	/* reserved 1*/ NULL,
        /* reserved 2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      array_type = gtk_type_unique (gtk_object_get_type(), &data_info);
    }
  return array_type;
}

static void
gtk_plot_array_class_init (GtkPlotArrayClass *klass)
{
  GtkObjectClass *object_class;
  GtkPlotArrayClass *array_class;

  array_parent_class = gtk_type_class (gtk_object_get_type ());

  object_class = (GtkObjectClass *) klass;
  array_class = (GtkPlotArrayClass *) klass;

  object_class->destroy = gtk_plot_array_destroy;
  object_class->set_arg = gtk_plot_array_set_arg;
  object_class->get_arg = gtk_plot_array_get_arg;

  gtk_object_add_arg_type ("GtkPlotArray::name",
                           GTK_TYPE_STRING,
                           GTK_ARG_READWRITE,
                           ARG_NAME);
  gtk_object_add_arg_type ("GtkPlotArray::label",
                           GTK_TYPE_STRING,
                           GTK_ARG_READWRITE,
                           ARG_LABEL);
  gtk_object_add_arg_type ("GtkPlotArray::description",
                           GTK_TYPE_STRING,
                           GTK_ARG_READWRITE,
                           ARG_DESCRIPTION);
  gtk_object_add_arg_type ("GtkPlotArray::type",
                           GTK_TYPE_ENUM,
                           GTK_ARG_READWRITE,
                           ARG_TYPE);
  gtk_object_add_arg_type ("GtkPlotArray::size",
                           GTK_TYPE_INT,
                           GTK_ARG_READWRITE,
                           ARG_SIZE);
  gtk_object_add_arg_type ("GtkPlotArray::own_data",
                           GTK_TYPE_BOOL,
                           GTK_ARG_READWRITE,
                           ARG_OWN);
  gtk_object_add_arg_type ("GtkPlotArray::required",
                           GTK_TYPE_BOOL,
                           GTK_ARG_READWRITE,
                           ARG_REQUIRED);
  gtk_object_add_arg_type ("GtkPlotArray::independent",
                           GTK_TYPE_BOOL,
                           GTK_ARG_READWRITE,
                           ARG_INDEPENDENT);
  gtk_object_add_arg_type ("GtkPlotArray::scale",
                           GTK_TYPE_DOUBLE,
                           GTK_ARG_READWRITE,
                           ARG_SIZE);
  gtk_object_add_arg_type ("GtkPlotArray::array_data",
                           GTK_TYPE_POINTER,
                           GTK_ARG_READWRITE,
                           ARG_DATA);

}

static void
gtk_plot_array_set_arg (GtkObject      *object,
                      GtkArg         *arg,
                      guint           arg_id)
{
  GtkPlotArray *data;
                                                                                
  data = GTK_PLOT_ARRAY (object);
                                                                                
  switch (arg_id)
    {
      case ARG_NAME:
        if(data->name) g_free(data->name);
        data->name = g_strdup(GTK_VALUE_STRING(*arg));
        break;
      case ARG_LABEL:
        if(data->label) g_free(data->label);
        data->label = g_strdup(GTK_VALUE_STRING(*arg));
        break;
      case ARG_DESCRIPTION:
        if(data->description) g_free(data->description);
        data->description = g_strdup(GTK_VALUE_STRING(*arg));
        break;
      case ARG_TYPE:
        data->type  = GTK_VALUE_ENUM(*arg);
        break;
      case ARG_SIZE:
        data->size  = GTK_VALUE_INT(*arg);
        break;
      case ARG_OWN:
        data->own_data  = GTK_VALUE_BOOL(*arg);
        break;
      case ARG_REQUIRED:
        data->required  = GTK_VALUE_BOOL(*arg);
        break;
      case ARG_INDEPENDENT:
        data->independent  = GTK_VALUE_BOOL(*arg);
        break;
      case ARG_SCALE:
        data->scale  = GTK_VALUE_DOUBLE(*arg);
        break;
      case ARG_DATA:
        data->data = *((GtkPlotArrayArg *)GTK_VALUE_POINTER(*arg));
        break;
    }
}

static void
gtk_plot_array_get_arg (GtkObject      *object,
                      GtkArg         *arg,
                      guint           arg_id)
{
  GtkPlotArray *data;

  data = GTK_PLOT_ARRAY (object);

  switch (arg_id)
    {
      case ARG_NAME:
        GTK_VALUE_STRING(*arg) = data->name;
        break;
      case ARG_LABEL:
        GTK_VALUE_STRING(*arg) = data->label;
        break;
      case ARG_DESCRIPTION:
        GTK_VALUE_STRING(*arg) = data->description;
        break;
      case ARG_TYPE:
        GTK_VALUE_ENUM(*arg) = data->type;
        break;
      case ARG_SIZE:
        GTK_VALUE_INT(*arg) = data->size;
        break;
      case ARG_OWN:
        GTK_VALUE_BOOL(*arg) = data->own_data;
        break;
      case ARG_REQUIRED:
        GTK_VALUE_BOOL(*arg) = data->required;
        break;
      case ARG_INDEPENDENT:
        GTK_VALUE_BOOL(*arg) = data->independent;
        break;
      case ARG_SCALE:
        GTK_VALUE_DOUBLE(*arg) = data->scale;
        break;
      case ARG_DATA:
        GTK_VALUE_POINTER(*arg) = &data->data;
        break;
      default:
        arg->type = GTK_TYPE_INVALID;
        break;
    }
}

static void
gtk_plot_array_init (GtkPlotArray *array)
{
  array->name = NULL;
  array->label = NULL;
  array->description = NULL;
  array->type = GTK_TYPE_DOUBLE;
  array->own_data = FALSE;
  array->required = FALSE;
  array->independent = FALSE;
  array->size = 0;
  array->scale = 1.;
}

GtkObject*
gtk_plot_array_new (const gchar *name, gpointer data, gint size, GtkType type, gboolean own_data)
{
  GtkObject *object;

  object = gtk_type_new (gtk_plot_array_get_type ());
  GTK_PLOT_ARRAY(object)->type = type;
  GTK_PLOT_ARRAY(object)->own_data = own_data;
  GTK_PLOT_ARRAY(object)->name = g_strdup(name);
  GTK_PLOT_ARRAY(object)->label = g_strdup(name);
  GTK_PLOT_ARRAY(object)->description = NULL;
  GTK_PLOT_ARRAY(object)->size = size;
  gtk_plot_array_set(GTK_PLOT_ARRAY(object), data, size, type);

  return (object);
}

static void
gtk_plot_array_destroy(GtkObject *object)
{
  GtkPlotArray *array = GTK_PLOT_ARRAY(object);
  if(array->name) g_free(array->name);
  array->name = NULL;
  if(array->label) g_free(array->label);
  array->label = NULL;
  if(array->description) g_free(array->description);
  array->description = NULL;
  if(array->own_data) gtk_plot_array_free(array);
  array->size = 0;
}

void
gtk_plot_array_set_label(GtkPlotArray *array, const gchar *label)
{
  if(array->label) g_free(array->label);
  array->label = NULL;
  if(label) array->label = g_strdup(label);
}

void
gtk_plot_array_set_description(GtkPlotArray *array, const gchar *description)
{
  if(array->description) g_free(array->description);
  array->description = NULL;
  if(description) array->description = g_strdup(description);
}

void
gtk_plot_array_set_scale(GtkPlotArray *array, gdouble scale)
{
  array->scale = scale;
}

void
gtk_plot_array_set_independent(GtkPlotArray *array, gboolean independent)
{
  array->independent = independent;
}

void
gtk_plot_array_set_required(GtkPlotArray *array, gboolean required)
{
  array->required = required;
}

void
gtk_plot_array_set(GtkPlotArray *array, gpointer data, gint size, GtkType type)
{
  if(array->own_data) gtk_plot_array_free(array);
  array->type = type;
  array->size = size;
  switch(type){
    case GTK_TYPE_DOUBLE:
      array->data.data_double = (gdouble *)data;
      break;
    case GTK_TYPE_FLOAT:
      array->data.data_float = (gfloat *)data;
      break;
    case GTK_TYPE_INT:
      array->data.data_int = (gint *)data;
      break;
    case GTK_TYPE_BOOL:
      array->data.data_bool = (gboolean *)data;
      break;
    case GTK_TYPE_STRING:
      array->data.data_string = (gchar **)data;
      break;
    default:
      break; 
  }
}

void
gtk_plot_array_free(GtkPlotArray *array)
{
  gint i = 0;

  switch(array->type){
    case GTK_TYPE_DOUBLE:
      g_free(array->data.data_double);
      array->data.data_double = NULL;
      break;
    case GTK_TYPE_FLOAT:
      g_free(array->data.data_float);
      array->data.data_float = NULL;
      break;
    case GTK_TYPE_INT:
      g_free(array->data.data_int);
      array->data.data_int = NULL;
      break;
    case GTK_TYPE_BOOL:
      g_free(array->data.data_bool);
      array->data.data_bool = NULL;
      break;
    case GTK_TYPE_STRING:
      for(i = 0; i < array->size; i++)
        if(array->data.data_string && array->data.data_string[i]) 
          g_free(array->data.data_string[i]);
      g_free(array->data.data_string);
      array->data.data_string = NULL;
      break;
    default:
      break; 
  }
}

gint
gtk_plot_array_get_size(GtkPlotArray *array)
{
  return array->size;
}

gdouble
gtk_plot_array_get_scale(GtkPlotArray *array)
{
  return array->scale;
}

gboolean
gtk_plot_array_required(GtkPlotArray *array)
{
  return array->required;
}

gboolean
gtk_plot_array_independent(GtkPlotArray *array)
{
  return array->independent;
}

GtkType
gtk_plot_array_get_data_type(GtkPlotArray *array)
{
  return array->type;
}

const gchar *
gtk_plot_array_get_name(GtkPlotArray *array)
{
  return array->name;
}

const gchar *
gtk_plot_array_get_label(GtkPlotArray *array)
{
  return array->label;
}

const gchar *
gtk_plot_array_get_description(GtkPlotArray *array)
{
  return array->description;
}

gboolean *
gtk_plot_array_get_bool(GtkPlotArray *array)
{
  if(array->type != GTK_TYPE_BOOL) return NULL;
  return array->data.data_bool;
}

gdouble *
gtk_plot_array_get_double(GtkPlotArray *array)
{
  if(array->type != GTK_TYPE_DOUBLE) return NULL;
  return array->data.data_double;
}

gfloat *
gtk_plot_array_get_float(GtkPlotArray *array)
{
  if(array->type != GTK_TYPE_FLOAT) return NULL;
  return array->data.data_float;
}

gint *
gtk_plot_array_get_int(GtkPlotArray *array)
{
  if(array->type != GTK_TYPE_INT) return NULL;
  return array->data.data_int;
}

gchar **
gtk_plot_array_get_string(GtkPlotArray *array)
{
  if(array->type != GTK_TYPE_STRING) return NULL;
  return array->data.data_string;
}

gpointer *
gtk_plot_array_get_pointer(GtkPlotArray *array)
{
  if(array->type != GTK_TYPE_POINTER) return NULL;
  return array->data.data_pointer;
}

GtkType
gtk_plot_array_list_get_type (void)
{
  static GtkType array_list_type = 0;

  if (!array_list_type)
    {
      GtkTypeInfo data_info =
      {
	"GtkPlotArrayList",
	sizeof (GtkPlotArrayList),
	sizeof (GtkPlotArrayListClass),
	(GtkClassInitFunc) gtk_plot_array_list_class_init,
	(GtkObjectInitFunc) gtk_plot_array_list_init,
	/* reserved 1*/ NULL,
        /* reserved 2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      array_list_type = gtk_type_unique (gtk_object_get_type(), &data_info);
    }
  return array_list_type;
}

static void
gtk_plot_array_list_class_init (GtkPlotArrayListClass *klass)
{
  GtkObjectClass *object_class;
  GtkPlotArrayListClass *array_list_class;

  array_list_parent_class = gtk_type_class (gtk_object_get_type ());

  object_class = (GtkObjectClass *) klass;
  array_list_class = (GtkPlotArrayListClass *) klass;

  object_class->destroy = gtk_plot_array_list_destroy;
}


static void
gtk_plot_array_list_init (GtkPlotArrayList *array_list)
{
  array_list->arrays = NULL;
}

GtkObject*
gtk_plot_array_list_new (void)
{
  GtkObject *object;

  object = gtk_type_new (gtk_plot_array_list_get_type ());

  return (object);
}

static void
gtk_plot_array_list_destroy(GtkObject *object)
{
  GtkPlotArrayList *array_list = GTK_PLOT_ARRAY_LIST(object);

  gtk_plot_array_list_clear(array_list);
}

GList *
find_array(GtkPlotArrayList *array_list, const gchar *name)
{
  GList *list;

  if(!array_list->arrays) return NULL;

  list = array_list->arrays;
  while(list){
    if(list->data && GTK_IS_PLOT_ARRAY(list->data))
      if(GTK_PLOT_ARRAY(list->data)->name && strcmp(GTK_PLOT_ARRAY(list->data)->name, name) == 0) return list;
    list = list->next;
  }
  return NULL;
}

void
gtk_plot_array_list_add(GtkPlotArrayList *array_list, GtkPlotArray *array)
{
  GList *list = NULL;
  list = find_array(array_list, array->name);
  if(list){
    gtk_object_unref(GTK_OBJECT(list->data));
    list->data = array;
    gtk_object_ref(GTK_OBJECT(array));
  } else {
    array_list->arrays = g_list_append(array_list->arrays, array);
    gtk_object_ref(GTK_OBJECT(array)); 
  }
}

void
gtk_plot_array_list_remove(GtkPlotArrayList *array_list, GtkPlotArray *array)
{
  GList *list;
  list = g_list_find(array_list->arrays, array);
  if(list){
    gtk_object_unref(GTK_OBJECT(array));
    array_list->arrays = g_list_remove_link(array_list->arrays, list);
    g_list_free_1(list); 
  }
}

GtkPlotArray *
gtk_plot_array_list_get(GtkPlotArrayList *array_list, const gchar *name)
{
  GList *list = NULL;;
  list = find_array(array_list, name);
  if(list) return GTK_PLOT_ARRAY(list->data);
  return NULL;
}

void
gtk_plot_array_list_clear(GtkPlotArrayList *array_list)
{
  GList *list;

  list = array_list->arrays;
  while(list){
    if(list->data && GTK_IS_OBJECT(list->data)) 
      gtk_object_unref(GTK_OBJECT(list->data));
    list->data = NULL;
    array_list->arrays = g_list_remove_link(array_list->arrays, list);
    g_list_free_1(list);

    list = array_list->arrays;
  }
  array_list->arrays = NULL;
}
