/*
 * manualtimer.c: Timing via key hitting
 *
 * This file is part of GTick
 *
 * Copyright (c) 1999, Alex Roberts
 * Copyright (c) 2003, Roland Stigge <stigge@antcom.de>
 *
 * GTick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GTick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GTick; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

/* GNU headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_SYS_TIME_H
  #include <sys/time.h>
#endif

/* GTK+ headers */
#include <gtk/gtk.h>

/* own headers */
#include "globals.h"
#include "util.h"
#include "manualtimer.h"

typedef struct tapping_t {
  GtkObject* adjustment;  /* location to set the resulting frequency */
  GtkWidget* label;       /* label to update */
  double frequency;       /* current frequency */

  struct timeval lasttap; /* time of last tap */
  int tapnumber;          /* number of occurred taps */
} tapping_t;

#define RESPONSE_TAP 1
#define RESPONSE_RESET 2

/*
 * creates tapping object
 * 
 * input: parent: parent window of tapping dialog
 *        adjustment: destination of new value
 * returns: new tapping object
 */
static tapping_t* tapping_new(GtkObject* adjustment) {
  tapping_t* result;

  if (!(result = malloc(sizeof(tapping_t)))) {
    fprintf(stderr, "Error allocating memory for tapping structure.\n");
    exit(1);
  }

  if (debug)
    printf("Creating tapping object at %p.\n", result);

  result->adjustment = adjustment;
  result->tapnumber = 0;
  result->frequency = 0.0;

  return result;
}

/*
 * destroys the specified tapping object
 */
static void tapping_delete(tapping_t* tapping) {
  if (debug)
    printf("Destroying tapping object at %p.\n", tapping);
  free(tapping);
}

/*
 * updates value in tapping dialog
 */
static void tapping_update_display(tapping_t* tapping) {
  char* text;
  
  if (tapping->tapnumber > 0) {
    if (asprintf(&text, "%.1f bpm", tapping->frequency * 60.0) < 0)
      text = NULL;
  } else {
    text = strdup("??? bpm");
  }
  if (text) {
    gtk_label_set_text(GTK_LABEL(tapping->label), text);
    free(text);
  } else {
    fprintf(stderr,
	    "tapping_update_value: "
	    "Error allocating buffer for output string.\n");
    exit(1);
  }
}

/*
 * callback invoked by response signal in dialog
 */
static void response_cb(GtkDialog *dialog, gint response, tapping_t* tapping) {
  struct timeval thistime = {0, 0};
  
  switch (response) {
    case RESPONSE_TAP:
      tapping->tapnumber++;
      gettimeofday(&thistime, NULL);
      if (tapping->tapnumber > 1) {
	struct timeval diffval;
	double timediff;

	timeval_subtract(&diffval, &thistime, &tapping->lasttap);
	timediff = diffval.tv_sec + (double)diffval.tv_usec / 1000000.0;
	tapping->frequency =
	  (tapping->frequency * (tapping->tapnumber - 2) + 1.0 / timediff) /
	  (tapping->tapnumber - 1);
      }
      tapping_update_display(tapping);
      tapping->lasttap = thistime;
      break;
    case RESPONSE_RESET:
      tapping->frequency = 0.0;
      tapping->tapnumber = 0;
      tapping_update_display(tapping);
      break;
    case GTK_RESPONSE_OK:
      if (tapping->frequency <
	  GTK_ADJUSTMENT(tapping->adjustment)->lower / 60.0)
	tapping->frequency = GTK_ADJUSTMENT(tapping->adjustment)->lower / 60.0;
      else if (tapping->frequency >
	  GTK_ADJUSTMENT(tapping->adjustment)->upper / 60.0)
	tapping->frequency = GTK_ADJUSTMENT(tapping->adjustment)->upper / 60.0;
      gtk_adjustment_set_value(GTK_ADJUSTMENT(tapping->adjustment),
	                       tapping->frequency * 60.0);
      tapping_delete(tapping);
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_CANCEL:
      tapping_delete(tapping);
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_DELETE_EVENT: /* ignore, window will just be closed */
      tapping_delete(tapping);
      break;
    default:
      fprintf(stderr, "Warning: Unhandled response: %d.\n", response);
  }
}

/*
 * callback invoked by menu selection
 */
GtkWidget* tapping_dialog_new(GtkWidget* parent, GtkObject* adjustment)
{
  tapping_t* tapping = tapping_new(adjustment);
  GtkWidget* window;
  GtkWidget* label;

  window = gtk_dialog_new_with_buttons(_("Manual Timer"),
      GTK_WINDOW(parent),
      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
      _("Tap"), RESPONSE_TAP,
      _("Reset"), RESPONSE_RESET,
      GTK_STOCK_OK, GTK_RESPONSE_OK,
      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
      NULL);

  label = gtk_label_new(_(
	"Here, you can approximate the desired\n"
	"speed by tapping at the \"Tap\" button.\n"
	"\n"
	"Just start tapping with the mouse or Enter.\n"
	"\n"
	"To use the calculated value, click \"OK\".\n"
	"To start again, click \"Reset\".\n"));
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), label,
		     TRUE, FALSE, 0);
  gtk_misc_set_padding(GTK_MISC(label), 10, 10);
  gtk_widget_show(label);

  tapping->label = gtk_label_new("");
  gtk_label_set_justify(GTK_LABEL(tapping->label), GTK_JUSTIFY_CENTER);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), tapping->label,
		     TRUE, FALSE, 0);
  gtk_misc_set_padding(GTK_MISC(tapping->label), 10, 10);
  gtk_widget_show(tapping->label);

  tapping_update_display(tapping);

  g_signal_connect(G_OBJECT(window),
                   "response",
		   G_CALLBACK(response_cb),
		   tapping);

  return window;
}

